#include <drive/backend/billing/ut/library/tests_helper.h>

#include <drive/backend/billing/trust/charge_logic.h>

#include <rtline/library/storage/structured.h>

Y_UNIT_TEST_SUITE(PerformanceSuite) {
    Y_UNIT_TEST(SimultaneousTasks) {
        TBillingTestEnvironment env("SQLite", TDuration::Seconds(0), 1);
        ui32 tasksCount = 20;
        const TBillingManager& manager = env.GetManager();
        {
            for (ui32 i = 0; i < tasksCount; ++i) {
                TString sessionId = DefaultSessionID + "_" + ToString(i);
                UNIT_ASSERT(env.CreateBillingTask(sessionId, DefaultUserId, EBillingType::Ticket, 0, 0, 13000));
            }
        }
        {
            auto session = manager.BuildSession(true);
            TRecordsSet payments;
            session->Exec("SELECT * FROM drive_payments", &payments);
            UNIT_ASSERT_VALUES_EQUAL(payments.GetRecords().size(), 0);
        }

        TInstant startFirst = Now();
        env.RunAllTasks();
        TInstant startSync = Now();
        env.RunAllTasks();
        INFO_LOG << "DurationStart: " << (startSync - startFirst).MilliSeconds() << Endl;
        INFO_LOG << "DurationSync: " << (Now() - startSync).MilliSeconds() << Endl;
        TVector<TBillingTask> tasks = env.GetBillingTasksByUser(DefaultUserId);
        UNIT_ASSERT(tasks.size() <= tasksCount);
        {
            auto session = manager.BuildSession(true);
            TRecordsSet payments;
            session->Exec("SELECT * FROM drive_payments", &payments);
            UNIT_ASSERT_VALUES_EQUAL(payments.GetRecords().size(), tasksCount);
        }
    }
}

Y_UNIT_TEST_SUITE(BillingInfoSuite) {
    Y_UNIT_TEST(UserInfo) {
        TBillingTestEnvironment env;
        env.LinkBonusAccount(DefaultUserId);
        const auto& billingManager = env.GetManager();
        TInstant start = TInstant::Seconds(Now().Seconds() + 1);
        {
            NJson::TJsonValue report;
            billingManager.FillUserReport(DefaultUserId, report, TDuration::Zero());
            INFO_LOG << report.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(report["bonus"].GetUInteger(), 0);
            UNIT_ASSERT_VALUES_EQUAL(report["limited_bonus"].GetArray().size(), 0);
            UNIT_ASSERT_VALUES_EQUAL(report["forced_tasks_count"].GetUInteger(), 0);
            UNIT_ASSERT_VALUES_EQUAL(report["tasks_count"].GetUInteger(), 0);
            UNIT_ASSERT_VALUES_EQUAL(report["debt"].GetUInteger(), 0);
        }

        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID + "_1", DefaultUserId, EBillingType::Ticket, 0, 0, 13000));
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 2000));
        {
            NJson::TJsonValue report;
            env.GetManager().FillUserReport(DefaultUserId, report, TDuration::Zero());
            INFO_LOG << report.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(report["bonus"].GetUInteger(), 2000);
            UNIT_ASSERT_VALUES_EQUAL(report["forced_tasks_count"].GetUInteger(), 1);
            UNIT_ASSERT_VALUES_EQUAL(report["tasks_count"].GetUInteger(), 1);
            UNIT_ASSERT_VALUES_EQUAL(report["debt"].GetUInteger(), 0);
        }

        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID + "_2", DefaultUserId, EBillingType::CarUsage, 0, 0, 30000));

        {
            NJson::TJsonValue report;
            env.GetManager().FillUserReport(DefaultUserId, report, TDuration::Zero());
            INFO_LOG << report.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(report["bonus"].GetUInteger(), 2000);
            UNIT_ASSERT_VALUES_EQUAL(report["forced_tasks_count"].GetUInteger(), 1);
            UNIT_ASSERT_VALUES_EQUAL(report["tasks_count"].GetUInteger(), 2);
            UNIT_ASSERT_VALUES_EQUAL(report["debt"].GetUInteger(), 0);
        }

        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID + "_3", DefaultUserId, EBillingType::Ticket, 0, 0, 109900));
        env.WaitForFinish(TDuration::Minutes(2), true);

        {
            NJson::TJsonValue report;
            env.GetManager().FillUserReport(DefaultUserId, report, TDuration::Zero());
            INFO_LOG << report.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(report["bonus"].GetUInteger(), 0);
            UNIT_ASSERT_VALUES_EQUAL(report["tasks_count"].GetUInteger(), 2);
            UNIT_ASSERT_VALUES_EQUAL(report["debt"].GetUInteger(), 109900);
            UNIT_ASSERT(report["forced_tasks_count"].GetUInteger() <= 1);
        }

        {
            auto baDescription = env.GetAccountsManager().GetDescriptionByName("bonus");
            UNIT_ASSERT(baDescription.Defined());
            auto descriptionJson = baDescription->SerializeToTableRecord().SerializeToJson();
            NJson::TJsonValue metaJson;
            UNIT_ASSERT(NJson::ReadJsonFastTree(descriptionJson["meta"].GetStringRobust(), &metaJson));
            metaJson["limited_policy"] = true;
            descriptionJson["meta"] = metaJson;
            NDrive::NBilling::TAccountDescriptionRecord newDescription;
            UNIT_ASSERT(newDescription.FromJson(descriptionJson, nullptr));

            auto session = env.BuildSession();
            UNIT_ASSERT(env.GetAccountsManager().UpsertAccountDescription(newDescription, DefaultUserId, session));
            UNIT_ASSERT(session.Commit());
        }

        TInstant bonusDeadline = start + TDuration::Days(1);
        TString bonusSource = "test";
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 2000));
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 2000, bonusDeadline, bonusSource));
        {
            NJson::TJsonValue report;
            billingManager.FillUserReport(DefaultUserId, report, TDuration::Zero());
            INFO_LOG << report.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(report["bonus"].GetUInteger(), 4000);
            UNIT_ASSERT_VALUES_EQUAL(report["limited_bonus"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(report["limited_bonus"][0]["deadline"], bonusDeadline.Seconds());
            UNIT_ASSERT_VALUES_EQUAL(report["limited_bonus"][0]["sum"], 2000);
            UNIT_ASSERT_VALUES_EQUAL(report["limited_bonus"][0]["source"], bonusSource);
            UNIT_ASSERT_VALUES_EQUAL(report["forced_tasks_count"].GetUInteger(), 0);
            UNIT_ASSERT_VALUES_EQUAL(report["tasks_count"].GetUInteger(), 2);
            UNIT_ASSERT_VALUES_EQUAL(report["debt"].GetUInteger(), 109900);
        }

        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID + "_4", DefaultUserId, EBillingType::CarUsage, 0, 0, 2000));

        {
            NJson::TJsonValue report;
            billingManager.FillUserReport(DefaultUserId, report, TDuration::Zero());
            INFO_LOG << report.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(report["bonus"].GetUInteger(), 4000);
            UNIT_ASSERT_VALUES_EQUAL(report["limited_bonus"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(report["limited_bonus"][0]["deadline"], bonusDeadline.Seconds());
            UNIT_ASSERT_VALUES_EQUAL(report["limited_bonus"][0]["sum"], 2000);
            UNIT_ASSERT_VALUES_EQUAL(report["limited_bonus"][0]["source"], bonusSource);
            UNIT_ASSERT_VALUES_EQUAL(report["forced_tasks_count"].GetUInteger(), 0);
            UNIT_ASSERT_VALUES_EQUAL(report["tasks_count"].GetUInteger(), 3);
            UNIT_ASSERT_VALUES_EQUAL(report["debt"].GetUInteger(), 109900);
        }
        env.WaitForFinish(TDuration::Minutes(1), true);
        {
            NJson::TJsonValue report;
            billingManager.FillUserReport(DefaultUserId, report, TDuration::Zero());
            INFO_LOG << report.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(report["bonus"].GetUInteger(), 2116);
            UNIT_ASSERT_VALUES_EQUAL(report["limited_bonus"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(report["limited_bonus"][0]["deadline"], bonusDeadline.Seconds());
            UNIT_ASSERT_VALUES_EQUAL(report["limited_bonus"][0]["sum"], 116);
            UNIT_ASSERT_VALUES_EQUAL(report["limited_bonus"][0]["source"], bonusSource);
            UNIT_ASSERT_VALUES_EQUAL(report["tasks_count"].GetUInteger(), 3);
            UNIT_ASSERT_VALUES_EQUAL(report["debt"].GetUInteger(), 109900);
        }
        UNIT_ASSERT(env.FinishTask(DefaultSessionID + "_4"));
        env.WaitForFinish(TDuration::Minutes(1), true);
        {
            NJson::TJsonValue report;
            billingManager.FillUserReport(DefaultUserId, report, TDuration::Zero());
            INFO_LOG << report.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(report["bonus"].GetUInteger(), 2000);
            UNIT_ASSERT_VALUES_EQUAL(report["limited_bonus"].GetArray().size(), 0);
            UNIT_ASSERT_VALUES_EQUAL(report["tasks_count"].GetUInteger(), 2);
            UNIT_ASSERT_VALUES_EQUAL(report["debt"].GetUInteger(), 109900);
        }

        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 2000, bonusDeadline, bonusSource));
        {
            NJson::TJsonValue report;
            billingManager.FillUserReport(DefaultUserId, report, TDuration::Zero());
            INFO_LOG << report.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(report["bonus"].GetUInteger(), 4000);
            UNIT_ASSERT_VALUES_EQUAL(report["limited_bonus"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(report["limited_bonus"][0]["deadline"], bonusDeadline.Seconds());
            UNIT_ASSERT_VALUES_EQUAL(report["limited_bonus"][0]["sum"], 2000);
            UNIT_ASSERT_VALUES_EQUAL(report["limited_bonus"][0]["source"], bonusSource);
            UNIT_ASSERT_VALUES_EQUAL(report["tasks_count"].GetUInteger(), 2);
            UNIT_ASSERT_VALUES_EQUAL(report["debt"].GetUInteger(), 109900);
        }
        TInstantGuard ig(bonusDeadline + TDuration::Minutes(1));
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID + "_5", DefaultUserId, EBillingType::CarUsage, 0, 0, 500));
        {
            NJson::TJsonValue report;
            billingManager.FillUserReport(DefaultUserId, report, TDuration::Zero());
            INFO_LOG << report.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(report["bonus"].GetUInteger(), 4000);
            UNIT_ASSERT_VALUES_EQUAL(report["limited_bonus"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(report["tasks_count"].GetUInteger(), 3);
            UNIT_ASSERT_VALUES_EQUAL(report["debt"].GetUInteger(), 109900);
        }
        UNIT_ASSERT(env.FinishTask(DefaultSessionID + "_5"));
        env.WaitForFinish(TDuration::Minutes(1), true);
        {
            NJson::TJsonValue report;
            billingManager.FillUserReport(DefaultUserId, report, TDuration::Zero());
            INFO_LOG << report.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(report["bonus"].GetUInteger(), 3500);
            UNIT_ASSERT_VALUES_EQUAL(report["limited_bonus"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(report["limited_bonus"][0]["sum"], 1500);
            UNIT_ASSERT_VALUES_EQUAL(report["tasks_count"].GetUInteger(), 2);
            UNIT_ASSERT_VALUES_EQUAL(report["debt"].GetUInteger(), 109900);
        }
        {
            auto session = env.BuildSession();
            auto bonusAccount = env.GetAccount(DefaultUserId, "bonus");
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->Refresh(DefaultUserId, ModelingNow(), session), ERefreshStatus::Skip);
            UNIT_ASSERT(session.Commit());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        }
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID + "_6", DefaultUserId, EBillingType::CarUsage, 0, 0, 3000));
        {
            NJson::TJsonValue report;
            billingManager.FillUserReport(DefaultUserId, report, TDuration::Zero());
            INFO_LOG << report.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(report["bonus"].GetUInteger(), 2000);
            UNIT_ASSERT_VALUES_EQUAL(report["limited_bonus"].GetArray().size(), 0);
            UNIT_ASSERT_VALUES_EQUAL(report["tasks_count"].GetUInteger(), 3);
            UNIT_ASSERT_VALUES_EQUAL(report["debt"].GetUInteger(), 109900);
        }
        UNIT_ASSERT(env.FinishTask(DefaultSessionID + "_6"));
        env.WaitForFinish(TDuration::Minutes(1), true);
        ig.Set(bonusDeadline + TDuration::Minutes(2));
        {
            NJson::TJsonValue report;
            billingManager.FillUserReport(DefaultUserId, report, TDuration::Zero());
            INFO_LOG << report.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(report["bonus"].GetUInteger(), 0);
            UNIT_ASSERT_VALUES_EQUAL(report["limited_bonus"].GetArray().size(), 0);
            UNIT_ASSERT_VALUES_EQUAL(report["tasks_count"].GetUInteger(), 3);
            UNIT_ASSERT_VALUES_EQUAL(report["debt"].GetUInteger(), 109900);
        }
        {
            auto session = billingManager.BuildSession(false);
            UNIT_ASSERT(billingManager.GetActiveTasksManager().BoostUserSessions(DefaultUserId, DefaultUserId, TBillingTask::TExecContext(), session));
            UNIT_ASSERT(session.Commit());
        }
        {
            NJson::TJsonValue report;
            env.GetManager().FillUserReport(DefaultUserId, report, TDuration::Zero());
            INFO_LOG << report.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(report["forced_tasks_count"].GetUInteger(), 3);
        }
    }
}


Y_UNIT_TEST_SUITE(BillingSuite) {
    Y_UNIT_TEST(PaymentsHistory) {
        TBillingTestEnvironment env("SQLite", TDuration::Seconds(1), 1);
        env.LinkBonusAccount(DefaultUserId);
        env.AddBonuses(DefaultUserId, 100000);
        {
            UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID + "_1", DefaultUserId, EBillingType::CarUsage, 0, 0, 10000));
            UNIT_ASSERT(env.FinishTask(DefaultSessionID + "_1"));
            env.WaitForFinish(TDuration::Seconds(30), true);
            UNIT_ASSERT_VALUES_EQUAL(env.GetTasks().size(), 0);

            UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID + "_2", DefaultUserId, EBillingType::CarUsage, 0, 0, 0));
            UNIT_ASSERT(env.FinishTask(DefaultSessionID + "_2"));
            env.WaitForFinish(TDuration::Seconds(30), true);
            UNIT_ASSERT_VALUES_EQUAL(env.GetTasks().size(), 0);

            UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID + "_3", DefaultUserId, EBillingType::CarUsage, 0, 0, 20000));
            UNIT_ASSERT(env.FinishTask(DefaultSessionID + "_3"));
            env.WaitForFinish(TDuration::Seconds(30), true);
            UNIT_ASSERT_VALUES_EQUAL(env.GetTasks().size(), 0);
        }

        {
            auto history = env.GetPaymentsHistory(DefaultUserId, false);
            UNIT_ASSERT_VALUES_EQUAL(history.size(), 3);
            UNIT_ASSERT_VALUES_EQUAL(history[0].GetBillingTask().GetId(), DefaultSessionID + "_1");
            UNIT_ASSERT_VALUES_EQUAL(history[1].GetBillingTask().GetId(), DefaultSessionID + "_2");
            UNIT_ASSERT_VALUES_EQUAL(history[2].GetBillingTask().GetId(), DefaultSessionID + "_3");
        }
        {
            auto history = env.GetPaymentsHistory(DefaultUserId, true);
            UNIT_ASSERT_VALUES_EQUAL(history.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(history[0].GetBillingTask().GetId(), DefaultSessionID + "_1");
            UNIT_ASSERT_VALUES_EQUAL(history[1].GetBillingTask().GetId(), DefaultSessionID + "_3");
            UNIT_ASSERT_VALUES_EQUAL(history[1].GetSnapshot().GetHeldSum(), 20000);
        }
    }

    Y_UNIT_TEST(EmptyOffer) {
        TBillingTestEnvironment env("SQLite", TDuration::Seconds(1), 1);
        env.LinkBonusAccount(DefaultUserId);
        env.LinkWallet(DefaultUserId, "simple_limited");
        env.AddBonuses(DefaultUserId, 10000);
        {
            auto session = env.BuildSession(false);
            auto offer = MakeAtomicShared<TFakeOffer>((ui32)0);
            offer->SetOfferId(DefaultSessionID).SetPaymentDiscretization(500);
            TVector<TString> accounts = { "simple_limited", "bonus" };
            offer->SetChargableAccounts(accounts);
            UNIT_ASSERT(env.GetManager().CreateBillingTask(DefaultUserId, offer, session));
            UNIT_ASSERT(session.Commit());
        }
        TVector<TBillingTask> tasks = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasks.size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(tasks.front().GetChargableAccounts().size(), 3);
        UNIT_ASSERT(tasks.front().GetChargableAccounts().contains("card"));
    }

    Y_UNIT_TEST(ProduceBillingTask) {
        TBillingTestEnvironment env("SQLite", TDuration::Seconds(1), 1);
        env.LinkBonusAccount(DefaultUserId);
        env.LinkWallet(DefaultUserId, "simple_limited");
        env.AddBonuses(DefaultUserId, 10000);
        {
            auto session = env.BuildSession(false);
            auto offer = MakeAtomicShared<TFakeOffer>((ui32)0);
            offer->SetOfferId(DefaultSessionID).SetPaymentDiscretization(500);
            TVector<TString> accounts = { "simple_limited", "bonus" };
            offer->SetChargableAccounts(accounts);
            UNIT_ASSERT(env.GetManager().CreateBillingTask(DefaultUserId, offer, session));
            UNIT_ASSERT(session.Commit());
        }

        auto walletAcc = env.GetAccount(DefaultUserId, "simple_limited");
        UNIT_ASSERT(walletAcc->GetBalance() > 0);
        env.AddMoney(DefaultSessionID, 2 * walletAcc->GetBalance() + 10000);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));

        {
            TVector<TBillingTask> tasks = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasks.size(), 1);
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto paymentsData = TPaymentsData::BuildPaymentsData(tasks.front(), std::move(snapshot));
            UNIT_ASSERT(paymentsData);
            TLazySession session(env.GetDatabase());
            env.GetManager().ProduceTask(*paymentsData, { env.GetAccount(DefaultUserId, "simple_limited"), env.GetAccount(DefaultUserId, "bonus") }, session, true);
        }
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 2);
        }
    }

    Y_UNIT_TEST(RunMultipleTasks) {
        TBillingTestEnvironment env;
        env.LinkBonusAccount(DefaultUserId);
        env.AddBonuses(DefaultUserId, 100000);
        ui32 tasksCount = 3;
        {
            for (ui32 i = 0; i < tasksCount; ++i) {
                TString sessionId = DefaultSessionID + "_" + ToString(i);
                UNIT_ASSERT(env.CreateBillingTask(sessionId, DefaultUserId, EBillingType::CarUsage, 0, 0, 13000));
                UNIT_ASSERT(env.FinishTask(sessionId));
            }
        }
        TVector<TBillingTask> tasksBefore = env.GetBillingTasksByUser(DefaultUserId);
        UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 3);
        env.WaitForFinish(TDuration::Seconds(30), true);
        TVector<TBillingTask> tasksAfter = env.GetBillingTasksByUser(DefaultUserId);
        UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 0);
    }

    Y_UNIT_TEST(RunMultipleTasks1) {
        TBillingTestEnvironment env("SQLite", TDuration::Seconds(20));
        ui32 tasksCount = 50;
        {
            for (ui32 i = 0; i < tasksCount; ++i) {
                TString sessionId = DefaultSessionID + "_" + ToString(i);
                UNIT_ASSERT(env.CreateBillingTask(sessionId, UserWithoutCards, EBillingType::Ticket, 0, 0, 13000));
            }
        }
        TVector<TBillingTask> tasksBefore = env.GetBillingTasksByUser(UserWithoutCards);
        TMap<TString, TInstant> taskUpdates;
        UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), tasksCount);
        for (auto&& task : tasksBefore) {
            taskUpdates[task.GetId()] = task.GetLastUpdate();
            UNIT_ASSERT_VALUES_EQUAL(task.GetTaskStatus(), EPaymentStatus::NotStarted);
        }
        Sleep(TDuration::Seconds(1));
        env.WaitForFinish(TDuration::Seconds(30), true);
        TVector<TBillingTask> tasksAfter = env.GetBillingTasksByUser(UserWithoutCards);
        UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), tasksCount);
        for (auto&& task : tasksAfter) {
            INFO_LOG << taskUpdates[task.GetId()] << "/" << task.GetLastUpdate();
            UNIT_ASSERT_VALUES_EQUAL(task.GetTaskStatus(), EPaymentStatus::NoCards);
            UNIT_ASSERT(taskUpdates[task.GetId()] < task.GetLastUpdate());
        }
    }

    Y_UNIT_TEST(SimpleCoinsPay) {
        TBillingTestEnvironment env;
        env.LinkCoinsAccount(DefaultUserId);
        UNIT_ASSERT(env.AddCoins(DefaultUserId, 2000));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("billing_account_history");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("billing_user_accounts");
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 0, 500));
        env.AddMoney(DefaultSessionID, 1000);
        UNIT_ASSERT_VALUES_EQUAL(env.GetCoins(DefaultUserId), 2000);
        TBillingManager& billingManager = env.GetManager();
        {
            TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
            env.RunAllTasks();
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto payments = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(payments.size(), 1);
        }

        UNIT_ASSERT_VALUES_EQUAL(env.GetCoins(DefaultUserId), 1500);
        env.AddMoney(DefaultSessionID, 2000);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetSum(), 500);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetPaymentType(), NDrive::NBilling::EAccount::Coins);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetSum(), 1500);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetPaymentType(), NDrive::NBilling::EAccount::Coins);
        }
        TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
        env.WaitForFinish(TDuration::Seconds(30), true);
        TVector<TBillingTask> tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 0);
        UNIT_ASSERT_VALUES_EQUAL(env.GetCoins(DefaultUserId), 0);
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 2);
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        UNIT_ASSERT_VALUES_EQUAL(billingManager.GetDefaultAccountName(DefaultUserId), "coins");
    }

    Y_UNIT_TEST(SimpleBonusPay) {
        TBillingTestEnvironment env;
        env.LinkBonusAccount(DefaultUserId);
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 2000));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("billing_account_history");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("billing_user_accounts");
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 0, 500));
        env.AddMoney(DefaultSessionID, 1000);
        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 2000);
        TBillingManager& billingManager = env.GetManager();
        {
            TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
            env.RunAllTasks();
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto payments = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(payments.size(), 1);
        }

        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 1500);
        env.AddMoney(DefaultSessionID, 2000);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetSum(), 500);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetPaymentType(), NDrive::NBilling::EAccount::Bonus);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetSum(), 1500);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetPaymentType(), NDrive::NBilling::EAccount::Bonus);
        }
        TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
        env.WaitForFinish(TDuration::Seconds(30), true);
        TVector<TBillingTask> tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 0);
        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 0);
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 2);
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        UNIT_ASSERT_VALUES_EQUAL(billingManager.GetDefaultAccountName(DefaultUserId), "card");
    }

    Y_UNIT_TEST(SimpleLimitedBonusPay) {
        TBillingTestEnvironment env;
        env.LinkBonusAccount(DefaultUserId, "limited_bonuses");
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 1000));
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 1000, Now() + TDuration::Hours(1), "test"));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("billing_account_history");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("billing_user_accounts");
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 0, 500, 0));
        env.AddMoney(DefaultSessionID, 1000);
        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 2000);
        UNIT_ASSERT_VALUES_EQUAL(env.GetLimitedBonuses(DefaultUserId).size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(env.GetLimitedBonuses(DefaultUserId).begin()->GetBalance(), 1000);
        TBillingManager& billingManager = env.GetManager();
        {
            TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
            env.RunAllTasks();
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto payments = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(payments.size(), 1);
        }

        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 1500);
        UNIT_ASSERT_VALUES_EQUAL(env.GetLimitedBonuses(DefaultUserId).size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(env.GetLimitedBonuses(DefaultUserId).begin()->GetBalance(), 500);
        env.AddMoney(DefaultSessionID, 2000);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetSum(), 500);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetPaymentType(), NDrive::NBilling::EAccount::Bonus);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetSum(), 1500);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetPaymentType(), NDrive::NBilling::EAccount::Bonus);
        }
        TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
        env.WaitForFinish(TDuration::Seconds(30), true);
        TVector<TBillingTask> tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 0);
        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 0);
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 2);
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        UNIT_ASSERT_VALUES_EQUAL(billingManager.GetDefaultAccountName(DefaultUserId), "card");
    }

    Y_UNIT_TEST(SimpleWalletPay) {
        TBillingTestEnvironment env;
        env.LinkWallet(DefaultUserId, "simple_limited");

        {
            auto session = env.BuildSession(false);
            auto offer = MakeAtomicShared<TFakeOffer>((ui32)0);
            offer->SetOfferId(DefaultSessionID).SetPaymentDiscretization(500);
            TVector<TString> accounts = { "simple_limited" };
            offer->SetChargableAccounts(accounts);
            UNIT_ASSERT(env.GetManager().CreateBillingTask(DefaultUserId, offer, session));
            UNIT_ASSERT(session.Commit());
        }
        env.AddMoney(DefaultSessionID, 1000);
        TBillingManager& billingManager = env.GetManager();
        {
            TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
            env.RunAllTasks();
            UNIT_ASSERT_VALUES_EQUAL(env.GetActualSnapshot(DefaultSessionID).GetPayments().size(), 1);
        }

        env.AddMoney(DefaultSessionID, 2000);
        TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        env.RunAllTasks();
        TVector<TBillingTask> tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 0);
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetTimeline().front().GetSum(), 500);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetTimeline().front().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetTimeline().front().GetPaymentType(), NDrive::NBilling::EAccount::Wallet);
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        auto session = billingManager.BuildSession(true);
        UNIT_ASSERT_VALUES_UNEQUAL(billingManager.GetLastUsedAccount(DefaultUserId, session), nullptr);
        UNIT_ASSERT_VALUES_EQUAL(billingManager.GetLastUsedAccount(DefaultUserId, session)->GetUniqueName(), "simple_limited");
    }

    Y_UNIT_TEST(SimpleBonusPayWithMinimalSum) {
        TBillingTestEnvironment env;
        env.LinkBonusAccount(DefaultUserId);
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 2000));
        env.SetSettings("billing.card_pay_min_sum", "100");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("billing_account_history");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("billing_user_accounts");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_settings_history");
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 0, 500));
        env.AddMoney(DefaultSessionID, 1000);
        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 2000);
        TBillingManager& billingManager = env.GetManager();
        {
            TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
            env.RunAllTasks();
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetSum(), 500);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetPaymentType(), NDrive::NBilling::EAccount::Bonus);
        }

        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 1500);
        env.AddMoney(DefaultSessionID, 2000);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(timeline[0].GetSum(), 500);
            UNIT_ASSERT_VALUES_EQUAL(timeline[0].GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(timeline[0].GetPaymentType(), NDrive::NBilling::EAccount::Bonus);
            UNIT_ASSERT_VALUES_EQUAL(timeline[1].GetSum(), 1400);
            UNIT_ASSERT_VALUES_EQUAL(timeline[1].GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(timeline[1].GetPaymentType(), NDrive::NBilling::EAccount::Bonus);
        }
        env.RunAllTasks();
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 3);
            UNIT_ASSERT_VALUES_EQUAL(timeline[0].GetSum(), 500);
            UNIT_ASSERT_VALUES_EQUAL(timeline[0].GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(timeline[0].GetPaymentType(), NDrive::NBilling::EAccount::Bonus);
            UNIT_ASSERT_VALUES_EQUAL(timeline[1].GetSum(), 1400);
            UNIT_ASSERT_VALUES_EQUAL(timeline[1].GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(timeline[1].GetPaymentType(), NDrive::NBilling::EAccount::Bonus);
            UNIT_ASSERT_VALUES_EQUAL(timeline[2].GetSum(), 100);
            UNIT_ASSERT_VALUES_EQUAL(timeline[2].GetPaymentType(), NDrive::NBilling::EAccount::Trust);
        }
        TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
        env.WaitForFinish(TDuration::Seconds(30), true);
        TVector<TBillingTask> tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 0);
        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 100);
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 3);
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        UNIT_ASSERT_VALUES_EQUAL(billingManager.GetDefaultAccountName(DefaultUserId), "card");
    }

    Y_UNIT_TEST(DepositFullBonusWithMinimalSum) {
        TBillingTestEnvironment env;
        env.LinkBonusAccount(DefaultUserId);
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 2000));
        env.SetSettings("billing.card_pay_min_sum", "100");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("billing_account_history");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("billing_user_accounts");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_settings_history");
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 1000, 500, 0));
        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 2000);
        {
            TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
            env.RunAllTasks();
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(timeline[0].GetSum(), 900);
            UNIT_ASSERT_VALUES_EQUAL(timeline[0].GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(timeline[0].GetPaymentType(), NDrive::NBilling::EAccount::Bonus);
            UNIT_ASSERT_VALUES_EQUAL(timeline[1].GetSum(), 100);
            UNIT_ASSERT_VALUES_EQUAL(timeline[1].GetPaymentType(), NDrive::NBilling::EAccount::Trust);
        }

        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 1100);
        env.AddMoney(DefaultSessionID, 800);
        env.RunAllTasks();
        {
            TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
            env.RunAllTasks();
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(timeline[0].GetCleared(), 900);
            UNIT_ASSERT_VALUES_EQUAL(timeline[0].GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(timeline[0].GetPaymentType(), NDrive::NBilling::EAccount::Bonus);
            UNIT_ASSERT_VALUES_EQUAL(timeline[1].GetSum(), 100);
            UNIT_ASSERT_VALUES_EQUAL(timeline[1].GetPaymentType(), NDrive::NBilling::EAccount::Trust);
            UNIT_ASSERT_VALUES_EQUAL(timeline[1].GetStatus(), NDrive::NTrustClient::EPaymentStatus::Authorized);
        }
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        env.WaitForFinish(TDuration::Seconds(30));
        env.RunClearing();
        env.WaitForFinish(TDuration::Seconds(30), true);
        TVector<TBillingTask> tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 0);
        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 1300);
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetBonusSum(), 700);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetCardSum(), 100);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetHeldSum(), 800);
        }
    }

    Y_UNIT_TEST(DepositPartBonusWithMinimalSum) {
        TBillingTestEnvironment env;
        env.LinkBonusAccount(DefaultUserId);
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 2000));
        env.SetSettings("billing.card_pay_min_sum", "100");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("billing_account_history");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("billing_user_accounts");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_settings_history");
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 3000, 500, 0));
        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 2000);
        {
            TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
            env.RunAllTasks();
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(timeline[0].GetSum(), 2000);
            UNIT_ASSERT_VALUES_EQUAL(timeline[0].GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(timeline[0].GetPaymentType(), NDrive::NBilling::EAccount::Bonus);
            UNIT_ASSERT_VALUES_EQUAL(timeline[1].GetSum(), 1000);
            UNIT_ASSERT_VALUES_EQUAL(timeline[1].GetPaymentType(), NDrive::NBilling::EAccount::Trust);
        }

        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 0);
        env.AddMoney(DefaultSessionID, 800);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        env.WaitForFinish(TDuration::Seconds(30));
        env.RunClearing();
        env.WaitForFinish(TDuration::Seconds(30), true);
        TVector<TBillingTask> tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 0);
        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 1300);
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 3);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetBonusSum(), 700);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetCardSum(), 100);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetHeldSum(), 800);
        }
    }
    Y_UNIT_TEST(DepositPartBonusWithMinimalSum1rub) {
        TBillingTestEnvironment env;
        env.LinkBonusAccount(DefaultUserId);
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 2000));
        env.SetSettings("billing.card_pay_min_sum", "100");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("billing_account_history");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("billing_user_accounts");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_settings_history");
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 2100, 500, 0));
        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 2000);
        {
            TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
            env.RunAllTasks();
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(timeline[0].GetSum(), 2000);
            UNIT_ASSERT_VALUES_EQUAL(timeline[0].GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(timeline[0].GetPaymentType(), NDrive::NBilling::EAccount::Bonus);
            UNIT_ASSERT_VALUES_EQUAL(timeline[1].GetSum(), 100);
            UNIT_ASSERT_VALUES_EQUAL(timeline[1].GetPaymentType(), NDrive::NBilling::EAccount::Trust);
        }

        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 0);
        env.AddMoney(DefaultSessionID, 800);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        env.WaitForFinish(TDuration::Seconds(30));
        env.RunClearing();
        env.WaitForFinish(TDuration::Seconds(30), true);
        TVector<TBillingTask> tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 0);
        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 1300);
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetBonusSum(), 700);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetCardSum(), 100);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetHeldSum(), 800);
        }
    }

    Y_UNIT_TEST(SimpleCoinsPayWithMinimalSum) {
        TBillingTestEnvironment env;
        env.LinkCoinsAccount(DefaultUserId);
        UNIT_ASSERT(env.AddCoins(DefaultUserId, 2000));
        env.SetSettings("billing.card_pay_min_sum", "100");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("billing_account_history");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("billing_user_accounts");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_settings_history");
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 0, 500));
        env.AddMoney(DefaultSessionID, 1000);
        UNIT_ASSERT_VALUES_EQUAL(env.GetCoins(DefaultUserId), 2000);
        TBillingManager& billingManager = env.GetManager();
        {
            TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
            env.RunAllTasks();
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetSum(), 500);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetPaymentType(), NDrive::NBilling::EAccount::Coins);
        }

        UNIT_ASSERT_VALUES_EQUAL(env.GetCoins(DefaultUserId), 1500);
        env.AddMoney(DefaultSessionID, 2000);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(timeline[0].GetSum(), 500);
            UNIT_ASSERT_VALUES_EQUAL(timeline[0].GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(timeline[0].GetPaymentType(), NDrive::NBilling::EAccount::Coins);
            UNIT_ASSERT_VALUES_EQUAL(timeline[1].GetSum(), 1400);
            UNIT_ASSERT_VALUES_EQUAL(timeline[1].GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(timeline[1].GetPaymentType(), NDrive::NBilling::EAccount::Coins);
        }
        env.RunAllTasks();
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 3);
            UNIT_ASSERT_VALUES_EQUAL(timeline[0].GetSum(), 500);
            UNIT_ASSERT_VALUES_EQUAL(timeline[0].GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(timeline[0].GetPaymentType(), NDrive::NBilling::EAccount::Coins);
            UNIT_ASSERT_VALUES_EQUAL(timeline[1].GetSum(), 1400);
            UNIT_ASSERT_VALUES_EQUAL(timeline[1].GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(timeline[1].GetPaymentType(), NDrive::NBilling::EAccount::Coins);
            UNIT_ASSERT_VALUES_EQUAL(timeline[2].GetSum(), 100);
            UNIT_ASSERT_VALUES_EQUAL(timeline[2].GetPaymentType(), NDrive::NBilling::EAccount::Trust);
        }
        TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
        env.WaitForFinish(TDuration::Seconds(30), true);
        TVector<TBillingTask> tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 0);
        UNIT_ASSERT_VALUES_EQUAL(env.GetCoins(DefaultUserId), 100);
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 3);
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        UNIT_ASSERT_VALUES_EQUAL(billingManager.GetDefaultAccountName(DefaultUserId), "card");
    }

    Y_UNIT_TEST(WalletPriority) {
        TBillingTestEnvironment env;
        auto simpleId = env.LinkWallet(DefaultUserId, "simple_limited");
        auto priorityId = env.LinkWallet(DefaultUserId, "priority_limited");

        {
            auto session = env.BuildSession(false);
            auto offer = MakeAtomicShared<TFakeOffer>((ui32)0);
            offer->SetOfferId(DefaultSessionID).SetPaymentDiscretization(500);
            TVector<TString> accounts = { "simple_limited", "priority_limited" };
            offer->SetChargableAccounts(accounts);
            UNIT_ASSERT(env.GetManager().CreateBillingTask(DefaultUserId, offer, session));
            UNIT_ASSERT(session.Commit());
        }
        env.AddMoney(DefaultSessionID, 1.5 * WalletHardLimit);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        env.RunAllTasks();
        TBillingManager& billingManager = env.GetManager();
        TVector<TBillingTask> tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 0);
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetTimeline().front().GetAccountId(), priorityId);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetTimeline().front().GetPaymentType(), NDrive::NBilling::EAccount::Wallet);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetTimeline().back().GetAccountId(), simpleId);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetTimeline().back().GetPaymentType(), NDrive::NBilling::EAccount::Wallet);
        }

        auto session = billingManager.BuildSession(true);
        auto lastUsed = billingManager.GetLastUsedAccount(DefaultUserId, session);
        UNIT_ASSERT_VALUES_UNEQUAL(lastUsed, nullptr);
        UNIT_ASSERT_VALUES_EQUAL(lastUsed->GetUniqueName(), "simple_limited");
    }


    Y_UNIT_TEST(SimpleCache) {
        TBillingTestEnvironment env;
        env.LinkBonusAccount(DefaultUserId);
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 2000));
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 0, 500));
        env.AddMoney(DefaultSessionID, 1000);
        env.WaitForFinish(TDuration::Seconds(1));
        env.AddMoney(DefaultSessionID, 2000);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        env.WaitForFinish(TDuration::Seconds(1));

        auto report = env.GetSessionReport(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(report->GetBill(), 2000);
    }

    Y_UNIT_TEST(Debt) {
        TBillingTestEnvironment env;
        env.LinkBonusAccount(UserWithoutCards);
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, UserWithoutCards, EBillingType::CarUsage, 0, 500));
        env.AddMoney(DefaultSessionID, 2000);
        UNIT_ASSERT_VALUES_EQUAL(env.GetDebt(UserWithoutCards), 0);
        env.WaitForFinish(TDuration::Seconds(5));
        UNIT_ASSERT_VALUES_EQUAL(env.GetDebt(UserWithoutCards), 2000);
        UNIT_ASSERT(env.AddBonuses(UserWithoutCards, 1000));
        env.WaitForFinish(TDuration::Seconds(5));
        UNIT_ASSERT_VALUES_EQUAL(env.GetDebt(UserWithoutCards), 2000);
    }

    Y_UNIT_TEST(DebtLimited) {
        TBillingTestEnvironment env;
        env.LinkBonusAccount(UserWithoutCards, "limited_bonuses");
        TInstant start = TInstant::Seconds(Now().Seconds()) + TDuration::Seconds(1);
        TInstant bonusDeadLine = start + TDuration::Hours(1);

        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, UserWithoutCards, EBillingType::CarUsage, 0, 500));
        env.AddMoney(DefaultSessionID, 2000);
        UNIT_ASSERT_VALUES_EQUAL(env.GetDebt(UserWithoutCards), 0);
        env.WaitForFinish(TDuration::Seconds(5));
        UNIT_ASSERT_VALUES_EQUAL(env.GetDebt(UserWithoutCards), 2000);
        UNIT_ASSERT(env.AddBonuses(UserWithoutCards, 1000, bonusDeadLine));
        env.WaitForFinish(TDuration::Seconds(5));
        UNIT_ASSERT_VALUES_EQUAL(env.GetDebt(UserWithoutCards), 2000);
    }

    Y_UNIT_TEST(DebtLimited1) {
        TBillingTestEnvironment env;
        env.LinkBonusAccount(UserWithoutCards, "limited_bonuses");
        TInstant start = TInstant::Seconds(Now().Seconds()) + TDuration::Seconds(1);
        TInstant bonusDeadLine = start + TDuration::Hours(1);
        UNIT_ASSERT(env.AddBonuses(UserWithoutCards, 1000, bonusDeadLine));
        TInstantGuard ig(bonusDeadLine + TDuration::Minutes(1));
        {
            auto session = env.BuildSession();
            auto bonusAccount = env.GetAccount(UserWithoutCards, "limited_bonuses");
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->Refresh(UserWithoutCards, ModelingNow(), session), ERefreshStatus::Ok);
            UNIT_ASSERT(session.Commit());
        }
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, UserWithoutCards, EBillingType::CarUsage, 0, 500));
        env.AddMoney(DefaultSessionID, 2000);
        UNIT_ASSERT_VALUES_EQUAL(env.GetDebt(UserWithoutCards), 0);
        env.WaitForFinish(TDuration::Seconds(5));
        UNIT_ASSERT_VALUES_EQUAL(env.GetDebt(UserWithoutCards), 2000);
    }

    Y_UNIT_TEST(CachedPaymentUpdate) {
        TCachedPayments snapshot;
        {
            TPaymentTask payment;
            TInstant ts = Now();
            NStorage::TTableRecord record;
            record.Set("sum", 500);
            record.Set("id", 1);
            record.Set("cleared", 500);
            record.Set("billing_type", "car_usage");
            record.Set("payment_type", ToString(NDrive::NBilling::EAccount::Bonus));
            record.Set("last_update_ts", ts);
            record.Set("created_at_ts", ts);
            record.Set("session_id", DefaultSessionID);
            record.Set("status", "cleared");
            record.Set("pay_method", ToString(NDrive::NBilling::EAccount::Bonus));
            record.Set("payment_id", "b_" + CreateGuidAsString());
            record.Set("meta", "");
            UNIT_ASSERT(payment.DeserializeFromTableRecord(record, nullptr));
            UNIT_ASSERT(snapshot.Update(payment));
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetHeldSum(), 500);
            UNIT_ASSERT(snapshot.Update(payment));
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetHeldSum(), 500);
        }
    }

    Y_UNIT_TEST(TaskBoosting) {
        TBillingTestEnvironment env("SQLite", TDuration::Minutes(5), 1);
        TBillingManager& billingManager = env.GetManager();
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, UserWithoutCards, EBillingType::Ticket, 0, 0, 1000));
        TInstant now = Now() - TDuration::Hours(10);
        TInstantGuard ig(now);
        env.WaitForFinish(TDuration::Seconds(10));
        UNIT_ASSERT_VALUES_EQUAL(env.GetDebt(UserWithoutCards), 1000);
        {
            TVector<TBillingTask> bTasks = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(bTasks.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(bTasks.back().GetLastUpdate().Seconds(), now.Seconds());
            UNIT_ASSERT(bTasks.back().IsDebt());
        }
        {
            auto session = billingManager.BuildSession(false);
            UNIT_ASSERT(billingManager.GetActiveTasksManager().BoostSession(DefaultSessionID, UserWithoutCards, TBillingTask::TExecContext(), session));
            UNIT_ASSERT(session.Commit());
        }
        UNIT_ASSERT_VALUES_EQUAL(env.GetBillingTasksBySession(DefaultSessionID).back().GetLastUpdate().Seconds(), 0);
        ig.Set(now + TDuration::Minutes(4));
        env.WaitForFinish(TDuration::Seconds(10));
        {
            TVector<TBillingTask> bTasks = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(bTasks.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(bTasks.back().GetLastUpdate().Seconds(), (now + TDuration::Minutes(4)).Seconds());
            UNIT_ASSERT(bTasks.back().IsDebt());
        }
    }

    Y_UNIT_TEST(DefaultCards) {
        TBillingTestEnvironment env;
        TBillingClient client(env.GetTrustClientConfig(), env.DB);

        TString uid = env.GetManager().GetPassportUid(UserWith2Cards);
        const TString terminal = client.GetTrustProduct(EBillingType::CarUsage)->GetTrustService();
        TVector<NDrive::NTrustClient::TPaymentMethod> methods;
        UNIT_ASSERT(client.GetPaymentMethods(terminal, uid, methods));
        UNIT_ASSERT_VALUES_EQUAL(methods.size(), 2);
        TVector<TString> cards = NDrive::NBilling::BuildCardList(methods, "", TSet<TString>());
        INFO_LOG << JoinVectorIntoString(cards, ",") << Endl;
        UNIT_ASSERT_VALUES_EQUAL(cards.size(), 2);

        NDrive::NBilling::TAccountsManager& accountsManager = env.GetAccountsManager();;
        auto userAccounts = accountsManager.GetUserAccounts(UserWith2Cards, Now());
        for (auto&& acc : userAccounts) {
            if (acc->GetType() != NDrive::NBilling::EAccount::Trust) {
                continue;
            }
            auto logic = env.GetManager().GetLogicByType(NDrive::NBilling::EAccount::Trust);
            UNIT_ASSERT(!!logic);
            auto session = env.BuildSession(false);
            auto trustLogic = logic->GetAsSafe<TTrustLogic>();
            UNIT_ASSERT(trustLogic);
            UNIT_ASSERT(!trustLogic->SetDefaultCard(acc, cards.back(), DefaultSessionID, session)); // old_logic
        }

        TVector<TString> cardsNew = NDrive::NBilling::BuildCardList(methods, cards.back(), TSet<TString>());
        UNIT_ASSERT_VALUES_EQUAL(cardsNew.size(), 2);
        UNIT_ASSERT_VALUES_EQUAL(cardsNew.front(), cards.back());

        cardsNew = NDrive::NBilling::BuildCardList(methods, cards.back(), { cards.back() });
        UNIT_ASSERT_VALUES_EQUAL(cardsNew.size(), 2);
        UNIT_ASSERT_VALUES_EQUAL(cardsNew.front(), cards.front());

        cardsNew = NDrive::NBilling::BuildCardList(methods, cards.back(), { cards.front() });
        UNIT_ASSERT_VALUES_EQUAL(cardsNew.size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(cardsNew.front(), cards.back());
/*
        TBillingTask task;
        TPaymentOpContext ctx(TDriveUserData, TChargeInfo, task, cards, cards.front());
        ctx.InitPayment(123, "", billingManager.GetTrustProduct(EBillingType::CarUsage));
        UNIT_ASSERT_VALUES_EQUAL(ctx.Payment.PaymethodId, cards.front());
        UNIT_ASSERT(ctx.BuildAlternatives());
        UNIT_ASSERT_VALUES_EQUAL(ctx.Payment.PaymethodId, cards.back());
        UNIT_ASSERT(!ctx.BuildAlternatives());
        */
    }

    Y_UNIT_TEST(YandexAccount) {
        TBillingTestEnvironment env;
        TBillingClient client(env.GetTrustClientConfig(), env.DB);

        // row client
        {
            TString uid = env.GetManager().GetPassportUid(UserWithYandexAccount);
            const TString terminal = client.GetTrustProduct(EBillingType::CarUsage)->GetTrustService();
            TVector<NDrive::NTrustClient::TPaymentMethod> methods;
            UNIT_ASSERT(client.GetPaymentMethods(terminal, uid, methods));
            UNIT_ASSERT_VALUES_EQUAL(methods.size(), 2);
        }

        {
            TBillingManager& billingManager = env.GetManager();
            auto methods = billingManager.GetUserCards(UserWithYandexAccount);
            UNIT_ASSERT(methods);
            UNIT_ASSERT_VALUES_EQUAL(methods->size(), 2);
            for (const auto& method : *methods) {
                if (TBillingGlobals::SupportedCashbackMethods.contains(method.GetPaymentMethod())) {
                    UNIT_ASSERT_VALUES_UNEQUAL(method.GetBalance(), 0);
                }
            }

            {
                auto filtredMethods = TBillingManager::GetUserPaymentCards(methods.Get());
                UNIT_ASSERT(filtredMethods);
                UNIT_ASSERT_VALUES_EQUAL(filtredMethods->size(), 1);
            }
        }
    }

    Y_UNIT_TEST(TrustCache) {
        TBillingTestEnvironment env;
        {
            TBillingManager& billingManager = env.GetManager();
            UNIT_ASSERT(billingManager.GetTrustCache());
            {
                auto methods = billingManager.GetUserCards(UserWith2Cards, /* withCache = */false);
                UNIT_ASSERT(methods);
                UNIT_ASSERT_VALUES_EQUAL(methods->size(), 2);
            }
            {
                auto methods = billingManager.GetUserCards(UserWith2Cards, /* withCache = */true);
                UNIT_ASSERT(methods);
                UNIT_ASSERT_VALUES_EQUAL(methods->size(), 2); // no empty first call
            }
            billingManager.GetTrustCache()->GetGuard().WaitAllTasks();
            {
                auto methods = billingManager.GetUserCards(UserWith2Cards, /* withCache = */true);
                UNIT_ASSERT(methods);
                UNIT_ASSERT_VALUES_EQUAL(methods->size(), 2);
            }
        }
    }

    Y_UNIT_TEST(CancelBillingTask) {
        TBillingTestEnvironment env;
        TBillingManager& billingManager = env.GetManager();
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, UserWithoutCards, EBillingType::CarUsage, 0, 500));
        env.AddMoney(DefaultSessionID, 1500);
        env.RunAllTasks();

        {
            auto session = billingManager.BuildSession(false);
            UNIT_ASSERT(!billingManager.GetActiveTasksManager().CancelTask(DefaultSessionID, DefaultUserId, session));
            INFO_LOG << session.GetStringReport() << Endl;
        }
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        {
            auto session = billingManager.BuildSession(false);
            UNIT_ASSERT(billingManager.GetActiveTasksManager().CancelTask(DefaultSessionID, DefaultUserId, session));
            UNIT_ASSERT(session.Commit());
        }
        {
            auto tasks = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasks.size(), 1);
            env.RunAllTasks();
            env.RunAllTasks();
            tasks = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasks.size(), 0);
        }
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().begin()->GetCleared(), 1500);
        }
    }


    Y_UNIT_TEST(CancelBillingTaskPartial) {
        TBillingTestEnvironment env;
        TBillingManager& billingManager = env.GetManager();
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, UserWithoutCards, EBillingType::CarUsage, 0, 500));
        env.AddMoney(DefaultSessionID, 1500);
        env.RunAllTasks();
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        {
            auto session = billingManager.BuildSession(false);
            UNIT_ASSERT(billingManager.GetActiveTasksManager().CancelTask(DefaultSessionID, DefaultUserId, session, 1000));
            UNIT_ASSERT(session.Commit());
        }
        {
            auto tasks = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasks.size(), 1);
            env.RunAllTasks();
            env.RunAllTasks();
            tasks = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasks.size(), 0);
        }
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().begin()->GetCleared(), 1000);
        }
        {
            auto refunds = env.GetSessionRefunds(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(refunds.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(refunds.begin()->GetBill(), 500);
        }
    }

    Y_UNIT_TEST(TooSmallBill) {
        TBillingTestEnvironment env;
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 0, 500));
        env.AddMoney(DefaultSessionID, 57.6);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        UNIT_ASSERT_VALUES_EQUAL(env.GetBillingTasksBySession(DefaultSessionID).size(), 1);
        env.WaitForFinish(TDuration::Seconds(1));
        UNIT_ASSERT_VALUES_EQUAL(env.GetBillingTasksBySession(DefaultSessionID).size(), 0);
    }

    Y_UNIT_TEST(TooSmallRest) {
        TBillingTestEnvironment env;
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 0, 31415));
        env.AddMoney(DefaultSessionID, 31430);
        env.WaitForFinish(TDuration::Seconds(1));
        UNIT_ASSERT_VALUES_EQUAL(env.GetActualSnapshot(DefaultSessionID).GetPayments().size(), 0);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        {
            env.WaitForFinish(TDuration::Seconds(1));
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto payments = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(payments.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(payments.back().GetSum(), 31430);
        }
    }

    Y_UNIT_TEST(SessionReport) {
        TBillingTestEnvironment env;
        env.LinkBonusAccount(DefaultUserId);
        env.LinkWallet(DefaultUserId, "simple_limited");

        UNIT_ASSERT(env.AddBonuses(DefaultUserId, WalletHardLimit));

        {
            auto session = env.BuildSession(false);
            auto offer = MakeAtomicShared<TFakeOffer>((ui32)0);
            offer->SetOfferId(DefaultSessionID).SetPaymentDiscretization(500);
            TVector<TString> accounts = { "simple_limited", "bonus" };
            offer->SetChargableAccounts(accounts);
            UNIT_ASSERT(env.GetManager().CreateBillingTask(DefaultUserId, offer, session));
            UNIT_ASSERT(session.Commit());
        }

        env.AddMoney(DefaultSessionID, 2 * WalletHardLimit);

        {
            auto sessionReport = env.GetSessionReport(DefaultSessionID);
            UNIT_ASSERT(!sessionReport.Defined());
        }
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        {
            auto sessionReport = env.GetSessionReport(DefaultSessionID);
            UNIT_ASSERT(!!sessionReport.Defined());
            UNIT_ASSERT_VALUES_EQUAL(sessionReport->GetBill(), 2 * WalletHardLimit);
            UNIT_ASSERT_VALUES_EQUAL(sessionReport->SumByAccount.size(), 2);
            for (auto&& accInfo : sessionReport->SumByAccount) {
                INFO_LOG << "Check " << accInfo.first << Endl;
                UNIT_ASSERT_VALUES_EQUAL(accInfo.second, WalletHardLimit);
            }
            UNIT_ASSERT_VALUES_EQUAL(env.GetDebt(DefaultUserId), 0);
        }
    }

    Y_UNIT_TEST(CompiledBill) {
        TBillingTestEnvironment env;
        env.LinkBonusAccount(DefaultUserId);
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 1000));
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 0, 100));
        env.AddMoney(DefaultSessionID, 2000);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        auto bill = env.GetCompiledBill(DefaultSessionID);
        UNIT_ASSERT(bill.Defined());
        INFO_LOG << bill->GetReport().GetStringRobust() << Endl;
        UNIT_ASSERT_VALUES_EQUAL(bill->GetBill(), 2000);
        UNIT_ASSERT_VALUES_EQUAL(bill->GetDetails().GetItems().size(), 2);
        ui32 held = 0;
        for (auto&& item : bill->GetDetails().GetItems()) {
            held += item.GetSum();
        }
        UNIT_ASSERT_VALUES_EQUAL(held, 2000);
    }


    Y_UNIT_TEST(CompiledBillWallet) {
        TBillingTestEnvironment env;
        env.LinkWallet(DefaultUserId, "simple_limited");
        {
            auto session = env.BuildSession(false);
            auto offer = MakeAtomicShared<TFakeOffer>((ui32)0);
            offer->SetOfferId(DefaultSessionID).SetPaymentDiscretization(500);
            TVector<TString> accounts = { "simple_limited", "bonus" };
            offer->SetChargableAccounts(accounts);
            UNIT_ASSERT(env.GetManager().CreateBillingTask(DefaultUserId, offer, session));
            UNIT_ASSERT(session.Commit());
        }
        env.AddMoney(DefaultSessionID, 700);
        env.RunAllTasks();
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        UNIT_ASSERT_VALUES_EQUAL(env.GetActualSnapshot(DefaultSessionID).GetPayments().size(), 2);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        auto bill = env.GetCompiledBill(DefaultSessionID);
        UNIT_ASSERT(bill.Defined());
        INFO_LOG << bill->GetReport().GetStringRobust() << Endl;
        UNIT_ASSERT_VALUES_EQUAL(bill->GetBill(), 700);
        UNIT_ASSERT_VALUES_EQUAL(bill->GetDetails().GetItems().size(), 1);
        ui32 held = 0;
        for (auto&& item : bill->GetDetails().GetItems()) {
            held += item.GetSum();
        }
        UNIT_ASSERT_VALUES_EQUAL(held, 700);
    }

    Y_UNIT_TEST(SessionReportDeposit) {
        TBillingTestEnvironment env;
        env.LinkBonusAccount(DefaultUserId);
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, WalletHardLimit));
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 10000, 100));
        env.WaitForFinish(TDuration::Seconds(30));
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        UNIT_ASSERT_VALUES_EQUAL(env.GetActualSnapshot(DefaultSessionID).GetPayments().size(), 1);
        {
            auto sessionReport = env.GetSessionReport(DefaultSessionID);
            UNIT_ASSERT(!!sessionReport.Defined());
            UNIT_ASSERT_VALUES_EQUAL(sessionReport->GetBill(), 0);
            UNIT_ASSERT_VALUES_EQUAL(sessionReport->SumByAccount.size(), 0);
        }
    }

    Y_UNIT_TEST(SessionReportDepositSmallBill) {
        TBillingTestEnvironment env;
        env.LinkBonusAccount(DefaultUserId);
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, WalletHardLimit));
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 10000, 100));
        env.WaitForFinish(TDuration::Seconds(30));
        env.AddMoney(DefaultSessionID, 5000);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        UNIT_ASSERT_VALUES_EQUAL(env.GetActualSnapshot(DefaultSessionID).GetPayments().size(), 1);
        {
            auto sessionReport = env.GetSessionReport(DefaultSessionID);
            UNIT_ASSERT(!!sessionReport.Defined());
            UNIT_ASSERT_VALUES_EQUAL(sessionReport->GetBill(), 5000);
            UNIT_ASSERT_VALUES_EQUAL(sessionReport->SumByAccount.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(sessionReport->SumByAccount.begin()->second, 5000);
        }
    }

    Y_UNIT_TEST(GetTasksByRealSessionId) {
        TBillingTestEnvironment env;
        env.LinkBonusAccount(DefaultUserId);
        TBillingManager& manager = env.GetManager();
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, WalletHardLimit));
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID + "_1", DefaultUserId, EBillingType::CarUsage, 10000, 100));
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID + "_2", DefaultUserId, EBillingType::TollRoad, 10000, 100, 0, DefaultSessionID + "_1"));
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID + "_3", DefaultUserId, EBillingType::TollRoad, 10000, 100, 0, DefaultSessionID + "_3"));

        auto tasks1 = env.GetBillingTasksBySession(DefaultSessionID + "_1");
        UNIT_ASSERT_VALUES_EQUAL(tasks1.size(), 1);

        {
            auto session = env.GetManager().BuildSession(true);
            auto sessionHistory1 = manager.GetHistoryManager().GetSessionHistory(DefaultSessionID + "_1", TInstant::Zero(), session);
            UNIT_ASSERT(sessionHistory1);
            UNIT_ASSERT_VALUES_EQUAL(sessionHistory1->size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(sessionHistory1->front().GetId(), DefaultSessionID + "_1");
            UNIT_ASSERT_VALUES_EQUAL(sessionHistory1->front().GetHistoryAction(), EObjectHistoryAction::Add);

            auto realSessionHistory = manager.GetHistoryManager().GetRealSessionHistory(DefaultSessionID + "_1", DefaultUserId, TInstant::Zero(), session);
            UNIT_ASSERT(realSessionHistory);
            UNIT_ASSERT_VALUES_EQUAL(realSessionHistory->size(), 4);
            UNIT_ASSERT_VALUES_EQUAL(realSessionHistory.GetRef()[0].GetId(), DefaultSessionID + "_1");
            UNIT_ASSERT_VALUES_EQUAL(realSessionHistory.GetRef()[0].GetHistoryAction(), EObjectHistoryAction::Add);
            UNIT_ASSERT_VALUES_EQUAL(realSessionHistory.GetRef()[1].GetId(), DefaultSessionID + "_2");
            UNIT_ASSERT_VALUES_EQUAL(realSessionHistory.GetRef()[1].GetHistoryAction(), EObjectHistoryAction::Add);
            UNIT_ASSERT_VALUES_EQUAL(realSessionHistory.GetRef()[2].GetId(), DefaultSessionID + "_2");
            UNIT_ASSERT_VALUES_EQUAL(realSessionHistory.GetRef()[2].GetHistoryAction(), EObjectHistoryAction::UpdateData);
            UNIT_ASSERT_VALUES_EQUAL(realSessionHistory.GetRef()[3].GetId(), DefaultSessionID + "_2");
            UNIT_ASSERT_VALUES_EQUAL(realSessionHistory.GetRef()[3].GetHistoryAction(), EObjectHistoryAction::UpdateData);
        }

        env.WaitForFinish(TDuration::Seconds(30));
        {
            auto session = env.GetManager().BuildSession(true);
            auto realSessionHistory = manager.GetHistoryManager().GetRealSessionHistory(DefaultSessionID + "_1", DefaultUserId, TInstant::Zero(), session);
            UNIT_ASSERT(realSessionHistory);
            UNIT_ASSERT_VALUES_EQUAL(realSessionHistory->size(), 5);
            UNIT_ASSERT_VALUES_EQUAL(realSessionHistory.GetRef()[4].GetId(), DefaultSessionID + "_2");
            UNIT_ASSERT_VALUES_EQUAL(realSessionHistory.GetRef()[4].GetHistoryAction(), EObjectHistoryAction::Remove);
        }

        env.AddMoney(DefaultSessionID + "_1", 5000);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID + "_1"));
        {
            auto session = env.GetManager().BuildSession(true);
            auto realSessionHistory = manager.GetHistoryManager().GetRealSessionHistory(DefaultSessionID + "_1", DefaultUserId, TInstant::Zero(), session);
            UNIT_ASSERT(realSessionHistory);
            UNIT_ASSERT_VALUES_EQUAL(realSessionHistory->size(), 7);
            UNIT_ASSERT_VALUES_EQUAL(realSessionHistory.GetRef()[5].GetId(), DefaultSessionID + "_1");
            UNIT_ASSERT_VALUES_EQUAL(realSessionHistory.GetRef()[5].GetHistoryAction(), EObjectHistoryAction::UpdateData);
            UNIT_ASSERT_VALUES_EQUAL(realSessionHistory.GetRef()[6].GetId(), DefaultSessionID + "_1");
            UNIT_ASSERT_VALUES_EQUAL(realSessionHistory.GetRef()[6].GetHistoryAction(), EObjectHistoryAction::UpdateData);
        }
    }

    Y_UNIT_TEST(SplitPaymentTask) {
        TBillingTestEnvironment env("SQLite", TDuration::Minutes(5));
        env.SetSettings("billing.split.billing_types", ToString(EBillingType::Ticket));
        env.SetSettings("billing.split.age", "1");
        env.SetSettings("billing.split.parts", "2");
        env.SetSettings("billing.split.min_sum", "0");
        env.SetSettings("billing.min_task_priority", "0");
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::Ticket, 0, 0, NotAuthorizedSum));
        env.WaitForFinish();
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT(snapshot.GetPayments().empty());
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetFailedPayMethods().size(), 1);
            TVector<TBillingTask> tasks = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasks.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(tasks.front().GetTaskStatus(), EPaymentStatus::NoFunds);
        }
        Sleep(TDuration::Seconds(5));
        env.WaitForFinish(TDuration::Minutes(5), /* finalize = */ true);
        env.RunAllTasks();
        Sleep(TDuration::Seconds(5));
        TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 2);
        UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().begin()->GetSum(), NotAuthorizedSum / 2);
    }

    Y_UNIT_TEST(SyncPayments) {
        TBillingTestEnvironment env("SQLite", TDuration::Minutes(5));
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 0, 500));
        env.AddMoney(DefaultSessionID, 1000);
        TBillingManager& billingManager = env.GetManager();
        billingManager.WaitBillingCycle(10, 2);
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetSum(), 500);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Authorized);
            TVector<TBillingTask> tasks = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasks.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(tasks.front().GetLastPaymentId(), snapshot.GetVersion());
        }
        env.AddMoney(DefaultSessionID, 2000);
        {
            auto session = billingManager.BuildSession(/* readOnly = */ false);
            UNIT_ASSERT(billingManager.GetActiveTasksManager().BoostSession(DefaultSessionID, DefaultUserId, TBillingTask::TExecContext(), session));
            UNIT_ASSERT(session.Commit());
        }
        billingManager.WaitBillingCycle(10, 1);
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetSum(), 500);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Authorized);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetSum(), 1000);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Started);
            TVector<TBillingTask> tasks = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasks.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(tasks.front().GetLastPaymentId(), snapshot.GetVersion());
            UNIT_ASSERT_VALUES_EQUAL(tasks.front().GetLastPaymentId(), timeline.back().GetId());
        }
        env.RunClearing();
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 2);
            auto& first = timeline.front().GetSum() == 500 ? timeline.front() : timeline.back();
            auto& second = timeline.front().GetSum() == 1000 ? timeline.front() : timeline.back();
            UNIT_ASSERT_VALUES_EQUAL(first.GetSum(), 500);
            UNIT_ASSERT_VALUES_EQUAL(first.GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(second.GetSum(), 1000);
            UNIT_ASSERT_VALUES_EQUAL(second.GetStatus(), NDrive::NTrustClient::EPaymentStatus::Started);
            TVector<TBillingTask> tasks = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasks.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(tasks.front().GetLastPaymentId(), snapshot.GetVersion());
            UNIT_ASSERT_VALUES_EQUAL(tasks.front().GetLastPaymentId(), first.GetId());
        }
        billingManager.WaitBillingCycle(10, 1);
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 2);
            auto& first = timeline.front().GetSum() == 500 ? timeline.front() : timeline.back();
            auto& second = timeline.front().GetSum() == 1000 ? timeline.front() : timeline.back();
            UNIT_ASSERT_VALUES_EQUAL(first.GetSum(), 500);
            UNIT_ASSERT_VALUES_EQUAL(first.GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(second.GetSum(), 1000);
            UNIT_ASSERT_VALUES_EQUAL(second.GetStatus(), NDrive::NTrustClient::EPaymentStatus::Authorized);
            TVector<TBillingTask> tasks = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasks.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(tasks.front().GetLastPaymentId(), snapshot.GetVersion());
            UNIT_ASSERT_VALUES_EQUAL(tasks.front().GetLastPaymentId(), second.GetId());
        }
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
    }

    Y_UNIT_TEST(UseDuplicates) {
        TBillingTestEnvironment env("SQLite", TDuration::Minutes(5));
        env.SetSettings("billing.min_task_priority", "0");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, UserWithoutCards, EBillingType::Ticket, 0, 0, 10000));

        env.WaitForFinish(TDuration::Seconds(30));
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT(snapshot.GetPayments().empty());
            UNIT_ASSERT(snapshot.GetFailedPayMethods().empty());
            TVector<TBillingTask> tasks = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasks.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(tasks.front().GetTaskStatus(), EPaymentStatus::NoCards);
        }

        const TString testPassNumberHash = "f3c266214dfa332bd6386c215c34c3043bb03291";
        const TString testUserName = "user_test_name";
        {
            auto session = env.GetManager().BuildSession(/* readOnly = */ false);
            auto updateUser = [&testPassNumberHash, &testUserName, &env] (const TString& userId, auto& session) {
                auto users = env.UsersDB.FetchInfo(userId, session);
                UNIT_ASSERT_C(users, session.GetStringReport());
                auto userData = users.GetResult().begin()->second;
                userData.SetPassportNumberHash(testPassNumberHash);
                userData.SetDrivingLicenseNumberHash(testPassNumberHash);
                userData.SetPassportNamesHash(testPassNumberHash);
                userData.SetLastName(testUserName);
                userData.SetFirstName(testUserName);
                userData.SetPName(testUserName);
                UNIT_ASSERT_C(env.UsersDB.UpdateUser(userData, "robot-frontend", session), session.GetStringReport());
            };
            updateUser(UserWithoutCards, session);
            updateUser(UserWith2Cards, session);
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        }
        {
            auto session = env.GetManager().BuildSession(/* readOnly = */ true);
            auto users = env.UsersDB.GetSamePersons(UserWithoutCards, session);
            UNIT_ASSERT_C(users, session.GetStringReport());
            UNIT_ASSERT_VALUES_EQUAL(users->size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(users->front().GetUserId(), UserWith2Cards);
        }

        env.WaitForFinish(TDuration::Seconds(30));
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT(snapshot.GetPayments().empty());
            UNIT_ASSERT(snapshot.GetFailedPayMethods().empty());
            TVector<TBillingTask> tasks = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasks.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(tasks.front().GetTaskStatus(), EPaymentStatus::NoCards);
        }

        env.SetSettings("billing.use_duplicates_age", "0");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        env.WaitForFinish(TDuration::Seconds(30));
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT(snapshot.GetPayments().empty());
            UNIT_ASSERT(snapshot.GetFailedPayMethods().empty());
            TVector<TBillingTask> tasks = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasks.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(tasks.front().GetTaskStatus(), EPaymentStatus::NoCards);
        }

        env.SetSettings("billing.use_extra_users", "true");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        env.WaitForFinish();
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT(snapshot.GetFailedPayMethods().empty());
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().begin()->GetStatus(), NDrive::NTrustClient::EPaymentStatus::Authorized);
        }
    }
}
