#include "trust_emulator.h"

#include <library/cpp/http/misc/parsed_request.h>
#include <library/cpp/json/json_value.h>
#include <library/cpp/logger/global/global.h>

#include <util/generic/guid.h>


class TTrustReplier : public TRequestReplier {
public:
    TTrustReplier(TTrustEmulator& owner)
        : Owner(owner) {}

    TString GetSimpleReply() const {
        NJson::TJsonValue json;
        json["status"] = "success";
        return json.GetStringRobust();
    }

    bool DoReply(const TReplyParams& params) override {
        TParsedHttpFull parsedRequest(params.Input.FirstLine());
        TString userId;
        for (auto&& header : params.Input.Headers()) {
            if (header.Name() == "X-Uid") {
                userId = header.Value();
            }
        }
        INFO_LOG << "X-Uid: " << userId << Endl;
        TString postData = params.Input.ReadAll();
        TString reply;
        if (parsedRequest.Path == "/trust-payments/v2/payment-methods") {
            if (userId == "1120000000005904") { // UserWithoutCards
                reply = ReplyNoCard;
            } else if (userId == "267356944" || userId == "1120000000019756") {
                reply = ReplyWithYandexAccount;
            } else {
                reply = ReplyTwoCards;
            }

            NJson::TJsonValue replyJson;
            if (!NJson::ReadJsonFastTree(reply, &replyJson)) {
                return false;
            }
            for (auto& methodJson : replyJson["bound_payment_methods"].GetArraySafe()) {
                if (methodJson["payment_method"].GetString() == "yandex_account") {
                    TString paymethodId = methodJson["id"].GetString();

                    ui32 spentSum = 0;
                    for(const auto& [id, payment] : Owner.GetPayments()) {
                        if (payment.PaymethodId == paymethodId && payment.Status == NDrive::NTrustClient::EPaymentStatus::Cleared) {
                            spentSum += payment.Sum;
                        }
                    }

                    methodJson["balance"] = ::ToString(FromString<double>(methodJson["balance"].GetString()) - spentSum / 100.);
                }
            }
            reply = replyJson.GetStringRobust();
        }

        if (parsedRequest.Path == "/trust-payments/v2/account") {
            if (userId == "1120000000019756") { // UserWithFailedYandexAccount
                reply = ReplyFailedCreateYandexAccount;
            } else {
                reply = ReplyCreateYandexAccount;
            }
        }

        if (parsedRequest.Path.StartsWith("/trust-payments/v2/payments")) {
            if (parsedRequest.Path == "/trust-payments/v2/payments") {
                NJson::TJsonValue request;
                CHECK_WITH_LOG(NJson::ReadJsonFastTree(postData, &request, false));
                double amount = request["amount"].GetDouble();
                TString paymethodId = request["paymethod_id"].GetString();
                TPaymentInfo info(amount * 100, paymethodId);
                info.Token = CreateGuidAsString();
                NJson::TJsonValue json;
                json["status"] = "success";
                json["purchase_token"] = info.Token;
                reply = json.GetStringRobust();
                Owner.RegisterPayment(info);
            } else {
                TFsPath path(parsedRequest.Path);
                if (path.GetName() == "start") {
                    Owner.StartPayment(path.Parent().GetName());
                    NJson::TJsonValue json;
                    json["status"] = "success";
                    NJson::TJsonValue orderJson;
                    orderJson["order_id"] = "123456";
                    json["orders"].AppendValue(orderJson);
                    reply = json.GetStringRobust();
                } else if (path.GetName() == "unhold") {
                    Owner.CancelPayment(path.Parent().GetName());
                    reply = GetSimpleReply();
                } else if (path.GetName() == "clear") {
                    Owner.ClearPayment(path.Parent().GetName());
                    reply = GetSimpleReply();
                } else if (path.GetName() == "resize") {
                    reply = GetSimpleReply();
                } else {
                    auto result = Owner.ProducePayment(path.GetName());
                    reply = result.GetReport();
                }
            }
        }
        if (parsedRequest.Path.StartsWith("/trust-payments/v2/refunds")) {
            NJson::TJsonValue json;
            TFsPath path(parsedRequest.Path);
            if (path.GetName() == "refunds") {
                json["trust_refund_id"] = CreateGuidAsString();
            }
            if (path.GetName() == "start") {
                json["status"] = "wait_for_notification";
            } else {
                json["status"] = "success";
            }
            reply = json.GetStringRobust();
        }
        params.Output << "HTTP/1.1 200 Ok" << Endl;
        params.Output << Endl;
        params.Output << reply << Endl;
        return true;
    }
private:
    TTrustEmulator& Owner;
};


void TTrustEmulator::Run(ui16 httpPort) {
    THttpServerOptions httpOptions(httpPort);
    Server = MakeHolder<THttpServer>(this, httpOptions);
    try {
        Server->Start();
    } catch (const std::exception& e) {
        ERROR_LOG << FormatExc(e);
        FAIL_LOG("Can't create emulator");
    }
}

void TTrustEmulator::RegisterPayment(const TPaymentInfo& info) {
    Payments.emplace(info.Token, info);
}

TPaymentInfo TTrustEmulator::CancelPayment(const TString& paymentId) {
    auto it = Payments.find(paymentId);
    CHECK_WITH_LOG(it != Payments.end());
    CHECK_WITH_LOG(it->second.Status == NDrive::NTrustClient::EPaymentStatus::Authorized || it->second.Status == NDrive::NTrustClient::EPaymentStatus::Canceled) << it->second.Status;
    it->second.Status = NDrive::NTrustClient::EPaymentStatus::Canceled;
    it->second.Sum = 0;
    return it->second;
}

TPaymentInfo TTrustEmulator::StartPayment(const TString& paymentId) {
    auto it = Payments.find(paymentId);
    CHECK_WITH_LOG(it != Payments.end());
    CHECK_WITH_LOG(it->second.Status == NDrive::NTrustClient::EPaymentStatus::NotStarted) << it->second.Status;
    it->second.Status = NDrive::NTrustClient::EPaymentStatus::Started;
    return it->second;
}

TPaymentInfo TTrustEmulator::ClearPayment(const TString& paymentId) {
    auto it = Payments.find(paymentId);
    CHECK_WITH_LOG(it != Payments.end());
    CHECK_WITH_LOG(it->second.Status == NDrive::NTrustClient::EPaymentStatus::Authorized || it->second.Status == NDrive::NTrustClient::EPaymentStatus::Cleared) << it->second.Status;
    it->second.Status = NDrive::NTrustClient::EPaymentStatus::Cleared;
    return it->second;
}

TPaymentInfo TTrustEmulator::ProducePayment(const TString& paymentId) {
    auto it = Payments.find(paymentId);
    CHECK_WITH_LOG(it != Payments.end());
    CHECK_WITH_LOG(it->second.Status != NDrive::NTrustClient::EPaymentStatus::NotStarted) << it->second.Status;

    if (it->second.Status == NDrive::NTrustClient::EPaymentStatus::Started) {
        if (it->second.PaymethodId.StartsWith("yandex_account")) {
            it->second.Status = NDrive::NTrustClient::EPaymentStatus::Cleared;
        } else if (it->second.Sum == NotAuthorizedSum) {
            it->second.Status = NDrive::NTrustClient::EPaymentStatus::NotAuthorized;
        } else {
            it->second.Status = NDrive::NTrustClient::EPaymentStatus::Authorized;
        }
    }
    return it->second;
}

TClientRequest* TTrustEmulator::CreateClient() {
    return new TTrustReplier(*this);
}

TTrustEmulator::~TTrustEmulator() {
    if (Server) {
        Server->Stop();
    }
}
