#include <drive/backend/billing/redis_proxy_cache.h>

#include <drive/library/cpp/trust/entity.h>
#include <drive/library/cpp/trust/lpm.h>
#include <drive/library/cpp/trust/trust_cache.h>

#include <kernel/daemon/config/daemon_config.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/writer/json_value.h>
#include <library/cpp/string_utils/base64/base64.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/threading/future/core/future.h>
#include <library/cpp/yconf/conf.h>

#include <rtline/library/json/parse.h>
#include <rtline/library/storage/config.h>
#include <rtline/library/storage/redis/abstract.h>
#include <rtline/library/storage/redis/support.h>

#include <util/datetime/base.h>
#include <util/generic/fwd.h>
#include <util/generic/ptr.h>
#include <util/generic/yexception.h>
#include <util/stream/output.h>
#include <util/system/compiler.h>
#include <util/system/yassert.h>

namespace {
    class TLocalTrustUpdater : public NDrive::ITrustUpdater {
    public:
        NThreading::TFuture<TVector<NDrive::NTrustClient::TPaymentMethod>> GetPaymentMethods(const TString& userId) const override {
            if (auto it = PaymentMethods.find(userId); it != PaymentMethods.end()) {
                return NThreading::MakeFuture<TVector<NDrive::NTrustClient::TPaymentMethod>>(it->second);
            }

            return NThreading::MakeErrorFuture<TVector<NDrive::NTrustClient::TPaymentMethod>>(std::make_exception_ptr(yexception() << "There is no such user"));
        }

        void AddPaymentMethod(const TString& userId, const TString& paymentMethodId) {
            NJson::TJsonValue json;
            NJson::InsertField(json, "id", paymentMethodId);
            NJson::InsertField(json, "account", TString{"fake"});

            NDrive::NTrustClient::TPaymentMethod paymentMethod;
            UNIT_ASSERT(paymentMethod.FromJson(json));

            PaymentMethods[userId].push_back(paymentMethod);
        }

        bool RemoveAnyPaymentMethod(const TString& userId) {
            if (auto it = PaymentMethods.find(userId); it != PaymentMethods.end()) {
                if (!it->second.empty()) {
                    it->second.pop_back();
                    return true;
                }
            }

            return false;
        }

    private:
        TMap<TString, TVector<NDrive::NTrustClient::TPaymentMethod>> PaymentMethods;
    };

    THolder<TLPMClient> CreateLPMClient(const TLPMClientConfig& config,
        TAtomicSharedPtr<TLocalTrustUpdater> updater,
        TAtomicSharedPtr<NDriveRedis::NKVAbstract::TRedisVersionedStorage> redisStorage = nullptr
    ) {
        TVector<THolder<NDrive::ITrustStorageOptions>> lpmCachesOps;
        for (const auto& storageConfig: config.GetCacheStorageConfigs()) {
            if (auto redisPtr = dynamic_cast<NDriveRedis::NCache::TTrustRedisCacheConfig*>(storageConfig.Get()); redisPtr && redisStorage) {
                lpmCachesOps.push_back(MakeHolder<NDriveRedis::NCache::TTrustRedisStorageOptions>(redisStorage));
            } else if (auto redisPtr = dynamic_cast<NDrive::NRedis::NCache::TTrustCacheConfig*>(storageConfig.Get())) {
                lpmCachesOps.push_back(MakeHolder<NDrive::NRedis::NCache::TTrustStorageOptions>());
            } else if (auto localPtr = dynamic_cast<NDrive::TLocalTrustStorageConfig*>(storageConfig.Get())) {
                lpmCachesOps.push_back(MakeHolder<NDrive::TLocalTrustStorageOptions>());
            } else {
                WARNING_LOG << "Unknown cache type " << storageConfig->GetCacheId() << Endl;
            }
        }
        return config.Construct(std::move(lpmCachesOps), updater);
    }

}

Y_UNIT_TEST_SUITE(RedisAuth) {
    NDriveRedis::NSession::TSessionConfig SessionConfig{GetEnv("REDIS_HOSTNAME", "localhost"),
                                                        FromStringWithDefault(GetEnv("REDIS_PORT"), 6379u),
                                                        GetEnv("REDIS_PASS")};
    NDriveRedis::NDatabases::TRedisDatabase Database(SessionConfig);
    const TString RedisCacheId = "redis1";

    TLPMClientConfig ConfigureLPM() {
        TAnyYandexConfig conf;
        TStringBuilder configData;
        configData  << "<Server>"
                    << "    <LPMClient>" << Endl
                    << "        Name: LPM" << Endl
                    << "        Host: Fake" << Endl
                    << "        ServiceToken: None" << Endl
                    << "        ServiceTokenPath: None" << Endl
                    << "        RequestTimeout: 600ms" << Endl
                    << "        <RequestConfig>" << Endl
                    << "            MaxAttempts: 1" << Endl
                    << "            TimeoutSendingms: 600" << Endl
                    << "        </RequestConfig>" << Endl
                    << "        <CacheStorages>" << Endl
                    << "            <RedisMain>" << Endl
                    << "                CacheType: Redis" << Endl
                    << "                DBName: Fake" << Endl
                    << "                RefreshInterval: 3s" << Endl
                    << "                CacheId: " << RedisCacheId << Endl
                    << "            </RedisMain>" << Endl
                    << "            <LocalMain>" << Endl
                    << "                CacheType: local" << Endl
                    << "                CacheId: local1" << Endl
                    << "            </LocalMain>" << Endl
                    << "        </CacheStorages>" << Endl
                    << "    </LPMClient>" << Endl
                    << "</Server>";
        bool success = conf.ParseMemory(configData);

        Y_ASSERT(success);

        TLPMClientConfig result;

        TYandexConfig::Section* serverSec = conf.GetFirstChild("Server");
        Y_ASSERT(serverSec);

        auto serverChildren = serverSec->GetAllChildren();
        auto lpmClientSec = serverChildren.find("LPMClient");
        Y_ASSERT(lpmClientSec != serverChildren.end());

        result.Init(lpmClientSec->second);

        return result;
    }

    TMaybe<TVector<NDrive::NTrustClient::TPaymentMethod>> GetPaymentMethods(TAtomicSharedPtr<NDriveRedis::NKVAbstract::TRedisVersionedStorage> storage, const TString& key) {
        TString jsonData;
        bool success = storage->GetValue(key, jsonData);

        if (!success || jsonData.Empty()) {
            return {};
        }

        NJson::TJsonValue json = NJson::ReadJsonFastTree(Base64Decode(jsonData));

        NDrive::ITrustStorage::TTimedMethods methods;
        success = methods.FromJson(json);
        if (!success) {
            return {};
        }

        return {methods.Methods};
    }

    TAtomicSharedPtr<NDriveRedis::NKVAbstract::TRedisVersionedStorage> CreateRedisClient() {
        NDriveRedis::NSession::TSessionConfig cfg{GetEnv("REDIS_HOSTNAME", "localhost"), FromStringWithDefault(GetEnv("REDIS_PORT"), 6379u), GetEnv("REDIS_PASS")};
        NRTLine::TStorageOptions ops{};
        return MakeAtomicShared<NDriveRedis::NKVAbstract::TRedisVersionedStorage>(ops, cfg);
    }

    TAtomicSharedPtr<TLocalTrustUpdater> CreateUpdater() {
        TAtomicSharedPtr<TLocalTrustUpdater> updater = MakeAtomicShared<TLocalTrustUpdater>();

        updater->AddPaymentMethod("Alice", "AliceId");
        return updater;
    }

    Y_UNIT_TEST(Correctness) {
        auto updater = CreateUpdater();
        auto redisCache = CreateRedisClient();
        auto config = ConfigureLPM();
        auto lpmClient = CreateLPMClient(config, updater, redisCache);
        UNIT_ASSERT(lpmClient);

        UNIT_ASSERT(!redisCache->ExistsNode("Alice"));

        auto futurePaymentMethods = lpmClient->GetPaymentMethods("Alice", RedisCacheId);
        auto paymentMethods = futurePaymentMethods.GetValueSync();

        UNIT_ASSERT(paymentMethods.size() == 1 && paymentMethods[0].GetId() == "AliceId");

        // Waiting when DoUpdate-Callback will be called
        Sleep(TDuration::MilliSeconds(300));

        auto mPaymentMethods = GetPaymentMethods(redisCache, "Alice");
        UNIT_ASSERT(!mPaymentMethods.Empty());

        paymentMethods = *mPaymentMethods;
        UNIT_ASSERT(paymentMethods.size() == 1 && paymentMethods[0].GetId() == "AliceId");

        // Cleanup
        redisCache->RemoveNode("Alice");
    }

    Y_UNIT_TEST(Lifetime) {
        auto updater = CreateUpdater();
        auto redisCache = CreateRedisClient();
        auto config = ConfigureLPM();
        auto lpmClient = CreateLPMClient(config, updater, redisCache);
        UNIT_ASSERT(lpmClient);

        UNIT_ASSERT(!redisCache->ExistsNode("Alice"));

        auto futurePaymentMethods = lpmClient->GetPaymentMethods("Alice", RedisCacheId);
        auto paymentMethods = futurePaymentMethods.GetValueSync();

        UNIT_ASSERT(paymentMethods.size() == 1 && paymentMethods[0].GetId() == "AliceId");

        // Wait when DoUpdate-Callback will be called
        Sleep(TDuration::MilliSeconds(300));

        auto mPaymentMethods = GetPaymentMethods(redisCache, "Alice");
        UNIT_ASSERT(!mPaymentMethods.Empty());

        paymentMethods = *mPaymentMethods;
        UNIT_ASSERT(paymentMethods.size() == 1 && paymentMethods[0].GetId() == "AliceId");

        // Waiting when value will be removed
        Sleep(TDuration::Seconds(3));

        mPaymentMethods = GetPaymentMethods(redisCache, "Alice");
        UNIT_ASSERT(mPaymentMethods.Empty());

        // Cleanup
        redisCache->RemoveNode("Alice");
    }

    Y_UNIT_TEST(Refreshness) {
        auto updater = CreateUpdater();
        auto redisCache = CreateRedisClient();
        auto config = ConfigureLPM();
        auto lpmClient = CreateLPMClient(config, updater, redisCache);
        UNIT_ASSERT(lpmClient);

        UNIT_ASSERT(!redisCache->ExistsNode("Alice"));

        auto futurePaymentMethods = lpmClient->GetPaymentMethods("Alice", RedisCacheId);
        auto paymentMethods = futurePaymentMethods.GetValueSync();

        UNIT_ASSERT(paymentMethods.size() == 1 && paymentMethods[0].GetId() == "AliceId");

        // Waiting when DoUpdate-Callback will be called
        Sleep(TDuration::MilliSeconds(300));

        auto mPaymentMethods = GetPaymentMethods(redisCache, "Alice");
        UNIT_ASSERT(!mPaymentMethods.Empty());

        paymentMethods = *mPaymentMethods;
        UNIT_ASSERT(paymentMethods.size() == 1 && paymentMethods[0].GetId() == "AliceId");

        updater->AddPaymentMethod("Alice", "AliceId2");

        // Obtain old values
        futurePaymentMethods = lpmClient->GetPaymentMethods("Alice", RedisCacheId);
        paymentMethods = futurePaymentMethods.GetValueSync();

        UNIT_ASSERT(paymentMethods.size() == 1 && paymentMethods[0].GetId() == "AliceId");

        // Wait when value will be removed
        Sleep(TDuration::Seconds(3));

        // Obtain new values
        futurePaymentMethods = lpmClient->GetPaymentMethods("Alice", RedisCacheId);
        paymentMethods = futurePaymentMethods.GetValueSync();

        UNIT_ASSERT(paymentMethods.size() == 2 && paymentMethods[0].GetId() == "AliceId" && paymentMethods[1].GetId() == "AliceId2");

        Sleep(TDuration::MilliSeconds(300));

        // Cleanup
        redisCache->RemoveNode("Alice");
    }
}

Y_UNIT_TEST_SUITE(RedisProxy) {
    TLPMClientConfig ConfigureLPM() {
        TAnyYandexConfig conf;
        TStringBuilder configData;
        configData  << "<Server>"
                    << "    <LPMClient>" << Endl
                    << "        Name: LPM" << Endl
                    << "        Host: Fake" << Endl
                    << "        ServiceToken: None" << Endl
                    << "        ServiceTokenPath: None" << Endl
                    << "        RequestTimeout: 600ms" << Endl
                    << "        <RequestConfig>" << Endl
                    << "            MaxAttempts: 1" << Endl
                    << "            TimeoutSendingms: 600" << Endl
                    << "        </RequestConfig>" << Endl
                    << "        <CacheStorages>" << Endl
                    << "            <RedisService>" << Endl
                    << "                CacheType: redis_service" << Endl
                    << "                RefreshInterval: 3s" << Endl
                    << "                CacheId: redis_service" << Endl
                    << "                Uri: " << GetEnv("RedisServiceUri") << Endl
                    << "                UriPath: " << GetEnv("RedisServiceUriPath") << Endl
                    << "                <RequestConfig>" << Endl
                    << "                    MaxAttempts: 1" << Endl
                    << "                </RequestConfig>" << Endl
                    << "            </RedisService>" << Endl
                    << "        </CacheStorages>" << Endl
                    << "    </LPMClient>" << Endl
                    << "</Server>";
        UNIT_ASSERT(conf.ParseMemory(configData));
        TYandexConfig::Section* serverSec = conf.GetFirstChild("Server");
        UNIT_ASSERT(serverSec);

        auto serverChildren = serverSec->GetAllChildren();
        auto lpmClientSec = serverChildren.find("LPMClient");
        UNIT_ASSERT(lpmClientSec != serverChildren.end());

        TLPMClientConfig result;
        result.Init(lpmClientSec->second);
        return result;
    }

    Y_UNIT_TEST(Lifetime) {
        const TString userId = "test_user_id";
        const TString firstPM = "first_payment_method";
        const TString secondPM = "second_payment_method";
        if (!GetEnv("RedisServiceUri")) {
            return;
        }
        auto localTrust = MakeAtomicShared<TLocalTrustUpdater>();
        localTrust->AddPaymentMethod(userId, firstPM);
        auto config = ConfigureLPM();
        auto lpmClient = CreateLPMClient(config, localTrust);
        UNIT_ASSERT(lpmClient);

        {
            auto futurePaymentMethods = lpmClient->GetPaymentMethods(userId, "redis_service");
            auto paymentMethods = futurePaymentMethods.GetValueSync();
            UNIT_ASSERT_VALUES_EQUAL(paymentMethods.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(paymentMethods[0].GetId(), firstPM);
        }
        localTrust->RemoveAnyPaymentMethod(userId);
        localTrust->AddPaymentMethod(userId, secondPM);
        Sleep(TDuration::MilliSeconds(300));
        {
            auto futurePaymentMethods = lpmClient->GetPaymentMethods(userId, "redis_service");
            auto paymentMethods = futurePaymentMethods.GetValueSync();
            UNIT_ASSERT(paymentMethods.size() == 1 && paymentMethods[0].GetId() == firstPM);
        }
        Sleep(TDuration::Seconds(3));
        {
            auto futurePaymentMethods = lpmClient->GetPaymentMethods(userId, "redis_service");
            auto paymentMethods = futurePaymentMethods.GetValueSync();
            UNIT_ASSERT(paymentMethods.size() == 1 && paymentMethods[0].GetId() == secondPM);
        }
    }
}
