#include <drive/backend/billing/ut/library/tests_helper.h>

#include <drive/backend/billing/bonus/charge_logic.h>


Y_UNIT_TEST_SUITE(TrustSuite) {
    Y_UNIT_TEST(CheckAvailableBonusPay) {
        TBillingTestEnvironment env("SQLite", TDuration::Zero(), 1);
        env.LinkBonusAccount(DefaultUserId);
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 200));
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 0, 500));
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 1800));
        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 2000);
        env.AddMoney(DefaultSessionID, 1000);
        env.WaitForFinish();
        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 1800);
        TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 2);
        for (auto&& payment : snapshot.GetPayments()) {
            if (payment.GetPaymentType() == NDrive::NBilling::EAccount::Bonus) {
                UNIT_ASSERT_VALUES_EQUAL(payment.GetSum(), 200);
            }
        }
    }

    Y_UNIT_TEST(CheckAvailableLimitedBonusPay) {
        TBillingTestEnvironment env("SQLite", TDuration::Zero(), 1);
        env.LinkBonusAccount(DefaultUserId, "limited_bonuses");
        TInstant start = TInstant::Seconds(Now().Seconds() + 1);
        TInstant bonusDeadline = start + TDuration::Hours(1);
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 200, bonusDeadline));
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 0, 500));
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 1800, bonusDeadline));
        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 2000);
        UNIT_ASSERT_VALUES_EQUAL(env.GetLimitedBonuses(DefaultUserId).size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(env.GetLimitedBonuses(DefaultUserId).begin()->GetBalance(), 2000);
        env.AddMoney(DefaultSessionID, 1000);
        env.WaitForFinish();
        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 1800);
        UNIT_ASSERT_VALUES_EQUAL(env.GetLimitedBonuses(DefaultUserId).size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(env.GetLimitedBonuses(DefaultUserId).begin()->GetBalance(), 1800);
        TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 2);
        for (auto&& payment : snapshot.GetPayments()) {
            if (payment.GetPaymentType() == NDrive::NBilling::EAccount::Bonus) {
                UNIT_ASSERT_VALUES_EQUAL(payment.GetSum(), 200);
                auto paymentMeta = dynamic_cast<const TBonusPaymentMeta*>(payment.GetMeta().Get());
                UNIT_ASSERT_VALUES_UNEQUAL(paymentMeta, nullptr);
                UNIT_ASSERT_VALUES_EQUAL(paymentMeta->GetLimitedBonuses().size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(paymentMeta->GetLimitedBonuses().begin()->GetBalance(), 200);
            }
        }
    }

    Y_UNIT_TEST(CheckAvailableLimitedBonusPayAfterDeadline) {
        TBillingTestEnvironment env("SQLite", TDuration::Zero(), 1);
        env.LinkBonusAccount(DefaultUserId, "limited_bonuses");
        TInstant start = TInstant::Seconds(Now().Seconds() + 1);
        TInstant bonusDeadline = start + TDuration::Hours(1);
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 200, bonusDeadline + TDuration::Hours(1)));
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 1800, bonusDeadline));
        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 2000);
        UNIT_ASSERT_VALUES_EQUAL(env.GetLimitedBonuses(DefaultUserId).size(), 2);

        TInstantGuard ig(bonusDeadline + TDuration::Minutes(1));
        {
            auto session = env.BuildSession();
            auto bonusAccount = env.GetAccount(DefaultUserId, "limited_bonuses");
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->Refresh(DefaultUserId, ModelingNow(), session), ERefreshStatus::Ok);
            UNIT_ASSERT(session.Commit());
        }
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 0, 500));
        env.AddMoney(DefaultSessionID, 1000);
        env.WaitForFinish();
        ig.Set(bonusDeadline + TDuration::Minutes(2));
        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 0);
        TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 2);
        for (auto&& payment : snapshot.GetPayments()) {
            if (payment.GetPaymentType() == NDrive::NBilling::EAccount::Bonus) {
                UNIT_ASSERT_VALUES_EQUAL(payment.GetSum(), 200);
                auto paymentMeta = dynamic_cast<const TBonusPaymentMeta*>(payment.GetMeta().Get());
                UNIT_ASSERT_VALUES_UNEQUAL(paymentMeta, nullptr);
                UNIT_ASSERT_VALUES_EQUAL(paymentMeta->GetLimitedBonuses().size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(paymentMeta->GetLimitedBonuses().begin()->GetBalance(), 200);
            }
        }
    }

    Y_UNIT_TEST(CheckAvailableLimitedBonusPayBeforeDeadline) {
        TBillingTestEnvironment env("SQLite", TDuration::Zero(), 1);
        env.LinkBonusAccount(DefaultUserId, "limited_bonuses");
        TInstant start = TInstant::Seconds(Now().Seconds() + 1);
        TInstant bonusDeadline = start + TDuration::Hours(1);
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 200, bonusDeadline + TDuration::Hours(1)));
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 1800, bonusDeadline));
        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 2000);
        UNIT_ASSERT_VALUES_EQUAL(env.GetLimitedBonuses(DefaultUserId).size(), 2);

        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 0, 500));
        TInstantGuard ig(bonusDeadline + TDuration::Minutes(1));
        {
            auto session = env.BuildSession();
            auto bonusAccount = env.GetAccount(DefaultUserId, "limited_bonuses");
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->Refresh(DefaultUserId, ModelingNow(), session), ERefreshStatus::Ok);
            UNIT_ASSERT(session.Commit());
        }
        env.AddMoney(DefaultSessionID, 1000);
        env.WaitForFinish();
        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 0);
        TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 2);
        for (auto&& payment : snapshot.GetPayments()) {
            if (payment.GetPaymentType() == NDrive::NBilling::EAccount::Bonus) {
                UNIT_ASSERT_VALUES_EQUAL(payment.GetSum(), 200);
                auto paymentMeta = dynamic_cast<const TBonusPaymentMeta*>(payment.GetMeta().Get());
                UNIT_ASSERT_VALUES_UNEQUAL(paymentMeta, nullptr);
                UNIT_ASSERT_VALUES_EQUAL(paymentMeta->GetLimitedBonuses().size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(paymentMeta->GetLimitedBonuses().begin()->GetBalance(), 200);
            }
        }
    }

    Y_UNIT_TEST(SimpleCardPay) {
        TBillingTestEnvironment env("SQLite");
        TBillingManager& billingManager = env.GetManager();
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 0, 500));
        env.AddMoney(DefaultSessionID, 1000);
        {
            TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
            env.RunAllTasks();
            TVector<TBillingTask> tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 1);
        }

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().begin()->GetStatus(), NDrive::NTrustClient::EPaymentStatus::Started);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().begin()->GetSum(), 500);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().begin()->GetPaymentType(), NDrive::NBilling::EAccount::Trust);
            UNIT_ASSERT(!snapshot.GetPayments().begin()->GetOrderId().empty());
        }

        TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
        env.WaitForFinish();

        TVector<TBillingTask> tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 1);
        //UNIT_ASSERT_VALUES_EQUAL(tasksAfter.front().GetTaskStatus(), EPaymentStatus::Ok);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));

        tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
        env.WaitForFinish();
        env.RunAllTasks(); // Remove closed tasks
        tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 0);

        Sleep(TDuration::Seconds(2));

        ui32 cleared = env.RunClearing(2);
        UNIT_ASSERT_VALUES_EQUAL(cleared, 0);

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetSum(), 500);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetCleared(), 500);
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        auto session = billingManager.BuildSession(true);
        UNIT_ASSERT_VALUES_UNEQUAL(billingManager.GetLastUsedAccount(DefaultUserId, session), nullptr);
        UNIT_ASSERT_VALUES_EQUAL(billingManager.GetLastUsedAccount(DefaultUserId, session)->GetUniqueName(), "card");
    }

    Y_UNIT_TEST(TwoCreditCards) {
        TBillingTestEnvironment env("SQLite", TDuration::Minutes(5), 1);
        TBillingManager& billingManager = env.GetManager();
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, UserWith2Cards, EBillingType::Ticket, 0, 0, 109900));

        TInstant now = Now() - TDuration::Hours(10);
        TInstantGuard ig(now);

        env.WaitForFinish();
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 0);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetFailedPayMethods().size(), 1);
        }

        UNIT_ASSERT_VALUES_EQUAL(env.GetDebt(UserWith2Cards), 109900);
        ig.Set(now + TDuration::Minutes(3));
        env.WaitForFinish();
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 0);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetFailedPayMethods().size(), 1);
        }
        ig.Set(now + TDuration::Minutes(10));
        env.WaitForFinish();
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetFailedPayMethods().size(), 2);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 0);
            TVector<TBillingTask> tasks = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasks.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(tasks.front().GetTaskStatus(), EPaymentStatus::NoFunds);
        }

        {
            auto session = billingManager.BuildSession(false);
            session->Exec("UPDATE billing_tasks SET bill=101000, event_id = nextval('billing_tasks_event_id_seq') WHERE session_id='" + DefaultSessionID + "'");
            UNIT_ASSERT(session.Commit());
        }
        UNIT_ASSERT_VALUES_EQUAL(env.GetDebt(UserWith2Cards), 101000);
        ig.Set(now + TDuration::Minutes(20));
        env.WaitForFinish();
        UNIT_ASSERT_VALUES_EQUAL(env.GetDebt(UserWith2Cards), 0);
    }

    Y_UNIT_TEST(SimpleYandexAccountPay) {
        TBillingTestEnvironment env("SQLite");
        env.LinkYandexAccount(UserWithYandexAccount);
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, UserWithYandexAccount, EBillingType::CarUsage, 0, 500));
        env.AddMoney(DefaultSessionID, 900);
        {
            TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
            env.RunAllTasks();
            TVector<TBillingTask> tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 1);
        }

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().begin()->GetStatus(), NDrive::NTrustClient::EPaymentStatus::Started);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().begin()->GetSum(), 500);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().begin()->GetPaymentType(), NDrive::NBilling::EAccount::YAccount);
            UNIT_ASSERT(!snapshot.GetPayments().begin()->GetOrderId().empty());
        }

        TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
        env.WaitForFinish();

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 1);
        }

        TVector<TBillingTask> tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 1);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));

        tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetSum(), 400);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetCleared(), 400);
        }

        env.RunAllTasks(); // Remove closed tasks
        tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 0);
    }

    Y_UNIT_TEST(YandexAccountCardPay) {
        TBillingTestEnvironment env("SQLite");
        env.LinkYandexAccount(UserWithYandexAccount);
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, UserWithYandexAccount, EBillingType::CarUsage, 0, 50000));
        env.AddMoney(DefaultSessionID, 100000);
        {
            TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
            env.RunAllTasks();
        }

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Started);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetSum(), 25127);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetPaymentType(), NDrive::NBilling::EAccount::YAccount);
        }

        TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
        env.WaitForFinish();
        env.RunAllTasks();
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetSum(), 25127);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetPaymentType(), NDrive::NBilling::EAccount::YAccount);

            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Started);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetSum(), 50000);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetPaymentType(), NDrive::NBilling::EAccount::Trust);
        }
        env.WaitForFinish();

        TVector<TBillingTask> tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 1);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));

        tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
        env.WaitForFinish();
        env.RunAllTasks(); // Remove closed tasks
        tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 0);

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 3);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetSum(), 24873);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Authorized);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetCleared(), 0);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetPaymentType(), NDrive::NBilling::EAccount::Trust);
        }
    }

    Y_UNIT_TEST(YandexAccountCardFinishPay) {
        TBillingTestEnvironment env("SQLite");
        env.LinkYandexAccount(UserWithYandexAccount);
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, UserWithYandexAccount, EBillingType::CarUsage, 0, 500));
        env.AddMoney(DefaultSessionID, 700);
        {
            TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
            env.RunAllTasks();
        }

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Started);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetSum(), 500);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetPaymentType(), NDrive::NBilling::EAccount::YAccount);
        }

        TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
        env.WaitForFinish();
        env.AddMoney(DefaultSessionID, 30000);

        TVector<TBillingTask> tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 1);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));

        tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetSum(), 24627);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetPaymentType(), NDrive::NBilling::EAccount::YAccount);
        }
    }

    Y_UNIT_TEST(YandexAccountFullFinishPay) {
        TBillingTestEnvironment env("SQLite");
        env.LinkYandexAccount(UserWithYandexAccount);
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, UserWithYandexAccount, EBillingType::CarUsage, 0, 500));
        env.AddMoney(DefaultSessionID, 400);
        {
            TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
            env.RunAllTasks();
            TVector<TBillingTask> tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 1);
        }

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 0);
        }

        UNIT_ASSERT(env.FinishTask(DefaultSessionID));

        TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetSum(), 400);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(timeline.back().GetCleared(), 400);
        }

        env.RunAllTasks(); // Remove closed tasks
        TVector<TBillingTask> tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 0);
    }

    Y_UNIT_TEST(YandexAccountFailedFinishPay) {
        TBillingTestEnvironment env("SQLite");
        env.LinkYandexAccount(UserWithFailedYandexAccount);
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, UserWithFailedYandexAccount, EBillingType::CarUsage, 0, 500));
        env.AddMoney(DefaultSessionID, 400);
        {
            TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
            env.RunAllTasks();
            TVector<TBillingTask> tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 1);
        }

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 0);
        }

        UNIT_ASSERT(!env.FinishTask(DefaultSessionID));
        {
            TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(tasksBefore.front().GetState(), "finishing");
        }
    }

    Y_UNIT_TEST(YandexAccountFinishProcessing) {
        TBillingTestEnvironment env("SQLite");
        env.LinkYandexAccount(UserWithYandexAccount);
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, UserWithYandexAccount, EBillingType::CarUsage, 0, 500));
        env.AddMoney(DefaultSessionID, 900);
        {
            TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
            env.RunAllTasks();
            TVector<TBillingTask> tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 1);
        }

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().begin()->GetStatus(), NDrive::NTrustClient::EPaymentStatus::Started);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().begin()->GetSum(), 500);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().begin()->GetPaymentType(), NDrive::NBilling::EAccount::YAccount);
        }

        TVector<TBillingTask> tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 1);
        UNIT_ASSERT(!env.FinishTask(DefaultSessionID));
        {
            TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(tasksBefore.front().GetState(), "finishing");
        }
    }

    Y_UNIT_TEST(YandexAccountDeposit) {
        TBillingTestEnvironment env("SQLite");
        env.LinkYandexAccount(UserWithYandexAccount);
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, UserWithYandexAccount, EBillingType::CarUsage, 1000, 500));
        env.AddMoney(DefaultSessionID, 900);
        {
            TVector<TBillingTask> tasksBefore = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksBefore.size(), 1);
            env.RunAllTasks();
            TVector<TBillingTask> tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 1);
        }

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().begin()->GetStatus(), NDrive::NTrustClient::EPaymentStatus::Started);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().begin()->GetSum(), 1000);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().begin()->GetPaymentType(), NDrive::NBilling::EAccount::YAccount);
        }

        env.WaitForFinish();

        TVector<TBillingTask> tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 1);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().begin()->GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().begin()->GetCleared(), 1000);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().begin()->GetPaymentType(), NDrive::NBilling::EAccount::YAccount);
        }

        env.RunAllTasks();
        env.WaitRefunds(1);
        env.WaitForFinish(TDuration::Minutes(2), true);
        tasksAfter = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasksAfter.size(), 0);
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(timeline.begin()->GetSum(), 1000);
            UNIT_ASSERT_VALUES_EQUAL(timeline.begin()->GetCleared(), 900);
            UNIT_ASSERT_VALUES_EQUAL(timeline.begin()->GetPaymentType(), NDrive::NBilling::EAccount::YAccount);
            UNIT_ASSERT_VALUES_EQUAL(timeline.begin()->GetStatus(), NDrive::NTrustClient::EPaymentStatus::Refunded);
        }
    }

    Y_UNIT_TEST(SessionReport) {
        TBillingTestEnvironment env("SQLite");
        env.LinkBonusAccount(DefaultUserId);
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 1000));
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 0, 500));
        env.AddMoney(DefaultSessionID, 2000);

        {
            auto sessionReport = env.GetSessionReport(DefaultSessionID);
            UNIT_ASSERT(!sessionReport.Defined());
        }
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        //WaitBonuses
        env.WaitForFinish();
        //WaitCard
        env.WaitForFinish();
        {
            auto sessionReport = env.GetSessionReport(DefaultSessionID);
            UNIT_ASSERT(!!sessionReport.Defined());
            UNIT_ASSERT_VALUES_EQUAL(sessionReport->GetBill(), 2000);
            UNIT_ASSERT_VALUES_EQUAL(sessionReport->SumByAccount.size(), 2);
            for (auto&& accInfo : sessionReport->SumByAccount) {
                INFO_LOG << "Check " << accInfo.first << Endl;
                UNIT_ASSERT_VALUES_EQUAL(accInfo.second, 1000);
            }
            UNIT_ASSERT_VALUES_EQUAL(env.GetDebt(DefaultUserId), 0);
        }
    }

    Y_UNIT_TEST(SessionReportDepositCardError) {
        TBillingTestEnvironment env("SQLite", TDuration::Minutes(5));
        env.LinkBonusAccount(DefaultUserId);
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 1000));
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 1000 + NotAuthorizedSum, 500));
        //WaitBonuses
        env.WaitForFinish();
        //WaitCard
        env.WaitForFinish();
        UNIT_ASSERT_VALUES_EQUAL(env.GetDebt(DefaultUserId), NotAuthorizedSum);
        env.AddMoney(DefaultSessionID, 500);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));

        {
            auto sessionReport = env.GetSessionReport(DefaultSessionID);
            UNIT_ASSERT(!!sessionReport.Defined());
            UNIT_ASSERT_VALUES_EQUAL(sessionReport->GetBill(), 500);
            UNIT_ASSERT_VALUES_EQUAL(sessionReport->SumByAccount.size(), 1);
            for (auto&& accInfo : sessionReport->SumByAccount) {
                INFO_LOG << "Check " << accInfo.first << Endl;
                UNIT_ASSERT_VALUES_UNEQUAL(accInfo.first, 0);
                UNIT_ASSERT_VALUES_EQUAL(accInfo.second, 500);
            }
            UNIT_ASSERT_VALUES_EQUAL(env.GetDebt(DefaultUserId), 0);
        }
    }

    Y_UNIT_TEST(SessionReportDepositCardErrorLimitedBonuses) {
        TBillingTestEnvironment env("SQLite", TDuration::Minutes(5));
        env.LinkBonusAccount(DefaultUserId, "limited_bonuses");
        TInstant bonusDeadline = TInstant::Seconds(Now().Seconds()) + TDuration::Hours(2);
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 1000, bonusDeadline));
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 1000 + NotAuthorizedSum, 500));
        //WaitBonuses
        env.WaitForFinish();
        //WaitCard
        env.WaitForFinish();
        UNIT_ASSERT_VALUES_EQUAL(env.GetDebt(DefaultUserId), NotAuthorizedSum);
        TInstantGuard ig(bonusDeadline + TDuration::Minutes(1));
        {
            auto session = env.BuildSession();
            auto bonusAccount = env.GetAccount(DefaultUserId, "limited_bonuses");
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->Refresh(DefaultUserId, ModelingNow(), session), ERefreshStatus::Skip);
            UNIT_ASSERT(session.Commit());
        }
        env.AddMoney(DefaultSessionID, 500);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));

        {
            auto sessionReport = env.GetSessionReport(DefaultSessionID);
            UNIT_ASSERT(!!sessionReport.Defined());
            UNIT_ASSERT_VALUES_EQUAL(sessionReport->GetBill(), 500);
            UNIT_ASSERT_VALUES_EQUAL(sessionReport->SumByAccount.size(), 1);
            for (auto&& accInfo : sessionReport->SumByAccount) {
                INFO_LOG << "Check " << accInfo.first << Endl;
                UNIT_ASSERT_VALUES_UNEQUAL(accInfo.first, 0);
                UNIT_ASSERT_VALUES_EQUAL(accInfo.second, 500);
            }
            UNIT_ASSERT_VALUES_EQUAL(env.GetDebt(DefaultUserId), 0);
        }
    }

    Y_UNIT_TEST(TaskBoostingContext) {
        TBillingTestEnvironment env("SQLite", TDuration::Minutes(5), 1);
        TBillingManager& billingManager = env.GetManager();
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, UserWith2Cards, EBillingType::Ticket, 0, 0, NotAuthorizedSum));
        env.WaitForFinish(TDuration::Seconds(20));
        UNIT_ASSERT_VALUES_EQUAL(env.GetDebt(DefaultUserId), NotAuthorizedSum);
        TString paymethod;
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetFailedPayMethods().size(), 1);
            paymethod = *snapshot.GetFailedPayMethods().begin();
            UNIT_ASSERT(!!paymethod);
        }
        {
            auto session = billingManager.BuildSession(false);
            UNIT_ASSERT(billingManager.GetActiveTasksManager().BoostSession(DefaultSessionID, DefaultUserId, TBillingTask::TExecContext().SetPaymethod(paymethod), session));
            UNIT_ASSERT(session.Commit());
        }
        env.WaitForFinish(TDuration::Seconds(20));
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetFailedPayMethods().size(), 1);
        }
        {
            auto session = billingManager.BuildSession(false);
            UNIT_ASSERT(billingManager.GetActiveTasksManager().BoostSession(DefaultSessionID, DefaultUserId, TBillingTask::TExecContext().SetPaymethod("card-x9378"), session));
            UNIT_ASSERT(session.Commit());
        }
        env.WaitForFinish(TDuration::Seconds(20));
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetFailedPayMethods().size(), 2);
        }
    }

    Y_UNIT_TEST(ReleaseLessThanRub) {
        TBillingTestEnvironment env("SQLite", TDuration::Minutes(5), 2);
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 31415, 31415));
        {
            env.WaitForFinish(TDuration::Seconds(1));
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto payments = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(payments.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(payments.back().GetSum(), 31415);
        }
        env.AddMoney(DefaultSessionID, 31430);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        env.WaitForFinish(TDuration::Minutes(2));
        UNIT_ASSERT_VALUES_EQUAL(env.GetActualSnapshot(DefaultSessionID).GetPayments().size(), 1);
        env.RunAllTasks(); // Remove finished tasks
        UNIT_ASSERT_VALUES_EQUAL(env.GetActualSnapshot(DefaultSessionID).GetPayments().size(), 1);
        auto tasks = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasks.size(), 0);
    }

    Y_UNIT_TEST(CreateTicket) {
        TBillingTestEnvironment env("SQLite");
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::Ticket, 0, 0, 13000));
        env.WaitForFinish();
        TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().begin()->GetStatus(), NDrive::NTrustClient::EPaymentStatus::Authorized);
    }

    Y_UNIT_TEST(TicketPaymentError) {
        TBillingTestEnvironment env("SQLite", TDuration::Minutes(5));
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::Ticket, 0, 0, 109900));
        env.WaitForFinish();
        TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 0);
        UNIT_ASSERT_VALUES_EQUAL(snapshot.GetFailedPayMethods().size(), 1);
        TVector<TBillingTask> tasks = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(tasks.size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(tasks.front().GetTaskStatus(), EPaymentStatus::NoFunds);
    }

    Y_UNIT_TEST(SnapshotActuality) {
        TBillingTestEnvironment env("SQLite", TDuration::Minutes(5), 1);
        env.LinkWallet(DefaultUserId, "simple_limited");
        env.LinkBonusAccount(DefaultUserId);
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 2000));
        {
            auto session = env.BuildSession(false);
            auto offer = MakeAtomicShared<TFakeOffer>();
            offer->SetOfferId(DefaultSessionID).SetPaymentDiscretization(500);
            TVector<TString> accounts = { "simple_limited" , "bonus", "card" };
            offer->SetChargableAccounts(accounts);
            UNIT_ASSERT(env.GetManager().CreateBillingTask(DefaultUserId, offer, session));
            UNIT_ASSERT(session.Commit());
        }
        {
            env.AddMoney(DefaultSessionID, WalletHardLimit + 2000 + 1000);
            UNIT_ASSERT(env.FinishTask(DefaultSessionID));
            env.WaitForFinish(TDuration::Minutes(2), true);
            UNIT_ASSERT_VALUES_EQUAL(env.GetTasks().size(), 0);

            TCachedPayments snapshot1 = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot1.GetPayments().size(), 3);

            env.RunClearing();

            TCachedPayments snapshot2 = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot2.GetPayments().size(), 3);
        }

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 3);
            auto refunds = snapshot.CalculateRefunds(4000, {}, 0);
            UNIT_ASSERT(refunds);
            UNIT_ASSERT_VALUES_EQUAL(refunds->size(), 3);
            auto session = env.BuildSession(false);
            ui32 refundSum = 0;
            for (auto issue : refunds.GetRef()) {
                INFO_LOG << "Try refund " << issue.GetPayment().GetPaymentId() << Endl;
                refundSum += issue.GetSum();
                if (issue.GetPayment().GetPaymentType() != NDrive::NBilling::EAccount::Trust) {
                    UNIT_ASSERT(env.GetManager().ProcessFiscalRefundIssue(issue.GetPayment(), DefaultUserId, issue.GetSum(), true, session));
                }
            }
            UNIT_ASSERT_VALUES_EQUAL(refundSum, 4000);
            UNIT_ASSERT(session.Commit());
        }
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetTimeline().size(), 3);
            for (auto&& payment : snapshot.GetTimeline()) {
                if (payment.GetPaymentType() == NDrive::NBilling::EAccount::Bonus) {
                    UNIT_ASSERT_VALUES_EQUAL(payment.GetCleared(), 0);
                }
                if (payment.GetPaymentType() == NDrive::NBilling::EAccount::Wallet) {
                    UNIT_ASSERT_VALUES_EQUAL(payment.GetCleared(), 99000);
                }
                if (payment.GetPaymentType() == NDrive::NBilling::EAccount::Trust) {
                    UNIT_ASSERT_VALUES_EQUAL(payment.GetCleared(), 1000);
                }
            }
        }
    }

    Y_UNIT_TEST(SnapshotActualityLimitedBonuses) {
        TBillingTestEnvironment env("SQLite", TDuration::Minutes(5), 1);
        env.LinkWallet(DefaultUserId, "simple_limited");
        env.LinkBonusAccount(DefaultUserId, "limited_bonuses");
        TInstant bonusDeadline = TInstant::Seconds(Now().Seconds()) + TDuration::Hours(1);
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 1300));
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 700, bonusDeadline));
        {
            auto session = env.BuildSession(false);
            auto offer = MakeAtomicShared<TFakeOffer>();
            offer->SetOfferId(DefaultSessionID).SetPaymentDiscretization(500);
            TVector<TString> accounts = { "simple_limited" , "limited_bonuses", "card" };
            offer->SetChargableAccounts(accounts);
            UNIT_ASSERT(env.GetManager().CreateBillingTask(DefaultUserId, offer, session));
            UNIT_ASSERT(session.Commit());
        }
        {
            env.AddMoney(DefaultSessionID, WalletHardLimit + 2000 + 1000);
            UNIT_ASSERT(env.FinishTask(DefaultSessionID));
            env.WaitForFinish(TDuration::Minutes(2), true);
            UNIT_ASSERT_VALUES_EQUAL(env.GetTasks().size(), 0);

            TCachedPayments snapshot1 = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot1.GetPayments().size(), 3);

            env.RunClearing();

            TCachedPayments snapshot2 = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot2.GetPayments().size(), 3);
            for (auto&& payment : snapshot2.GetTimeline()) {
                if (payment.GetPaymentType() == NDrive::NBilling::EAccount::Bonus) {
                    UNIT_ASSERT_VALUES_EQUAL(payment.GetCleared(), 2000);
                    auto paymentMeta = dynamic_cast<const TBonusPaymentMeta*>(payment.GetMeta().Get());
                    UNIT_ASSERT_VALUES_UNEQUAL(paymentMeta, nullptr);
                    UNIT_ASSERT_VALUES_EQUAL(paymentMeta->GetLimitedBonuses().size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(paymentMeta->GetLimitedBonuses().begin()->GetBalance(), 700);
                }
                if (payment.GetPaymentType() == NDrive::NBilling::EAccount::Wallet) {
                    UNIT_ASSERT_VALUES_EQUAL(payment.GetCleared(), 100000);
                }
            }
        }

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 3);
            auto refunds = snapshot.CalculateRefunds(1200, {}, 0);
            UNIT_ASSERT(refunds);
            UNIT_ASSERT_VALUES_EQUAL(refunds->size(), 2);
            auto session = env.BuildSession(false);
            ui32 refundSum = 0;
            for (auto issue : refunds.GetRef()) {
                INFO_LOG << "Try refund " << issue.GetPayment().GetPaymentId() << Endl;
                refundSum += issue.GetSum();
                if (issue.GetPayment().GetPaymentType() != NDrive::NBilling::EAccount::Trust) {
                    UNIT_ASSERT(env.GetManager().ProcessFiscalRefundIssue(issue.GetPayment(), DefaultUserId, issue.GetSum(), false, session));
                }
            }
            UNIT_ASSERT_VALUES_EQUAL(refundSum, 1200);
            UNIT_ASSERT(session.Commit());
        }
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetTimeline().size(), 3);
            for (auto&& payment : snapshot.GetTimeline()) {
                if (payment.GetPaymentType() == NDrive::NBilling::EAccount::Bonus) {
                    UNIT_ASSERT_VALUES_EQUAL(payment.GetCleared(), 1800);
                    auto paymentMeta = dynamic_cast<const TBonusPaymentMeta*>(payment.GetMeta().Get());
                    UNIT_ASSERT_VALUES_UNEQUAL(paymentMeta, nullptr);
                    UNIT_ASSERT_VALUES_EQUAL(paymentMeta->GetLimitedBonuses().size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(paymentMeta->GetLimitedBonuses().begin()->GetBalance(), 700);
                }
                if (payment.GetPaymentType() == NDrive::NBilling::EAccount::Wallet) {
                    UNIT_ASSERT_VALUES_EQUAL(payment.GetCleared(), 100000);
                }
                if (payment.GetPaymentType() == NDrive::NBilling::EAccount::Trust) {
                    UNIT_ASSERT_VALUES_EQUAL(payment.GetCleared(), 1000);
                }
            }
        }
    }

    Y_UNIT_TEST(TasksPriority) {
        TBillingTestEnvironment env("SQLite", TDuration::Seconds(5), 1);
        TInstant now = TInstant::Now();
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::Ticket, 0, 0, 109900));
        env.WaitForFinish();
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetFailedPayMethods().size(), 1);
        }
        env.WaitForFinish();
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetFailedPayMethods().size(), 1);
        }

        const TBillingManager& manager = env.GetManager();
        {
            auto actualTask = manager.GetPaymentData(DefaultSessionID);
            UNIT_ASSERT(actualTask);
            UNIT_ASSERT_EQUAL(manager.GetTaskPriority(*actualTask, Now()), TBillingLogic::ETaskPriority::Ignore);
            UNIT_ASSERT_EQUAL(manager.GetTaskPriority(*actualTask, now + TDuration::Days(3)), TBillingLogic::ETaskPriority::RegularDebts);
            UNIT_ASSERT_VALUES_EQUAL(actualTask->GetBillingTask().GetTaskStatus(), EPaymentStatus::NoFunds);
            UNIT_ASSERT_EQUAL(manager.GetTaskPriority(*actualTask, now + TDuration::Days(41)), TBillingLogic::ETaskPriority::DeepDebts);
        }
    }

    Y_UNIT_TEST(SelectorAutoUpdate) {
        TBillingTestEnvironment env;
        env.SetSettings("billing.fast_tasks_selector", "1");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_settings_history");
        const TBillingManager& manager = env.GetManager();
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 20, 0));

        auto payments = manager.GetPriorityTasks(1000, TInstant::Now());
        UNIT_ASSERT_VALUES_EQUAL(payments.size(), 1);

        env.AddMoney(DefaultSessionID, 1000);
        payments = manager.GetPriorityTasks(1000, TInstant::Now());
        UNIT_ASSERT_VALUES_EQUAL(payments.size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(payments.front().GetBillingTask().GetBillCorrected(), 1000);
    }

    Y_UNIT_TEST(TooOldDebts) {
        TBillingTestEnvironment env("SQLite", TDuration::Seconds(5), 1);
        TInstant now = TInstant::Now();

        TInstant fixedLastUpdate = now - TDuration::Days(100);
        const auto period = TDuration::Days(7).Seconds();
        i64 hashDiff = fixedLastUpdate.Seconds() % period - now.Seconds() % period;
        if (hashDiff >= 0) {
            fixedLastUpdate -= TDuration::Seconds(SafeIntegerCast<ui64>(hashDiff) + 1);
        }

        TInstantGuard ig(fixedLastUpdate);

        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::Ticket, 0, 0, NotAuthorizedSum));

        env.WaitForFinish();
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetFailedPayMethods().size(), 1);
        }

        const TBillingManager& manager = env.GetManager();
        {
            auto actualTask = manager.GetPaymentData(DefaultSessionID);
            UNIT_ASSERT(actualTask);
            UNIT_ASSERT(actualTask->GetBillingTask().IsDebt());
            UNIT_ASSERT_EQUAL(manager.GetTaskPriority(*actualTask, now), TBillingLogic::ETaskPriority::DeepDebts);
        }
        ig.Set(now);
        {
            env.WaitForFinish();
            TVector<TBillingTask> actualTasks = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT(!actualTasks.empty());
            UNIT_ASSERT_VALUES_EQUAL(actualTasks.front().GetLastUpdate().Seconds(), fixedLastUpdate.Seconds());
        }

        TTimeRestriction timeRestriction;
        auto currentHour = now.Hours() % 24;
        const ui32 startTime = currentHour == 0 ? 0 : (currentHour - 1) * 100;
        const ui32 finishTime = currentHour == 23 ? 2400 : (currentHour + 1) * 100;
        timeRestriction.SetTimeRestriction(startTime, finishTime);
        env.SetSettings("billing.deep_debt.time_restriction", NJson::WriteJson(timeRestriction.SerializeToJson()));
        env.SetSettings("billing.min_task_priority", ToString((ui32)TBillingLogic::ETaskPriority::DeepDebts));
        env.SetSettings("billing.deep_debt.limit", "1");
        {
            env.WaitForFinish(TDuration::Seconds(10), true);
            TVector<TBillingTask> actualTasks = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT(!actualTasks.empty());
            UNIT_ASSERT_VALUES_UNEQUAL(actualTasks.front().GetLastUpdate().Seconds(), fixedLastUpdate.Seconds());
        }
    }

    Y_UNIT_TEST(TraverseFailedMethods) {
        auto filter = [](const TCachedPayments::TErrorInfo& errorInfo) {
            return errorInfo.Date <= ModelingNow() - TDuration::Days(7);
        };
        TCachedPayments snapshot;
        const TString firstPayMethod = "card-x3366";
        {
            NDrive::NTrustClient::TPayment payment;
            payment.PaymentStatus = NDrive::NTrustClient::EPaymentStatus::NotAuthorized;
            payment.PaymethodId = firstPayMethod;
            UNIT_ASSERT(snapshot.Update(TPaymentTask::CreateFromPayment(payment, Now())));
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetFailedPayMethods().size(), 1);
            UNIT_ASSERT(snapshot.GetFailedPayMethodsQueue(filter).empty());
        }

        const TString payMethod = "card-x8823";
        {
            NDrive::NTrustClient::TPayment payment;
            payment.PaymentStatus = NDrive::NTrustClient::EPaymentStatus::NotAuthorized;
            payment.PaymethodId = payMethod;
            payment.PaymentError = "expired_card";
            UNIT_ASSERT(snapshot.Update(TPaymentTask::CreateFromPayment(payment, Now() - TDuration::Days(9))));
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetFailedPayMethods().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetFailedPayMethodsQueue(filter).size(), 1);
        }

        const TString secondPayMethod = "card-x9378";
        {
            NDrive::NTrustClient::TPayment payment;
            payment.PaymentStatus = NDrive::NTrustClient::EPaymentStatus::NotAuthorized;
            payment.PaymethodId = secondPayMethod;
            UNIT_ASSERT(snapshot.Update(TPaymentTask::CreateFromPayment(payment, Now() - TDuration::Days(8))));
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetFailedPayMethods().size(), 3);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetFailedPayMethodsQueue(filter).size(), 2);
        }

        TVector<TString> failedPayMethods;
        {
            failedPayMethods = snapshot.GetFailedPayMethodsQueue(filter);
            UNIT_ASSERT_VALUES_EQUAL(failedPayMethods.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(failedPayMethods.front(), payMethod);
        }

        TSet<TString> failedCards;
        {
            failedCards = snapshot.GetFailedPayMethods();
            UNIT_ASSERT_VALUES_EQUAL(failedCards.size(), 3);
        }

        TVector<NDrive::NTrustClient::TPaymentMethod> methods;
        {
            NJson::TJsonValue json;
            UNIT_ASSERT(NJson::ReadJsonFastTree(ReplyTwoCards, &json));
            for (auto&& payment : json["bound_payment_methods"].GetArray()) {
                NDrive::NTrustClient::TPaymentMethod method;
                method.FromJson(payment);
                methods.push_back(method);
            }
        }

        {
            TVector<TString> cardsList = NDrive::NBilling::BuildCardList(methods, "", failedCards, failedPayMethods);
            UNIT_ASSERT_VALUES_EQUAL(cardsList.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(cardsList.front(), secondPayMethod);
        }

        {
            TVector<TString> cardsList = NDrive::NBilling::BuildCardList(methods, firstPayMethod, failedCards, failedPayMethods);
            UNIT_ASSERT_VALUES_EQUAL(cardsList.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(cardsList.front(), secondPayMethod);
        }

        failedPayMethods.push_back(firstPayMethod);
        {
            TVector<TString> cardsList = NDrive::NBilling::BuildCardList(methods, firstPayMethod, failedCards, failedPayMethods);
            UNIT_ASSERT_VALUES_EQUAL(cardsList.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(cardsList.front(), firstPayMethod);
        }

        {
            NDrive::NTrustClient::TPayment payment;
            payment.PaymentStatus = NDrive::NTrustClient::EPaymentStatus::NotAuthorized;
            payment.PaymethodId = secondPayMethod;
            UNIT_ASSERT(snapshot.Update(TPaymentTask::CreateFromPayment(payment, Now())));
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetFailedPayMethods().size(), 3);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetFailedPayMethodsQueue(filter).size(), 1);
        }
    }
}

Y_UNIT_TEST_SUITE(RefundSuite) {

    Y_UNIT_TEST(CancelPayment) {
        TBillingTestEnvironment env("SQLite");
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 0, 500));
        env.AddMoney(DefaultSessionID, 1000);
        env.WaitForFinish();

        {
            auto session = env.BuildSession(false);
            session->Exec("update clearing_tasks set refund=500");
            UNIT_ASSERT(session.Commit());
        }

        {
            auto session = env.BuildSession();
            TVector<TClearingTask> clearing = env.GetManager().GetClearingTasks(TDuration::Zero(), session).GetRef();
            UNIT_ASSERT_VALUES_EQUAL(clearing.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(clearing.front().GetRefund(), clearing.front().GetSum());
        }

        ui32 tCount = env.RunClearing(1);
        UNIT_ASSERT_VALUES_EQUAL(tCount, 0);

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().begin()->GetStatus(), NDrive::NTrustClient::EPaymentStatus::Canceled);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().begin()->GetCleared(), 0);
        }
    }

    Y_UNIT_TEST(RefundNotCleared) {
        TBillingTestEnvironment env("SQLite");
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 0, 500));
        env.AddMoney(DefaultSessionID, 13000);
        env.WaitForFinish();
        env.AddMoney(DefaultSessionID, 34000);
        env.WaitForFinish();


        {
            TCachedPayments payments = env.GetActualSnapshot(DefaultSessionID);
            auto session = env.BuildSession(false);
            for (auto&& pay : payments.GetPayments()) {
                UNIT_ASSERT(env.GetManager().ProcessFiscalRefundIssue(pay, DefaultUserId, 12500, true, session));
            }
            UNIT_ASSERT(session.Commit());
        }

        {
            auto session = env.BuildSession(true);
            auto clearing = env.GetManager().GetClearingTasks(TDuration::Zero(), session).GetRef();
            UNIT_ASSERT_VALUES_EQUAL(clearing.size(), 2);
            for (auto&& task : clearing) {
                UNIT_ASSERT(task.GetSum() >= task.GetRefund());
                UNIT_ASSERT_VALUES_EQUAL(task.GetRefund(), 12500);
            }
        }

        UNIT_ASSERT_VALUES_EQUAL(env.RunClearing(), 0);

        {
            auto session = env.BuildSession(true);
            TRecordsSet payments;
            session->Exec("SELECT * FROM drive_payments WHERE session_id='" + DefaultSessionID + "'", &payments);
            UNIT_ASSERT_VALUES_EQUAL(payments.GetRecords().size(), 2);
            for (auto&& pay : payments.GetRecords()) {
                TPaymentTask pTask;
                pTask.DeserializeFromTableRecord(pay, nullptr);
                if (pTask.GetStatus() == NDrive::NTrustClient::EPaymentStatus::Canceled) {
                    UNIT_ASSERT_VALUES_EQUAL(pTask.GetCleared(), 0);
                } else if (pTask.GetStatus() == NDrive::NTrustClient::EPaymentStatus::Cleared) {
                    UNIT_ASSERT_VALUES_EQUAL(pTask.GetCleared(), 21000 - 12500);
                }
            }
        }
        {
            auto refunds = env.GetSessionRefunds(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(refunds.size(), 2);
        }
    }

    Y_UNIT_TEST(RefundCleared) {
        TBillingTestEnvironment env("SQLite");
        const TBillingManager& manager = env.GetManager();
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 0, 500));

        env.AddMoney(DefaultSessionID, 13000);
        env.WaitForFinish();
        env.AddMoney(DefaultSessionID, 34000);
        env.WaitForFinish();
        env.RunClearing();

        {
            TCachedPayments payments = env.GetActualSnapshot(DefaultSessionID);
            auto session = env.BuildSession(false);
            for (auto&& pay : payments.GetPayments()) {
                UNIT_ASSERT(manager.ProcessFiscalRefundIssue(pay, DefaultUserId, 10000, true, session));
            }
            UNIT_ASSERT(session.Commit());
        }

        env.WaitRefunds(2);
        {
            auto refunds = env.GetSessionRefunds(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(refunds.size(), 2);
        }
    }

    Y_UNIT_TEST(RefundWallet) {
        TBillingTestEnvironment env("SQLite", TDuration::Seconds(1), 1);
        env.LinkWallet(DefaultUserId, "simple_limited");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            auto session = env.BuildSession(false);
            auto offer = MakeAtomicShared<TFakeOffer>((ui32)10000);
            offer->SetOfferId(DefaultSessionID).SetPaymentDiscretization(500);
            TVector<TString> accounts;
            accounts.push_back("simple_limited");
            offer->SetChargableAccounts(accounts);
            UNIT_ASSERT(env.GetManager().CreateBillingTask(DefaultUserId, offer, session));
            UNIT_ASSERT(session.Commit());
        }
        env.WaitForFinish(TDuration::Seconds(5));
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetCleared(), 10000);
        }
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        env.WaitForFinish(TDuration::Seconds(2000), true);
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetCleared(), 0);
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        auto wallet = env.GetAccount(DefaultUserId, "simple_limited");
        UNIT_ASSERT_VALUES_EQUAL(wallet->GetBalance(), WalletSoftLimit);
        UNIT_ASSERT_VALUES_EQUAL(env.GetTasks().size(), 0);
        {
            auto refunds = env.GetSessionRefunds(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(refunds.size(), 0);
        }
    }

    Y_UNIT_TEST(RefundLimitedBonuses) {
        TBillingTestEnvironment env("SQLite", TDuration::Seconds(1), 1);
        env.LinkBonusAccount(DefaultUserId, "limited_bonuses");
        TInstant deadline = TInstant::Seconds(Now().Seconds()) + TDuration::Hours(2);
        TString source = "test";
        env.AddBonuses(DefaultUserId, 10000, deadline, source);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            auto session = env.BuildSession(false);
            auto offer = MakeAtomicShared<TFakeOffer>((ui32)10000);
            offer->SetOfferId(DefaultSessionID).SetPaymentDiscretization(500);
            UNIT_ASSERT(env.GetManager().CreateBillingTask(DefaultUserId, offer, session));
            UNIT_ASSERT(session.Commit());
        }
        env.WaitForFinish(TDuration::Seconds(5));
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetCleared(), 10000);
            UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 0);
            UNIT_ASSERT_VALUES_EQUAL(env.GetLimitedBonuses(DefaultUserId).size(), 0);
        }
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        env.WaitForFinish(TDuration::Seconds(2000), true);
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetCleared(), 0);
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        UNIT_ASSERT_VALUES_EQUAL(env.GetTasks().size(), 0);
        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 10000);
        UNIT_ASSERT_VALUES_EQUAL(env.GetLimitedBonuses(DefaultUserId).size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(env.GetLimitedBonuses(DefaultUserId).begin()->GetBalance(), 10000);
        UNIT_ASSERT_VALUES_EQUAL(env.GetLimitedBonuses(DefaultUserId).begin()->GetDeadline(), deadline);
        UNIT_ASSERT_VALUES_EQUAL(env.GetLimitedBonuses(DefaultUserId).begin()->GetSource(), source);
    }

    Y_UNIT_TEST(RefundLimitedBonusesAfterDeadline) {
        TBillingTestEnvironment env("SQLite", TDuration::Seconds(1), 1);
        env.LinkBonusAccount(DefaultUserId, "limited_bonuses");
        TInstant deadline = TInstant::Seconds(Now().Seconds()) + TDuration::Hours(2);
        TString source = "test";
        env.AddBonuses(DefaultUserId, 10000, deadline, source);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            auto session = env.BuildSession(false);
            auto offer = MakeAtomicShared<TFakeOffer>((ui32)10000);
            offer->SetOfferId(DefaultSessionID).SetPaymentDiscretization(500);
            UNIT_ASSERT(env.GetManager().CreateBillingTask(DefaultUserId, offer, session));
            UNIT_ASSERT(session.Commit());
        }
        env.WaitForFinish(TDuration::Seconds(5));
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetCleared(), 10000);
            UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 0);
            UNIT_ASSERT_VALUES_EQUAL(env.GetLimitedBonuses(DefaultUserId).size(), 0);
        }
        TInstantGuard ig(deadline + TDuration::Minutes(1));
        {
            auto session = env.BuildSession();
            auto bonusAccount = env.GetAccount(DefaultUserId, "limited_bonuses");
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->Refresh(DefaultUserId, ModelingNow(), session), ERefreshStatus::Skip);
            UNIT_ASSERT(session.Commit());
        }
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        env.WaitForFinish(TDuration::Seconds(2000), true);
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto timeline = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(timeline.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(timeline.front().GetCleared(), 0);
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        UNIT_ASSERT_VALUES_EQUAL(env.GetTasks().size(), 0);
        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 10000);
        UNIT_ASSERT_VALUES_EQUAL(env.GetLimitedBonuses(DefaultUserId).size(), 1);
        {
            auto session = env.BuildSession();
            auto bonusAccount = env.GetAccount(DefaultUserId, "limited_bonuses");
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->Refresh(DefaultUserId, ModelingNow(), session), ERefreshStatus::Ok);
            UNIT_ASSERT(session.Commit());

            UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 0);
            UNIT_ASSERT_VALUES_EQUAL(env.GetLimitedBonuses(DefaultUserId).size(), 0);
        }
    }
}

Y_UNIT_TEST_SUITE(DepositSuite) {
    Y_UNIT_TEST(DepositBonusCancel) {
        TBillingTestEnvironment env("SQLite");
        env.LinkBonusAccount(DefaultUserId);
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 2000));
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 1000, 500));
        env.WaitForFinish(TDuration::Seconds(1));

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto payments = snapshot.GetTimeline();

            UNIT_ASSERT_VALUES_EQUAL(payments.size(), 1);

            UNIT_ASSERT_VALUES_EQUAL(payments.front().GetSum(), 1000);
            UNIT_ASSERT_VALUES_EQUAL(payments.front().GetCleared(), 1000);
            UNIT_ASSERT_VALUES_EQUAL(payments.front().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(payments.front().GetBillingType(), EBillingType::CarUsage);

            auto bTasks = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(bTasks.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(bTasks.front().GetTaskStatus(), EPaymentStatus::Ok);
        }
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        env.WaitForFinish(TDuration::Seconds(1));

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto payments = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(payments.size(), 1);

            UNIT_ASSERT_VALUES_EQUAL(payments.front().GetCleared(), 0);

        //    auto bTasks = env.GetBillingTasksBySession(DefaultSessionID);
        //    UNIT_ASSERT_VALUES_EQUAL(bTasks.size(), 1);
        //    UNIT_ASSERT_VALUES_EQUAL(bTasks.front().GetTaskStatus(), EPaymentStatus::RefundDeposit);
        }

        env.WaitForFinish(TDuration::Seconds(1));
        auto bTasks = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(bTasks.size(), 0);
    }

    Y_UNIT_TEST(DepositBonusCancelPartial) {
        TBillingTestEnvironment env("SQLite", TDuration::Seconds(0), 1);
        env.LinkBonusAccount(DefaultUserId);
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 2000));
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 1000, 500));

        env.WaitForFinish(TDuration::Seconds(1), true);
        env.AddMoney(DefaultSessionID, 500);

        //Finish task
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        env.WaitForFinish(TDuration::Seconds(1), true);

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto payments = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(payments.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(payments.front().GetCleared(), 500);

       //     auto bTasks = env.GetBillingTasksBySession(DefaultSessionID);
       //     UNIT_ASSERT_VALUES_EQUAL(bTasks.size(), 1);
       //     UNIT_ASSERT_VALUES_EQUAL(bTasks.front().GetTaskStatus(), EPaymentStatus::RefundDeposit);
        }
        env.WaitForFinish(TDuration::Seconds(10), true);
        auto bTasks = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(bTasks.size(), 0);
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto payments = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(payments.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(payments.front().GetCleared(), 500);
        }
    }

    Y_UNIT_TEST(DepositLimitedBonusCancelPartial) {
        TBillingTestEnvironment env("SQLite", TDuration::Seconds(0), 1);
        env.LinkBonusAccount(DefaultUserId, "limited_bonuses");
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 1000));
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 700, Now() + TDuration::Hours(1)));
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 1000, 500));

        env.WaitForFinish(TDuration::Seconds(1), true);
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto payments = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(payments.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(payments.front().GetSum(), 1000);
            auto paymentMeta = dynamic_cast<const TBonusPaymentMeta*>(payments.front().GetMeta().Get());
            UNIT_ASSERT_VALUES_UNEQUAL(paymentMeta, nullptr);
            UNIT_ASSERT_VALUES_EQUAL(paymentMeta->GetLimitedBonuses().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(paymentMeta->GetLimitedBonuses().begin()->GetBalance(), 700);
        }
        env.AddMoney(DefaultSessionID, 500);

        //Finish task
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        env.WaitForFinish(TDuration::Seconds(1), true);

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto payments = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(payments.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(payments.front().GetCleared(), 500);
            auto paymentMeta = dynamic_cast<const TBonusPaymentMeta*>(payments.front().GetMeta().Get());
            UNIT_ASSERT_VALUES_UNEQUAL(paymentMeta, nullptr);
            UNIT_ASSERT_VALUES_EQUAL(paymentMeta->GetLimitedBonuses().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(paymentMeta->GetLimitedBonuses().begin()->GetBalance(), 500);

        //    auto bTasks = env.GetBillingTasksBySession(DefaultSessionID);
        //    UNIT_ASSERT_VALUES_EQUAL(bTasks.size(), 1);
        //    UNIT_ASSERT_VALUES_EQUAL(bTasks.front().GetTaskStatus(), EPaymentStatus::RefundDeposit);
        }
        env.WaitForFinish(TDuration::Seconds(10), true);
        auto bTasks = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(bTasks.size(), 0);
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto payments = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(payments.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(payments.front().GetCleared(), 500);
        }
        UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 1200);
        UNIT_ASSERT_VALUES_EQUAL(env.GetLimitedBonuses(DefaultUserId).size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(env.GetLimitedBonuses(DefaultUserId).begin()->GetBalance(), 200);
    }

    Y_UNIT_TEST(DepositBonusGet) {
        TBillingTestEnvironment env("SQLite");
        env.LinkBonusAccount(DefaultUserId);
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 2000));
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 1000, 500));

        // Wait deposit
        env.WaitForFinish(TDuration::Seconds(1));

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto payments = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(payments.size(), 1);
        }
        env.AddMoney(DefaultSessionID, 2000);
        // Finish task
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        env.WaitForFinish(TDuration::Seconds(1), true);

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto payments = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(payments.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(payments.front().GetCleared(), 1000);
            UNIT_ASSERT_VALUES_EQUAL(payments.back().GetCleared(), 1000);

            auto bTasks = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(bTasks.size(), 0);
        }
    }


    Y_UNIT_TEST(GetDepositError) {
        TBillingTestEnvironment env("SQLite");
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, UserWithoutCards, EBillingType::CarUsage, 1000, 500));

        env.WaitForFinish(TDuration::Seconds(1));

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto payments = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(payments.size(), 0);
            auto bTasks = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(bTasks.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(bTasks.front().GetTaskStatus(), EPaymentStatus::NoCards);
        }
    }

    Y_UNIT_TEST(BonusAndCardDeposit) {
        TBillingTestEnvironment env("SQLite");
        env.LinkBonusAccount(DefaultUserId);
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 1000));
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 2000, 500));

        env.WaitForFinish(TDuration::Minutes(2));
        env.WaitForFinish(TDuration::Minutes(2));

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto payments = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(payments.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(payments.front().GetPaymentType(), NDrive::NBilling::EAccount::Bonus);
            UNIT_ASSERT_VALUES_EQUAL(payments.back().GetPaymentType(), NDrive::NBilling::EAccount::Trust);
            auto bTasks = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(bTasks.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(bTasks.front().GetTaskStatus(), EPaymentStatus::Ok);
        }
        env.AddMoney(DefaultSessionID, 3000);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));

        env.WaitForFinish();
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto payments = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(payments.size(), 3);
        }
    }

    Y_UNIT_TEST(BonusAndCardDepositCancel) {
        TBillingTestEnvironment env("SQLite");
        env.LinkBonusAccount(DefaultUserId);
        TBillingManager& billingManager = env.GetManager();
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 1000));
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 2000, 500));

        env.WaitForFinish(TDuration::Minutes(2));
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        env.WaitForFinish();

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto payments = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(payments.size(), 2);

            UNIT_ASSERT_VALUES_EQUAL(payments.front().GetCleared(), 0);

            auto session = env.BuildSession(false);
            TVector<TClearingTask> clearing = billingManager.GetClearingTasks(TDuration::Zero(), session).GetRef();
            UNIT_ASSERT_VALUES_EQUAL(clearing.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(clearing.back().GetRefund(), 1000);

            auto bTasks = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(bTasks.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(bTasks.front().GetTaskStatus(), EPaymentStatus::RefundDeposit);
        }
    }


    Y_UNIT_TEST(CardDepositCancel) {
        TBillingTestEnvironment env("SQLite");
        TBillingManager& billingManager = env.GetManager();
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 1000, 500));
        env.WaitForFinish();
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        env.RunAllTasks();
        {
            auto session = billingManager.BuildSession(false);
            auto clearing = billingManager.GetClearingTasks(TDuration::Zero(), session).GetRef();
            UNIT_ASSERT_VALUES_EQUAL(clearing.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(clearing.back().GetSum(), clearing.back().GetRefund());
            UNIT_ASSERT(session.Commit());
        }
        ui32 result = env.RunClearing(1);
        UNIT_ASSERT_VALUES_EQUAL(result, 0);

        {
            auto session = billingManager.BuildSession(true);
            TRecordsSet payments;
            session->Exec("SELECT * FROM drive_payments WHERE session_id='" + DefaultSessionID + "'", &payments);
            UNIT_ASSERT_VALUES_EQUAL(payments.GetRecords().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(payments.GetRecords().front().Get("status"), ::ToString(NDrive::NTrustClient::EPaymentStatus::Canceled));
            UNIT_ASSERT_VALUES_EQUAL(payments.GetRecords().front().Get("cleared"), "0");
        }
    }

    Y_UNIT_TEST(CancelDepositComplex) {
        TBillingTestEnvironment env("SQLite");
        env.LinkBonusAccount(DefaultUserId);
        TBillingManager& billingManager = env.GetManager();
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 1000));
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 2000, 500));

        env.WaitForFinish();
        env.AddMoney(DefaultSessionID, 500);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));

        env.WaitForFinish();

        {
            auto session = billingManager.BuildSession(true);
            TVector<TClearingTask> clearing = billingManager.GetClearingTasks(TDuration::Zero(), session).GetRef();
            UNIT_ASSERT_VALUES_EQUAL(clearing.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(clearing.back().GetSum(), clearing.back().GetRefund());
        }
        ui32 result = env.RunClearing(1);
        UNIT_ASSERT_VALUES_EQUAL(result, 0);

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto payments = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(payments.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(payments.back().GetCleared(), 0);
            UNIT_ASSERT_VALUES_EQUAL(payments.back().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Canceled);
            UNIT_ASSERT_VALUES_EQUAL(payments.front().GetCleared(), 500);
            UNIT_ASSERT_VALUES_EQUAL(payments.front().GetPaymentType(), NDrive::NBilling::EAccount::Bonus);
            UNIT_ASSERT_VALUES_EQUAL(payments.front().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);

            UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 500);
        }

        env.WaitForFinish();

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto payments = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(payments.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(payments[0].GetCleared(), 500);
            UNIT_ASSERT_VALUES_EQUAL(payments[0].GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(payments[0].GetPaymentType(), NDrive::NBilling::EAccount::Bonus);

            UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 500);
        }
    }

    Y_UNIT_TEST(CancelDepositPartialCard) {
        TBillingTestEnvironment env("SQLite");
        env.LinkBonusAccount(DefaultUserId);
        TBillingManager& billingManager = env.GetManager();
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 1000));
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 2000, 500));

        env.WaitForFinish();
        env.AddMoney(DefaultSessionID, 1500);
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        env.WaitForFinish();

        {
            auto session = billingManager.BuildSession(false);
            TVector<TClearingTask> clearing = billingManager.GetClearingTasks(TDuration::Zero(), session).GetRef();
            UNIT_ASSERT_VALUES_EQUAL(clearing.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(clearing.back().GetRefund(), 1000);
            UNIT_ASSERT(session.Commit());
        }

        ui32 result = env.RunClearing(1);
        UNIT_ASSERT_VALUES_EQUAL(result, 0);

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto payments = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(payments.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(payments.back().GetCleared(), 0);
            UNIT_ASSERT_VALUES_EQUAL(payments.back().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Canceled);
            UNIT_ASSERT_VALUES_EQUAL(payments.front().GetCleared(), 1000);
            UNIT_ASSERT_VALUES_EQUAL(payments.front().GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(payments.front().GetPaymentType(), NDrive::NBilling::EAccount::Bonus);
            UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 0);
        }

        env.WaitForFinish();

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto payments = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(payments.size(), 3);
            UNIT_ASSERT_VALUES_EQUAL(payments[2].GetSum(), 500);
            UNIT_ASSERT_VALUES_EQUAL(payments[2].GetStatus(), NDrive::NTrustClient::EPaymentStatus::Authorized);
            UNIT_ASSERT_VALUES_EQUAL(payments[0].GetCleared(), 1000);
            UNIT_ASSERT_VALUES_EQUAL(payments[0].GetStatus(), NDrive::NTrustClient::EPaymentStatus::Cleared);
            UNIT_ASSERT_VALUES_EQUAL(env.GetBonuses(DefaultUserId), 0);
        }
    }

    Y_UNIT_TEST(FailedDepositRefund) {
        TBillingTestEnvironment env("SQLite", TDuration::Seconds(0), 1);
        env.LinkBonusAccount(DefaultUserId);

        TInstant now = Now() - TDuration::Hours(10);
        TInstantGuard ig(now);
        UNIT_ASSERT(env.AddBonuses(DefaultUserId, 2000));
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 1000, 500));

        env.WaitForFinish(TDuration::Seconds(1), true);

        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetTimeline().size(), 1);
        }
        UNIT_ASSERT(env.FinishTask(DefaultSessionID));
        env.WaitForFinish(TDuration::Seconds(1), true);
        {
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto payments = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(payments.size(), 1);

            UNIT_ASSERT_VALUES_EQUAL(payments.front().GetCleared(), 0);

            auto bTasks = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(bTasks.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(bTasks.front().GetTaskStatus(), EPaymentStatus::RefundDeposit);
        }

        env.WaitForFinish(TDuration::Seconds(1), true);
        auto bTasks = env.GetBillingTasksBySession(DefaultSessionID);
        UNIT_ASSERT_VALUES_EQUAL(bTasks.size(), 0);
    }
}

Y_UNIT_TEST_SUITE(BillingTaskQueueSuite) {
    Y_UNIT_TEST(SimpleQueueSwitch) {
        TBillingTestEnvironment env("SQLite", TDuration::Seconds(5), 1);
        TBillingManager& manager = env.GetManager();
        {
            UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 0, 500, 1000));
            TVector<TBillingTask> actualTasks = env.GetBillingTasksBySession(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(actualTasks.size(), 1);
            actualTasks.front().SetNextQueue(EBillingQueue::Prestable);
            auto session = manager.BuildSession();
            UNIT_ASSERT(manager.UpdateBillingTask(actualTasks.front(), "unittest_init", session));
            UNIT_ASSERT(session.Commit());
        }

        {
            env.RunAllTasks();
            auto actualTask = manager.GetPaymentData(DefaultSessionID);
            UNIT_ASSERT(actualTask);

            auto& billingTask = actualTask->GetBillingTask();
            UNIT_ASSERT_VALUES_EQUAL(billingTask.GetQueue(), EBillingQueue::Prestable);
            UNIT_ASSERT_VALUES_EQUAL(billingTask.GetTaskStatus(), EPaymentStatus::NotStarted);
            UNIT_ASSERT(actualTask->GetSnapshot().GetPayments().empty());
        }
    }

    Y_UNIT_TEST(StartedQueueSwitch) {
        TBillingTestEnvironment env("SQLite", TDuration::Seconds(5), 1);
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 0, 500, 1000));

        {
            env.RunAllTasks();
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().begin()->GetStatus(), NDrive::NTrustClient::EPaymentStatus::Started);
            UNIT_ASSERT_VALUES_UNEQUAL(snapshot.GetProcessingPayment(), nullptr);
        }

        TBillingManager& manager = env.GetManager();
        {
            auto task = env.GetBillingTaskByDefaultSession();
            task.SetNextQueue(EBillingQueue::Prestable);
            auto session = manager.BuildSession();
            UNIT_ASSERT(manager.UpdateBillingTask(task, "unittest_init", session));
            UNIT_ASSERT(session.Commit());
        }

        {
            env.RunAllTasks();
            UNIT_ASSERT_VALUES_EQUAL(env.GetBillingTaskByDefaultSession().GetQueue(), env.Config.GetActiveQueue());
        }
    }

    Y_UNIT_TEST(NotStartedQueueSwitch) {
        TBillingTestEnvironment env("SQLite", TDuration::Seconds(5), 1);
        UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 0, 500, 1000));
        TBillingManager& manager = env.GetManager();

        {
            auto session = manager.BuildSession();
            auto task = env.GetBillingTaskByDefaultSession();
            task.SetNextQueue(EBillingQueue::Prestable);
            UNIT_ASSERT(manager.UpdateBillingTask(task, "unittest_init", session));
            UNIT_ASSERT(session.Commit());
        }

        {
            ui64 ts = ModelingNow().Seconds();
            NStorage::TTableRecord record;
            record.Set("sum", 500);
            record.Set("billing_type", ToString(env.GetBillingTaskByDefaultSession().GetBillingType()));
            record.Set("payment_type", ToString(NDrive::NBilling::EAccount::Trust));
            record.Set("last_update_ts", ts);
            record.Set("created_at_ts", ts);
            record.Set("session_id", DefaultSessionID);
            record.Set("status", NDrive::NTrustClient::EPaymentStatus::NotStarted);
            record.Set("pay_method", "card");
            const TString paymentId = NUnitTest::RandomString(DefaultUserId.size());
            record.Set("payment_id", paymentId);
            TPaymentTask paymentTask;
            auto session = manager.BuildSession(false);
            UNIT_ASSERT_VALUES_EQUAL(manager.GetPaymentsManager().RegisterPayment(record, paymentTask, session), EDriveOpResult::Ok);
            UNIT_ASSERT(session.Commit());
            TPaymentInfo paymentInfo(500, "card");
            paymentInfo.Token = paymentId;
            env.TrustEmulator.RegisterPayment(paymentInfo);
        }

        {
            auto actualTask = manager.GetPaymentData(DefaultSessionID);
            UNIT_ASSERT(actualTask);

            UNIT_ASSERT_VALUES_EQUAL(actualTask->GetSnapshot().GetLastStatus(), NDrive::NTrustClient::EPaymentStatus::NotStarted);
            UNIT_ASSERT_VALUES_UNEQUAL(actualTask->GetSnapshot().GetProcessingPayment(), nullptr);
            UNIT_ASSERT_VALUES_EQUAL(actualTask->GetSnapshot().GetProcessingPayment()->GetStatus(), NDrive::NTrustClient::EPaymentStatus::NotStarted);
        }

        {
            env.RunAllTasks();
            auto actualTask = manager.GetPaymentData(DefaultSessionID);
            UNIT_ASSERT(actualTask);

            UNIT_ASSERT_VALUES_EQUAL(actualTask->GetBillingTask().GetQueue(), manager.GetConfig().GetActiveQueue());
            UNIT_ASSERT_VALUES_UNEQUAL(actualTask->GetSnapshot().GetProcessingPayment(), nullptr);
            UNIT_ASSERT_VALUES_EQUAL(actualTask->GetSnapshot().GetProcessingPayment()->GetStatus(), NDrive::NTrustClient::EPaymentStatus::Started);
        }
    }

    Y_UNIT_TEST(HardQueueSwitch) {
        TBillingTestEnvironment env("SQLite", TDuration::Seconds(5), 1);

        TPaymentTask pTask;
        {
            UNIT_ASSERT(env.CreateBillingTask(DefaultSessionID, DefaultUserId, EBillingType::CarUsage, 0, 500, 1000));
            env.WaitForFinish();
            TCachedPayments snapshot = env.GetActualSnapshot(DefaultSessionID);
            auto payments = snapshot.GetTimeline();
            UNIT_ASSERT_VALUES_EQUAL(payments.size(), 1);
            pTask = payments.front();
        }

        TBillingManager& manager = env.GetManager();
        {
            auto bTask = env.GetBillingTaskByDefaultSession();
            auto session = manager.BuildSession();
            UNIT_ASSERT_VALUES_EQUAL(manager.GetPaymentsManager().UpdatePaymentStatus(pTask, {}, session, bTask.GetLastPaymentId()), EDriveOpResult::Ok);
            UNIT_ASSERT(session.Commit());
        }

        {
            auto session = manager.BuildSession();
            UNIT_ASSERT_VALUES_EQUAL(manager.GetPaymentsManager().UpdatePaymentStatus(pTask, {}, session, 0), EDriveOpResult::LogicError);
        }
    }
}
