#pragma once

#include <drive/backend/billing/client/callback.h>

class TYAccountLogic;

template <class TBase>
class TCreateAccountCallback: public TBase {
public:
    template <class... TArgs>
    TCreateAccountCallback(TYAccountLogic& trustLogic, const TPaymentOpContext& context, NDrive::NBilling::IBillingAccount::TPtr account, const TCachedPayments& payments, const TPaymentsManager& manager, TArgs&... args)
        : TBase(trustLogic, args...)
        , Payments(payments)
        , Context(context)
        , Account(account)
        , Owner(manager)
    {}

    static NDrive::NTrustClient::TPayment BuildPayment(const TPaymentOpContext& context, const TChargeInfo& charge, const TVirtualTerminal& product, const TString& card) {
        NDrive::NTrustClient::TPayment payment;
        payment.Amount = charge.Sum;
        payment.ProductId = product.GetTrustProduct();
        payment.Currency = "RUB";
        payment.FiscalTitle = product.GetFiscalTitle();
        payment.FiscalNDS = product.GetFiscalNDS();
        payment.Email = context.GetUserData().GetEmail();
        payment.PaymethodId = card;
        return payment;
    }

protected:
    virtual void OnResult(const NJson::TJsonValue& json, bool isOk, ui16 /*code*/, const TBillingClient& trustClient) override {
        auto syncGuard = this->GetGuard();
        if (isOk) {
            auto charge = Context.GetCharge();
            if (charge.Sum == 0) {
                return;
            }

            const TString paymethodId = json["payment_method_id"].GetString();
            if (!paymethodId) {
                auto session = Owner.BuildSession(false);
                if (Owner.UpdateTaskStatus(Payments.GetSessionId(), EPaymentStatus::IncorrectYAccount, session) == EDriveOpResult::Ok) {
                    Y_UNUSED(session.Commit());
                }
                return;
            }

            auto product = trustClient.GetTrustProduct(charge.Type);
            if (!product.Defined()) {
                TBase::Notify("Incorrect terminal " + ToString(charge.Type));
                return;
            }

            NDrive::NTrustClient::TPayment payment = BuildPayment(Context, charge, product.GetRef(), paymethodId);
            Context.SetPayment(payment);
            THolder<TBillingCallback> callback;
            if (syncGuard) {
                callback = MakeHolder<TCreatePaymentCallback<TGuardedCallback>>(const_cast<ITrustLogic&>(this->TrustLogic), Context, Account, Owner, *syncGuard);
            } else {
                callback = MakeHolder<TCreatePaymentCallback<TLogicCallback>>(const_cast<ITrustLogic&>(this->TrustLogic), Context, Account, Owner);
            }

            TBillingClient::TOperation operation(ETrustOperatinType::PaymentCreate, callback.Release());
            operation.SetPayment(payment).SetPassportId(Context.GetPassportUid()).SetBillingType(charge.Type);

            if (syncGuard) {
                trustClient.RunOperation(operation);
            } else {
                TBase::TrustLogic.AddOperation(std::move(operation));
            }
        } else {
            if (syncGuard) {
                syncGuard->SetResult(EProduceResult::Error);
            }
            TBase::Notify(json.GetStringRobust());
        }
    }

    virtual TString GetOperation() const override {
        return "create_account";
    }

private:
    TCachedPayments Payments;
    TPaymentOpContext Context;
    NDrive::NBilling::IBillingAccount::TPtr Account;
    const TPaymentsManager& Owner;
};
