#include "charge_logic.h"

#include "callback.h"

#include <drive/backend/billing/accounts/trust.h>

ILogicConfig::TFactory::TRegistrator<TYAccountLogicConfig> TYAccountLogicConfig::Registrator(NDrive::NBilling::EAccount::YAccount);

IChargeLogic::TPtr TYAccountLogicConfig::ConstructLogic(const ISettings& settings, TUsersDB& users, TAtomicSharedPtr<NDrive::INotifier> notifier) const {
    return new TYAccountLogic(*this, settings, users, notifier);
}

TYAccountLogic::TYAccountLogic(const TYAccountLogicConfig& trustConfig, const ISettings& settings, TUsersDB& usersDB, TAtomicSharedPtr<NDrive::INotifier> notifier)
    : ITrustLogic(trustConfig, settings, usersDB, notifier)
{}

TExpectedCharge TYAccountLogic::ProducePayments(NDrive::NBilling::IBillingAccount::TPtr account, const TPaymentsData& snapshot, const TPaymentsManager& paymentsManager,
                                             const TChargeInfo& chargeInfo, TLazySession& session, const bool sync) const {

    TChargeInfo charge = CorrectCharge(account, snapshot, chargeInfo);
    if (charge.Sum == 0) {
        return charge;
    }
    const TString userId = snapshot.GetBillingTask().GetUserId();
    auto gUser = UsersDB.FetchInfo(userId, session.Get());
    auto userData = gUser.GetResultPtr(userId);
    if (!userData) {
        return MakeUnexpected(EProduceResult::Error);
    }

    TString passportUid = GetPassportUid(userData->GetUserId(), userData);
    TPaymentOpContext pContext(*userData, charge, snapshot.GetBillingTask(), passportUid);
    THolder<TBillingCallback> callback;

    auto result = EProduceResult::Wait;
    {
        TBillingSyncGuard g(result);
        if (sync) {
            callback = MakeHolder<TCreateAccountCallback<TGuardedCallback>>(const_cast<TYAccountLogic&>(*this), pContext, account, snapshot.GetSnapshot(), paymentsManager, g);
        } else {
            callback = MakeHolder<TCreateAccountCallback<TLogicCallback>>(const_cast<TYAccountLogic&>(*this), pContext, account, snapshot.GetSnapshot(), paymentsManager);
        }

        TBillingClient::TOperation operation(ETrustOperatinType::YAccountCreate, callback.Release());
        operation.SetPassportId(passportUid).SetBillingType(snapshot.GetBillingTask().GetBillingType());
        if (TBillingGlobals::SupportedCashbackCurrency) {
            operation.MutablePayment().Currency = *TBillingGlobals::SupportedCashbackCurrency.begin();
        }

        if (sync) {
            TrustClient.RunOperation(operation);
        } else {
            AddOperation(std::move(operation));
        }
    }
    if (result == EProduceResult::Ok) {
        return charge;
    }
    return MakeUnexpected(result);
}

TChargeInfo TYAccountLogic::CorrectCharge(const NDrive::NBilling::IBillingAccount::TPtr /*account*/, const TPaymentsData& snapshot, const TChargeInfo& chargeInfo) const {
    if (snapshot.IsClosed()) {
        return TChargeInfo(chargeInfo.Type);
    }
    auto payMethods = GetUserCards(snapshot.GetBillingTask().GetUserId());
    if (!payMethods) {
        return TChargeInfo(chargeInfo.Type);
    }

    if (snapshot.GetSnapshot().GetYandexAccountSum() >= snapshot.GetBillingTask().GetAvailableYandexBonus()) {
        return TChargeInfo(chargeInfo.Type);
    }

    TChargeInfo bonusCharge = CorrectChargeForBonus(snapshot, chargeInfo);
    for (auto&& method : *payMethods) {
        if (TBillingGlobals::SupportedCashbackMethods.contains(method.GetPaymentMethod()) && TBillingGlobals::SupportedCashbackCurrency.contains(method.GetCurrency())) {
            bonusCharge.Sum = Min<ui32>(bonusCharge.Sum, Max<i64>(method.GetBalance(), 0), snapshot.GetBillingTask().GetAvailableYandexBonus() - snapshot.GetSnapshot().GetYandexAccountSum());
            break;
        }
    }
    return bonusCharge;
}
