#include "charge_logic.h"

#include "callback.h"

#include <drive/backend/billing/accounts/trust.h>

ILogicConfig::TFactory::TRegistrator<TYCashbackLogicConfig> TYCashbackLogicConfig::Registrator(NDrive::NBilling::EAccount::YCashback);

IChargeLogic::TPtr TYCashbackLogicConfig::ConstructLogic(const ISettings& settings, TUsersDB& users, TAtomicSharedPtr<NDrive::INotifier> notifier) const {
    return new TYCashbackLogic(*this, settings, users, notifier);
}

TYCashbackLogic::TYCashbackLogic(const TYCashbackLogicConfig& trustConfig, const ISettings& settings, TUsersDB& usersDB, TAtomicSharedPtr<NDrive::INotifier> notifier)
    : ITrustLogic(trustConfig, settings, usersDB, notifier)
{}

EProduceResult TYCashbackLogic::SyncPayments(const TPaymentsData& snapshot, const TPaymentsManager& paymentsManager, const NDrive::NBilling::TAccountsManager& /*accountsManager*/, ITrustStorage* cache, const bool sync) const {
    return SyncPaymentsImpl(snapshot, paymentsManager, ETrustOperatinType::TopupStart, ETrustOperatinType::TopupInfo, cache, sync);
}

TChargeInfo TYCashbackLogic::CorrectCharge(const NDrive::NBilling::IBillingAccount::TPtr /*account*/, const TPaymentsData& snapshot, const TChargeInfo& chargeInfo) const {
    TChargeInfo result(EBillingType::YCashback);
    if (snapshot.IsClosed()
    && chargeInfo.Sum == 0
    && snapshot.GetBillingTask().GetBillCorrected() >= snapshot.GetSnapshot().GetHeldSum()
    && snapshot.GetBillingTask().GetBillCorrected() < snapshot.GetSnapshot().GetHeldSum() + 100
    && snapshot.GetActualCashback() > snapshot.GetSnapshot().GetCashbackSum()) {
        result.Sum = snapshot.GetActualCashback() - snapshot.GetSnapshot().GetCashbackSum();
        result.BaseSum = snapshot.GetSnapshot().GetCardSum();
        result.IsPlusUser = snapshot.GetBillingTask().OptionalIsPlusUser();
        result.OfferType = snapshot.GetBillingTask().GetOfferType();
        result.OfferName = snapshot.GetBillingTask().GetOfferName();
        result.SessionId = snapshot.GetBillingTask().GetId();
        result.PayloadType = Settings.GetValueDef<TString>("billing.cashback.payload.type", "drive");
        if (auto info = snapshot.GetBillingTask().GetCashbacksInfoPtr()) {
            info->CorrectCashback(snapshot.GetSnapshot().GetCashbackSumByTypes(), result.Sum, result.PayloadExtra, result.Type);
        } else {
            result.PayloadExtra = Settings.GetJsonValue("billing.cashback.payload.extra");
        }
    }
    return result;
}

TExpectedCharge TYCashbackLogic::ProducePayments(NDrive::NBilling::IBillingAccount::TPtr account, const TPaymentsData& snapshot, const TPaymentsManager& paymentsManager,
                                             const TChargeInfo& chargeInfo, TLazySession& session, const bool /*sync*/) const {
    TChargeInfo charge = CorrectCharge(account, snapshot, chargeInfo);
    if (charge.Sum == 0) {
        return charge;
    }
    const TString userId = snapshot.GetBillingTask().GetUserId();
    auto gUser = UsersDB.FetchInfo(userId, session.Get());
    auto userData = gUser.GetResultPtr(userId);
    if (!userData) {
        return MakeUnexpected(EProduceResult::Error);
    }

    TString passportUid = GetPassportUid(userData->GetUserId(), userData);
    TPaymentOpContext pContext(*userData, charge, snapshot.GetBillingTask(), passportUid);
    TBillingClient::TOperation operation(ETrustOperatinType::YAccountCreate, new TCreateCashbackAccountCallback(const_cast<TYCashbackLogic&>(*this), pContext, snapshot.GetSnapshot(), paymentsManager));
    operation.SetPassportId(passportUid).SetBillingType(snapshot.GetBillingTask().GetBillingType());
    if (TBillingGlobals::SupportedCashbackCurrency) {
        operation.MutablePayment().Currency = *TBillingGlobals::SupportedCashbackCurrency.begin();
    }
    AddOperation(std::move(operation));
    return MakeUnexpected(EProduceResult::Wait);
}
