#pragma once

#include <library/cpp/json/json_value.h>
#include <library/cpp/object_factory/object_factory.h>

#include <rtline/util/types/accessor.h>

enum EDocumentAttachmentType : ui32 {
    Unknown = 1 << 0                        /* "unknown" */,
    CarHardwareSim = 1 << 1                 /* "car_hardware_sim" */,
    CarHardwareBeacon = 1 << 2              /* "car_hardware_beacon" */,
    CarHardwareModem = 1 << 3               /* "car_hardware_modem" */,
    CarHardwareHead = 1 << 4                /* "car_hardware_head" */,
    CarHardwareVega = 1 << 5                /* "car_hardware_vega" */,
    CarAirportPass = 1 << 6                 /* "car_airport_pass" */,
    CarTransponder = 1 << 7                 /* "car_transponder" */,
    CarTransponderSpb = 1 << 8              /* "car_transponder_spb" */,
    CarInsurancePolicy = 1 << 9             /* "car_insurance_policy" */,
    CarRegistryDocument = 1 << 10           /* "car_registry_document" */,
    CarAirportPassSpb = 1 << 11             /* "car_airport_pass_spb" */,
    CarHardwareHeadSerialNumber = 1 << 12   /* "car_hardware_head_sn" */,
    CarHardwareHeadNew = 1 << 13            /* "car_hardware_head_new" */,
    CarFuelCard = 1 << 14                   /* "car_fuel_card" */,
    CarHardwareDashcam = 1 << 15            /* "car_hardware_dashcam" */,
    CarQrCode = 1 << 16                     /* "car_qr_code" */,
    CarSignalDevice = 1 << 17               /* "car_signal_device" */,
};

namespace NAttachmentReport {
    using TReportTraits = ui32;
    using TStructureReportTraits = ui32;
    static const TReportTraits NoTraits = 0;
    static const TStructureReportTraits NoStructureTraits = 0;
    static constexpr TReportTraits ReportAll = Max<TReportTraits>();
    static constexpr TStructureReportTraits ReportAllStructure = Max<TStructureReportTraits>();
    static constexpr TReportTraits ReportServiceApp =
        EDocumentAttachmentType::CarFuelCard |
        EDocumentAttachmentType::CarQrCode |
        EDocumentAttachmentType::CarHardwareBeacon | EDocumentAttachmentType::CarHardwareHead | EDocumentAttachmentType::CarHardwareModem | EDocumentAttachmentType::CarHardwareVega |
        EDocumentAttachmentType::CarAirportPass | EDocumentAttachmentType::CarTransponder | EDocumentAttachmentType::CarTransponderSpb | EDocumentAttachmentType::CarInsurancePolicy |
        EDocumentAttachmentType::CarAirportPassSpb | EDocumentAttachmentType::CarHardwareHeadNew | EDocumentAttachmentType::CarHardwareHeadSerialNumber |
        EDocumentAttachmentType::CarSignalDevice;
};

namespace NRegistryDocumentReport {
    using TReportTraits = NAttachmentReport::TStructureReportTraits;
    static const TReportTraits NoTraits = 0;
    static constexpr TReportTraits ReportAll = Max<TReportTraits>();
    enum EReportTraits : TReportTraits {
        ReportBasicInfo = 1 << 0,
        ReportContractIdAndDate = 1 << 1,
        ReportRegistrationDate = 1 << 2,
        ReportTransferDate = 1 << 3,
        ReportOsagoDetails = 1 << 4,
        ReportPtsDetails = 1 << 5,
        ReportDeptransDocumentsDate = 1 << 6,
        ReportParkingPermitStartDate = 1 << 7,
        ReportCascoNumber = 1 << 8,
        ReportDkpPrice = 1 << 9,
        ReportDsPayment = 1 << 10,
        ReportLeaseTermMonths = 1 << 11,
        ReportReturnDate = 1 << 12,
        ReportContractMileage = 1 << 13,
        ReportTotalDetails = 1 << 14,
        ReportSendingCarTransporterDetails = 1 << 15,
        ReportCity = 1 << 16,
    };
};

class ICarAttachmentReportContext {
public:
    virtual ~ICarAttachmentReportContext() = default;
    virtual NAttachmentReport::TStructureReportTraits GetRegistryDocumentTraits() const = 0;
    virtual TString GetMdsLink(const TString& bucketName, const TString& path) const = 0;
};

class TCarAttachmentReportImplContext : public ICarAttachmentReportContext {
public:
    using ICarAttachmentReportContext::ICarAttachmentReportContext;

    virtual NAttachmentReport::TStructureReportTraits GetRegistryDocumentTraits() const override {
        return NAttachmentReport::ReportAllStructure;
    }
    virtual TString GetMdsLink(const TString& /*bucketName*/, const TString& path) const override {
        return path;
    }
};

class IJsonBlobSerializableCarAttachment {
public:
    enum EUniquePolicy {
        NotUnique = 0,
        Unique = 1
    };

    using TFactory = NObjectFactory::TParametrizedObjectFactory<IJsonBlobSerializableCarAttachment, EDocumentAttachmentType>;
    virtual NJson::TJsonValue SerializeToJsonBlob() const {
        return BuildReport(TCarAttachmentReportImplContext());
    }

    virtual EUniquePolicy GetUniquePolicy() const = 0;

    virtual bool DeserializeFromJsonBlob(const NJson::TJsonValue& blob) = 0;
    virtual NJson::TJsonValue BuildReport(const ICarAttachmentReportContext& context) const = 0;

    virtual TString GetAbsenceTagName() const {
        return "";
    }

    virtual TString GetPresenceTagName() const {
        return "";
    }

    virtual TString GetServiceAppSlug() const {
        return "";
    }
    virtual TVector<TString> GetUsedICCNumbers() const {
        return {};
    }
    virtual ~IJsonBlobSerializableCarAttachment() {
    }

    virtual EDocumentAttachmentType GetTypeName() const = 0;

    virtual NAttachmentReport::TReportTraits GetReportTrait() const {
        return GetTypeName();
    }
protected:
    void ParseDateIntoField(const NJson::TJsonValue& payload, const TString& fieldName, TInstant& resultField);

    bool UpdateDateField(TInstant& field, const TString& value);

    template<typename TField>
    bool UpdateField(TField& field, const TField& value) {
        if (field == value) {
            return false;
        }
        field = value;
        return true;
    }

    template <typename TField, typename TStringValue>
    bool ParseAndUpdateField(TField& field, const TStringValue& value) {
        TField parsed;
        if (!TryFromString(value, parsed)) {
            return false;
        }
        return UpdateField(field, parsed);
    }
    template <>
    bool ParseAndUpdateField(TInstant& field, const TString& value) {
        return UpdateDateField(field, value);
    }
};

class ISimpleJsonBodyAttachment : public IJsonBlobSerializableCarAttachment {
protected:
    NJson::TJsonValue Blob;

public:
    virtual NJson::TJsonValue BuildReport(const ICarAttachmentReportContext& /*context*/) const override {
        return Blob;
    }

    bool DeserializeFromJsonBlob(const NJson::TJsonValue& blob) override {
        Blob = blob;
        return true;
    }
};

class ISimpleAttachment : public IJsonBlobSerializableCarAttachment {
    // According to https://st.yandex-team.ru/DRIVEBACK-568
protected:
    TString Code = "";

public:
    ISimpleAttachment() = default;
    ISimpleAttachment(const TString& code);

    virtual NJson::TJsonValue BuildReport(const ICarAttachmentReportContext& context) const override;
    bool DeserializeFromJsonBlob(const NJson::TJsonValue& blob) override;

    virtual TString GetServiceAppSlug() const override;
};
