#include "fuel_card.h"

#include <rtline/library/json/parse.h>
#include <rtline/util/json_processing.h>

#include <util/string/subst.h>

TFuelCard::TFactory::TRegistrator<TFuelCard> TFuelCard::Registrator(EDocumentAttachmentType::CarFuelCard);

TFuelCard::TFuelCard(const TString& code, const TString& pin, const TString& station)
    : Code(code)
    , Pin(pin)
    , Station(station)
{}


IJsonBlobSerializableCarAttachment::EUniquePolicy TFuelCard::GetUniquePolicy() const {
    return EUniquePolicy::NotUnique;
}

EDocumentAttachmentType TFuelCard::GetTypeName() const {
    return EDocumentAttachmentType::CarFuelCard;
}

NJson::TJsonValue TFuelCard::BuildReport(const ICarAttachmentReportContext& /*context*/) const {
    NJson::TJsonValue result;
    NJson::InsertField(result, "code", Code);
    NJson::InsertNonNull(result, "pin", Pin);
    NJson::InsertNonNull(result, "station", Station);
    return result;
}

bool TFuelCard::DeserializeFromJsonBlob(const NJson::TJsonValue& jsonValue) {
    return NJson::ParseField(jsonValue["code"], Code, true)
        && NJson::ParseField(jsonValue["pin"], Pin)
        && NJson::ParseField(jsonValue["station"], Station);
}

TVector<THolder<TFuelCard>> TFuelCard::ConstructFromAdmin(const NJson::TJsonValue& rawData, const TVector<TCardFields>& customFields) {
    static const TVector<TCardFields> defaultFields = {
        {"газпромнефть", "гпн пин", "gazpromneft"},
        {"роснефть", "рн пин", "rosneft"},
        {"татнефть", "т пин", "tatneft"},
        {"лукойл", "л пин", "lukoil"}
    };
    const auto& cardFields = customFields.empty() ? defaultFields : customFields;

    TVector<THolder<TFuelCard>> result;
    for (const auto& fields : cardFields) {
        const TString lowCaseCodeField = ToLowerUTF8(fields.Code);
        const TString lowCasePinField = ToLowerUTF8(fields.Pin);
        if (!rawData.Has(lowCaseCodeField) || rawData[lowCaseCodeField].GetStringRobust().empty()) {
            continue;
        }
        result.emplace_back(new TFuelCard(rawData[lowCaseCodeField].GetStringRobust(), rawData.Has(lowCasePinField) ? rawData[lowCasePinField].GetStringRobust() : "", fields.Station));
    }
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TFuelCard::TCardFields& result) {
    return NJson::ParseField(value["code_field"], result.Code, true)
        && NJson::ParseField(value["pin_field"], result.Pin, true)
        && NJson::ParseField(value["station_id"], result.Station, true);
}
