#include "insurance.h"

#include <rtline/library/json/adapters.h>
#include <rtline/util/json_processing.h>

#include <util/string/subst.h>

TCarInsurancePolicy::TFactory::TRegistrator<TCarInsurancePolicy> TCarInsurancePolicy::Registrator(EDocumentAttachmentType::CarInsurancePolicy);

TCarInsurancePolicy::TCarInsurancePolicy(const TString& agreementPartnerNumber, const TString& agreementNumber, ui32 baseCost, ui32 perMinuteCost, const TInstant validFrom, const TInstant validUntil)
    : AgreementPartnerNumber(agreementPartnerNumber)
    , AgreementNumber(agreementNumber)
    , BaseCost(baseCost)
    , PerMinuteCost(perMinuteCost)
    , ValidFrom(validFrom)
    , ValidUntil(validUntil)
{
}

NJson::TJsonValue TCarInsurancePolicy::BuildReport(const ICarAttachmentReportContext& /*context*/) const {
    NJson::TJsonValue result;
    result["agreement_partner_number"] = AgreementPartnerNumber;
    result["agreement_number"] = AgreementNumber;

    result["base_cost"] = BaseCost;
    result["per_minute_cost"] = PerMinuteCost;
    if (ValidFrom != TInstant::Zero()) {
        result["valid_from"] = ValidFrom.ToString();
    }
    if (ValidUntil != TInstant::Zero()) {
        result["valid_until"] = ValidUntil.ToString();
    }
    result["provider"] = ::ToString(Provider);
    return result;
}

bool TCarInsurancePolicy::DeserializeFromJsonBlob(const NJson::TJsonValue& blob) {
    JREAD_STRING(blob, "agreement_partner_number", AgreementPartnerNumber);
    JREAD_STRING(blob, "agreement_number", AgreementNumber);

    if (!blob.Has("base_cost")) {
        return false;
    }
    if (blob["base_cost"].IsUInteger()) {
        JREAD_UINT(blob, "base_cost", BaseCost);
    } else {
        TString baseCostStr;
        JREAD_STRING(blob, "base_cost", baseCostStr);
        double tmp = 0.0;
        if (!TryFromString(baseCostStr, tmp)) {
            return false;
        }
        BaseCost = tmp * 100;
    }

    if (blob.Has("provider")) {
        if (!TryFromString(blob["provider"].GetString(), Provider)) {
            return false;
        }
    }

    if (!blob.Has("per_minute_cost")) {
        return false;
    }
    if (blob["per_minute_cost"].IsUInteger()) {
        JREAD_UINT(blob, "per_minute_cost", PerMinuteCost);
    } else {
        TString perMinuteCostStr;
        JREAD_STRING(blob, "per_minute_cost", perMinuteCostStr);
        double tmp = 0.0;
        if (!TryFromString(perMinuteCostStr, tmp)) {
            return false;
        }
        PerMinuteCost = tmp * 100;
    }

    JREAD_INSTANT_ISOFORMAT_NULLABLE_OPT(blob, "valid_from", ValidFrom);
    JREAD_INSTANT_ISOFORMAT_NULLABLE_OPT(blob, "valid_until", ValidUntil);
    return true;
}

bool TCarInsurancePolicy::ConstructFromAdmin(const NJson::TJsonValue& rawData, const NDrive::EInsuranceProvider provider, const ui32 baseCost) {
    Provider = provider;
    if (Provider == NDrive::EInsuranceProvider::Renins || Provider == NDrive::EInsuranceProvider::Ingos) {
        BaseCost = baseCost;

        JREAD_STRING(rawData, "agreementpartnernum", AgreementPartnerNumber);
        JREAD_STRING(rawData, "agreementnum", AgreementNumber);

        const auto& perMinuteCost = rawData["поминутное"];
        if (perMinuteCost.IsDefined()) {
            TString perMinuteCostStr;
            double perMinuteCostDouble;
            JREAD_STRING(rawData, "поминутное", perMinuteCostStr);
            SubstGlobal(perMinuteCostStr, ",", ".");
            if (!TryFromString(perMinuteCostStr, perMinuteCostDouble)) {
                return false;
            }
            PerMinuteCost = (ui32)(perMinuteCostDouble * 100 + 0.1);
        }

        TString validFromStr;
        TString validUntilStr;
        JREAD_STRING(rawData, "дата начала страхования", validFromStr);
        JREAD_STRING(rawData, "дата окончания страхования", validUntilStr);

        if (!UpdateDateField(ValidFrom, validFromStr) || !UpdateDateField(ValidUntil, validUntilStr)) {
            return false;
        }

        // Moscow time is used in Renins policies
        ValidFrom -= TDuration::Hours(3);
        ValidUntil += TDuration::Hours(21) - TDuration::Seconds(1);
    } else if (Provider == NDrive::EInsuranceProvider::Fake) {
        ValidFrom = TInstant::Zero();
        ValidUntil = Now() + TDuration::Days(3560);
        if (rawData.Has("дата начала страхования") && rawData["дата начала страхования"].IsString()) {
            auto validFromStr = rawData["дата начала страхования"].GetString();
            if (!UpdateDateField(ValidFrom, validFromStr)) {
                return false;
            }
        }
        if (rawData.Has("дата окончания страхования") && rawData["дата окончания страхования"].IsString()) {
            auto validUntilStr = rawData["дата окончания страхования"].GetString();
            if (!UpdateDateField(ValidUntil, validUntilStr)) {
                return false;
            }
        }
    }

    return true;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::EInsuranceProvider& object) {
    return NJson::ToJson(NJson::Stringify(object));
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::EInsuranceProvider& result) {
    return NJson::TryFromJson(value, NJson::Stringify(result));
}
