#include "hardware.h"

#include <rtline/util/json_processing.h>

TCarHardwareBeacon::TFactory::TRegistrator<TCarHardwareBeacon> TCarHardwareBeacon::Registrator(EDocumentAttachmentType::CarHardwareBeacon);
TCarHardwareDashcam::TFactory::TRegistrator<TCarHardwareDashcam> TCarHardwareDashcam::Registrator(EDocumentAttachmentType::CarHardwareDashcam);
TCarHardwareHead::TFactory::TRegistrator<TCarHardwareHead> TCarHardwareHead::Registrator(EDocumentAttachmentType::CarHardwareHead);
TCarHardwareModem::TFactory::TRegistrator<TCarHardwareModem> TCarHardwareModem::Registrator(EDocumentAttachmentType::CarHardwareModem);
TCarHardwareSim::TFactory::TRegistrator<TCarHardwareSim> TCarHardwareSim::Registrator(EDocumentAttachmentType::CarHardwareSim);
TCarHardwareVega::TFactory::TRegistrator<TCarHardwareVega> TCarHardwareVega::Registrator(EDocumentAttachmentType::CarHardwareVega);
TCarSignalDevice::TFactory::TRegistrator<TCarSignalDevice> TCarSignalDevice::Registrator(EDocumentAttachmentType::CarSignalDevice);

TCarHardwareHeadSerialNumber::TFactory::TRegistrator<TCarHardwareHeadSerialNumber> TCarHardwareHeadSerialNumber::Registrator(EDocumentAttachmentType::CarHardwareHeadSerialNumber);
TCarHardwareHeadNew::TFactory::TRegistrator<TCarHardwareHeadNew> TCarHardwareHeadNew::Registrator(EDocumentAttachmentType::CarHardwareHeadNew);



TCarHardwareSim::TCarHardwareSim(const TString& icc, const TString& phoneNumber)
    : ICC(icc)
    , PhoneNumber(phoneNumber)
{
}

NJson::TJsonValue TCarHardwareSim::BuildReport(const ICarAttachmentReportContext& /*context*/) const {
    NJson::TJsonValue result;
    JWRITE(result, "icc", ICC);
    JWRITE_DEF(result, "phone_number", PhoneNumber, "");
    return result;
}

bool TCarHardwareSim::DeserializeFromJsonBlob(const NJson::TJsonValue& blob) {
    if (blob["icc"].IsUInteger()) {
        ICC = ToString(blob["icc"].GetUInteger());
    } else {
        JREAD_STRING(blob, "icc", ICC);
    }
    if (blob.Has("phone_number") && blob["phone_number"].IsDefined()) {
        JREAD_STRING_OPT(blob, "phone_number", PhoneNumber);
    }
    return true;
}

TString TCarHardwareSim::NormalizeICC(const TString& icc) {
    /*
        One ICC can have several forms, as in with control digit and without.
        Also, operator-dependant. Unify.
    */
    if (icc.StartsWith("8970199")) {  // beeline
        return icc.substr(0, 18);
    } else if (icc.StartsWith("8970102")) {  // megafon
        return icc.substr(0, 17);
    }
    return icc;
}

TCarHardwareBeacon::TCarHardwareBeacon(const TString& imei, const TString& serialNumber)
    : IMEI(imei)
    , SerialNumber(serialNumber)
{
}

NJson::TJsonValue TCarHardwareBeacon::BuildReport(const ICarAttachmentReportContext& /*context*/) const {
    NJson::TJsonValue result;
    JWRITE(result, "imei", IMEI);
    if (SerialNumber != "") {
        JWRITE(result, "serial_number", SerialNumber);
    } else {
        result["serial_number"] = NJson::JSON_NULL;
    }
    if (Sim) {
        result["sim"] = Sim->SerializeToJsonBlob();
    }
    return result;
}

bool TCarHardwareBeacon::DeserializeFromJsonBlob(const NJson::TJsonValue& blob) {
    if (blob["imei"].IsUInteger()) {
        IMEI = ToString(blob["imei"].GetUInteger());
    } else {
        JREAD_STRING(blob, "imei", IMEI);
    }
    if (blob["serial_number"].IsString()) {
        JREAD_STRING(blob, "serial_number", SerialNumber);
    }
    if (blob["sim"] != NJson::JSON_NULL) {
        Sim = new TCarHardwareSim();
        Sim->DeserializeFromJsonBlob(blob["sim"]);
    }
    return true;
}

TVector<TString> TCarHardwareBeacon::GetUsedICCNumbers() const {
    if (Sim != nullptr && !!Sim->GetICC()) {
        return {Sim->GetICC()};
    }
    return {};
}

TString TCarHardwareBeacon::GetServiceAppSlug() const {
    return IMEI;
}

TCarHardwareModem::TCarHardwareModem(const TString& simICC) {
    Sim.Reset(new TCarHardwareSim(simICC));
}

NJson::TJsonValue TCarHardwareModem::BuildReport(const ICarAttachmentReportContext& /*context*/) const {
    NJson::TJsonValue result;
    result["sim"] = Sim->SerializeToJsonBlob();
    return result;
}

bool TCarHardwareModem::DeserializeFromJsonBlob(const NJson::TJsonValue& blob) {
    if (blob["sim"] != NJson::JSON_NULL) {
        Sim = new TCarHardwareSim();
        return Sim->DeserializeFromJsonBlob(blob["sim"]);
    }
    return false;
}

TVector<TString> TCarHardwareModem::GetUsedICCNumbers() const {
    if (Sim != nullptr && !!Sim->GetICC()) {
        return {Sim->GetICC()};
    }
    return {};
}

TString TCarHardwareModem::GetServiceAppSlug() const {
    return Sim->GetICC();
}

TCarHardwareHead::TCarHardwareHead(const TString& headId, const TString& deviceId)
    : HeadId(headId)
    , DeviceId(deviceId)
{
}

NJson::TJsonValue TCarHardwareHead::BuildReport(const ICarAttachmentReportContext& /*context*/) const {
    NJson::TJsonValue result;
    JWRITE(result, "head_id", HeadId);
    if (DeviceId != "") {
        JWRITE(result, "device_id", DeviceId);
    } else {
        result["device_id"] = NJson::JSON_NULL;
    }
    return result;
}

bool TCarHardwareHead::DeserializeFromJsonBlob(const NJson::TJsonValue& blob)  {
    JREAD_STRING(blob, "head_id", HeadId);
    if (blob["device_id"] != NJson::JSON_NULL) {
        JREAD_STRING_OPT(blob, "device_id", DeviceId);
    }
    return true;
}

TString TCarHardwareHead::GetServiceAppSlug() const {
    return HeadId;
}

TCarHardwareHeadNew::TCarHardwareHeadNew(const TString& headId)
    : HeadId(headId)
{
}

NJson::TJsonValue TCarHardwareHeadNew::BuildReport(const ICarAttachmentReportContext& /*context*/) const {
    NJson::TJsonValue result;
    JWRITE(result, "head_id", HeadId);
    return result;
}

bool TCarHardwareHeadNew::DeserializeFromJsonBlob(const NJson::TJsonValue& blob)  {
    JREAD_STRING(blob, "head_id", HeadId);
    return true;
}

TString TCarHardwareHeadNew::GetServiceAppSlug() const {
    return HeadId;
}

TCarHardwareHeadSerialNumber::TCarHardwareHeadSerialNumber(const TString& serialNumber)
    : SerialNumber(serialNumber)
{
}

NJson::TJsonValue TCarHardwareHeadSerialNumber::BuildReport(const ICarAttachmentReportContext& /*context*/) const {
    NJson::TJsonValue result;
    JWRITE(result, "serial_number", SerialNumber);
    return result;
}

bool TCarHardwareHeadSerialNumber::DeserializeFromJsonBlob(const NJson::TJsonValue& blob)  {
    JREAD_STRING(blob, "serial_number", SerialNumber);
    return true;
}

TString TCarHardwareHeadSerialNumber::GetServiceAppSlug() const {
    return SerialNumber;
}

TCarHardwareVega::TCarHardwareVega(const TString& imei)
    : IMEI(imei)
{
}

NJson::TJsonValue TCarHardwareVega::BuildReport(const ICarAttachmentReportContext& /*context*/) const {
    NJson::TJsonValue result;
    JWRITE(result, "imei", IMEI);
    if (PrimarySim != nullptr) {
        result["primary_sim"] = PrimarySim->SerializeToJsonBlob();
    }
    if (SecondarySim != nullptr) {
        result["secondary_sim"] = SecondarySim->SerializeToJsonBlob();
    }
    return result;
}

bool TCarHardwareVega::DeserializeFromJsonBlob(const NJson::TJsonValue& blob) {
    if (blob["imei"].IsUInteger()) {
        IMEI = ToString(blob["imei"].GetUInteger());
    } else {
        JREAD_STRING(blob, "imei", IMEI);
    }
    if (blob["primary_sim"] != NJson::JSON_NULL) {
        PrimarySim = new TCarHardwareSim();
        PrimarySim->DeserializeFromJsonBlob(blob["primary_sim"]);
    }
    if (blob["secondary_sim"] != NJson::JSON_NULL) {
        SecondarySim = new TCarHardwareSim();
        SecondarySim->DeserializeFromJsonBlob(blob["secondary_sim"]);
    }
    return true;
}

TVector<TString> TCarHardwareVega::GetUsedICCNumbers() const {
    TVector<TString> result;
    if (PrimarySim != nullptr && !!PrimarySim->GetICC()) {
        result.push_back(PrimarySim->GetICC());
    }
    if (SecondarySim != nullptr && !!SecondarySim->GetICC()) {
        result.push_back(SecondarySim->GetICC());
    }
    return result;
}

void TCarHardwareVega::UpdateSim(const TString& icc, const TString& phoneNumber, const bool isPrimary) {
    /*
        Here we assume that these sim cards weren't used anywhere else at the moment.
        Should be checked prior to update.
    */
    if (isPrimary) {
        PrimarySim.Reset(new TCarHardwareSim(icc, phoneNumber));
    } else {
        SecondarySim.Reset(new TCarHardwareSim(icc, phoneNumber));
    }
}

TString TCarHardwareVega::GetServiceAppSlug() const {
    return IMEI;
}

TCarSignalDevice::TCarSignalDevice(const TString& serialNumber, const TString& imei)
    : SerialNumber(serialNumber)
    , Imei(imei)
{
}

NJson::TJsonValue TCarSignalDevice::BuildReport(const ICarAttachmentReportContext& /*context*/) const {
    NJson::TJsonValue result;
    JWRITE(result, "serial_number", SerialNumber);
    if (!Imei.empty()) {
        JWRITE(result, "imei", Imei);
    }
    return result;
}

bool TCarSignalDevice::DeserializeFromJsonBlob(const NJson::TJsonValue& blob)  {
    JREAD_STRING(blob, "serial_number", SerialNumber);
    JREAD_STRING_OPT(blob, "imei", Imei);
    return true;
}

TString TCarSignalDevice::GetServiceAppSlug() const {
    return SerialNumber;
}
