#pragma once

#include <drive/backend/car_attachments/abstract/attachment.h>

class TCarHardwareSim : public IJsonBlobSerializableCarAttachment {
    R_READONLY(TString, ICC);
    R_READONLY(TString, PhoneNumber);

    static TFactory::TRegistrator<TCarHardwareSim> Registrator;
public:
    TCarHardwareSim() = default;
    TCarHardwareSim(const TString& icc, const TString& phoneNumber = "");

    virtual NJson::TJsonValue BuildReport(const ICarAttachmentReportContext& context) const override;
    virtual bool DeserializeFromJsonBlob(const NJson::TJsonValue& blob) override;

    EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Unique;
    }

    static TString NormalizeICC(const TString& icc);

    EDocumentAttachmentType GetTypeName() const override {
        return EDocumentAttachmentType::CarHardwareSim;
    }
};

class TCarHardwareBeacon : public IJsonBlobSerializableCarAttachment {
    R_READONLY(TString, IMEI);
    R_READONLY(TString, SerialNumber);

    static TFactory::TRegistrator<TCarHardwareBeacon> Registrator;
private:
    TAtomicSharedPtr<TCarHardwareSim> Sim;

public:
    TCarHardwareBeacon() = default;
    TCarHardwareBeacon(const TString& imei, const TString& serialNumber);

    EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Unique;
    }

    virtual TString GetAbsenceTagName() const override {
        return "need_beacon";
    }

    virtual NJson::TJsonValue BuildReport(const ICarAttachmentReportContext& context) const override;
    virtual bool DeserializeFromJsonBlob(const NJson::TJsonValue& blob) override;

    virtual TVector<TString> GetUsedICCNumbers() const override;
    virtual TString GetServiceAppSlug() const override;

    void SetSim(const TAtomicSharedPtr<TCarHardwareSim>& sim) {
        Sim = sim;
    }

    EDocumentAttachmentType GetTypeName() const override {
        return EDocumentAttachmentType::CarHardwareBeacon;
    }
};

class TCarHardwareDashcam : public TCarHardwareBeacon {
public:
    TCarHardwareDashcam() = default;

    virtual EDocumentAttachmentType GetTypeName() const override {
        return EDocumentAttachmentType::CarHardwareDashcam;
    }

private:
    static TFactory::TRegistrator<TCarHardwareDashcam> Registrator;
};

class TCarHardwareModem : public IJsonBlobSerializableCarAttachment {
private:
    TAtomicSharedPtr<TCarHardwareSim> Sim;

    static TFactory::TRegistrator<TCarHardwareModem> Registrator;
public:
    TCarHardwareModem() = default;
    TCarHardwareModem(const TString& simICC);

    virtual NJson::TJsonValue BuildReport(const ICarAttachmentReportContext& context) const override;
    virtual bool DeserializeFromJsonBlob(const NJson::TJsonValue& blob) override;

    virtual TVector<TString> GetUsedICCNumbers() const override;
    virtual TString GetServiceAppSlug() const override;

    EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Unique;
    }

    void SetSim(const TAtomicSharedPtr<TCarHardwareSim>& sim) {
        Sim = sim;
    }

    virtual TString GetAbsenceTagName() const override {
        return "need_modem";
    }

    EDocumentAttachmentType GetTypeName() const override {
        return EDocumentAttachmentType::CarHardwareModem;
    }
};

class TCarHardwareHead : public IJsonBlobSerializableCarAttachment {
    R_READONLY(TString, HeadId);
    R_READONLY(TString, DeviceId);

    static TFactory::TRegistrator<TCarHardwareHead> Registrator;
public:
    TCarHardwareHead() = default;
    TCarHardwareHead(const TString& headId, const TString& deviceId = "");

    EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Unique;
    }

    virtual TString GetAbsenceTagName() const override {
        return "need_head";
    }

    virtual NJson::TJsonValue BuildReport(const ICarAttachmentReportContext& context) const override;
    virtual bool DeserializeFromJsonBlob(const NJson::TJsonValue& blob) override;

    virtual TString GetServiceAppSlug() const override;

    EDocumentAttachmentType GetTypeName() const override {
        return EDocumentAttachmentType::CarHardwareHead;
    }
};

class TCarHardwareHeadSerialNumber : public IJsonBlobSerializableCarAttachment {
    R_READONLY(TString, SerialNumber);

    static TFactory::TRegistrator<TCarHardwareHeadSerialNumber> Registrator;
public:
    TCarHardwareHeadSerialNumber() = default;
    TCarHardwareHeadSerialNumber(const TString& serialNumber);

    EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Unique;
    }

    virtual NJson::TJsonValue BuildReport(const ICarAttachmentReportContext& context) const override;
    virtual bool DeserializeFromJsonBlob(const NJson::TJsonValue& blob) override;

    virtual TString GetServiceAppSlug() const override;

    EDocumentAttachmentType GetTypeName() const override {
        return EDocumentAttachmentType::CarHardwareHeadSerialNumber;
    }
};

class TCarHardwareHeadNew : public IJsonBlobSerializableCarAttachment {
    R_READONLY(TString, HeadId);

    static TFactory::TRegistrator<TCarHardwareHeadNew> Registrator;
public:
    TCarHardwareHeadNew() = default;
    TCarHardwareHeadNew(const TString& headId);

    EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Unique;
    }

    virtual NJson::TJsonValue BuildReport(const ICarAttachmentReportContext& context) const override;
    virtual bool DeserializeFromJsonBlob(const NJson::TJsonValue& blob) override;

    virtual TString GetServiceAppSlug() const override;

    EDocumentAttachmentType GetTypeName() const override {
        return EDocumentAttachmentType::CarHardwareHeadNew;
    }
};

class TCarHardwareVega : public IJsonBlobSerializableCarAttachment {
    R_READONLY(TString, IMEI);

    static TFactory::TRegistrator<TCarHardwareVega> Registrator;
private:
    TAtomicSharedPtr<TCarHardwareSim> PrimarySim;
    TAtomicSharedPtr<TCarHardwareSim> SecondarySim;

public:
    TCarHardwareVega() = default;
    TCarHardwareVega(const TString& imei);

    EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Unique;
    }

    virtual TString GetAbsenceTagName() const override {
        return "need_vega";
    }

    virtual NJson::TJsonValue BuildReport(const ICarAttachmentReportContext& context) const override;
    bool DeserializeFromJsonBlob(const NJson::TJsonValue& blob) override;

    virtual TVector<TString> GetUsedICCNumbers() const override;
    virtual TString GetServiceAppSlug() const override;

    void SetPrimarySim(const TAtomicSharedPtr<TCarHardwareSim>& sim) {
        PrimarySim = sim;
    }

    void SetSecondarySim(const TAtomicSharedPtr<TCarHardwareSim>& sim) {
        SecondarySim = sim;
    }

    void UpdateSim(const TString& icc, const TString& phoneNumber="", const bool isPrimary=true);

    EDocumentAttachmentType GetTypeName() const override {
        return EDocumentAttachmentType::CarHardwareVega;
    }
};

class TCarSignalDevice : public IJsonBlobSerializableCarAttachment {
    R_READONLY(TString, SerialNumber);
    R_READONLY(TString, Imei);

    static TFactory::TRegistrator<TCarSignalDevice> Registrator;
public:
    TCarSignalDevice() = default;
    TCarSignalDevice(const TString& serialNumber, const TString& imei);

    EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Unique;
    }

    virtual NJson::TJsonValue BuildReport(const ICarAttachmentReportContext& context) const override;
    virtual bool DeserializeFromJsonBlob(const NJson::TJsonValue& blob) override;

    virtual TString GetServiceAppSlug() const override;

    EDocumentAttachmentType GetTypeName() const override {
        return EDocumentAttachmentType::CarSignalDevice;
    }
};
