#include "registry.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/major/client.h>

#include <rtline/util/json_processing.h>

TCarRegistryDocument::TFactory::TRegistrator<TCarRegistryDocument> TCarRegistryDocument::Registrator(EDocumentAttachmentType::CarRegistryDocument);

bool TCarRegistryDocument::DeserializeFromJsonBlob(const NJson::TJsonValue& regDocument) {
    JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::Vin), Vin);
    JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::ContractId), ContractId);
    JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::Dealer), Dealer);
    JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::Lessor), Lessor);
    JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::Manufacturer), Manufacturer);
    JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::Model), Model);
    JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::Number), Number);
    JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::PurchaseContractNumber), PurchaseContractNumber);

    {
        TString registrationIdStr = "";
        JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::RegistrationId), registrationIdStr);
        double valueDouble;
        if (TryFromString(registrationIdStr, valueDouble)) {
            RegistrationId = (ui64)valueDouble;
        }
    }

    JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::RegistrationMDSKey), RegistrationMDSKey);
    JREAD_DOUBLE_NULLABLE_OPT(regDocument, ToString(EFields::DagoCost), DagoCost);
    JREAD_DOUBLE_NULLABLE_OPT(regDocument, ToString(EFields::OsagoCost), OsagoCost);
    JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::OsagoNumber), OsagoNumber);
    JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::OsagoMDSKey), OsagoMDSKey);
    JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::MarkByPts), MarkByPts);
    JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::ProductionYear), ProductionYear);
    JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::PtsNumber), PtsNumber);
    JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::AntitheftSystem), AntitheftSystem);
    JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::NumberOfKeys), NumberOfKeys);
    JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::IMEI), IMEI);
    JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::FuelCardNumber), FuelCardNumber);
    JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::CascoNumber), CascoNumber);
    JREAD_DOUBLE_NULLABLE_OPT(regDocument, ToString(EFields::DkpPrice), DkpPrice);
    JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::Color), Color);
    JREAD_DOUBLE_NULLABLE_OPT(regDocument, ToString(EFields::DsPayment), DsPayment);
    JREAD_DOUBLE_NULLABLE_OPT(regDocument, ToString(EFields::RecontractingDsPayment), RecontractingDsPayment);
    JREAD_DOUBLE_NULLABLE_OPT(regDocument, ToString(EFields::RecontractingContractMileage), RecontractingContractMileage);
    JREAD_UINT_NULLABLE_OPT(regDocument, ToString(EFields::RecontractingLeaseTermMonths), RecontractingLeaseTermMonths);
    JREAD_DOUBLE_NULLABLE_OPT(regDocument, ToString(EFields::RecontractingUnderOverrunRate), RecontractingUnderOverrunRate);
    JREAD_UINT_NULLABLE_OPT(regDocument, ToString(EFields::LeaseTermMonths), LeaseTermMonths);
    JREAD_DOUBLE_NULLABLE_OPT(regDocument, ToString(EFields::ContractMileage), ContractMileage);
    JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::MajorAppMDSKey), MajorAppMDSKey);
    JREAD_DOUBLE_NULLABLE_OPT(regDocument, ToString(EFields::ParkingPermitCost), ParkingPermitCost);
    JREAD_DOUBLE_NULLABLE_OPT(regDocument, ToString(EFields::SendingCarTransporterPrice), SendingCarTransporterPrice);
    JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::TotalIneligiblePartyAtFault), TotalIneligiblePartyAtFault);
    JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::TotalIneligibleStatus), TotalIneligibleStatus);
    JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::TotalIneligibleTicketLink), TotalIneligibleTicketLink);
    JREAD_BOOL_OPT(regDocument, ToString(EFields::TotalInsuranceEligible), TotalInsuranceEligible);
    JREAD_DOUBLE_NULLABLE_OPT(regDocument, ToString(EFields::OverRunRate), OverRunRate);
    JREAD_DOUBLE_NULLABLE_OPT(regDocument, ToString(EFields::UnderRunRate), UnderRunRate);
    JREAD_DOUBLE_NULLABLE_OPT(regDocument, ToString(EFields::UnderOverRunRate), UnderOverRunRate);
    JREAD_STRING_NULLABLE_OPT(regDocument, ToString(EFields::City), City);

    ParseDateIntoField(regDocument, ToString(EFields::LeasingEndDate), LeasingEndDate);
    ParseDateIntoField(regDocument, ToString(EFields::DeptransDocumentsDate), DeptransDocumentsDate);
    ParseDateIntoField(regDocument, ToString(EFields::ParkingPermitStartDate), ParkingPermitStartDate);
    ParseDateIntoField(regDocument, ToString(EFields::ParkingPermitExpirationDate), ParkingPermitExpirationDate);
    ParseDateIntoField(regDocument, ToString(EFields::ContractDate), ContractDate);
    ParseDateIntoField(regDocument, ToString(EFields::ContractIdDate), ContractIdDate);
    ParseDateIntoField(regDocument, ToString(EFields::PurchaseContractDate), PurchaseContractDate);
    ParseDateIntoField(regDocument, ToString(EFields::RecontractingDate), RecontractingDate);
    ParseDateIntoField(regDocument, ToString(EFields::RecontractingContractMileageDate), RecontractingContractMileageDate);
    ParseDateIntoField(regDocument, ToString(EFields::RecontractingLeaseTermMonthsDate), RecontractingLeaseTermMonthsDate);
    ParseDateIntoField(regDocument, ToString(EFields::RecontractingUnderOverrunRateDate), RecontractingUnderOverrunRateDate);
    ParseDateIntoField(regDocument, ToString(EFields::RegistrationDate), RegistrationDate);
    ParseDateIntoField(regDocument, ToString(EFields::TransferDate), TransferDate);
    ParseDateIntoField(regDocument, ToString(EFields::DagoDate), DagoDate);
    ParseDateIntoField(regDocument, ToString(EFields::DagoDateTo), DagoDateTo);
    ParseDateIntoField(regDocument, ToString(EFields::OsagoDate), OsagoDate);
    ParseDateIntoField(regDocument, ToString(EFields::OsagoDateTo), OsagoDateTo);
    ParseDateIntoField(regDocument, ToString(EFields::ReturnDate), ReturnDate);
    ParseDateIntoField(regDocument, ToString(EFields::PretermReturnDate), PretermReturnDate);
    ParseDateIntoField(regDocument, ToString(EFields::EstimatedReturnDate), EstimatedReturnDate);
    ParseDateIntoField(regDocument, ToString(EFields::LeaseFirstPaymentDate), LeaseFirstPaymentDate);
    ParseDateIntoField(regDocument, ToString(EFields::TotalInsuranceAckDate), TotalInsuranceAckDate);
    ParseDateIntoField(regDocument, ToString(EFields::TotalIneligibleDate), TotalIneligibleDate);
    ParseDateIntoField(regDocument, ToString(EFields::SendingCarTransporterDate), SendingCarTransporterDate);
    ParseDateIntoField(regDocument, ToString(EFields::AvailableForLongTermUntil), AvailableForLongTermUntil);

    if (OsagoDateTo == TInstant::Zero() && OsagoDate != TInstant::Zero()) {
        OsagoDateTo = OsagoDate + TDuration::Days(365);
    }

    return true;
}

NJson::TJsonValue TCarRegistryDocument::BuildReport(const ICarAttachmentReportContext& context) const {
    NJson::TJsonValue result;
    auto traits = context.GetRegistryDocumentTraits();
    if (traits & NRegistryDocumentReport::EReportTraits::ReportBasicInfo) {
        JWRITE_DEF(result, ToString(EFields::Vin), Vin, "");
        JWRITE_DEF(result, ToString(EFields::Manufacturer), Manufacturer, "");
        JWRITE_DEF(result, ToString(EFields::Model), Model, "");
        JWRITE_DEF(result, ToString(EFields::Number), Number, "");
        JWRITE_DEF(result, ToString(EFields::RegistrationId), ToString(RegistrationId), "0");
        JWRITE_DEF(result, ToString(EFields::ProductionYear), ProductionYear, "");
        JWRITE_DEF(result, ToString(EFields::IMEI), IMEI, "");
        JWRITE_DEF(result, ToString(EFields::FuelCardNumber), FuelCardNumber, "");
        JWRITE_DEF(result, ToString(EFields::AntitheftSystem), AntitheftSystem, "");
        JWRITE_DEF(result, ToString(EFields::NumberOfKeys), NumberOfKeys, "");
        JWRITE_DEF(result, ToString(EFields::Color), Color, "");
        if (MajorAppMDSKey) {
            result.InsertValue(ToString(EFields::MajorAppMDSKey), MajorAppMDSKey);
            result.InsertValue(ToString(EFields::MajorAppMDSHRKey), context.GetMdsLink("carsharing-car-documents", MajorAppMDSKey));
        }
    }
    if (traits & NRegistryDocumentReport::EReportTraits::ReportContractIdAndDate) {
        JWRITE_DEF(result, ToString(EFields::ContractId), ContractId, "");
        JWRITE_DEF(result, ToString(EFields::Dealer), Dealer, "");
        JWRITE_DEF(result, ToString(EFields::Lessor), Lessor, "");
        JWRITE_DEF(result, ToString(EFields::PurchaseContractNumber), PurchaseContractNumber, "");
        JWRITE_INSTANT_ISOFORMAT_DEF(result, ToString(EFields::ContractDate), ContractDate, TInstant::Zero());
        JWRITE_INSTANT_ISOFORMAT_DEF(result, ToString(EFields::ContractIdDate), ContractIdDate, TInstant::Zero());
        JWRITE_INSTANT_ISOFORMAT_DEF(result, ToString(EFields::PurchaseContractDate), PurchaseContractDate, TInstant::Zero());
        JWRITE_INSTANT_ISOFORMAT_DEF(result, ToString(EFields::RecontractingDate), RecontractingDate, TInstant::Zero());
    }
    if (traits & NRegistryDocumentReport::EReportTraits::ReportRegistrationDate) {
        JWRITE_INSTANT_ISOFORMAT_DEF(result, ToString(EFields::RegistrationDate), RegistrationDate, TInstant::Zero());
        if (RegistrationMDSKey) {
            result.InsertValue(ToString(EFields::RegistrationMDSKey), RegistrationMDSKey);
            result.InsertValue(ToString(EFields::RegistrationMDSHRKey), context.GetMdsLink("carsharing-car-documents", RegistrationMDSKey));
        }
    }
    if (traits & NRegistryDocumentReport::EReportTraits::ReportTransferDate) {
        JWRITE_INSTANT_ISOFORMAT_DEF(result, ToString(EFields::TransferDate), TransferDate, TInstant::Zero());
    }
    if (traits & NRegistryDocumentReport::EReportTraits::ReportOsagoDetails) {
        JWRITE_DEF(result, ToString(EFields::DagoCost), DagoCost, -1.0);
        JWRITE_INSTANT_ISOFORMAT_DEF(result, ToString(EFields::DagoDate), DagoDate, TInstant::Zero());
        JWRITE_INSTANT_ISOFORMAT_DEF(result, ToString(EFields::DagoDateTo), DagoDateTo, TInstant::Zero());

        JWRITE_DEF(result, ToString(EFields::OsagoCost), OsagoCost, -1.0);
        JWRITE_DEF(result, ToString(EFields::OsagoNumber), OsagoNumber, "");
        if (OsagoMDSKey) {
            result.InsertValue(ToString(EFields::OsagoMDSKey), OsagoMDSKey);
            result.InsertValue(ToString(EFields::OsagoMDSHRKey), context.GetMdsLink("carsharing-car-documents", OsagoMDSKey));
        }
        JWRITE_INSTANT_ISOFORMAT_DEF(result, ToString(EFields::OsagoDate), OsagoDate, TInstant::Zero());
        JWRITE_INSTANT_ISOFORMAT_DEF(result, ToString(EFields::OsagoDateTo), OsagoDateTo, TInstant::Zero());
    }
    if (traits & NRegistryDocumentReport::EReportTraits::ReportPtsDetails) {
        JWRITE_DEF(result, ToString(EFields::MarkByPts), MarkByPts, "");
        JWRITE_DEF(result, ToString(EFields::PtsNumber), PtsNumber, "");
    }
    if (traits & NRegistryDocumentReport::EReportTraits::ReportDeptransDocumentsDate) {
        JWRITE_INSTANT_ISOFORMAT_DEF(result, ToString(EFields::DeptransDocumentsDate), DeptransDocumentsDate, TInstant::Zero());
    }
    if (traits & NRegistryDocumentReport::EReportTraits::ReportParkingPermitStartDate) {
        JWRITE_DEF(result, ToString(EFields::ParkingPermitCost), ParkingPermitCost, -1.0);
        JWRITE_INSTANT_ISOFORMAT_DEF(result, ToString(EFields::ParkingPermitStartDate), ParkingPermitStartDate, TInstant::Zero());
        JWRITE_INSTANT_ISOFORMAT_DEF(result, ToString(EFields::ParkingPermitExpirationDate), ParkingPermitExpirationDate, TInstant::Zero());
    }
    if (traits & NRegistryDocumentReport::EReportTraits::ReportCascoNumber) {
        JWRITE_DEF(result, ToString(EFields::CascoNumber), CascoNumber, "");
    }
    if (traits & NRegistryDocumentReport::EReportTraits::ReportDkpPrice) {
        JWRITE_DEF(result, ToString(EFields::DkpPrice), DkpPrice, -1.0);
    }
    if (traits & NRegistryDocumentReport::EReportTraits::ReportDsPayment) {
        JWRITE_DEF(result, ToString(EFields::DsPayment), DsPayment, -1.0);
        JWRITE_DEF(result, ToString(EFields::RecontractingDsPayment), RecontractingDsPayment, -1.0);
    }
    if (traits & NRegistryDocumentReport::EReportTraits::ReportLeaseTermMonths) {
        JWRITE_DEF(result, ToString(EFields::LeaseTermMonths), LeaseTermMonths, 0);
        JWRITE_INSTANT_ISOFORMAT_DEF(result, ToString(EFields::LeaseFirstPaymentDate), LeaseFirstPaymentDate, TInstant::Zero());
        JWRITE_INSTANT_ISOFORMAT_DEF(result, ToString(EFields::LeasingEndDate), LeasingEndDate, TInstant::Zero());
        JWRITE_DEF(result, ToString(EFields::RecontractingLeaseTermMonths), RecontractingLeaseTermMonths, 0);
        JWRITE_INSTANT_ISOFORMAT_DEF(result, ToString(EFields::RecontractingLeaseTermMonthsDate), RecontractingLeaseTermMonthsDate, TInstant::Zero());
    }
    if (traits & NRegistryDocumentReport::EReportTraits::ReportReturnDate) {
        JWRITE_INSTANT_ISOFORMAT_DEF(result, ToString(EFields::ReturnDate), ReturnDate, TInstant::Zero());
        JWRITE_INSTANT_ISOFORMAT_DEF(result, ToString(EFields::PretermReturnDate), PretermReturnDate, TInstant::Zero());
        JWRITE_INSTANT_ISOFORMAT_DEF(result, ToString(EFields::EstimatedReturnDate), EstimatedReturnDate, TInstant::Zero());
    }
    if (traits & NRegistryDocumentReport::EReportTraits::ReportContractMileage) {
        JWRITE_DEF(result, ToString(EFields::ContractMileage), ContractMileage, -1.0);
        JWRITE_DEF(result, ToString(EFields::RecontractingContractMileage), RecontractingContractMileage, -1.0);
        JWRITE_INSTANT_ISOFORMAT_DEF(result, ToString(EFields::RecontractingContractMileageDate), RecontractingContractMileageDate, TInstant::Zero());
    }
    if (traits & NRegistryDocumentReport::EReportTraits::ReportTotalDetails) {
        JWRITE(result, ToString(EFields::TotalInsuranceEligible), TotalInsuranceEligible);
        JWRITE_DEF(result, ToString(EFields::TotalIneligiblePartyAtFault), TotalIneligiblePartyAtFault, "");
        JWRITE_DEF(result, ToString(EFields::TotalIneligibleStatus), TotalIneligibleStatus, "");
        JWRITE_DEF(result, ToString(EFields::TotalIneligibleTicketLink), TotalIneligibleTicketLink, "");
        JWRITE_INSTANT_ISOFORMAT_DEF(result, ToString(EFields::TotalInsuranceAckDate), TotalInsuranceAckDate, TInstant::Zero());
        JWRITE_INSTANT_ISOFORMAT_DEF(result, ToString(EFields::TotalIneligibleDate), TotalIneligibleDate, TInstant::Zero());
    }
    if (traits & NRegistryDocumentReport::EReportTraits::ReportSendingCarTransporterDetails) {
        JWRITE_INSTANT_ISOFORMAT_DEF(result, ToString(EFields::SendingCarTransporterDate), SendingCarTransporterDate, TInstant::Zero());
        JWRITE_DEF(result, ToString(EFields::SendingCarTransporterPrice), SendingCarTransporterPrice, -1.0);
    }
    if (traits & NRegistryDocumentReport::EReportTraits::ReportCity) {
        JWRITE_DEF(result, ToString(EFields::City), City, "");
    }
    JWRITE_DEF(result, ToString(EFields::OverRunRate), OverRunRate, 0.0);
    JWRITE_DEF(result, ToString(EFields::UnderRunRate), UnderRunRate, 0.0);
    JWRITE_DEF(result, ToString(EFields::UnderOverRunRate), UnderOverRunRate, 0.0);
    JWRITE_INSTANT_ISOFORMAT_DEF(result, ToString(EFields::AvailableForLongTermUntil), AvailableForLongTermUntil, TInstant::Zero());
    JWRITE_DEF(result, ToString(EFields::RecontractingUnderOverrunRate), RecontractingUnderOverrunRate, 0.0);
    JWRITE_INSTANT_ISOFORMAT_DEF(result, ToString(EFields::RecontractingUnderOverrunRateDate), RecontractingUnderOverrunRateDate, TInstant::Zero());
    return result;
}

bool TCarRegistryDocument::PatchWithRawRegistryData(const NJson::TJsonValue& registryData) {
    NJson::TJsonValue::TMapType registryMap;
    bool hasChanges = false;
    registryData.GetMap(&registryMap);
    for (auto&& entry : registryMap) {
        auto key = ToLowerUTF8(entry.first);

        if (!entry.second.IsString()) {
            return false;
        }

        auto value = entry.second.GetString();
        if (key == "vin" || key == ToString(TCarRegistryDocument::EFields::Vin)) {
            hasChanges |= UpdateField(Vin, value);
        } else if (key == "договор №" || key == ToString(TCarRegistryDocument::EFields::ContractId)) {
            hasChanges |= UpdateField(ContractId, value);
        } else if (key == "дата дс" || key == ToString(TCarRegistryDocument::EFields::ContractDate)) {
            hasChanges |= UpdateDateField(ContractDate, value);
        } else if (key == "дата договора" || key == ToString(TCarRegistryDocument::EFields::ContractIdDate)) {
            hasChanges |= UpdateDateField(ContractIdDate, value);
        } else if (key == ToString(TCarRegistryDocument::EFields::Lessor)) {
            hasChanges |= UpdateField(Lessor, value);
        } else if (key == ToString(TCarRegistryDocument::EFields::PurchaseContractNumber)) {
            hasChanges |= UpdateField(PurchaseContractNumber, value);
        } else if (key == ToString(TCarRegistryDocument::EFields::PurchaseContractDate)) {
            hasChanges |= UpdateDateField(PurchaseContractDate, value);
        } else if (key == "марка" || key == ToString(TCarRegistryDocument::EFields::Manufacturer)) {
            hasChanges |= UpdateField(Manufacturer, value);
        } else if (key == "модель" || key == ToString(TCarRegistryDocument::EFields::Model)) {
            hasChanges |= UpdateField(Model, value);
        } else if (key == "грз" || key == ToString(TCarRegistryDocument::EFields::Number)) {
            hasChanges |= UpdateField(Number, ToLowerUTF8(value));
        } else if (key == "стс" || key == ToString(TCarRegistryDocument::EFields::RegistrationId)) {
            double valueDouble;
            if (TryFromString(value, valueDouble)) {
                hasChanges |= UpdateField(RegistrationId, (ui64)valueDouble);
            }
        } else if (key == "дата регистрации" || key == ToString(TCarRegistryDocument::EFields::RegistrationDate)) {
            hasChanges |= UpdateDateField(RegistrationDate, value);
        } else if (key == "дата передачи" || key == ToString(TCarRegistryDocument::EFields::TransferDate)) {
            hasChanges |= UpdateDateField(TransferDate, value);
        } else if (key == ToString(TCarRegistryDocument::EFields::DagoCost)) {
            hasChanges |= ParseAndUpdateField(DagoCost, value);
        } else if (key == ToString(TCarRegistryDocument::EFields::DagoDate)) {
            hasChanges |= UpdateDateField(DagoDate, value);
        } else if (key == ToString(TCarRegistryDocument::EFields::DagoDateTo)) {
            hasChanges |= UpdateDateField(DagoDateTo, value);
        } else if (key == ToString(TCarRegistryDocument::EFields::OsagoCost)) {
            hasChanges |= ParseAndUpdateField(OsagoCost, value);
        } else if (key == "осаго №" || key == ToString(TCarRegistryDocument::EFields::OsagoNumber)) {
            hasChanges |= UpdateField(OsagoNumber, value);
        } else if (key == "осаго дата" || key == ToString(TCarRegistryDocument::EFields::OsagoDate)) {
            hasChanges |= UpdateDateField(OsagoDate, value);
        } else if (key == ToString(TCarRegistryDocument::EFields::OsagoDateTo)) {
            hasChanges |= UpdateDateField(OsagoDateTo, value);
        } else if (key == "птс.модель" || key == ToString(TCarRegistryDocument::EFields::MarkByPts)) {
            hasChanges |= UpdateField(MarkByPts, value);
        } else if (key == "год.вып." || key == ToString(TCarRegistryDocument::EFields::ProductionYear)) {
            hasChanges |= UpdateField(ProductionYear, value);
        } else if (key == "№птс" || key == ToString(TCarRegistryDocument::EFields::PtsNumber)) {
            hasChanges |= UpdateField(PtsNumber, value);
        } else if (key == "противоугонная система" || key == ToString(TCarRegistryDocument::EFields::AntitheftSystem)) {
            hasChanges |= UpdateField(AntitheftSystem, value);
        } else if (key == "количество ключей"  || key == ToString(TCarRegistryDocument::EFields::NumberOfKeys)) {
            hasChanges |= UpdateField(NumberOfKeys, value);
        } else if (key == "дата подачи документов в дептранс" || key == ToString(TCarRegistryDocument::EFields::DeptransDocumentsDate)) {
            hasChanges |= UpdateDateField(DeptransDocumentsDate, value);
        } else if (key == ToString(TCarRegistryDocument::EFields::ParkingPermitCost)) {
            hasChanges |= ParseAndUpdateField(ParkingPermitCost, value);
        } else if (key == "дата начала парковочного разрешения" || key == ToString(TCarRegistryDocument::EFields::ParkingPermitStartDate)) {
            hasChanges |= UpdateDateField(ParkingPermitStartDate, value);
        } else if (key == ToString(TCarRegistryDocument::EFields::ParkingPermitExpirationDate)) {
            hasChanges |= UpdateDateField(ParkingPermitExpirationDate, value);
        } else if (key == "вега" || key == ToString(TCarRegistryDocument::EFields::IMEI)) {
            hasChanges |= UpdateField(IMEI, value);
        } else if (key == "топливная карта" || key == ToString(TCarRegistryDocument::EFields::FuelCardNumber)) {
            hasChanges |= UpdateField(FuelCardNumber, value);
        } else if (key == "каско" || key == ToString(TCarRegistryDocument::EFields::CascoNumber)) {
            hasChanges |= UpdateField(CascoNumber, value);
        } else if (key == "цена по дкп" || key == ToString(TCarRegistryDocument::EFields::DkpPrice)) {
            float price;
            if (TryFromString(value, price)) {
                hasChanges |= UpdateField(DkpPrice, price);
            }
        } else if (key == "цвет" || key == "color") {
            hasChanges |= UpdateField(Color, value);
        } else if (key == "платеж по дс (без ндс)" || key == "платеж по дс без ндс" || key == "ежемесячный платеж без ндс" || key == ToString(TCarRegistryDocument::EFields::DsPayment)) {
            float payment;
            if (TryFromString(value, payment)) {
                hasChanges |= UpdateField(DsPayment, payment);
            }
        } else if (key == "срок аренды (месяцев)" || key == "срок аренды" || key == ToString(TCarRegistryDocument::EFields::LeaseTermMonths)) {
            ui32 leaseTermMonths;
            if (TryFromString(value, leaseTermMonths)) {
                hasChanges |= UpdateField(LeaseTermMonths, leaseTermMonths);
            }
        } else if (key == "дата окончания лизинга" || key == ToString(TCarRegistryDocument::EFields::LeasingEndDate)) {
            hasChanges |= UpdateDateField(LeasingEndDate, value);
        } else if (key == "дата первого арендного платежа" || key == ToString(TCarRegistryDocument::EFields::LeaseFirstPaymentDate)) {
            hasChanges |= UpdateDateField(LeaseFirstPaymentDate, value);
        } else if (key == "дата признания статуса тотал" || key == ToString(TCarRegistryDocument::EFields::TotalInsuranceAckDate)) {
            hasChanges |= UpdateDateField(TotalInsuranceAckDate, value);
        } else if (key == "дата возврата" || key == ToString(TCarRegistryDocument::EFields::ReturnDate)) {
            hasChanges |= UpdateDateField(ReturnDate, value);
        } else if (key == "дата досрочного возврата" || key == ToString(TCarRegistryDocument::EFields::PretermReturnDate)) {
            hasChanges |= UpdateDateField(PretermReturnDate, value);
        } else if (key == "рассчетная дата возврата" || key == ToString(TCarRegistryDocument::EFields::EstimatedReturnDate)) {
            hasChanges |= UpdateDateField(EstimatedReturnDate, value);
        } else if (key == "контрактный пробег" || key == ToString(TCarRegistryDocument::EFields::ContractMileage)) {
            float contractMileage;
            if (TryFromString(value, contractMileage)) {
                hasChanges |= UpdateField(ContractMileage, contractMileage);
            }
        } else if (key == "дата отправки автовоза" || key == ToString(TCarRegistryDocument::EFields::SendingCarTransporterDate)) {
            hasChanges |= UpdateDateField(SendingCarTransporterDate, value);
        } else if (key == "стоимость доставки" || key == ToString(TCarRegistryDocument::EFields::SendingCarTransporterPrice)) {
            float price;
            if (TryFromString(value, price)) {
                hasChanges |= UpdateField(SendingCarTransporterPrice, price);
            }
        } else if (key == ToString(TCarRegistryDocument::EFields::TotalIneligibleDate)) {
            hasChanges |= UpdateDateField(TotalIneligibleDate, value);
        } else if (key == ToString(TCarRegistryDocument::EFields::TotalInsuranceEligible)) {
            bool eligible = false;
            hasChanges |= TryFromString(value, eligible) && UpdateField(TotalInsuranceEligible, eligible);
        } else if (key == ToString(TCarRegistryDocument::EFields::TotalIneligiblePartyAtFault)) {
            hasChanges |= UpdateField(TotalIneligiblePartyAtFault, value);
        } else if (key == ToString(TCarRegistryDocument::EFields::TotalIneligibleStatus)) {
            hasChanges |= UpdateField(TotalIneligibleStatus, value);
        } else if (key == ToString(TCarRegistryDocument::EFields::TotalIneligibleTicketLink)) {
            hasChanges |= UpdateField(TotalIneligibleTicketLink, value);
        } else if (key == ToString(TCarRegistryDocument::EFields::UnderOverRunRate) || key == "ставка перепробега") {
            float rate;
            if (TryFromString(value, rate)) {
                hasChanges |= UpdateField(UnderOverRunRate, rate);
            }
        } else if (key == ToString(TCarRegistryDocument::EFields::OverRunRate)) {
            hasChanges |= ParseAndUpdateField(OverRunRate, value);
        } else if (key == ToString(TCarRegistryDocument::EFields::UnderRunRate)) {
            hasChanges |= ParseAndUpdateField(UnderRunRate, value);
        } else if (key == ToString(TCarRegistryDocument::EFields::AvailableForLongTermUntil)) {
            hasChanges |= UpdateDateField(AvailableForLongTermUntil, value);
        } else if (key == ToString(TCarRegistryDocument::EFields::RecontractingDate)) {
            hasChanges |= UpdateDateField(RecontractingDate, value);
        } else if (key == ToString(TCarRegistryDocument::EFields::RecontractingDsPayment)) {
            float payment;
            if (TryFromString(value, payment)) {
                hasChanges |= UpdateField(RecontractingDsPayment, payment);
            }
        } else if (key == ToString(TCarRegistryDocument::EFields::RecontractingContractMileage)) {
            hasChanges |= ParseAndUpdateField(RecontractingContractMileage, value);
        } else if (key == ToString(TCarRegistryDocument::EFields::RecontractingContractMileageDate)) {
            hasChanges |= UpdateDateField(RecontractingContractMileageDate, value);
        } else if (key == ToString(TCarRegistryDocument::EFields::RecontractingLeaseTermMonths)) {
            hasChanges |= ParseAndUpdateField(RecontractingLeaseTermMonths, value);
        } else if (key == ToString(TCarRegistryDocument::EFields::RecontractingLeaseTermMonthsDate)) {
            hasChanges |= UpdateDateField(RecontractingLeaseTermMonthsDate, value);
        } else if (key == ToString(TCarRegistryDocument::EFields::RecontractingUnderOverrunRate)) {
            hasChanges |= ParseAndUpdateField(RecontractingUnderOverrunRate, value);
        } else if (key == ToString(TCarRegistryDocument::EFields::RecontractingUnderOverrunRateDate)) {
            hasChanges |= UpdateDateField(RecontractingUnderOverrunRateDate, value);
        } else if (key == ToString(TCarRegistryDocument::EFields::Dealer)) {
            hasChanges |= UpdateField(Dealer, value);
        } else if (key == ToString(TCarRegistryDocument::EFields::City)) {
            hasChanges |= UpdateField(City, value);
        }
    }

    return hasChanges;
}

bool TCarRegistryDocument::PatchWithCarJSON(const NJson::TJsonValue& payload, const TModelsDB* modelsData, const TInstant actuality) {
    NJson::TJsonValue::TMapType asMap;
    bool hasChanges = false;
    payload.GetMap(&asMap);
    for (auto&& entry : asMap) {
        auto key = ToLowerUTF8(entry.first);

        if (!entry.second.IsString()) {
            continue;
        }

        auto value = entry.second.GetString();
        if (key == "vin") {
            hasChanges |= UpdateField(Vin, value);
        } else if (key == "number") {
            hasChanges |= UpdateField(Number, value);
        } else if (key == "registration_id") {
            ui64 valueInt;
            if (TryFromString(value, valueInt)) {
                hasChanges |= UpdateField(RegistrationId, valueInt);
            }
        } else if (key == "model_code" && modelsData) {
            auto modelFR = modelsData->FetchInfo(value, actuality);
            auto modelPtr = modelFR.GetResultPtr(value);
            if (modelPtr && (modelPtr->GetRegistryManufacturer() || modelPtr->GetRegistryModel())) {
                hasChanges |= UpdateField(Manufacturer, modelPtr->GetRegistryManufacturer());
                hasChanges |= UpdateField(Model, modelPtr->GetRegistryModel());
            }
        } else if (key == "osago_mds_key") {
            hasChanges |= UpdateField(OsagoMDSKey, value);
        } else if (key == "registration_mds_key") {
            hasChanges |= UpdateField(RegistrationMDSKey, value);
        } else if (key == "major_app_mds_key") {
            hasChanges |= UpdateField(MajorAppMDSKey, value);
        }
    }

    return hasChanges;
}

bool TCarRegistryDocument::UpdateOSAGO(bool& isUpdate, const TS3Client::TBucket& mdsBucket, const NDrive::IServer& server, TMessagesCollector& errors, const bool isCurrent) {
    NMajorClient::TOSAGORequest::TOSAGODocument majorDocument;
    if (!server.GetDriveDatabase().GetMajorClient().GetOSAGO(Vin, isCurrent, majorDocument, errors)) {
        return false;
    }

    if (OsagoMDSKey && majorDocument.GetNumber() == OsagoNumber) {
        isUpdate = false;
        return true;
    }

    TString fileName = "OSAGO/" + Vin + "/majorAPI/" + ToString(ModelingNow().Seconds()) + ".pdf";
    if (mdsBucket.PutKey(fileName, majorDocument.GetFile(), errors) != HTTP_OK) {
        return false;
    }

    OsagoNumber = majorDocument.GetNumber();
    OsagoDateTo = majorDocument.GetDateTo();
    OsagoDate = majorDocument.GetDateFrom();
    OsagoMDSKey = fileName;

    isUpdate = true;
    return true;
}
