#include "car.h"
#include "car_model.h"

#include <drive/backend/database/drive/url.h>
#include <drive/backend/database/entity/search_index.h>
#include <drive/backend/database/transaction/assert.h>

TDriveCarInfo::TDriveCarInfoDecoder::TDriveCarInfoDecoder(const TMap<TString, ui32>& decoderBase) {
    Id = GetFieldDecodeIndex("id", decoderBase);
    Number = GetFieldDecodeIndex("number", decoderBase);
    Vin = GetFieldDecodeIndex("vin", decoderBase);
    Model = GetFieldDecodeIndex("model_code", decoderBase);
    RegistrationID = GetFieldDecodeIndex("registration_id", decoderBase);
    IMEI = GetFieldDecodeIndex("imei", decoderBase);
    Specifications = GetFieldDecodeIndex("model_specifications", decoderBase);
}

bool TDriveCarInfo::DeserializeWithDecoder(const TDriveCarInfoDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, Id);
    READ_DECODER_VALUE(decoder, values, Number);
    READ_DECODER_VALUE(decoder, values, Vin);
    READ_DECODER_VALUE(decoder, values, Model);
    READ_DECODER_VALUE_DEF(decoder, values, RegistrationID, 0);
    READ_DECODER_VALUE(decoder, values, IMEI);
    if (decoder.GetSpecifications() >= 0) {
        NJson::TJsonValue specifications;
        const auto& value = values[decoder.GetSpecifications()];
        if (value && !NJson::ReadJsonFastTree(values[decoder.GetSpecifications()], &specifications)) {
            return false;
        }
        if (value && !Specifications.Deserialize(specifications, Id)) {
            return false;
        }
    }
    return true;
}

void TDriveCarInfo::DoBuildReportItem(NJson::TJsonValue& item) const {
    item.InsertValue("object", GetReport(DefaultLocale, NDeviceReport::ReportHistory));
}

const TString& TDriveCarInfo::GetObjectId(const TDriveCarInfo& object) {
    return object.GetId();
}

bool TDriveCarInfo::Parse(const NStorage::TTableRecord& row) {
    return TBaseDecoder::DeserializeFromTableRecord(*this, row);
}

TDriveCarInfo TDriveCarInfo::FromHistoryEvent(const TObjectEvent<TDriveCarInfo>& historyEvent) {
    return historyEvent;
}

NStorage::TTableRecord TDriveCarInfo::SerializeToTableRecord() const {
    NStorage::TTableRecord result;

    // required fields
    result.Set("id", Id);

    if (Vin != "") {
        result.Set("vin", ToUpperUTF8(Vin));
    }

    // optional fields
    if (Number != "") {
        result.Set("number", Number);
    } else {
        result.Set("number", "get_null()");
    }

    if (IMEI != "") {
        result.Set("imei", IMEI);
    } else {
        result.Set("imei", "get_null()");
    }

    result.Set("status", "available");
    if (Model != "unknown" && Model != "") {
        result.Set("model_code", Model);
    }

    if (RegistrationID != 0) {
        result.Set("registration_id", RegistrationID);
    } else {
        result.Set("registration_id", "get_null()");
    }

    if (Specifications) {
        result.Set("model_specifications", Specifications.Serialize());
    }

    return result;
}

namespace {
    void FillHeavySearchProp(TVector<TBasicSearchIndex::TSearchEntityProp>& result, const TString indexingStr, const bool isUndying, const TMaybe<TSearchTraits> traits = {}) {
        size_t maxLength = indexingStr.length() > 2 ? indexingStr.length() - 2 : indexingStr.length();
        for (size_t i = 0; i < maxLength; i++) {
            result.push_back(TBasicSearchIndex::TSearchEntityProp(TString(SubstrUTF8(indexingStr, i, indexingStr.length())), isUndying));
            if (traits) {
                result.back().SetTraits(*traits);
            }
        }
    }
}

void TDriveCarInfo::Index(TBasicSearchIndex& index, bool heavyIndex) const {
    TVector<TBasicSearchIndex::TSearchEntityProp> result;
    if (Vin != "" && Vin != "get_null()") {
        if (Vin.size() == 17) {
            FillHeavySearchProp(result, Vin, false, NDeviceReport::ReportVIN);
        } else {
            result.push_back(TBasicSearchIndex::TSearchEntityProp(Vin).SetTraits(NDeviceReport::ReportVIN));
        }
    }
    if (Number != "" && Number != "get_null()") {
        if (heavyIndex) {
            FillHeavySearchProp(result, Number, true);
        } else if (Number.length() > 1) {
            result.push_back(TBasicSearchIndex::TSearchEntityProp(Number, true));
            result.push_back(TBasicSearchIndex::TSearchEntityProp(TString(SubstrUTF8(Number, 1, Number.length() - 1)), true));
        }
    }
    if (Model != "") {
        if (heavyIndex) {
            FillHeavySearchProp(result, Model, false,  NDeviceReport::ReportModels);
        } else {
            result.push_back(TBasicSearchIndex::TSearchEntityProp(Model).SetTraits(NDeviceReport::ReportModels));
        }
    }
    if (IMEI != "" && IMEI != "get_null()") {
        result.push_back(TBasicSearchIndex::TSearchEntityProp(IMEI).SetTraits(NDeviceReport::ReportIMEI));
    }
    if (RegistrationID != 0) {
        result.push_back(TBasicSearchIndex::TSearchEntityProp(ToString(RegistrationID)).SetTraits(NDeviceReport::ReportRegistrationId));
    }
    if (Id != "") {
        FillHeavySearchProp(result, Id, false,  NDeviceReport::ReportCarId);
    }

    index.Refresh(Id, result);
}

const NJson::TJsonValue& TDriveCarInfo::GetOriginalReport() const {
    return OriginalReport;
}

TString TDriveCarInfo::GetHRReport() const {
    return Sprintf("<a href=\"%s\">%s %s</a>", TCarsharingUrl().CarInfo(Id).data(), Number ? Number.data() : Id.data(), Model.data());
}

TString TDriveCarInfo::GetDeepLink() const {
    return "yandexdrive://cars/" + Number;
}

NJson::TJsonValue TDriveCarInfo::GetReport(ELocalization locale, NDeviceReport::TReportTraits traits) const {
    NJson::TJsonValue result;
    result.InsertValue("number", Number);
    result.InsertValue("model_id", Model);
    if (Specifications) {
        result.InsertValue("model_specifications", Specifications.GetReport(locale));
    }
    if (traits & NDeviceReport::EReportTraits::ReportCarId) {
        result.InsertValue("id", Id);
    }
    if (traits & NDeviceReport::EReportTraits::DeepLink) {
        result.InsertValue("deeplink", GetDeepLink());
    }
    if (traits & NDeviceReport::EReportTraits::ReportIMEI) {
        result.InsertValue("imei", IMEI);
    }
    if (traits & NDeviceReport::EReportTraits::ReportVIN) {
        result.InsertValue("vin", Vin);
    }
    if (traits & NDeviceReport::EReportTraits::ReportRegistrationId) {
        if (RegistrationID != 0) {
            result.InsertValue("registration_id", ToString(RegistrationID));
        } else {
            result.InsertValue("registration_id", "");
        }
    }
    return result;
}

void TDriveCarInfo::FillReport(ELocalization locale, NDeviceReport::TReportTraits traits, NJson::TJsonWriter& result) const {
    result.Write("number", Number);
    result.Write("model_id", Model);
    if (Specifications) {
        result.Write("model_specifications", Specifications.GetReport(locale));
    }
    if (traits & NDeviceReport::EReportTraits::ReportCarId) {
        result.Write("id", Id);
    }
    if (traits & NDeviceReport::EReportTraits::DeepLink) {
        result.Write("deeplink", "yandexdrive://cars/" + Number);
    }
    if (traits & NDeviceReport::EReportTraits::ReportIMEI) {
        result.Write("imei", IMEI);
    }
    if (traits & NDeviceReport::EReportTraits::ReportVIN) {
        result.Write("vin", Vin);
    }
    if (traits & NDeviceReport::EReportTraits::ReportRegistrationId) {
        if (RegistrationID != 0) {
            result.Write("registration_id", ToString(RegistrationID));
        } else {
            result.Write("registration_id", "");
        }
    }
}

NJson::TJsonValue TDriveCarInfo::GetSearchReport() const {
    return GetReport(DefaultLocale,
        EReportTraits::ReportVIN |
        EReportTraits::ReportCarId |
        EReportTraits::ReportModels
    );
}

bool TDriveCarInfo::Patch(const NJson::TJsonValue& info) {
    JREAD_STRING_OPT(info, "id", Id);

    if (info.Has("number")) {
        JREAD_STRING_NULLABLE_OPT(info, "number", Number);
        if (info["number"].IsNull()) {
            Number = "get_null()";
        }
    }

    if (info.Has("vin")) {
        JREAD_STRING_NULLABLE_OPT(info, "vin", Vin);
        if (info["vin"].IsNull()) {
            Vin = "get_null()";
        }
    }

    if (info.Has("imei")) {
        JREAD_STRING_NULLABLE_OPT(info, "imei", IMEI);
        if (info["imei"].IsNull()) {
            IMEI = "get_null()";
        }
    }

    JREAD_STRING_OPT(info, "model_code", Model);

    if (info.Has("registration_id")) {
        if (info["registration_id"].IsString()) {
            TString registrationIdStr = "";
            JREAD_STRING_NULLABLE_OPT(info, "registration_id", registrationIdStr);
            TryFromString(registrationIdStr, RegistrationID);
        } else if (info["registration_id"].IsNull()) {
            RegistrationID = 0;
        }
    }

    if (info.Has("model_specifications")) {
        if (!Specifications.Deserialize(info["model_specifications"], Id)) {
            return false;
        }
    }

    return true;
}

bool TDriveCarInfo::Parse(const NJson::TJsonValue& info, TDriveCarInfo& result) {
    result.OriginalReport = info;
    JREAD_STRING(info, "id", result.Id);
    JREAD_STRING(info, "vin", result.Vin);
    JREAD_STRING(info, "model_code", result.Model);
    JREAD_STRING(info, "number", result.Number);
    JREAD_STRING(info, "imei", result.IMEI);

    TString registrationIdStr;
    JREAD_STRING_OPT(info, "registration_id", registrationIdStr);
    TryFromString(registrationIdStr, result.RegistrationID);

    return true;
}

double NDrive::GetFuelTankVolume(const TDriveCarInfo* carInfo, const TDriveModelData* modelInfo) {
    if (carInfo) {
        auto volume = carInfo->GetSpecifications().GetFuelTankVolume();
        if (volume) {
            return volume;
        }
    }
    if (modelInfo) {
        auto volume = modelInfo->GetSpecifications().GetFuelTankVolume();
        if (volume) {
            return volume;
        }
    }
    return 50;
}

double NDrive::GetNeedFuelingLiters(i32 currentFuelLevelPercent, double fuelTankVolume) {
    if (currentFuelLevelPercent < 0) {
        return fuelTankVolume;
    } else if (currentFuelLevelPercent < 100) {
        return fuelTankVolume * (100.0 - currentFuelLevelPercent) / 100.0;
    } else {
        return 0;
    }
}

double NDrive::GetNeedFuelingLiters(i32 currentFuelLevelPercent, const TDriveCarInfo* carInfo, const TDriveModelData* modelInfo) {
    auto fuelTankVolume = GetFuelTankVolume(carInfo, modelInfo);
    return GetNeedFuelingLiters(currentFuelLevelPercent, fuelTankVolume);
}

const TDriveModelSpecification* NDrive::GetSpecification(std::string_view name, const TDriveCarInfo* carInfo, const TDriveModelData* modelInfo) {
    const TDriveModelSpecification* result = nullptr;
    if (!result && carInfo) {
        result = carInfo->GetSpecifications().Get(name);
    }
    if (!result && modelInfo) {
        result = modelInfo->GetSpecifications().Get(name);
    }
    return result;
}

TDriveCarInfoHistoryManager::TOptionalEvents TDriveCarInfoHistoryManager::GetEvents(const TString& carId, TRange<TInstant> timestampRange, NDrive::TEntitySession& tx, const TQueryOptions& queryOptions) const {
    auto qo = queryOptions;
    qo.AddGenericCondition("id", carId);
    TEventId since = 0;
    return GetEvents(since, timestampRange, tx, qo);
}

bool TDriveCarInfoHistoryManager::GetEventsSinceId(TBase::TEventId id, TBase::TEventPtrs& results, const TInstant actuality) const {
    if (Config.GetUseCache()) {
        return TBase::GetEventsSinceId(id, results, actuality);
    } else {
        return TBase::GetEventsSinceIdDirect(id, results);
    }
}

NJson::TJsonValue TCarsDB::TVinValidationResult::SerializeToJson() {
    NJson::TJsonValue result = NJson::JSON_MAP;

    // Validation status
    switch (ValidationStatus) {
        case EVinValidationStatus::VinOK:
            result["validation_status"] = "ok";
            break;
        case EVinValidationStatus::VinSuspicious:
            result["validation_status"] = "suspicious";
            break;
        case EVinValidationStatus::VinInvalid:
            result["validation_status"] = "invalid";
            break;
        default:
            ERROR_LOG << "Unable to match vin validation status" << Endl;
            break;
    }

    // Recognized model
    if (RecognizedModel != "") {
        result["recognized_model"] = RecognizedModel;
    } else {
        result["recognized_model"] = NJson::JSON_NULL;
    }

    // Suspicious indices
    {
        result["suspicious_indices"] = NJson::JSON_ARRAY;
        for (auto&& index : SuspiciousIndices) {
            result["suspicious_indices"].AppendValue(index);
        }
    }

    return result;
}

void TCarsDBConfig::Init(const TYandexConfig::Section* section) {
    IsSearchEnabled = section->GetDirectives().Value<bool>("IsSearchEnabled", IsSearchEnabled);
    HeavyIndexEnabled = section->GetDirectives().Value<bool>("HeavyIndexEnabled", HeavyIndexEnabled);
}

void TCarsDBConfig::ToString(IOutputStream& os) const {
    os << "IsSearchEnabled: " << IsSearchEnabled << Endl;
    os << "HeavyIndexEnabled: " << HeavyIndexEnabled << Endl;
}

TCarsDB::TCarsDB(const IHistoryContext& context, const TCarsDBConfig& config)
    : TCacheBase("car", context, THistoryConfig()
        .SetDeep(TDuration::Zero())
        .SetUseCache(false)
    )
    , Database(context.GetDatabase())
    , Config(config)
    , CarsTable(MakeHolder<TDBTable>(Database))
    , Index(config.GetIsSearchEnabled() ? MakeHolder<TBasicSearchIndex>() : nullptr)
{
}

TCarsDB::~TCarsDB() {
}

bool TCarsDB::DoStart() {
    return
        CarsTable->SetPeriod(TDuration::Minutes(1)).Start() &&
        TCacheBase::DoStart();
}

bool TCarsDB::DoStop() {
    return
        CarsTable->Stop() &&
        TCacheBase::DoStop();
}

void TCarsDB::AcceptHistoryEventUnsafe(const TObjectEvent<TRecordType>& ev) const {
    auto entry = TDriveCarInfo::FromHistoryEvent(ev);
    auto carId = entry.GetId();
    if (ev.GetHistoryAction() == EObjectHistoryAction::Remove) {
        if (Index) {
            Index->Remove(carId);
        }
        Objects.erase(ev.GetId());
    } else {
        if (Index) {
            entry.Index(*Index, Config.GetHeavyIndexEnabled());
        }
        Objects[ev.GetId()] = ev;
    }
    Objects[carId] = std::move(entry);
}

bool TCarsDB::DoRebuildCacheUnsafe() const {
    Objects.clear();
    auto tx = GetHistoryManager().BuildTx<NSQL::ReadOnly>();
    auto table = Database->GetTable(CarsTable->GetTableName());
    auto transaction = tx.GetTransaction();
    NStorage::TObjectRecordsSet<TRecordType> records;
    {
        auto result = table->GetRows("", records, transaction);
        if (!result->IsSucceed()) {
            ERROR_LOG << "Cannot refresh data for " << CarsTable->GetTableName() << Endl;
            return false;
        }
    }
    {
        auto optionalEvents = GetHistoryManager().GetEvents(0, tx);
        if (!optionalEvents) {
            ERROR_LOG << "cannot GetEvents: " << tx.GetStringReport() << Endl;
            return false;
        }
        TSet<std::pair<TString, TString>> knownCarNumbers;
        for (auto&& event : *optionalEvents) {
            if (!event.GetNumber()) {
                continue;
            }
            auto knownNumberPair = std::make_pair(event.GetId(), event.GetNumber());
            if (knownCarNumbers.emplace(knownNumberPair).second) {
                TVector<TBasicSearchIndex::TSearchEntityProp> props;
                props.push_back(TBasicSearchIndex::TSearchEntityProp(event.GetNumber(), true));
                if (Index) {
                    Index->Refresh(event.GetId(), props);
                }
            }
        }
    }
    for (auto&& i : records) {
        auto carId = i.GetId();
        if (Index) {
            i.Index(*Index, Config.GetHeavyIndexEnabled());
        }
        Objects.emplace(carId, i);
    }
    return true;
}

TCarsDB::TOptionalFormerNumbers TCarsDB::FetchFormerNumbers(const TString& carId, NDrive::TEntitySession& tx) const {
    auto optionalEvents = HistoryManager->GetEvents(carId, {}, tx);
    if (!optionalEvents) {
        return {};
    }
    TFormerNumbers result;
    {
        TString currentNumber = "";
        for (auto&& event : *optionalEvents) {
            auto newNumber = event.GetNumber();
            if (newNumber != currentNumber) {
                currentNumber = newNumber;
                TFormerNumber formerNumber;
                formerNumber.Value = event.GetNumber();
                formerNumber.Timestamp = event.GetHistoryInstant();
                result.push_back(std::move(formerNumber));
            }
        }
    }
    return result;
}

TVector<TString> TCarsDB::GetMatchingIds(const TSearchRequest& searchRequest, const std::function<bool(const TString&)>& entityFilter) const {
   return Yensured(Index)->GetMatchingIds(searchRequest, entityFilter);
}

TStringBuf TCarsDB::GetEventObjectId(const TObjectEvent<TRecordType>& ev) const {
    return ev.GetId();
}

TCarsDB::TFetchResult TCarsDB::FetchInfo(const TInstant actuality) const {
    Y_ENSURE_BT(RefreshCache(actuality));
    auto rg = MakeObjectReadGuard();
    TFetchResult result;
    for (auto&& objectIt : Objects) {
        result.AddResult(objectIt.first, objectIt.second);
    }
    return result;
}

TCarsDB::TFetchResult TCarsDB::FetchCarsByVIN(const TSet<TString>& vins, NDrive::TEntitySession& session) const {
    return CarsTable->FetchInfoByField(vins, "vin", session);
}

TCarsDB::TFetchResult TCarsDB::FetchInfo(const TString& id, const TInstant actuality) const {
    return FetchInfo(TSet<TString>({ id }), actuality);
}

TCarsDB::TFetchResult TCarsDB::FetchInfo(const TSet<TString>& ids, const TInstant actuality) const {
    Y_ENSURE_BT(RefreshCache(actuality));
    auto rg = MakeObjectReadGuard();
    TFetchResult result;
    for (auto&& id : ids) {
        auto entryIt = Objects.find(id);
        if (entryIt != Objects.end()) {
            result.AddResult(id, entryIt->second);
        }
    }
    return result;
}

TString TCarsDB::RegisterNewCar(const TString& userId, const TString& number, const TString& imei, const TString& model, NDrive::TEntitySession& session, const TString& carId, const TString& vin) const {
    TDriveCarInfo newCar;
    newCar.SetNumber(number).SetIMEI(imei).SetModel(model).SetVin(vin);
    if (carId) {
        newCar.SetId(carId);
    }
    TDriveCarInfo upsertedCar;
    if (!UpdateCar(newCar, userId, session, &upsertedCar, true)) {
        return "";
    }
    return upsertedCar.GetId();
}

TString TCarsDB::RegisterNewCar(const TString& userId, const TString& number, const TString& imei, const TString& model) const {
    NDrive::TEntitySession session(Database->CreateTransaction(false));
    auto carId = RegisterNewCar(userId, number, imei, model, session);
    if (!carId || !session.Commit()) {
        return "";
    }
    return carId;
}

TCarsDB::TVinValidationResult TCarsDB::ValidateVIN(TString vin, NDrive::TEntitySession& session) const {
    // According to https://st.yandex-team.ru/DRIVEBACK-425
    const TVector<size_t> VIN_BLOCK_LENGTHS = {3, 5, 1, 1, 1};
    TVinValidationResult result;
    vin.to_upper();

    // Check the rules which does not require matching with DB entries first.
    {
        if (vin.size() != 17) {
            result.ValidationStatus = TVinValidationResult::EVinValidationStatus::VinInvalid;
            return result;
        }

        for (size_t i = 0; i < vin.size(); ++i) {
            bool isLetter = (vin[i] >= 'A' && vin[i] <= 'Z');
            bool isDigit = (vin[i] >= '0' && vin[i] <= '9');
            bool isBadLetter = (vin[i] == 'I' || vin[i] == 'O' || vin[i] == 'Q');
            if (!(isLetter || isDigit) || isBadLetter) {
                result.ValidationStatus = TVinValidationResult::EVinValidationStatus::VinInvalid;
                result.SuspiciousIndices.push_back(i);
            }
        }
        if (result.ValidationStatus == TVinValidationResult::EVinValidationStatus::VinInvalid) {
            return result;
        }
    }

    // Precalculate block occurrences, find most likely model
    TFetchResult fetchResult = FetchInfo(session);

    TVector<TMap<TString, ui32>> blockElementOccurrences(VIN_BLOCK_LENGTHS.size());
    TMap<TString, ui32> modelFrequencies;
    TString vinModelCode = vin.substr(VIN_BLOCK_LENGTHS[0], VIN_BLOCK_LENGTHS[1]);

    for (auto&& entry : fetchResult.GetResult()) {
        auto car = entry.second;
        TString vin = car.GetVin();

        if (vin.size() != 17) {
            // Incorrect vin in the DB
            continue;
        }

        TString carVinModelCode = vin.substr(VIN_BLOCK_LENGTHS[0], VIN_BLOCK_LENGTHS[1]);
        if (carVinModelCode == vinModelCode) {
            ++modelFrequencies[car.GetModel()];
        }

        size_t currentBlockStart = 0;
        for (size_t i = 0; i < VIN_BLOCK_LENGTHS.size(); ++i) {
            TString block = vin.substr(currentBlockStart, VIN_BLOCK_LENGTHS[i]);
            ++blockElementOccurrences[i][block];
            currentBlockStart += VIN_BLOCK_LENGTHS[i];
        }
    }

    // Find the most frequent model
    ui32 maxFrequency = 0;
    TString modelCode = "";
    for (auto&& entry : modelFrequencies) {
        if (entry.second > maxFrequency) {
            maxFrequency = entry.second;
            modelCode = entry.first;
        }
    }
    if (maxFrequency > 0) {
        result.RecognizedModel = modelCode;
    }

    // Compare blocks with calculated
    result.ValidationStatus = TVinValidationResult::EVinValidationStatus::VinOK;
    size_t currentBlockStart = 0;
    for (size_t i = 0; i < VIN_BLOCK_LENGTHS.size(); ++i) {
        TString block = vin.substr(currentBlockStart, VIN_BLOCK_LENGTHS[i]);
        if (blockElementOccurrences[i][block] < 5) {
            result.ValidationStatus = TVinValidationResult::EVinValidationStatus::VinSuspicious;
            for (size_t j = 0; j < VIN_BLOCK_LENGTHS[i]; ++j) {
                result.SuspiciousIndices.push_back(currentBlockStart + j);
            }
        }
        currentBlockStart += VIN_BLOCK_LENGTHS[i];
    }

    return result;
}

bool TCarsDB::UpdateCarFromJSON(const NJson::TJsonValue& newCarJSON, const TString& carId, const TString& operatorUserId, NDrive::TEntitySession& session) const {
    TDriveCarInfo currentCarInfo;
    bool isNewCar = false;

    {
        auto gCarsData = FetchInfo(carId, session);
        auto result = gCarsData.GetResult();
        if (!result.empty()) {
            currentCarInfo = result.find(carId)->second;
        } else {
            isNewCar = true;
        }
    }

    if (!currentCarInfo.Patch(newCarJSON)) {
        ERROR_LOG << "unable to patch car info with JSON" << Endl;
        return false;
    }

    return UpdateCar(currentCarInfo, operatorUserId, session, nullptr, isNewCar);
}

bool TCarsDB::IsSearchEnabled() const {
    return Config.GetIsSearchEnabled();
}

TMap<TString, TString> TCarsDB::GetAllVins(NDrive::TEntitySession& session) const {
    TFetchResult fetchResult = FetchInfo(session);
    TMap<TString, TString> result;
    for (auto&& entry : fetchResult.GetResult()) {
        auto car = entry.second;
        result.emplace(car.GetVin(), car.GetId());
    }
    return result;
}

bool TCarsDB::UpdateCar(TDriveCarInfo carInfo, const TString& operatorUserId, NDrive::TEntitySession& session, TDriveCarInfo* upsertedCarPtr, bool isNewCar) const {
    isNewCar |= (carInfo.GetId() == "" || carInfo.GetId() == "uuid_generate_v4()");

    NStorage::TObjectRecordsSet<TDriveCarInfo> upsertedCars;
    if (!isNewCar) {
        auto optionalEvents = HistoryManager->GetEvents(carInfo.GetId(), {}, session);
        if (!optionalEvents) {
            return false;
        }
        if (optionalEvents->empty()) {
            auto fetchResult = FetchInfo(carInfo.GetId(), session);
            if (!fetchResult) {
                session.AddErrorMessage("CarsDB::UpdateCar", "unable to FetchInfo of " + carInfo.GetId());
                return false;
            }
            auto currentState = fetchResult.GetResultPtr(carInfo.GetId());
            if (!currentState || !HistoryManager->AddHistory(*currentState, operatorUserId, EObjectHistoryAction::UpdateData, session)) {
                session.AddErrorMessage("car", "unable to write initial history");
                return false;
            }
        }

        if (!CarsTable->Upsert(carInfo, session, &upsertedCars)) {
            session.AddErrorMessage("car", "unable to upsert car in db table");
            return false;
        }
    } else {
        if (!carInfo.GetId()) {
            carInfo.SetId("uuid_generate_v4()");
        }
        if (!CarsTable->Insert(carInfo, session, &upsertedCars)) {
            session.AddErrorMessage("car", "unable to insert car in db table");
            return false;
        }
    }

    if (upsertedCars.size() != 1) {
        session.SetErrorInfo("car", "upsertedCars.size() != 1", EDriveSessionResult::InternalError);
        return false;
    }

    if (!HistoryManager->AddHistory(*upsertedCars.begin(), operatorUserId, isNewCar ? EObjectHistoryAction::Add : EObjectHistoryAction::UpdateData, session)) {
        session.AddErrorMessage("car", "unable to write history");
        return false;
    }

    if (upsertedCarPtr != nullptr) {
        *upsertedCarPtr = *upsertedCars.begin();
    }

    return true;
}

TString TCarNumbersDB::GetTableName() const {
    return "car";
}

TString TCarNumbersDB::GetMainId(const TDriveCarInfo& e) const {
    return e.GetNumber();
}

TString TCarNumbersDB::GetColumnName() const {
    return "number";
}

TString TDBImei::GetTableName() const {
    return "car";
}

TString TDBImei::GetMainId(const TDriveCarInfo& e) const {
    return e.GetIMEI();
}

TString TDBImei::GetColumnName() const {
    return "imei";
}

TString TDBVin::GetTableName() const {
    return "car";
}

TString TDBVin::GetMainId(const TDriveCarInfo& e) const {
    return e.GetVin();
}

TString TDBVin::GetColumnName() const {
    return "vin";
}

template <>
NJson::TJsonValue NJson::ToJson(const TObjectEvent<TDriveCarInfo>& object) {
    return object.SerializeToJson();
}

template <>
NJson::TJsonValue NJson::ToJson(const TCarsDB::TFormerNumber& object) {
    NJson::TJsonValue result;
    NJson::InsertField(result, "number", object.Value);
    NJson::InsertField(result, "timestamp", object.Timestamp.Seconds());
    return result;
}
