#pragma once

#include "car_report_traits.h"
#include "specification.h"

#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/entity/search_request.h>
#include <drive/backend/database/history/cache.h>
#include <drive/backend/database/history/manager.h>
#include <drive/backend/database/transaction/tx.h>

#include <rtline/util/json_processing.h>
#include <rtline/util/types/accessor.h>

using TReportTraits = NDeviceReport::TReportTraits;
using EReportTraits = NDeviceReport::EReportTraits;

class TBasicSearchIndex;
class TDriveModelData;

class TDriveCarInfo {
public:
    using TId = TString;

private:
    NJson::TJsonValue OriginalReport = NJson::JSON_NULL;
    R_FIELD(TId, Id);
    R_FIELD(TString, Number);
    R_FIELD(TString, Vin);
    R_FIELD(TString, IMEI);
    R_FIELD(TString, Model, "unknown");
    R_FIELD(ui64, RegistrationID, 0);
    R_FIELD(TDriveModelSpecifications, Specifications);

public:
    class TDriveCarInfoDecoder: public TBaseDecoder {
        R_FIELD(i32, Id, -1);
        R_FIELD(i32, Number, -1);
        R_FIELD(i32, Vin, -1);
        R_FIELD(i32, Model, -1);
        R_FIELD(i32, RegistrationID, -1);
        R_FIELD(i32, IMEI, -1);
        R_FIELD(i32, Specifications, -1);

    public:
        TDriveCarInfoDecoder() = default;
        TDriveCarInfoDecoder(const TMap<TString, ui32>& decoderBase);
    };
    using TDecoder = TDriveCarInfoDecoder;

public:
    static TDriveCarInfo FromHistoryEvent(const TObjectEvent<TDriveCarInfo>& historyEvent);
    static const TString& GetObjectId(const TDriveCarInfo& object);

    static bool Parse(const NJson::TJsonValue& info, TDriveCarInfo& result);

public:
    bool DeserializeWithDecoder(const TDriveCarInfoDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);

    void DoBuildReportItem(NJson::TJsonValue& item) const;

    TString GetDeepLink() const;

    bool Parse(const NStorage::TTableRecord& row);
    NStorage::TTableRecord SerializeToTableRecord() const;

    void Index(TBasicSearchIndex& index, bool heavyIndex) const;

    const NJson::TJsonValue& GetOriginalReport() const;
    TString GetHRReport() const;
    NJson::TJsonValue GetSearchReport() const;
    NJson::TJsonValue GetReport(ELocalization locale, NDeviceReport::TReportTraits traits) const;
    void FillReport(ELocalization locale, NDeviceReport::TReportTraits traits, NJson::TJsonWriter& result) const;

    bool Patch(const NJson::TJsonValue& info);
};

namespace NDrive {
    double GetFuelTankVolume(const TDriveCarInfo* carInfo, const TDriveModelData* modelInfo);
    double GetNeedFuelingLiters(i32 currentFuelLevelPercent, double fuelTankVolume);
    double GetNeedFuelingLiters(i32 currentFuelLevelPercent, const TDriveCarInfo* carInfo, const TDriveModelData* modelInfo);

    const TDriveModelSpecification* GetSpecification(std::string_view name, const TDriveCarInfo* carInfo, const TDriveModelData* modelInfo);
}

class TDriveCarInfoHistoryManager: public TIndexedAbstractHistoryManager<TDriveCarInfo> {
private:
    using TBase = TIndexedAbstractHistoryManager<TDriveCarInfo>;
    using TIdRef = typename TIdTypeSelector<typename TDriveCarInfo::TId>::TIdRef;

public:
    TDriveCarInfoHistoryManager(const IHistoryContext& context, const THistoryConfig& hConfig)
        : TBase(context, "drive_car_info_history", hConfig)
    {
    }

    using TBase::GetEvents;
    TOptionalEvents GetEvents(const TString& carId, TRange<TInstant> timestampRange, NDrive::TEntitySession& tx, const TQueryOptions& queryOptions = {}) const;

    bool GetEventsSinceId(TBase::TEventId id, TBase::TEventPtrs& results, const TInstant actuality) const override;
};

class TCarsDBConfig {
    R_READONLY(bool, IsSearchEnabled, false);
    R_READONLY(bool, HeavyIndexEnabled, false);
public:
    void Init(const TYandexConfig::Section* section);
    void ToString(IOutputStream& os) const;
};

class TCarsTable: public TCachedEntityManager<TDriveCarInfo, true, TString, TEmptyFilter> {
private:
    using TBase = TCachedEntityManager<TDriveCarInfo, true, TString, TEmptyFilter>;

public:
    using TBase::TBase;

    virtual TString GetTableName() const override {
        return "car";
    }

    virtual TString GetMainId(const TDriveCarInfo& e) const override {
        return e.GetId();
    }
};

class TCarsDB: public TDBCacheWithHistoryOwner<TDriveCarInfoHistoryManager, TDriveCarInfo> {
private:
    using TCacheBase = TDBCacheWithHistoryOwner<TDriveCarInfoHistoryManager, TDriveCarInfo>;
    using TDBTable = TCarsTable;

public:
    using TFetchResult = TDBTable::TFetchResult;
    using TRecordType = TDriveCarInfo;

private:
    NStorage::IDatabase::TPtr Database;
    TCarsDBConfig Config;

    THolder<TDBTable> CarsTable;
    THolder<TBasicSearchIndex> Index;

    virtual void AcceptHistoryEventUnsafe(const TObjectEvent<TRecordType>& ev) const override;
    virtual bool DoRebuildCacheUnsafe() const override;
    virtual TStringBuf GetEventObjectId(const TObjectEvent<TRecordType>& ev) const override;

public:
    TCarsDB(const IHistoryContext& context, const TCarsDBConfig& config = TCarsDBConfig());
    ~TCarsDB();

public:
    struct TFormerNumber {
        TString Value;
        TInstant Timestamp;
    };
    using TFormerNumbers = TVector<TFormerNumber>;
    using TOptionalFormerNumbers = TMaybe<TFormerNumbers>;

    struct TVinValidationResult {
        enum EVinValidationStatus {
            VinOK,
            VinSuspicious,
            VinInvalid
        };

        EVinValidationStatus ValidationStatus = VinOK;
        TString RecognizedModel;
        TVector<ui32> SuspiciousIndices;

        TVinValidationResult() = default;
        TVinValidationResult(EVinValidationStatus validationStatus, TString recognizedModel, TVector<ui32> suspiciousIndices)
            : ValidationStatus(validationStatus)
            , RecognizedModel(recognizedModel)
            , SuspiciousIndices(suspiciousIndices)
        {
        }

        NJson::TJsonValue SerializeToJson();
    };

public:
    TVinValidationResult ValidateVIN(TString vin, NDrive::TEntitySession& session) const;

    TString RegisterNewCar(const TString& userId, const TString& number, const TString& imei, const TString& model, NDrive::TEntitySession& session, const TString& carId = Default<TString>(), const TString& vin = Default<TString>()) const;
    TString RegisterNewCar(const TString& userId, const TString& number, const TString& imei, const TString& model = "kia_rio") const;
    bool UpdateCarFromJSON(const NJson::TJsonValue& newCarJSON, const TString& carId, const TString& operatorUserId, NDrive::TEntitySession& session) const;
    bool UpdateCar(TDriveCarInfo carInfo, const TString& operatorUserId, NDrive::TEntitySession& session, TDriveCarInfo* upsertedCarPtr = nullptr, bool isNewCar = false) const;

    template <class... TArgs>
    TDBTable::TFetchResult GetCached(TArgs&&... args) const {
        return CarsTable->GetCached(std::forward<TArgs>(args)...);
    }
    template <class... TArgs>
    TDBTable::TFetchResult GetCachedOrFetch(TArgs&&... args) const {
        return CarsTable->GetCachedOrFetch(std::forward<TArgs>(args)...);
    }
    TDBTable::TFetchResult FetchInfo() const {
        return CarsTable->FetchInfo();
    }
    template <class TArg>
    TDBTable::TFetchResult FetchInfo(TArg&& arg) const {
        return CarsTable->FetchInfo(std::forward<TArg>(arg));
    }
    template <class TArg>
    TDBTable::TFetchResult FetchInfo(TArg&& arg, NDrive::TEntitySession& session) const {
        return CarsTable->FetchInfo(std::forward<TArg>(arg), session);
    }

    TMap<TString, TString> GetAllVins(NDrive::TEntitySession& session) const;
    TFetchResult FetchCarsByVIN(const TSet<TString>& vins, NDrive::TEntitySession& session) const;
    TOptionalFormerNumbers FetchFormerNumbers(const TString& carId, NDrive::TEntitySession& tx) const;

    TFetchResult FetchInfo(const TInstant actuality) const;
    TFetchResult FetchInfo(const TString& id, const TInstant actuality) const;
    TFetchResult FetchInfo(const TSet<TString>& ids, const TInstant actuality) const;

    bool IsSearchEnabled() const;
    TVector<TString> GetMatchingIds(const TSearchRequest& searchRequest, const std::function<bool(const TString&)>& entityFilter) const;

protected:
    virtual bool DoStart() override;
    virtual bool DoStop() override;
};

class TCarNumbersDB: public TCachedEntityManager<TDriveCarInfo> {
private:
    using TBase = TCachedEntityManager<TDriveCarInfo>;
public:
    using TBase::TBase;

    virtual TString GetTableName() const override;
    virtual TString GetMainId(const TDriveCarInfo& e) const override;
    virtual TString GetColumnName() const override;
};

class TDBImei: public TCachedEntityManager<TDriveCarInfo> {
private:
    using TBase = TCachedEntityManager<TDriveCarInfo>;

public:
    using TBase::TBase;

    virtual TString GetTableName() const override;
    virtual TString GetMainId(const TDriveCarInfo& e) const override;
    virtual TString GetColumnName() const override;
};

class TDBVin: public TCachedEntityManager<TDriveCarInfo> {
private:
    using TBase = TCachedEntityManager<TDriveCarInfo>;

public:
    using TBase::TBase;

    virtual TString GetTableName() const override;
    virtual TString GetMainId(const TDriveCarInfo& e) const override;
    virtual TString GetColumnName() const override;
};
