#include "car_model.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/common/localization.h>

#include <rtline/util/json_processing.h>

#include <util/generic/serialized_enum.h>

namespace {
    template <class T>
    bool GetOptional(const NStorage::TTableRecord& row, const TString& key, T& value) {
        if (const TString& str = row.Get(key)) {
            typename T::TValueType parsed;
            if (!TryFromString(str, parsed)) {
                return false;
            }
            value = parsed;
        }
        return true;
    };
}

template <>
NJson::TJsonValue NJson::ToJson(const TDriveModelData::TDefaultTagDescription& object) {
    return object.SerializeToJson();
}

bool TDriveModelData::TDefaultTagDescription::DeserializeFromJson(const NJson::TJsonValue& json) {
    if (json.GetType() != NJson::EJsonValueType::JSON_MAP) {
        return false;
    }
    JREAD_STRING(json, "tag_name", TagName);
    JREAD_STRING(json, "comment", Comment);
    JREAD_INT(json, "priority", Priority);

    if (TagName == "" || Comment == "") {
        return false;
    }

    return true;
}

NJson::TJsonValue TDriveModelData::TDefaultTagDescription::SerializeToJson() const {
    NJson::TJsonValue result;
    result["tag_name"] = TagName;
    result["comment"] = Comment;
    result["priority"] = Priority;
    return result;
}

bool TDriveModelData::Parse(const NStorage::TTableRecord& row) {
    Name = row.Get("name");
    ShortName = row.Get("short_name");
    Code = row.Get("code");
    Manufacturer = row.Get("manufacturer");
    FuelDistanceEnabled = FromStringWithDefault(row.Get("fuel_distance_enabled"), FuelDistanceEnabled);
    FuelIconUrl = row.Get("fuel_icon_url");
    FuelType = row.Get("fuel_type");
    ImageAngleUrl = row.Get("image_angle_url");
    ImageBackgroundUrl = row.Get("image_background_url");
    ImageLargeUrl = row.Get("image_large_url");
    ImageSmallUrl = row.Get("image_small_url");
    ImageMapUrl2X = row.Get("image_map_url_2x");
    ImageMapUrl3X = row.Get("image_map_url_3x");
    FuelCapSide = row.Get("fuel_cap_side");
    ImagePinUrl2X = row.Get("image_pin_url_2x");
    ImagePinUrl3X = row.Get("image_pin_url_3x");
    RegistryManufacturer = row.Get("registry_manufacturer");
    RegistryModel = row.Get("registry_model");
    Rank = FromStringWithDefault(row.Get("rank"), Rank);
    FullCostCasco = FromStringWithDefault(row.Get("full_cost_casco"), FullCostCasco);
    Segment = row.Get("segment");
    GetOptional(row, "deprecated", Deprecated);

    if (row.Get("default_tags")) {
        TString defaultTagsStr = row.Get("default_tags");
        if (defaultTagsStr == "") {
            defaultTagsStr = "[]";
        }

        NJson::TJsonValue tagsArray;
        bool isReadSuccessful = NJson::ReadJsonFastTree(defaultTagsStr, &tagsArray);
        if (!isReadSuccessful || !ParseDefaultTags(tagsArray)) {
            return false;
        }
    }

    if (const auto& specificationsString = row.Get("model_specifications")) {
        NJson::TJsonValue specifications;
        if (!NJson::ReadJsonFastTree(specificationsString, &specifications)) {
            return false;
        }
        if (!Specifications.Deserialize(specifications, Code)) {
            return false;
        }
    }

    if (const TString& metaString = row.Get("meta")) {
        NJson::TJsonValue meta;
        if (!NJson::ReadJsonFastTree(metaString, &meta)) {
            return false;
        }
        if (!NJson::ParseField(meta, NJson::Dictionary(Meta))) {
            return false;
        }
    }
    if (const TString& visual = row.Get("visual")) {
        if (!NJson::ReadJsonFastTree(visual, &Visual)) {
            return false;
        }
    }

    return GetOptional(row, "z_index", ZIndex)
        && GetOptional(row, "first_maintenance_mileage", FirstMaintenanceMileage)
        && GetOptional(row, "maintenance_mileage", MaintenanceMileage)
        && GetOptional(row, "intermediate_maintenance_mileage", IntermediateMaintenanceMileage)
        && GetOptional(row, "maintenance_period", MaintenancePeriod);
}

bool TDriveModelData::Patch(const NJson::TJsonValue& updatePayload) {
    JREAD_STRING_OPT(updatePayload, "name", Name);
    JREAD_STRING_OPT(updatePayload, "short_name", ShortName);
    JREAD_STRING_OPT(updatePayload, "code", Code);
    JREAD_STRING_OPT(updatePayload, "manufacturer", Manufacturer);
    JREAD_BOOL_OPT  (updatePayload, "fuel_distance_enabled", FuelDistanceEnabled);
    JREAD_STRING_OPT(updatePayload, "fuel_icon_url", FuelIconUrl);
    JREAD_STRING_OPT(updatePayload, "fuel_type", FuelType);
    JREAD_STRING_OPT(updatePayload, "image_angle_url", ImageAngleUrl);
    JREAD_STRING_OPT(updatePayload, "image_background_url", ImageBackgroundUrl);
    JREAD_STRING_OPT(updatePayload, "image_large_url", ImageLargeUrl);
    JREAD_STRING_OPT(updatePayload, "image_small_url", ImageSmallUrl);
    JREAD_STRING_OPT(updatePayload, "image_map_url_2x", ImageMapUrl2X);
    JREAD_STRING_OPT(updatePayload, "image_map_url_3x", ImageMapUrl3X);
    JREAD_STRING_OPT(updatePayload, "fuel_cap_side", FuelCapSide);
    JREAD_STRING_OPT(updatePayload, "image_pin_url_2x", ImagePinUrl2X);
    JREAD_STRING_OPT(updatePayload, "image_pin_url_3x", ImagePinUrl3X);
    JREAD_STRING_OPT(updatePayload, "registry_manufacturer", RegistryManufacturer);
    JREAD_STRING_OPT(updatePayload, "registry_model", RegistryModel);
    JREAD_STRING_OPT(updatePayload, "segment", Segment);
    JREAD_INT_OPT   (updatePayload, "rank", Rank);
    JREAD_UINT_OPT  (updatePayload, "full_cost_casco", FullCostCasco);
    JREAD_BOOL_OPT  (updatePayload, "deprecated", Deprecated);
    if (updatePayload.Has("default_tags")) {
        auto defaultTagsJson = updatePayload["default_tags"];
        if (!ParseDefaultTags(defaultTagsJson)) {
            return false;
        }
    }
    if (updatePayload.Has("model_specifications")) {
        auto specificationsJson = updatePayload["model_specifications"];
        if (!Specifications.Deserialize(specificationsJson, Code)) {
            return false;
        }
    }
    if (updatePayload.Has("meta")) {
        TMeta meta;
        if (!NJson::ParseField(updatePayload["meta"], NJson::Dictionary(meta))) {
            return false;
        }
        Meta = std::move(meta);
    }
    if (updatePayload.Has("visual")) {
        Visual = updatePayload["visual"];
    }

    if (updatePayload.Has("first_maintenance_mileage")) {
        if (!NJson::ParseField(updatePayload["first_maintenance_mileage"], FirstMaintenanceMileage)) {
            return false;
        }
    } else {
        FirstMaintenanceMileage = Nothing();
    }
    if (updatePayload.Has("maintenance_mileage")) {
        if (!NJson::ParseField(updatePayload["maintenance_mileage"], MaintenanceMileage)) {
            return false;
        }
    } else {
        MaintenanceMileage = Nothing();
    }
    if (updatePayload.Has("intermediate_maintenance_mileage")) {
        if (!NJson::ParseField(updatePayload["intermediate_maintenance_mileage"], IntermediateMaintenanceMileage)) {
            return false;
        }
    } else {
        IntermediateMaintenanceMileage = Nothing();
    }
    if (updatePayload.Has("maintenance_period")) {
        if (!NJson::ParseField(updatePayload["maintenance_period"], MaintenancePeriod)) {
            return false;
        }
    } else {
        MaintenancePeriod = Nothing();
    }

    return NJson::ParseField(updatePayload["z_index"], ZIndex);
}

bool TDriveModelData::DeserializeFromJson(const NJson::TJsonValue& json) {
    NStorage::TTableRecord record;
    return record.DeserializeFromJson(json) && Parse(record);
}

NStorage::TTableRecord TDriveModelData::SerializeToTableRecord() const {
    NStorage::TTableRecord result;
    result.Set("name", Name);
    result.Set("short_name", ShortName);
    result.Set("code", Code);
    result.Set("manufacturer", Manufacturer);
    result.Set("fuel_type", FuelType);
    result.Set("image_large_url", ImageLargeUrl);
    result.Set("image_small_url", ImageSmallUrl);
    result.Set("image_map_url_2x", ImageMapUrl2X);
    result.Set("image_map_url_3x", ImageMapUrl3X);
    result.Set("fuel_cap_side", FuelCapSide);
    result.Set("image_pin_url_2x", ImagePinUrl2X);
    result.Set("image_pin_url_3x", ImagePinUrl3X);
    result.Set("registry_manufacturer", RegistryManufacturer);
    result.Set("registry_model", RegistryModel);

    result.SetNonNull("fuel_icon_url", FuelIconUrl);
    result.SetNonNull("image_angle_url", ImageAngleUrl);
    result.SetNonNull("image_background_url", ImageBackgroundUrl);
    result.SetNonNull("rank", Rank);
    result.SetNonNull("full_cost_casco", FullCostCasco);
    result.SetOptional("deprecated", Deprecated);

    if (!Meta.empty()) {
        result.Set("meta", NJson::ToJson(NJson::Dictionary(Meta)));
    }

    result.SetNonNull("segment", Segment);
    if (Visual.IsDefined()) {
        result.Set("visual", Visual);
    }
    result.SetOptional("z_index", ZIndex);

    result.SetOptional("first_maintenance_mileage", FirstMaintenanceMileage, /* setNull = */ true);
    result.SetOptional("maintenance_mileage", MaintenanceMileage, /* setNull = */ true);
    result.SetOptional("intermediate_maintenance_mileage", IntermediateMaintenanceMileage, /* setNull = */ true);
    if (MaintenancePeriod) {
        result.Set("maintenance_period", MaintenancePeriod->Seconds());
    } else {
        result.SetOptional("maintenance_period", MaintenancePeriod, /* setNull = */ true);
    }

    result.Set("fuel_distance_enabled", FuelDistanceEnabled);

    if (!DefaultTags.empty()) {
        result.Set("default_tags", NJson::ToJson(DefaultTags));
    }
    result.Set("model_specifications", Specifications.Serialize());

    return result;
}

NJson::TJsonValue TDriveModelData::GetReport(ELocalization locale, NDriveModelReport::TReportTraits traits /* = NDriveModelReport::ReportAll */) const {
    NJson::TJsonValue result;
    NJson::InsertField(result, "code", Code);
    if (traits & NDriveModelReport::ReportNames) {
        NJson::InsertField(result, "name", Name);
        NJson::InsertField(result, "short_name", ShortName);
        NJson::InsertField(result, "manufacturer", Manufacturer);
    }
    if (traits & NDriveModelReport::ReportRegistryInfo) {
        NJson::InsertField(result, "registry_manufacturer", RegistryManufacturer);
        NJson::InsertField(result, "registry_model", RegistryModel);
    }
    if (traits & NDriveModelReport::ReportImages) {
        NJson::InsertField(result, "image_large_url", ImageLargeUrl);
        NJson::InsertField(result, "image_small_url", ImageSmallUrl);
        NJson::InsertField(result, "image_map_url_2x", ImageMapUrl2X);
        NJson::InsertField(result, "image_map_url_3x", ImageMapUrl3X);
        NJson::InsertField(result, "image_pin_url_2x", ImagePinUrl2X);
        NJson::InsertField(result, "image_pin_url_3x", ImagePinUrl3X);
        NJson::InsertNonNull(result, "image_angle_url", ImageAngleUrl);
        NJson::InsertNonNull(result, "image_background_url", ImageBackgroundUrl);
    }
    if (traits & NDriveModelReport::ReportFuel) {
        NJson::InsertField(result, "fuel_type", FuelType);
        NJson::InsertField(result, "fuel_cap_side", FuelCapSide);
        if (traits & NDriveModelReport::ReportImages) {
            NJson::InsertNonNull(result, "fuel_icon_url", FuelIconUrl);
        }
    }
    if (traits & NDriveModelReport::ReportPriority) {
        NJson::InsertNonNull(result, "rank", Rank);
        NJson::InsertNonNull(result, "segment", Segment);
        NJson::InsertNonNull(result, "z_index", ZIndex);
    }
    if (traits & NDriveModelReport::ReportKASKO) {
        NJson::InsertNonNull(result, "full_cost_casco", FullCostCasco);
    }
    if (traits & NDriveModelReport::ReportFullMeta && !Meta.empty()) {
        NJson::InsertField(result, "meta", NJson::Dictionary(Meta));
    }
    if (traits & NDriveModelReport::ReportVisual && Visual.IsDefined()) {
        NJson::InsertField(result, "visual", Visual);
    }
    if (traits & NDriveModelReport::ReportServiceInfo) {
        NJson::InsertField(result, "first_maintenance_mileage", FirstMaintenanceMileage);
        NJson::InsertField(result, "maintenance_mileage", MaintenanceMileage);
        NJson::InsertField(result, "intermediate_maintenance_mileage", IntermediateMaintenanceMileage);
        NJson::InsertNonNull(result, "maintenance_period", NJson::Hr(MaintenancePeriod));

        NJson::InsertField(result, "fuel_distance_enabled", FuelDistanceEnabled);
        NJson::InsertField(result, "default_tags", DefaultTags);
    }
    if (traits & (NDriveModelReport::ReportServiceInfo | NDriveModelReport::ReportModelSpecifications)) {
        NJson::InsertField(result, "model_specifications", Specifications.GetReport(locale));
    }
    NJson::InsertNonNull(result, "deprecated", Deprecated);
    return result;
}

bool TDriveModelData::ParseDefaultTags(const NJson::TJsonValue& tagsArray) {
    DefaultTags.clear();
    if (tagsArray.GetType() != NJson::EJsonValueType::JSON_ARRAY) {
        return false;
    }
    for (auto&& element : tagsArray.GetArray())  {
        TDefaultTagDescription defaultTagDescription;
        if (!defaultTagDescription.DeserializeFromJson(element)) {
            return false;
        }
        DefaultTags.push_back(std::move(defaultTagDescription));
    }
    return true;
}

TString TDriveModelData::FormDescriptionElement(const TString& value, ELocalization locale, const ILocalization* localization) const {
    auto result = value;
    SubstGlobal(result, "_CarModelName_", Name);
    SubstGlobal(result, "_CarModelShortName_", ShortName);
    SubstGlobal(result, "_CarModelManufacturer_", Manufacturer);
    if (localization) {
        localization->ApplyResources(result, locale);
    }
    return result;
}

TMap<std::pair<TString, TString>, TString> TModelsDB::GetRegistryModelCodesMapping() const {
    TModelsDB::TFetchResult fetchResult = FetchInfo();
    TMap<TPair, TString> result;
    for (auto&& entry : fetchResult.GetResult()) {
        auto model = entry.second;
        auto manufacturerName = ToLowerUTF8(model.GetRegistryManufacturer());
        auto modelName = ToLowerUTF8(model.GetRegistryModel());
        result[std::make_pair(manufacturerName, modelName)] = model.GetCode();
    }
    return result;
}

TSet<TString> TModelsDB::GetModelCodes() const {
    TModelsDB::TFetchResult fetchResult = FetchInfo();
    TSet<TString> result;
    for (auto&& [code, model] : fetchResult.GetResult()) {
        result.insert(model.GetCode());
    }
    return result;
}
