#pragma once

#include "specification.h"

#include <drive/backend/abstract/localization.h>
#include <drive/backend/database/entity/manager.h>

namespace NDriveModelReport {
    using TReportTraits = ui32;
    enum EReportTraits: TReportTraits {
        ReportServiceInfo = 1 << 0,
        ReportNames = 1 << 1,
        ReportImages = 1 << 2,
        ReportFuel = 1 << 3,
        ReportRegistryInfo = 1 << 4,
        ReportFullMeta = 1 << 5,
        ReportVisual = 1 << 6,
        ReportKASKO = 1 << 7,
        ReportPriority = 1 << 8,
        ReportModelSpecifications = 1 << 9,
    };

    static constexpr TReportTraits NoTraits = 0;
    static constexpr TReportTraits ReportAll = Max<TReportTraits>();
    static constexpr TReportTraits UserReport = Max<TReportTraits>()
        - ReportServiceInfo
        - ReportModelSpecifications
    ;
}

class TDriveModelData {
public:
    struct TDefaultTagDescription {
        TString TagName;
        TString Comment;
        ui32 Priority;

        TDefaultTagDescription() = default;

        bool DeserializeFromJson(const NJson::TJsonValue& json);
        NJson::TJsonValue SerializeToJson() const;
    };

    using TMeta = TMap<TString, TString>;

private:
    R_FIELD(TString, Name);
    R_FIELD(TString, ShortName);
    R_FIELD(TString, Code);
    R_FIELD(TString, Manufacturer);
    R_READONLY(TMeta, Meta);
    R_FIELD(TString, RegistryManufacturer);
    R_FIELD(TString, RegistryModel);
    R_READONLY(bool, FuelDistanceEnabled, true);
    R_READONLY(TString, FuelIconUrl);
    R_READONLY(TString, FuelType);
    R_READONLY(TString, ImageAngleUrl);
    R_READONLY(TString, ImageBackgroundUrl);
    R_READONLY(TString, ImageLargeUrl);
    R_READONLY(TString, ImageSmallUrl);
    R_READONLY(TString, ImageMapUrl2X);
    R_READONLY(TString, ImageMapUrl3X);
    R_READONLY(TString, FuelCapSide);
    R_READONLY(TString, ImagePinUrl2X);
    R_READONLY(TString, ImagePinUrl3X);
    R_READONLY(TString, Segment);
    R_READONLY(i64, Rank, 0);
    R_READONLY(ui32, FullCostCasco, 0);
    R_OPTIONAL(i64, ZIndex);

    R_FIELD(TDriveModelSpecifications, Specifications);

    R_READONLY(TVector<TDefaultTagDescription>, DefaultTags);
    R_READONLY(NJson::TJsonValue, Visual);

    R_OPTIONAL(ui64, FirstMaintenanceMileage);
    R_OPTIONAL(ui64, MaintenanceMileage);
    R_OPTIONAL(ui64, IntermediateMaintenanceMileage);
    R_OPTIONAL(TDuration, MaintenancePeriod);
    R_OPTIONAL(bool, Deprecated);

public:
    bool Parse(const NStorage::TTableRecord& row);
    bool Patch(const NJson::TJsonValue& updatePayload);
    bool DeserializeFromJson(const NJson::TJsonValue& json);
    NStorage::TTableRecord SerializeToTableRecord() const;
    NJson::TJsonValue GetReport(ELocalization locale, NDriveModelReport::TReportTraits traits = NDriveModelReport::ReportAll) const;

    TString FormDescriptionElement(const TString& value, ELocalization locale, const ILocalization* localization) const;

private:
    bool ParseDefaultTags(const NJson::TJsonValue& tagsArray);
};

class TModelsDB: public TCachedEntityManager<TDriveModelData> {
private:
    using TBase = TCachedEntityManager<TDriveModelData>;
    using TPair = std::pair<TString, TString>;

public:
    using TBase::TBase;

    virtual TString GetMainId(const TDriveModelData& e) const override {
        return e.GetCode();
    }

    virtual TString GetTableName() const override {
        return "car_model";
    }

    TString GetColumnName() const override {
        return "code";
    }

    TMap<std::pair<TString, TString>, TString> GetRegistryModelCodesMapping() const;
    TSet<TString> GetModelCodes() const;
};
