#include "hardware.h"

#include <drive/backend/abstract/drive_database.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/data/leasing/leasing.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/util/types/uuid.h>

namespace {
    bool IsAllDigits(const TString& str) {
        for (char c : str) {
            if (c < '0' || c > '9') {
                return false;
            }
        }
        return true;
    }

    bool IsAllHex(const TString& str) {
        for (char c : str) {
            if (!((c >= '0' && c <= '9') || (c >= 'a' && c <= 'f') || (c >= 'A' || c <= 'F'))) {
                return false;
            }
        }
        return true;
    }

    bool IsAllAlnum(const TString& str) {
        for (char c : str) {
            if (!((c >= '0' && c <= '9') || (c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z'))) {
                return false;
            }
        }
        return true;
    }

    bool IsSignalDeviceSN(const TString& serviceAppCode) {
        static constexpr TStringBuf possible = "0123456789ABCDEF";
        static constexpr size_t signalDeviceSNSize = 13;

        if (serviceAppCode.size() != signalDeviceSNSize) {
            return false;
        }
        size_t factor = 1;
        size_t sum = 0;
        const auto possibleSize = possible.size();
        for (int i = serviceAppCode.size() - 1; i >= 0; i--) {
            const auto codePoint = possible.find(serviceAppCode[i]);
            if (codePoint == TString::npos) {
                return false;
            }
            size_t addend = factor * codePoint;
            factor = (factor == 2) ? 1 : 2;
            addend = addend / possibleSize + addend % possibleSize;
            sum += addend;
        }
        const auto remainder = sum % possibleSize;
        return remainder == 0;
    }
};

TCarGenericAttachment::TCarGenericAttachment(TAtomicSharedPtr<IJsonBlobSerializableCarAttachment> impl)
    : Type(impl != nullptr ? impl->GetTypeName() : EDocumentAttachmentType::Unknown)
    , AttachmentImpl(impl)
{
    Id = NUtil::CreateUUID();
}

IJsonBlobSerializableCarAttachment::EUniquePolicy TCarGenericAttachment::GetUniquePolicy() const {
    return !!AttachmentImpl ? AttachmentImpl->GetUniquePolicy() : IJsonBlobSerializableCarAttachment::EUniquePolicy::Unique;
}

bool TCarGenericAttachment::Parse(const NStorage::TTableRecord& row) {
    return TBaseDecoder::DeserializeFromTableRecord(*this, row);
}

NStorage::TTableRecord TCarGenericAttachment::SerializeToTableRecord() const {
    NStorage::TTableRecord result;
    result.Set("id", Id);
    result.Set("type", ToString(Type));
    if (AttachmentImpl) {
        result.Set("blob", AttachmentImpl->SerializeToJsonBlob());
    }
    return result;
}

TString TCarGenericAttachment::GetServiceAppSlug() const {
    if (AttachmentImpl) {
        return AttachmentImpl->GetServiceAppSlug();
    }
    return "";
}

TAtomicSharedPtr<IJsonBlobSerializableCarAttachment> TCarGenericAttachment::GetImpl() const {
    return AttachmentImpl;
}

NJson::TJsonValue TCarGenericAttachment::BuildReport(const ICarAttachmentReportContext& context) const {
    if (AttachmentImpl != nullptr) {
        NJson::TJsonValue result;
        result["cpp_type"] = ToString(Type);
        result["type"] = CarAttachmentType(Type);
        result["data"] = AttachmentImpl->BuildReport(context);
        result["id"] = GetId();
        return result;
    }
    return NJson::JSON_NULL;
}

TCarAttachmentAssignment::TCarAttachmentAssignment(const TString& carId, const TString& genericAttachmentId)
    : CarId(carId)
    , GenericAttachmentId(genericAttachmentId)
{
    Id = NUtil::CreateUUID();
}

TCarAttachmentAssignment::TCarAttachmentAssignment(const TString& id, const TString& carId, const TString& genericAttachmentId)
    : Id(id)
    , CarId(carId)
    , GenericAttachmentId(genericAttachmentId)
{
}

NStorage::TTableRecord TCarAttachmentAssignment::SerializeToTableRecord() const {
    NStorage::TTableRecord result;
    result.Set("id", Id);
    result.Set("document_id", GenericAttachmentId);
    result.Set("car_id", CarId);
    return result;
}

bool TCarAttachmentAssignment::Parse(const NStorage::TTableRecord& row) {
    return TBaseDecoder::DeserializeFromTableRecord(*this, row);
}


bool TCarGenericAttachment::ParseBlob(const TString& blob) {
    NJson::TJsonValue blobJsonValue;
    if (!NJson::ReadJsonFastTree(blob, &blobJsonValue)) {
        ERROR_LOG << "corrupt json in blob : " << blob << Endl;
        return false;
    }

    TAtomicSharedPtr<IJsonBlobSerializableCarAttachment> attachmentImpl = IJsonBlobSerializableCarAttachment::TFactory::Construct(Type);

    if (!!attachmentImpl && !attachmentImpl->DeserializeFromJsonBlob(blobJsonValue)) {
        ERROR_LOG << "incorrect blob for type " << Type << " : " << blob << Endl;
        return false;
    }

    AttachmentImpl = attachmentImpl;
    return true;
}

bool TCarGenericAttachment::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, Id);
    READ_DECODER_VALUE(decoder, values, Type);
    {
        TString blob;
        READ_DECODER_VALUE_TEMP(decoder, values, blob, Blob);
        if (!!blob) {
            return ParseBlob(blob);
        }
    }
    return true;
}

void TCarGenericAttachment::DoBuildReportItem(NJson::TJsonValue& item) const {
    item.InsertValue("object", BuildReport(TCarAttachmentReportImplContext()));
}

TString TCarGenericAttachment::GetObjectId(const TCarGenericAttachment& object) {
    return object.GetId();
}

TCarGenericAttachment TCarGenericAttachment::FromHistoryEvent(const TObjectEvent<TCarGenericAttachment>& historyEvent) {
    /*
        Use that the set of fields, generated by TObjectEvent is a superset of the fields,
        which can be parsed by Parse.
    */
    NStorage::TTableRecord tr = historyEvent.SerializeToTableRecord();
    TCarGenericAttachment attachment;
    attachment.Parse(tr);
    return attachment;
}

bool TCarAttachmentAssignment::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, Id);
    READ_DECODER_VALUE(decoder, values, CarId);
    READ_DECODER_VALUE(decoder, values, GenericAttachmentId);
    READ_DECODER_VALUE_INSTANT_ISOFORMAT_OPT(decoder, values, UnassignedAt);
    return true;
}

void TCarAttachmentAssignment::DoBuildReportItem(NJson::TJsonValue& item) const {
    item.InsertValue("object", BuildReport());
}

TString TCarAttachmentAssignment::GetObjectId(const TCarAttachmentAssignment& object) {
    return object.GetId();
}

TCarAttachmentAssignment TCarAttachmentAssignment::FromHistoryEvent(const TObjectEvent<TCarAttachmentAssignment>& historyEvent) {
    return TCarAttachmentAssignment(historyEvent.GetId(), historyEvent.GetCarId(), historyEvent.GetGenericAttachmentId());
}

NJson::TJsonValue TCarAttachmentAssignment::BuildReport() const {
    NJson::TJsonValue report;
    report["id"] = Id;
    report["car_id"] = CarId;
    report["generic_attachment_id"] = GenericAttachmentId;
    return report;
}

bool TCarAttachmentHistoryManager::GetEventsSinceId(TBase::TEventId id, TBase::TEventPtrs& results, const TInstant actuality) const {
    if (Config.GetUseCache()) {
        return TBase::GetEventsSinceId(id, results, actuality);
    } else {
        return TBase::GetEventsSinceIdDirect(id, results);
    }
}

bool TCarGenericAttachmentDB::UpsertImpl(TCarGenericAttachment& candidate, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session) {
    if (candidate->GetServiceAppSlug() && RecognizedDevices.contains(candidate->GetServiceAppSlug())) {
        result = RecognizedDevices[candidate->GetServiceAppSlug()];
        return true;
    }
    if (!UpsertAttachment(candidate, operatorUserId, session)) {
        session.SetErrorInfo("attachment", "unable to upsert attachment", EDriveSessionResult::InternalError);
        return false;
    }
    result = candidate;
    return true;
}

bool TCarGenericAttachmentDB::UpsertModemBySimICC(const TString& normalizedICC, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session) {
    if (RecognizedDevices.contains(normalizedICC)) {
        result = RecognizedDevices[normalizedICC];
        return true;
    }
    TAtomicSharedPtr<IJsonBlobSerializableCarAttachment> modemImpl;
    modemImpl.Reset(new TCarHardwareModem(normalizedICC));
    TCarGenericAttachment newModem(modemImpl);
    return UpsertImpl(newModem, result, operatorUserId, session);
}

bool TCarGenericAttachmentDB::UpsertVegaByIMEI(const TString& vegaImei, const TString& primaryICC, const TString& secondaryICC, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session) {
    TAtomicSharedPtr<IJsonBlobSerializableCarAttachment> vegaImpl;
    auto vegaPtr = new TCarHardwareVega(vegaImei);
    vegaImpl.Reset(vegaPtr);
    TCarGenericAttachment newVega(vegaImpl);
    if (!!primaryICC) {
        TString normalizedICC = TCarHardwareSim::NormalizeICC(primaryICC);
        TString phoneNumber;
        if (PhoneNumberByICC.contains(normalizedICC)) {
            phoneNumber = PhoneNumberByICC[normalizedICC];
        }

        vegaPtr->UpdateSim(normalizedICC, phoneNumber);
    }
    if (!!secondaryICC) {
        TString normalizedICC = TCarHardwareSim::NormalizeICC(secondaryICC);
        TString phoneNumber;
        if (PhoneNumberByICC.contains(normalizedICC)) {
            phoneNumber = PhoneNumberByICC[normalizedICC];
        }

        vegaPtr->UpdateSim(normalizedICC, phoneNumber, false);
    }
    return UpsertImpl(newVega, result, operatorUserId, session);
}

bool TCarGenericAttachmentDB::UpsertBeaconByIMEI(const TString& beaconImei, const TString& beaconSerialNumber, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session) {
    TAtomicSharedPtr<IJsonBlobSerializableCarAttachment> beaconImpl;
    beaconImpl.Reset(new TCarHardwareBeacon(beaconImei, beaconSerialNumber));
    TCarGenericAttachment newBeacon(beaconImpl);
    return UpsertImpl(newBeacon, result, operatorUserId, session);
}

bool TCarGenericAttachmentDB::UpsertFuelCardByNumber(const TString& number, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session) {
    TAtomicSharedPtr<IJsonBlobSerializableCarAttachment> fuelCardImpl;
    fuelCardImpl.Reset(new TFuelCard(number));
    TCarGenericAttachment newFuelCard(fuelCardImpl);
    return UpsertImpl(newFuelCard, result, operatorUserId, session);
}

bool TCarGenericAttachmentDB::UpsertHeadByHeadId(const TString& headId, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session) {
    TAtomicSharedPtr<IJsonBlobSerializableCarAttachment> headImpl;
    headImpl.Reset(new TCarHardwareHead(headId));
    TCarGenericAttachment newHead(headImpl);
    return UpsertImpl(newHead, result, operatorUserId, session);
}

bool TCarGenericAttachmentDB::UpsertHeadSNByNumber(const TString& serialNumber, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session) {
    TAtomicSharedPtr<IJsonBlobSerializableCarAttachment> headSerialNumberImpl;
    headSerialNumberImpl.Reset(new TCarHardwareHeadSerialNumber(serialNumber));
    TCarGenericAttachment newHead(headSerialNumberImpl);
    return UpsertImpl(newHead, result, operatorUserId, session);
}

bool TCarGenericAttachmentDB::UpsertHeadNewById(const TString& headId, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session) {
    TAtomicSharedPtr<IJsonBlobSerializableCarAttachment> headNewImpl;
    headNewImpl.Reset(new TCarHardwareHeadNew(headId));
    TCarGenericAttachment newHead(headNewImpl);
    return UpsertImpl(newHead, result, operatorUserId, session);
}

bool TCarGenericAttachmentDB::UpsertAirportPassByCode(const TString& code, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session) {
    TAtomicSharedPtr<IJsonBlobSerializableCarAttachment> passImpl;
    passImpl.Reset(new TYTaxiAirportPass(code));
    TCarGenericAttachment newPass(passImpl);
    return UpsertImpl(newPass, result, operatorUserId, session);
}

bool TCarGenericAttachmentDB::UpsertTransponderByCode(const TString& code, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session) {
    TAtomicSharedPtr<IJsonBlobSerializableCarAttachment> transponderImpl;
    transponderImpl.Reset(new TTransponder(code));
    TCarGenericAttachment newTransponder(transponderImpl);
    return UpsertImpl(newTransponder, result, operatorUserId, session);
}

bool TCarGenericAttachmentDB::UpsertTransponderSpbByCode(const TString& code, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session) {
    TAtomicSharedPtr<IJsonBlobSerializableCarAttachment> transponderSpbImpl;
    transponderSpbImpl.Reset(new TTransponderSpb(code));
    TCarGenericAttachment newTransponderSpb(transponderSpbImpl);
    return UpsertImpl(newTransponderSpb, result, operatorUserId, session);
}

bool TCarGenericAttachmentDB::UpsertAirportPassSpbByCode(const TString& code, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session) {
    TAtomicSharedPtr<IJsonBlobSerializableCarAttachment> airportPassSpbImpl;
    airportPassSpbImpl.Reset(new TAirportPassSpb(code));
    TCarGenericAttachment newAirportPassSpb(airportPassSpbImpl);
    return UpsertImpl(newAirportPassSpb, result, operatorUserId, session);
}

bool TCarGenericAttachmentDB::UpsertQrCode(const TString& value, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session) {
    auto impl = MakeAtomicShared<TCarQrCode>(value);
    TCarGenericAttachment attachment(impl);
    return UpsertImpl(attachment, result, operatorUserId, session);
}

bool TCarGenericAttachmentDB::UpsertSignalDeviceByNumber(const TString& signalDeviceSN, const TString& signalDeviceImei, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session) {
    TAtomicSharedPtr<IJsonBlobSerializableCarAttachment> signalDeviceImpl;
    signalDeviceImpl.Reset(new TCarSignalDevice(signalDeviceSN, signalDeviceImei));
    TCarGenericAttachment newSignalDevice(signalDeviceImpl);
    return UpsertImpl(newSignalDevice, result, operatorUserId, session);
}

TVector<TString> TCarGenericAttachmentDB::GetUsedICC(const TCarGenericAttachment& attachment) const {
    if (attachment) {
        auto usedIcc = attachment->GetUsedICCNumbers();
        TVector<TString> normalizedResult;
        for (auto&& icc : usedIcc) {
            normalizedResult.push_back(TCarHardwareSim::NormalizeICC(icc));
        }
        return normalizedResult;
    }
    return {};
}

void TCarGenericAttachmentDB::UpsertEntryUnsafe(const TCarGenericAttachment& entry) const {
    auto newlyUsedICC = GetUsedICC(entry);
    if (ActiveAttachments.contains(entry.GetId())) {
        if (IdToAttachment.contains(entry.GetId())) {
            for (auto icc : IccById[entry.GetId()]) {
                IccOwner.erase(icc);
            }
        }
        for (auto icc : newlyUsedICC) {
            IccOwner[icc] = entry.GetId();
        }
    }

    IdToAttachment[entry.GetId()] = entry;
    IccById[entry.GetId()] = std::move(newlyUsedICC);

    if (!!entry.GetServiceAppSlug()) {
        RecognizedDevices[entry.GetServiceAppSlug()] = entry;
    }

    if (entry.GetType() == EDocumentAttachmentType::CarHardwareSim) {
        auto simImpl = dynamic_cast<const TCarHardwareSim*>(entry.Get());
        if (simImpl) {
            PhoneNumberByICC[simImpl->GetICC()] = simImpl->GetPhoneNumber();
        }
    }
}

void TCarGenericAttachmentDB::RemoveEntryUnsafe(const TCarGenericAttachment& entry) const {
    if (IdToAttachment.find(entry.GetId()) != IdToAttachment.end()) {
        IdToAttachment.erase(IdToAttachment.find(entry.GetId()));
    }

    if (ActiveAttachments.contains(entry.GetId())) {
        for (auto icc : IccById[entry.GetId()]) {
            IccOwner.erase(icc);
        }
    }

    if (IccById.find(entry.GetId()) != IccById.end()) {
        IccById.erase(IccById.find(entry.GetId()));
    }

    if (!!entry.GetServiceAppSlug() && RecognizedDevices.find(entry.GetServiceAppSlug()) != RecognizedDevices.end()) {
        RecognizedDevices.erase(RecognizedDevices.find(entry.GetServiceAppSlug()));
    }

    ActiveAttachments.erase(entry.GetId());
}

void TCarGenericAttachmentDB::AcceptHistoryEventUnsafe(const TObjectEvent<TCarGenericAttachment>& ev) const {
    TCarGenericAttachment entry = TCarGenericAttachment::FromHistoryEvent(ev);
    if (ev.GetHistoryAction() == EObjectHistoryAction::Remove) {
        RemoveEntryUnsafe(entry);
    } else {
        UpsertEntryUnsafe(entry);
    }
}

bool TCarGenericAttachmentDB::DoRebuildCacheUnsafe() const {
    IdToAttachment.clear();
    RecognizedDevices.clear();
    IccById.clear();
    IccOwner.clear();
    ActiveAttachments.clear();

    auto allEntries = AttachmentsTable.FetchInfo();
    for (auto entry : allEntries) {
        UpsertEntryUnsafe(entry.second);
    }

    return true;
}

TStringBuf TCarGenericAttachmentDB::GetEventObjectId(const TObjectEvent<TCarGenericAttachment>& ev) const {
    return ev.GetId();
}

TOptionalCarGenericAttachments TCarGenericAttachmentDB::GetAttachments(TConstArrayRef<TString> ids, NDrive::TEntitySession& tx, TMaybe<EDocumentAttachmentType> type) const {
    NSQL::TQueryOptions queryOptions;
    queryOptions.SetGenericCondition("id", MakeSet<TString>(ids));
    if (type) {
        queryOptions.AddGenericCondition("type", ToString(type));
    }
    return AttachmentsTable.Fetch(tx, queryOptions);
}

bool TCarGenericAttachmentDB::TryGetAttachment(const TString& id, TCarGenericAttachment& result, const TInstant actuality) const {
    if (!RefreshCache(actuality)) {
        return false;
    }
    auto rg = MakeObjectReadGuard();
    if (!IdToAttachment.contains(id)) {
        return false;
    } else {
        result = IdToAttachment[id];
        return true;
    }
}

bool TCarGenericAttachmentDB::TryGetAttachmentByServiceAppSlug(const TString& slug, TCarGenericAttachment& result, const TInstant actuality) const {
    if (!RefreshCache(actuality)) {
        return false;
    }
    auto rg = MakeObjectReadGuard();
    if (RecognizedDevices.contains(slug)) {
        result = RecognizedDevices[slug];
        return true;
    }
    return false;
}

void TCarGenericAttachmentDB::ActivateAttachment(const TString& id) const {
    auto wg = MakeObjectWriteGuard();
    if (ActiveAttachments.emplace(id).second) {
        for (auto&& icc : IccById[id]) {
            IccOwner[icc] = id;
        }
    }
}

void TCarGenericAttachmentDB::DeactivateAttachment(const TString& id) const {
    auto wg = MakeObjectWriteGuard();
    ActiveAttachments.erase(id);
    for (auto&& icc : IccById[id]) {
        auto it = IccOwner.find(icc);
        if (it != IccOwner.end() && it->second == id) {
            IccOwner.erase(it);
        }
    }
}


size_t TCarGenericAttachmentDB::Size(const TInstant actuality) const {
    Y_ENSURE_BT(RefreshCache(actuality));
    auto rg = MakeObjectReadGuard();
    return IdToAttachment.size();
}

bool TCarGenericAttachmentDB::UpsertAttachment(TCarGenericAttachment& attachment, const TString& operatorUserId, NDrive::TEntitySession& session) const {
    /*
        Add new or change existing attachment.

        Id of attachment may change in case it clones some existing attachment. That is the case for
        hardware kind of attachments, because for them the keys are service app slugs (we obviously can't
        have two of Vega with same IMEI, although you can pass another one with different generic document Id).
        Not the case for insurance kind of attachment or generic car information.

        May fail if either DB insertion fail, or conflict by ICC numbers happen.
    */
    if (!RefreshCache(Now())) {
        session.SetErrorInfo("CarGenericAttachmentDB::UpsertAttachment", "cannot refresh cache");
        return false;
    }

    auto wg = MakeObjectWriteGuard();
    bool isNew = true;
    TCarGenericAttachment currentAttachmentState;
    if (IdToAttachment.contains(attachment.GetId())) {
        currentAttachmentState = IdToAttachment[attachment.GetId()];
        attachment.SetId(currentAttachmentState.GetId());
        isNew = false;
    } else if (!!attachment.GetServiceAppSlug() && RecognizedDevices.contains(attachment.GetServiceAppSlug())) {
        currentAttachmentState = RecognizedDevices[attachment.GetServiceAppSlug()];
        attachment.SetId(currentAttachmentState.GetId());
        isNew = false;
    }
    TVector<TString> attachmentUsedICC = GetUsedICC(currentAttachmentState);

    if (!AttachmentsTable.Upsert(attachment, session)) {
        session.SetErrorInfo("attachment", "unable to upsert attachment into db", EDriveSessionResult::TransactionProblem);
        return false;
    }

    return HistoryManager->AddHistory(attachment, operatorUserId, isNew ? EObjectHistoryAction::Add : EObjectHistoryAction::UpdateData, session);
}

bool TCarGenericAttachmentDB::UpsertFromServiceApp(const TString& serviceAppCode, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session, TMaybe<EDocumentAttachmentType> type) {
    /*
        Parse scanned barcode value from service app and create the respective attachment if it was missing.
        If it's impossible to create, returns false.
    */
    if (!RefreshCache(Now())) {
        session.SetErrorInfo("CarGenericAttachmentDB::UpsertFromServiceApp", "cannot refresh cache");
        return false;
    }

    TVector<TString> commaSeparatedCode = SplitString(serviceAppCode, ",");
    if ((!type || *type == EDocumentAttachmentType::CarSignalDevice) && IsSignalDeviceSN(commaSeparatedCode[0])) {
        TString signalDeviceSN;
        TString signalDeviceImei;
        if (commaSeparatedCode.size() > 1) {
            signalDeviceSN = std::move(commaSeparatedCode[0]);
            signalDeviceImei = std::move(commaSeparatedCode[1]);
        } else {
            signalDeviceSN = serviceAppCode;
        }
        return UpsertSignalDeviceByNumber(signalDeviceSN, signalDeviceImei, result, operatorUserId, session);
    } else if ((!type || *type == EDocumentAttachmentType::CarHardwareModem) && serviceAppCode.StartsWith("89701") && serviceAppCode.size() >= 17 && serviceAppCode.size() <= 20) {
        TString normalizedICC = TCarHardwareSim::NormalizeICC(serviceAppCode);
        return UpsertModemBySimICC(normalizedICC, result, operatorUserId, session);
    } else if ((!type || *type == EDocumentAttachmentType::CarHardwareVega) && ((serviceAppCode.size() >= 14 && serviceAppCode.size() <= 15 && IsAllDigits(serviceAppCode)) || serviceAppCode.StartsWith("MT-32K LTE"))) {
        TString vegaImei;
        TString primaryICC;
        TString secondaryICC;
        if (serviceAppCode.StartsWith("MT-32K LTE")) {
            TVector<TString> tokens = SplitString(serviceAppCode, ";");
            vegaImei = SplitString(serviceAppCode, ";")[1];
            if (tokens.size() == 4) {
                primaryICC = tokens[2];
                secondaryICC = tokens[3];
            }
        } else {
            vegaImei = serviceAppCode;
        }
        return UpsertVegaByIMEI(vegaImei, primaryICC, secondaryICC, result, operatorUserId, session);
    } else if ((!type || *type == EDocumentAttachmentType::CarHardwareBeacon) && serviceAppCode.StartsWith("SN")) {
        if (commaSeparatedCode.size() != 2) {
            session.SetErrorInfo("UpsertFromServiceApp", TStringBuilder() << "cannot split " << serviceAppCode << " by , to 2 parts: " << commaSeparatedCode.size());
            return false;
        }
        TString beaconSerialNumber = std::move(commaSeparatedCode[0]);
        TString beaconImei = std::move(commaSeparatedCode[1]);
        return UpsertBeaconByIMEI(beaconImei, beaconSerialNumber, result, operatorUserId, session);
    } else if ((!type || *type == EDocumentAttachmentType::CarHardwareHead) && serviceAppCode.size() == 12 && IsAllAlnum(serviceAppCode)) {
        return UpsertHeadByHeadId(ToLowerUTF8(serviceAppCode), result, operatorUserId, session);
    } else if ((!type || *type == EDocumentAttachmentType::CarAirportPass) && serviceAppCode.size() == 10 && serviceAppCode[2] == '-') {
        return UpsertAirportPassByCode(ToUpperUTF8(serviceAppCode), result, operatorUserId, session);
    } else if ((!type || *type == EDocumentAttachmentType::CarTransponder) && serviceAppCode.size() == 35 || (serviceAppCode.size() == 16 && serviceAppCode.StartsWith("06"))) {
        return UpsertTransponderByCode(ToUpperUTF8(serviceAppCode), result, operatorUserId, session);
    } else if ((!type || *type == EDocumentAttachmentType::CarTransponderSpb) && serviceAppCode.size() == 19 && serviceAppCode.StartsWith("63628750000")) {
        return UpsertTransponderSpbByCode(ToUpperUTF8(serviceAppCode), result, operatorUserId, session);
    } else if ((!type || *type == EDocumentAttachmentType::CarAirportPassSpb) && serviceAppCode.size() == 6) {
        return UpsertAirportPassSpbByCode(ToUpperUTF8(serviceAppCode), result, operatorUserId, session);
    } else if ((!type || *type == EDocumentAttachmentType::CarHardwareHeadSerialNumber) &&
            ((serviceAppCode.size() == 18 && IsAllDigits(serviceAppCode)) || (serviceAppCode.size() == 15 && serviceAppCode.StartsWith("DVM") && IsAllDigits(serviceAppCode.substr(3, 12))))) {
        return UpsertHeadSNByNumber(ToUpperUTF8(serviceAppCode), result, operatorUserId, session);
    } else if ((!type || *type == EDocumentAttachmentType::CarHardwareHeadNew) && serviceAppCode.size() == 32 && IsAllHex(serviceAppCode)) {
        return UpsertHeadNewById(ToLowerUTF8(serviceAppCode), result, operatorUserId, session);
    } else if ((!type || *type == EDocumentAttachmentType::CarFuelCard) && IsAllDigits(serviceAppCode) && (serviceAppCode.size() == 16 || serviceAppCode.size() == 19)) {
        return UpsertFuelCardByNumber(serviceAppCode, result, operatorUserId, session);
    } else if ((!type || *type == EDocumentAttachmentType::CarQrCode) && serviceAppCode.StartsWith("QR_")) {
        return UpsertQrCode(serviceAppCode, result, operatorUserId, session);
    }
    session.SetErrorInfo("assignment", "unable to recognize device type", EDriveSessionResult::DataCorrupted);
    return false;
}

TString TCarGenericAttachmentDB::GetOwnerByIcc(const TString& icc, const TInstant actuality) const {
    Y_ENSURE_BT(RefreshCache(actuality));
    auto rg = MakeObjectReadGuard();
    auto it = IccOwner.find(icc);
    if (it == IccOwner.end()) {
        return "";
    }
    return it->second;
}

void TCarGenericAttachmentDB::ForceUpdate(const TInstant actuality) {
    Y_ENSURE_BT(RefreshCache(actuality));
}

bool TCarAttachmentAssignmentHistoryManager::GetEventsSinceId(TBase::TEventId id, TBase::TEventPtrs& results, const TInstant actuality) const {
    if (Config.GetUseCache()) {
        return TBase::GetEventsSinceId(id, results, actuality);
    } else {
        return TBase::GetEventsSinceIdDirect(id, results);
    }
}

bool TCarAttachmentAssignmentDB::DetachWithoutRefreshUnsafe(const TString& assignmentRecordId, const TString& operatorUserId, NDrive::TEntitySession& session, const NDrive::IServer* server) const {
    if (!AssignmentRecords.contains(assignmentRecordId)) {
        ERROR_LOG << "there is no active attachment with id : " << assignmentRecordId << Endl;
        return true;
    }
    auto record = AssignmentRecords[assignmentRecordId];
    record.SetUnassignedAt(Now());

    if (!AssignmentsTable.Remove({assignmentRecordId}, session)) {
        session.SetErrorInfo("detachment", "unable to remove the old attachment connection", EDriveSessionResult::InternalError);
        return false;
    }

    TString carId = AssignmentRecords[assignmentRecordId].GetCarId();
    TCarGenericAttachment attachment;
    if (!GenericAttachmentDB.TryGetAttachment(AssignmentRecords[assignmentRecordId].GetGenericAttachmentId(), attachment, Now())) {
        session.SetErrorInfo("detachment", "attachment with such id is not present", EDriveSessionResult::InternalError);
        return false;
    }

    // If it's a Vega, we should update the IMEI of the car
    if (attachment.GetType() == EDocumentAttachmentType::CarHardwareVega) {
        auto fetchResult = CarsDB.FetchInfo(carId, session);
        auto carInfoPtr = fetchResult.GetResultPtr(carId);
        if (!carInfoPtr || !attachment.GetImpl()) {
            session.SetErrorInfo("detachment", "update for nonexistent car or with empty data", EDriveSessionResult::DataCorrupted);
            return false;
        }
        auto carInfo = *carInfoPtr;
        carInfo.SetIMEI("get_null()");
        if (!CarsDB.UpdateCar(carInfo, operatorUserId, session)) {
            session.SetErrorInfo("detachment", "unable to upsert the updated car", EDriveSessionResult::TransactionProblem);
            return false;
        }
    } else if (server && attachment.GetType() == EDocumentAttachmentType::CarSignalDevice) {
        const auto& deviceTagsManager = server->GetDriveDatabase().GetTagsManager().GetDeviceTags();
        auto carObject = deviceTagsManager.RestoreObject(carId, session);
        R_ENSURE(carObject, {}, "cannot restore car tags " << carId, session);

        for (auto&& dbTag : carObject->GetTags()) {
            if (dbTag.Is<NDrivematics::THasSignalqTag>()) {
                if (!deviceTagsManager.RemoveTagSimple(dbTag, operatorUserId, session, false)) {
                    session.SetErrorInfo("detachment", "can not remove signalq tag from car", EDriveSessionResult::InternalError);
                    return false;
                }
            }
        }
    }

    return HistoryManager->AddHistory(record, operatorUserId, EObjectHistoryAction::Remove, session);
}

void TCarAttachmentAssignmentDB::AcceptHistoryEventUnsafe(const TObjectEvent<TCarAttachmentAssignment>& ev) const {
    TCarAttachmentAssignment entry = TCarAttachmentAssignment::FromHistoryEvent(ev);
    TString carId = entry.GetCarId();
    TString recordId = entry.GetId();

    bool isForDeletion = (entry.GetUnassignedAt() != TInstant::Zero() || ev.GetHistoryAction() == EObjectHistoryAction::Remove);
    if (isForDeletion && !AssignmentRecords.contains(recordId)) {
        GenericAttachmentDB.DeactivateAttachment(entry.GetGenericAttachmentId());
        return;
    }
    auto& entriesByCar = CarToAssignments[carId];

    size_t changedPosition;
    bool isPresent = false;

    for (size_t i = 0; i < entriesByCar.size(); ++i) {
        if (entriesByCar[i].GetId() == recordId) {
            changedPosition = i;
            isPresent = true;
            break;
        }
    }

    if (isPresent) {
        if (!isForDeletion) {
            /*
                Something has changed with the edge, it is not yet deleted. Update.
            */

            entriesByCar[changedPosition] = entry;
            AssignmentRecords[entry.GetId()] = entry;
            GenericAttachmentDB.ActivateAttachment(entry.GetGenericAttachmentId());
            AttachmentToAssignmentRecord[entry.GetGenericAttachmentId()] = std::move(entry);
        } else {
            entriesByCar.erase(entriesByCar.begin() + changedPosition);
            AssignmentRecords.erase(AssignmentRecords.find(recordId));
            if (AttachmentToAssignmentRecord.contains(entry.GetGenericAttachmentId())) {
                AttachmentToAssignmentRecord.erase(AttachmentToAssignmentRecord.find(entry.GetGenericAttachmentId()));
            }
            TotalAssignments -= 1;
            GenericAttachmentDB.DeactivateAttachment(entry.GetGenericAttachmentId());
        }
    } else if (!isForDeletion) {
        /*
            If not found and the assignment is alive, this is a new car->device edge. Register.
        */

        entriesByCar.push_back(entry);
        TotalAssignments += 1;
        AssignmentRecords[entry.GetId()] = entry;
        GenericAttachmentDB.ActivateAttachment(entry.GetGenericAttachmentId());
        AttachmentToAssignmentRecord[entry.GetGenericAttachmentId()] = std::move(entry);
    }
}

bool TCarAttachmentAssignmentDB::DoRebuildCacheUnsafe() const {
    CarToAssignments.clear();
    AttachmentToAssignmentRecord.clear();
    AssignmentRecords.clear();
    TotalAssignments = 0;

    auto session = GetHistoryManager().BuildSession(true);
    auto allEntries = AssignmentsTable.FetchInfo(session).GetResult();
    for (auto&& entryPair : allEntries) {
        auto entry = entryPair.second;
        if (entry.GetUnassignedAt() != TInstant::Zero()) {
            continue;
        }

        auto& entriesByCar = CarToAssignments[entry.GetCarId()];
        entriesByCar.push_back(entry);
        TotalAssignments += 1;
        AssignmentRecords[entry.GetId()] = entry;
        GenericAttachmentDB.ActivateAttachment(entry.GetGenericAttachmentId());
        AttachmentToAssignmentRecord[entry.GetGenericAttachmentId()] = std::move(entry);
    }

    return true;
}

TStringBuf TCarAttachmentAssignmentDB::GetEventObjectId(const TObjectEvent<TCarAttachmentAssignment>& ev) const {
    return ev.GetId();
}

TVector<TCarGenericAttachment> TCarAttachmentAssignmentDB::GetActiveAttachments(const TString& carId, const NAttachmentReport::TReportTraits traits, const TInstant actuality) const {
    Y_ENSURE_BT(RefreshCache(actuality));
    auto rg = MakeObjectReadGuard();
    auto assignments = CarToAssignments.find(carId);
    if (assignments == CarToAssignments.end()) {
        return {};
    }
    const auto& history = assignments->second;
    TVector<TString> attachmentIds;
    for (auto& record : history) {
        if (record.GetUnassignedAt() == TInstant::Zero()) {
            attachmentIds.push_back(record.GetGenericAttachmentId());
        }
    }

    TVector<TCarGenericAttachment> attachments;
    for (auto attachmentId : attachmentIds) {
        TCarGenericAttachment attachment;
        if (!GenericAttachmentDB.TryGetAttachment(attachmentId, attachment, actuality)) {
            ERROR_LOG << "incorrect attachment id: " << attachmentId << Endl;
            continue;
        }
        if (attachment && (traits & attachment->GetReportTrait())) {
            attachments.push_back(std::move(attachment));
        }
    }

    return attachments;
}

bool TCarAttachmentAssignmentDB::TryGetActiveAttachmentAssignment(const TString& attachmentId, TCarAttachmentAssignment& assignment, TInstant actuality) const {
    if (!RefreshCache(actuality)) {
        return false;
    }
    auto rg = MakeObjectReadGuard();
    if (AttachmentToAssignmentRecord.contains(attachmentId)) {
        assignment = AttachmentToAssignmentRecord[attachmentId];
        return true;
    }
    return false;
}

bool TCarAttachmentAssignmentDB::Detach(const TString& assignmentRecordId, const TString& operatorUserId, NDrive::TEntitySession& session, const NDrive::IServer* server) const {
    auto actuality = Now();
    if (!RefreshCache(actuality)) {
        session.SetErrorInfo("CarAttachmentAssignmentDB::Detach", "cannot refresh cache");
        return false;
    }
    auto wg = MakeObjectWriteGuard();

    auto assignmentIt = AssignmentRecords.find(assignmentRecordId);
    if (assignmentIt == AssignmentRecords.end()) {
        session.SetErrorInfo("detachment", "no such assignment record", EDriveSessionResult::InternalError);
        return false;
    }

    TCarGenericAttachment attachment;
    if (!GenericAttachmentDB.TryGetAttachment(assignmentIt->second.GetGenericAttachmentId(), attachment, actuality)) {
        session.SetErrorInfo("detachment", "attachment with such id is not present", EDriveSessionResult::InternalError);
        return false;
    }

    if (!MaybeSetTag(attachment, assignmentIt->second.GetCarId(), attachment->GetAbsenceTagName(), operatorUserId, session, server)) {
        session.AddErrorMessage("detachment", "unable to set absence tag");
        return false;
    }
    if (!MaybeDropTag(attachment, assignmentIt->second.GetCarId(), attachment->GetPresenceTagName(), operatorUserId, session, server)) {
        session.AddErrorMessage("detachment", "unable to drop presence tag");
        return false;
    }

    return DetachWithoutRefreshUnsafe(assignmentRecordId, operatorUserId, session, server);
}

bool TCarAttachmentAssignmentDB::Attach(TCarGenericAttachment& attachment, const TString& carId, const TString& operatorUserId, NDrive::TEntitySession& session, const NDrive::IServer* server, bool wasDetached) const {
    TInstant actuality = Now();
    if (!RefreshCache(actuality)) {
        session.SetErrorInfo("CarAttachmentAssignmentDB::Attach", "cannot refresh cache");
        return false;
    }

    auto wg = MakeObjectWriteGuard();

    // Check if this attachment conflicts with something else by ICC
    if (attachment) {
        ERROR_LOG << "attachment has impl" << Endl;
        for (auto&& icc : attachment->GetUsedICCNumbers()) {
            auto attachmentId = GenericAttachmentDB.GetOwnerByIcc(TCarHardwareSim::NormalizeICC(icc), actuality);
            if (!!attachmentId && !wasDetached) {
                session.SetErrorInfo("assignment", "one of attachment icc occupied", EDriveSessionResult::ResourceLocked);
                return false;
            }
        }
    }

    // Check if this attachment is used
    if (AttachmentToAssignmentRecord.contains(attachment.GetId()) && !wasDetached) {
        ERROR_LOG << "attachment with id " << attachment.GetId() << " is occupied" << Endl;
        session.SetErrorInfo("assignment", "attachment occupied", EDriveSessionResult::ResourceLocked);
        return false;
    }

    // Detach the attachment of this type from the car
    if (attachment.GetUniquePolicy() == IJsonBlobSerializableCarAttachment::EUniquePolicy::Unique) {
        const auto p = CarToAssignments.find(carId);
        const auto& assignments = p != CarToAssignments.end() ? p->second : Default<TVector<TCarAttachmentAssignment>>();
        for (auto presentAssignment : assignments) {
            auto attachmentId = presentAssignment.GetGenericAttachmentId();
            TCarGenericAttachment presentAttachment;
            if (GenericAttachmentDB.TryGetAttachment(attachmentId, presentAttachment, Now())) {
                if (presentAttachment.GetType() == attachment.GetType()) {
                    if (!DetachWithoutRefreshUnsafe(presentAssignment.GetId(), operatorUserId, session, server)) {
                        session.AddErrorMessage("assignment", "unable to detach the present attachment of this kind");
                        return false;
                    }
                }
            }
        }
    }

    // Upsert updated attachment. This may duplicate the existing state, but still is better than to take an attachment and not to update it
    if (!GenericAttachmentDB.UpsertAttachment(attachment, operatorUserId, session)) {
        return false;
    }

    // Now, when we're sure it's free and the car doesn't have anything else attached to the slot
    TCarAttachmentAssignment newAssignment(carId, attachment.GetId());
    if (!AssignmentsTable.Upsert(newAssignment, session)) {
        return false;
    }

    // If it's a Vega, we should update the IMEI of the car
    if (attachment.GetType() == EDocumentAttachmentType::CarHardwareVega) {
        auto fetchResult = CarsDB.FetchInfo(carId, session);
        auto carInfoPtr = fetchResult.GetResultPtr(carId);
        if (!carInfoPtr || !attachment.GetImpl()) {
            session.SetErrorInfo("assignment", "update for nonexistent car or with empty data", EDriveSessionResult::DataCorrupted);
            return false;
        }
        auto carInfo = *carInfoPtr;
        auto vegaImpl = dynamic_cast<const TCarHardwareVega*>(attachment.Get());
        if (!vegaImpl) {
            session.SetErrorInfo("assignment", "empty vega impl", EDriveSessionResult::DataCorrupted);
            return false;
        }
        carInfo.SetIMEI(vegaImpl->GetIMEI());
        if (!CarsDB.UpdateCar(carInfo, operatorUserId, session)) {
            session.AddErrorMessage("assignment", "unable to upsert updated car into car table");
            return false;
        }
        const auto& database = Yensured(server)->GetDriveDatabase();
        auto telematicsConfigurationTag = database.GetTagsManager().GetTagsMeta().CreateTag("telematics_configuration");
        if (telematicsConfigurationTag) {
            if (!database.GetTagsManager().GetDeviceTags().AddTag(telematicsConfigurationTag, operatorUserId, carId, server, session)) {
                session.AddErrorMessage("assignment", "unable to add tag " + telematicsConfigurationTag->GetName());
                return false;
            }
        }
    } else if (attachment.GetType() == EDocumentAttachmentType::CarRegistryDocument) {
        auto fetchResult = CarsDB.FetchInfo(carId, session);
        auto carInfoPtr = fetchResult.GetResultPtr(carId);
        if (carInfoPtr) {  // if nullptr, the car is just added and no need to update its own data
            if (!attachment.GetImpl()) {
                session.SetErrorInfo("assignment", "update with empty data", EDriveSessionResult::DataCorrupted);
                return false;
            }
            auto carInfo = *carInfoPtr;

            auto regDocument = dynamic_cast<const TCarRegistryDocument*>(attachment.Get());

            if (regDocument) {
                bool hasChanged = false;
                if (!regDocument->GetNumber().empty() && carInfo.GetNumber() != regDocument->GetNumber()) {
                    hasChanged = true;
                    carInfo.SetNumber(regDocument->GetNumber());
                }
                if (regDocument->GetRegistrationId() != 0 && carInfo.GetRegistrationID() != regDocument->GetRegistrationId()) {
                    hasChanged = true;
                    carInfo.SetRegistrationID(regDocument->GetRegistrationId());
                }
                if (hasChanged && !CarsDB.UpdateCar(carInfo, operatorUserId, session)) {
                    session.AddErrorMessage("assignment", "unable to upsert updated car into car table");
                    return false;
                }
            }
        }
    } else if (attachment.GetType() == EDocumentAttachmentType::CarSignalDevice) {
        const auto& database = Yensured(server)->GetDriveDatabase();
        const auto& hasSignalqTagName = NDrivematics::THasSignalqTag::TypeName;
        ITag::TPtr hasSignalqTag = database.GetTagsManager().GetTagsMeta().CreateTag(hasSignalqTagName);
        R_ENSURE(hasSignalqTag, HTTP_INTERNAL_SERVER_ERROR, "cannot create tag " << hasSignalqTagName, session);

        if (!database.GetTagsManager().GetDeviceTags().AddTag(hasSignalqTag, operatorUserId, carId, server, session)) {
            session.AddErrorMessage("assignment", "unable to add tag " + hasSignalqTagName);
            return false;
        }
    }

    if (!MaybeDropTag(attachment, carId, attachment->GetAbsenceTagName(), operatorUserId, session, server)) {
        session.AddErrorMessage("assignment", "unable to drop absence tag");
        return false;
    }
    if (!MaybeSetTag(attachment, carId, attachment->GetPresenceTagName(), operatorUserId, session, server)) {
        session.AddErrorMessage("assignment", "unable to set presence tag");
        return false;
    }

    return HistoryManager->AddHistory(newAssignment, operatorUserId, EObjectHistoryAction::Add, session);
}

size_t TCarAttachmentAssignmentDB::AttachmentsCount(const TString& carId, EDocumentAttachmentType type) const {
    auto allAssignments = GetAssignmentsOfTypeUnsafe(TSet<TString>({ carId }), type);
    if (allAssignments.size() == 0) {
        return 0;
    }
    return allAssignments.size();
}

TVector<TString> TCarAttachmentAssignmentDB::GetAssignmentIdsForAttachments(const TVector<TString>& removalCandidates, const TInstant actuality) const {
    Y_ENSURE_BT(RefreshCache(actuality));
    auto rg = MakeObjectReadGuard();
    TVector<TString> result;
    for (auto&& id : removalCandidates) {
        auto it = AttachmentToAssignmentRecord.find(id);
        if (it == AttachmentToAssignmentRecord.end()) {
            continue;
        }
        result.push_back(it->second.GetId());
    }
    return result;
}

bool TCarAttachmentAssignmentDB::MaybeSetTag(const TCarGenericAttachment& attachment, const TString& carId, const TString& tagName, const TString& operatorUserId, NDrive::TEntitySession& session, const NDrive::IServer* server) const {
    auto uniquePolicy = attachment->GetUniquePolicy();
    if (!tagName || (uniquePolicy != IJsonBlobSerializableCarAttachment::EUniquePolicy::Unique && AttachmentsCount(carId, attachment.GetType()) > 1)) {
        return true;
    }

    auto tag = server->GetDriveDatabase().GetTagsManager().GetTagsMeta().CreateTag(tagName, "set automatically");
    NDrive::TEntitySession tagSession(session.GetTransaction());

    if (!tag) {
        return true;
    }

    if (!server->GetDriveDatabase().GetTagsManager().GetDeviceTags().AddTag(tag, operatorUserId, carId, server, tagSession)) {
        return false;
    }

    return true;
}

bool TCarAttachmentAssignmentDB::MaybeDropTag(const TCarGenericAttachment& attachment, const TString& carId, const TString& tagName, const TString& operatorUserId, NDrive::TEntitySession& session, const NDrive::IServer* server) const {
    auto uniquePolicy = attachment->GetUniquePolicy();
    if (!tagName || (uniquePolicy != IJsonBlobSerializableCarAttachment::EUniquePolicy::Unique && AttachmentsCount(carId, attachment.GetType()) > 1)) {
        return true;
    }

    TVector<TDBTag> dbTags;
    if (!server->GetDriveDatabase().GetTagsManager().GetDeviceTags().RestoreTags({ carId }, { tagName }, dbTags, session)) {
        return false;
    }
    if (dbTags.size() && !server->GetDriveDatabase().GetTagsManager().GetDeviceTags().RemoveTagsSimple(dbTags, operatorUserId, session, true)) {
        return false;
    }

    return true;
}

TMaybe<TVector<TString>> TCarAttachmentAssignmentDB::GetActiveAttachmentIds(const TString& carId, NDrive::TEntitySession& tx) const {
    auto optionalAssignments = AssignmentsTable.Fetch(tx, NSQL::TQueryOptions()
        .AddGenericCondition("car_id", carId)
    );
    if (!optionalAssignments) {
        return {};
    }

    TVector<TString> attachmentIds;
    for (auto&& assignment : *optionalAssignments) {
        Y_ASSERT(assignment.GetCarId() == carId);
        if (assignment.GetUnassignedAt() != TInstant::Zero()) {
            continue;
        }
        attachmentIds.push_back(assignment.GetGenericAttachmentId());
    }
    return attachmentIds;
}

TOptionalCarGenericAttachments TCarAttachmentAssignmentDB::GetActiveAttachments(const TString& carId, NAttachmentReport::TReportTraits traits, NDrive::TEntitySession& tx) const {
    auto optionalAttachmentIds = GetActiveAttachmentIds(carId, tx);
    if (!optionalAttachmentIds) {
        return {};
    }
    auto optionalAttachments = GenericAttachmentDB.GetAttachments(*optionalAttachmentIds, tx);
    if (!optionalAttachments) {
        return {};
    }
    TCarGenericAttachments result;
    for (auto&& attachment : *optionalAttachments) {
        if (attachment && (attachment->GetReportTrait() & traits)) {
            result.push_back(std::move(attachment));
        }
    }
    return result;
}

TOptionalCarAttachmentAssignment TCarAttachmentAssignmentDB::GetActiveAttachmentAssignment(const TString& attachmentId, NDrive::TEntitySession& tx) const {
    auto optionalAssignments = AssignmentsTable.Fetch(tx, NSQL::TQueryOptions()
        .AddGenericCondition("document_id", attachmentId)
    );
    if (!optionalAssignments) {
        return {};
    }
    for (auto&& assignment : *optionalAssignments) {
        Y_ASSERT(assignment.GetGenericAttachmentId() == attachmentId);
        if (assignment.GetUnassignedAt() == TInstant::Zero()) {
            return std::move(assignment);
        }
    }
    return TCarAttachmentAssignment();
}

TOptionalCarGenericAttachments TCarAttachmentAssignmentDB::GetAttachmentOfType(const TString& carId, EDocumentAttachmentType type, NDrive::TEntitySession& tx) const {
    auto optionalAttachmentIds = GetActiveAttachmentIds(carId, tx);
    if (!optionalAttachmentIds) {
        return {};
    }
    auto attachmentIds = std::move(*optionalAttachmentIds);
    return GenericAttachmentDB.GetAttachments(attachmentIds, tx, type);
}

TOptionalCarGenericAttachments TCarAttachmentAssignmentDB::GetAttachmentsByInstant(const TString& carId, TInstant date, NDrive::TEntitySession& tx, TMaybe<EDocumentAttachmentType> type) const {
    TSet<TString> removed;
    TSet<TString> attachments;
    auto events = HistoryManager->GetEvents({}, { TInstant::Zero(), date}, tx, NSQL::TQueryOptions(0, true)
        .AddGenericCondition("car_id", carId)
        .SetOrderBy({ "history_event_id" })
    );
    if (!events) {
        return {};
    }
    for (auto&& event : *events) {
        if (event.GetHistoryAction() == EObjectHistoryAction::Remove) {
            removed.insert(event.GetId());
        } else if (removed.contains(event.GetId())) {
            continue;
        } else if (event.GetHistoryAction() == EObjectHistoryAction::Add && (!event.GetUnassignedAt() || event.GetUnassignedAt() > date)) {
            attachments.insert(event.GetGenericAttachmentId());
        }
    }
    auto optionalAssignments = AssignmentsTable.Fetch(tx, NSQL::TQueryOptions()
        .AddGenericCondition("car_id", carId)
        .AddCustomCondition(TStringBuilder()
            << "assigned_at < " << tx->Quote(date.ToString())
            << " AND "
            << "(unassigned_at IS NULL OR unassigned_at > " << tx->Quote(date.ToString()) << ")"
        )
    );
    if (!optionalAssignments) {
        return {};
    }
    for (auto&& assignment : *optionalAssignments) {
        if (!removed.contains(assignment.GetId())) {
            attachments.insert(assignment.GetGenericAttachmentId());
        }
    }
    return GenericAttachmentDB.GetAttachments(MakeVector(attachments), tx, type);
}

TOptionalAttachmentsByCar TCarAttachmentAssignmentDB::GetAttachmentOfType(const TSet<TString>& carIds, EDocumentAttachmentType type, NDrive::TEntitySession& tx) const {
    auto optionalAssignments = AssignmentsTable.Fetch(tx, NSQL::TQueryOptions()
        .SetGenericCondition("car_id", carIds)
    );
    if (!optionalAssignments) {
        return {};
    }
    TMap<TString, TString> map;
    Transform(
        optionalAssignments->begin(), optionalAssignments->end(),std::inserter(map, map.begin()),
        [](const auto& item) -> std::pair<TString, TString> {
            return { item.GetGenericAttachmentId(), item.GetCarId() };
        }
    );
    auto attachments = GenericAttachmentDB.GetAttachments(MakeVector(NContainer::Keys(map)), tx, type);
    if (!attachments) {
        return {};
    }
    TAttachmentsByCar result;
    for (auto&& attachment : *attachments) {
        if (auto carId = map.FindPtr(attachment.GetId())) {
            result[*carId].emplace_back(std::move(attachment));
        }
    }
    return result;
}

bool TCarAttachmentAssignmentDB::TryGetAttachmentOfType(const TString& carId, EDocumentAttachmentType attachmentType, TCarGenericAttachment& attachment, const TInstant actuality) const {
    TVector<TCarGenericAttachment> allAttachments = GetActiveAttachments(carId, NAttachmentReport::ReportAll, actuality);
    for (const auto& item : allAttachments) {
        if (item.GetType() == attachmentType) {
            attachment = item;
            return true;
        }
    }
    return false;
}

TString TCarAttachmentAssignmentDB::GetAttachmentOwnerByServiceAppSlug(const TString& slug, const TInstant actuality) const {
    Y_ENSURE_BT(RefreshCache(actuality));
    auto rg = MakeObjectReadGuard();
    TCarGenericAttachment attachment;
    if (GenericAttachmentDB.TryGetAttachmentByServiceAppSlug(slug, attachment, actuality)) {
        TString attachmentId = attachment.GetId();
        TCarAttachmentAssignment assignment;
        if (TryGetActiveAttachmentAssignment(attachment.GetId(), assignment, actuality)) {
            return assignment.GetCarId();
        }
        return "";
    }
    return "";
}

TString TCarAttachmentAssignmentDB::GetCarByHeadId(const TString& headId, const TInstant actuality) const {
    return GetAttachmentOwnerByServiceAppSlug(headId, actuality);
}

TString TCarAttachmentAssignmentDB::GetCarByTransponderSpbCode(const TString& transponderCode, const TInstant actuality) const {
    return GetAttachmentOwnerByServiceAppSlug(transponderCode, actuality);
}

size_t TCarAttachmentAssignmentDB::Size(const TInstant actuality) const {
    Y_ENSURE_BT(RefreshCache(actuality));
    auto rg = MakeObjectReadGuard();
    return TotalAssignments;
}

bool TCarAttachmentAssignmentDB::GetEffectiveInsurancePolicy(const TString& carId, const TInstant queryInstant, TCarInsurancePolicy& policy, const TInstant actuality) const {
    auto rg = MakeObjectReadGuard();
    auto activeAttachments = GetActiveAttachments(carId, EDocumentAttachmentType::CarInsurancePolicy, actuality);

    bool hasDefaultInsurance = false;
    TCarInsurancePolicy defaultInsurance;

    for (auto&& element : activeAttachments) {
        auto insurance = dynamic_cast<const TCarInsurancePolicy*>(element.Get());
        if (!insurance) {
            ERROR_LOG << "INCONSISTENT ATTACHMENT: " << element.GetId() << Endl;
            continue;
        }
        if (insurance->GetValidFrom() <= queryInstant && queryInstant <= insurance->GetValidUntil()) {
            policy = *insurance;
            return true;
        }
        if (insurance->GetValidFrom() == TInstant::Zero() && insurance->GetValidUntil() == TInstant::Zero()) {
            hasDefaultInsurance = true;
            defaultInsurance = *insurance;
        }
    }

    // old-style behaviour
    if (hasDefaultInsurance) {
        policy = std::move(defaultInsurance);
        return true;
    }

    return false;
}

TString CarAttachmentType(EDocumentAttachmentType type) {
    switch (type) {
    case EDocumentAttachmentType::CarHardwareBeacon:
        return "Маяк";
    case EDocumentAttachmentType::CarHardwareModem:
        return "Модем";
    case EDocumentAttachmentType::CarHardwareHead:
        return "Голова";
    case EDocumentAttachmentType::CarHardwareVega:
        return "Вега";
    case EDocumentAttachmentType::CarHardwareSim:
        return "Сим-карта";
    case EDocumentAttachmentType::CarAirportPass:
        return "Метка-пропуск в аэропорты";
    case EDocumentAttachmentType::CarTransponder:
        return "Транспондер";
    case EDocumentAttachmentType::CarTransponderSpb:
        return "Транспондер (СПб)";
    case EDocumentAttachmentType::CarRegistryDocument:
        return "Базовые данные об авто";
    case EDocumentAttachmentType::CarInsurancePolicy:
        return "Страховой полис";
    case EDocumentAttachmentType::CarAirportPassSpb:
        return "Метка-пропуск в Пулково (СПб)";
    case EDocumentAttachmentType::CarHardwareHeadSerialNumber:
        return "Серийный номер головы";
    case EDocumentAttachmentType::CarHardwareHeadNew:
        return "Голова (новый формат)";
    case EDocumentAttachmentType::CarFuelCard:
        return "Топливная карта";
    case EDocumentAttachmentType::CarQrCode:
        return "QR";
    case EDocumentAttachmentType::CarSignalDevice:
        return "Камера SignalQ";
    default:
        return ToString(type);
    }
}
