#pragma once

#include "car.h"

#include <drive/backend/car_attachments/abstract/attachment.h>
#include <drive/backend/car_attachments/documents/fuel_card.h>
#include <drive/backend/car_attachments/documents/insurance.h>
#include <drive/backend/car_attachments/documents/pass.h>
#include <drive/backend/car_attachments/hardware/hardware.h>
#include <drive/backend/car_attachments/registry/registry.h>
#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/history/cache.h>

#include <rtline/util/json_processing.h>

class TCarGenericAttachment: public TPointerCommon<TCarGenericAttachment, const IJsonBlobSerializableCarAttachment> {
public:
    R_FIELD(TString, Id);
    R_READONLY(EDocumentAttachmentType, Type, EDocumentAttachmentType::Unknown);

private:
    TAtomicSharedPtr<IJsonBlobSerializableCarAttachment> AttachmentImpl;

private:
    bool ParseBlob(const TString& blob);

public:
    TCarGenericAttachment() = default;
    TCarGenericAttachment(TAtomicSharedPtr<IJsonBlobSerializableCarAttachment> impl);

    explicit operator bool() const {
        return AttachmentImpl != nullptr;
    }
    const IJsonBlobSerializableCarAttachment* Get() const {
        return Yensured(AttachmentImpl.Get());
    }

    class TCarGenericAttachmentDecoder: public TBaseDecoder {
        R_FIELD(i32, Id, -1);
        R_FIELD(i32, Type, -1);
        R_FIELD(i32, Blob, -1);

    public:
        TCarGenericAttachmentDecoder() = default;

        TCarGenericAttachmentDecoder(const TMap<TString, ui32>& decoderBase) {
            Id = GetFieldDecodeIndex("id", decoderBase);
            Type = GetFieldDecodeIndex("type", decoderBase);
            Blob = GetFieldDecodeIndex("blob", decoderBase);
        }
    };
    using TDecoder = TCarGenericAttachmentDecoder;

    bool DeserializeWithDecoder(const TCarGenericAttachmentDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);
    void DoBuildReportItem(NJson::TJsonValue& item) const;
    static TString GetObjectId(const TCarGenericAttachment& object);
    bool Parse(const NStorage::TTableRecord& row);
    NStorage::TTableRecord SerializeToTableRecord() const;
    static TCarGenericAttachment FromHistoryEvent(const TObjectEvent<TCarGenericAttachment>& historyEvent);

    TAtomicSharedPtr<IJsonBlobSerializableCarAttachment> GetImpl() const;
    TString GetServiceAppSlug() const;
    NJson::TJsonValue BuildReport(const ICarAttachmentReportContext& context) const;
    IJsonBlobSerializableCarAttachment::EUniquePolicy GetUniquePolicy() const;
};
using TCarGenericAttachments = TVector<TCarGenericAttachment>;
using TOptionalCarGenericAttachments = TMaybe<TCarGenericAttachments>;
using TAttachmentsByCar = TMap<TString, TCarGenericAttachments>;
using TOptionalAttachmentsByCar = TMaybe<TAttachmentsByCar>;

class TCarAttachmentHistoryManager : public TIndexedAbstractHistoryManager<TCarGenericAttachment> {
private:
    using TBase = TIndexedAbstractHistoryManager<TCarGenericAttachment>;

public:
    TCarAttachmentHistoryManager(const IHistoryContext& context, const THistoryConfig& hConfig)
        : TBase(context, "drive_car_document_history", hConfig)
    {
    }

    bool GetEventsSinceId(TBase::TEventId id, TBase::TEventPtrs& results, const TInstant actuality) const override;
};

class TCarGenericAttachmentTable : public TCachedEntityManager<TCarGenericAttachment> {
private:
    using TBase = TCachedEntityManager<TCarGenericAttachment>;

public:
    using TBase::TBase;

    virtual TString GetTableName() const override {
        return "car_document";
    }

    virtual TString GetMainId(const TCarGenericAttachment& e) const override {
        return e.GetId();
    }
};

class TCarGenericAttachmentDB : protected TDBCacheWithHistoryOwner<TCarAttachmentHistoryManager, TCarGenericAttachment> {
private:
    using TDBTable = TCarGenericAttachmentTable;
    using TCacheBase = TDBCacheWithHistoryOwner<TCarAttachmentHistoryManager, TCarGenericAttachment>;

private:
    TDBTable AttachmentsTable;

    mutable TInstant LastRefreshTimestamp = TInstant::Zero();
    mutable TMap<TString, TCarGenericAttachment> IdToAttachment;
    mutable TMap<TString, TCarGenericAttachment> RecognizedDevices;
    mutable TMap<TString, TVector<TString>> IccById;
    mutable TMap<TString, TString> IccOwner;
    mutable TMap<TString, TString> PhoneNumberByICC;
    mutable TSet<TString> ActiveAttachments;

private:
    bool UpsertImpl(TCarGenericAttachment& candidate, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session);
    bool UpsertModemBySimICC(const TString& normalizedICC, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session);
    bool UpsertVegaByIMEI(const TString& vegaImei, const TString& primaryICC, const TString& secondaryICC, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session);
    bool UpsertBeaconByIMEI(const TString& beaconImei, const TString& beaconSerialNumber, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session);
    bool UpsertHeadByHeadId(const TString& headId, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session);
    bool UpsertHeadNewById(const TString& headId, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session);
    bool UpsertHeadSNByNumber(const TString& serialNumber, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session);
    bool UpsertAirportPassByCode(const TString& code, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session);
    bool UpsertTransponderByCode(const TString& code, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session);
    bool UpsertTransponderSpbByCode(const TString& code, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session);
    bool UpsertAirportPassSpbByCode(const TString& code, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session);
    bool UpsertFuelCardByNumber(const TString& number, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session);
    bool UpsertQrCode(const TString& value, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session);
    bool UpsertSignalDeviceByNumber(const TString& signalDeviceSN, const TString& signalDeviceImei, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session);

    TVector<TString> GetUsedICC(const TCarGenericAttachment& attachment) const;

    void UpsertEntryUnsafe(const TCarGenericAttachment& entry) const;
    void RemoveEntryUnsafe(const TCarGenericAttachment& entry) const;

    virtual void AcceptHistoryEventUnsafe(const TObjectEvent<TCarGenericAttachment>& ev) const override;
    bool DoRebuildCacheUnsafe() const override;
    virtual TStringBuf GetEventObjectId(const TObjectEvent<TCarGenericAttachment>& ev) const override;

public:
    TCarGenericAttachmentDB(NStorage::IDatabase::TPtr db, const IHistoryContext& historyContext)
        : TCacheBase("car_document", historyContext, THistoryConfig()
            .SetDeep(TDuration::Zero())
            .SetUseCache(false)
        )
        , AttachmentsTable(db)
    {
    }

    using TCacheBase::Start;
    using TCacheBase::Stop;
    using TCacheBase::RefreshCache;

    TOptionalCarGenericAttachments GetAttachments(TConstArrayRef<TString> ids, NDrive::TEntitySession& tx, TMaybe<EDocumentAttachmentType> type = {}) const;

    bool TryGetAttachment(const TString& id, TCarGenericAttachment& result, const TInstant actuality = TInstant::Zero()) const;
    bool TryGetAttachmentByServiceAppSlug(const TString& slug, TCarGenericAttachment& result, const TInstant actuality = TInstant::Zero()) const;

    template<class TContainer>
    bool TryGetAttachments(const TContainer& ids, TVector<TCarGenericAttachment>& result, const TInstant actuality = TInstant::Zero()) const {
        if (!RefreshCache(actuality)) {
            return false;
        }
        auto rg = MakeObjectReadGuard();
        bool fetchAllOK = true;
        for (const auto& id : ids) {
            auto itAttachment = IdToAttachment.find(id);
            if (itAttachment == IdToAttachment.end()) {
                fetchAllOK = false;
            } else {
                result.push_back(itAttachment->second);
            }
        }
        return fetchAllOK;
    }

    size_t Size(const TInstant actuality = TInstant::Zero()) const;
    bool UpsertAttachment(TCarGenericAttachment& attachment, const TString& operatorUserId, NDrive::TEntitySession& session) const;
    bool UpsertFromServiceApp(const TString& serviceAppCode, TCarGenericAttachment& result, const TString& operatorUserId, NDrive::TEntitySession& session, TMaybe<EDocumentAttachmentType> type = {});
    void ForceUpdate(const TInstant actuality);

    TString GetOwnerByIcc(const TString& icc, const TInstant actuality) const;

    void ActivateAttachment(const TString& id) const;
    void DeactivateAttachment(const TString& id) const;
};

class TCarAttachmentAssignment {
private:
    R_READONLY(TString, Id);
    R_READONLY(TString, CarId);
    R_READONLY(TString, GenericAttachmentId);

    // Backwards compatibility.
    R_FIELD(TInstant, UnassignedAt, TInstant::Zero());

public:
    TCarAttachmentAssignment() = default;
    TCarAttachmentAssignment(const TString& carId, const TString& genericAttachmentId);
    TCarAttachmentAssignment(const TString& id, const TString& carId, const TString& genericAttachmentId);

    explicit operator bool() const {
        return !Id.empty();
    }

    class TCarAttachmentAssignmentDecoder: public TBaseDecoder {
        R_FIELD(i32, Id, -1);
        R_FIELD(i32, CarId, -1);
        R_FIELD(i32, GenericAttachmentId, -1);
        R_FIELD(i32, UnassignedAt, -1);

    public:
        TCarAttachmentAssignmentDecoder() = default;

        TCarAttachmentAssignmentDecoder(const TMap<TString, ui32>& decoderBase) {
            Id = GetFieldDecodeIndex("id", decoderBase);
            CarId = GetFieldDecodeIndex("car_id", decoderBase);
            GenericAttachmentId = GetFieldDecodeIndex("document_id", decoderBase);
            UnassignedAt = GetFieldDecodeIndex("unassigned_at", decoderBase);
        }
    };
    using TDecoder = TCarAttachmentAssignmentDecoder;

    bool DeserializeWithDecoder(const TCarAttachmentAssignmentDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);
    void DoBuildReportItem(NJson::TJsonValue& item) const;
    static TString GetObjectId(const TCarAttachmentAssignment& object);
    static TCarAttachmentAssignment FromHistoryEvent(const TObjectEvent<TCarAttachmentAssignment>& historyEvent);
    NJson::TJsonValue BuildReport() const;
    NStorage::TTableRecord SerializeToTableRecord() const;
    bool Parse(const NStorage::TTableRecord& row);
};
using TOptionalCarAttachmentAssignment = TMaybe<TCarAttachmentAssignment>;

class TCarAttachmentAssignmentHistoryManager : public TIndexedAbstractHistoryManager<TCarAttachmentAssignment> {
private:
    using TBase = TIndexedAbstractHistoryManager<TCarAttachmentAssignment>;

public:
    TCarAttachmentAssignmentHistoryManager(const IHistoryContext& context, const THistoryConfig& hConfig)
        : TBase(context, "drive_car_document_assignment_history", hConfig)
    {
    }

    bool GetEventsSinceId(TBase::TEventId id, TBase::TEventPtrs& results, const TInstant actuality) const override;
};

class TCarAttachmentAssignmentTable : public TCachedEntityManager<TCarAttachmentAssignment> {
private:
    using TBase = TCachedEntityManager<TCarAttachmentAssignment>;

public:
    using TBase::TBase;

    virtual TString GetTableName() const override {
        return "car_document_assignment";
    }

    virtual TString GetMainId(const TCarAttachmentAssignment& e) const override {
        return e.GetId();
    }
};

class TCarAttachmentAssignmentDB : protected TDBCacheWithHistoryOwner<TCarAttachmentAssignmentHistoryManager, TCarAttachmentAssignment> {
private:
    using TCacheBase = TDBCacheWithHistoryOwner<TCarAttachmentAssignmentHistoryManager, TCarAttachmentAssignment>;
    using TDBTable = TCarAttachmentAssignmentTable;

private:
    TDBTable AssignmentsTable;
    TCarGenericAttachmentDB& GenericAttachmentDB;
    TCarsDB& CarsDB;

    mutable TMap<TString, TVector<TCarAttachmentAssignment>> CarToAssignments;
    mutable TMap<TString, TCarAttachmentAssignment> AttachmentToAssignmentRecord;
    mutable TMap<TString, TCarAttachmentAssignment> AssignmentRecords;
    mutable size_t TotalAssignments = 0;

private:
    bool DetachWithoutRefreshUnsafe(const TString& assignmentRecordId, const TString& operatorUserId, NDrive::TEntitySession& session, const NDrive::IServer* server) const;

    virtual void AcceptHistoryEventUnsafe(const TObjectEvent<TCarAttachmentAssignment>& ev) const override;
    bool DoRebuildCacheUnsafe() const override;
    TStringBuf GetEventObjectId(const TObjectEvent<TCarAttachmentAssignment>& ev) const override;

    size_t AttachmentsCount(const TString& carId, const EDocumentAttachmentType type) const;

    bool MaybeSetTag(const TCarGenericAttachment& attachment, const TString& carId, const TString& tagName, const TString& operatorUserId, NDrive::TEntitySession& session, const NDrive::IServer* server) const;
    bool MaybeDropTag(const TCarGenericAttachment& attachment, const TString& carId, const TString& tagName, const TString& operatorUserId, NDrive::TEntitySession& session, const NDrive::IServer* server) const;

    template<class TContainer>
    std::multimap<TString, TCarGenericAttachment> GetAssignmentsOfTypeUnsafe(const TContainer& ids, const EDocumentAttachmentType type, const TInstant actuality = TInstant::Zero()) const {
        TVector<TString> attachmentIds;
        if (ids.empty()) {
            for (const auto& it : AssignmentRecords) {
                attachmentIds.push_back(it.second.GetGenericAttachmentId());
            }
        } else {
            for (const auto& carId : ids) {
                auto assignments = CarToAssignments.find(carId);
                if (assignments == CarToAssignments.end()) {
                    continue;
                }
                for (const auto& assignment : assignments->second) {
                    attachmentIds.push_back(assignment.GetGenericAttachmentId());
                }
            }
        }
        TVector<TCarGenericAttachment> attachments;
        if (!GenericAttachmentDB.TryGetAttachments(attachmentIds, attachments, actuality)) {
            ERROR_LOG << "Not all attachments were fetched" << Endl;
        }
        TMap<TString, TCarGenericAttachment> idToAttachment;
        for (const auto& attachment : attachments) {
            if (attachment.GetType() == type) {
                idToAttachment[attachment.GetId()] = attachment;
            }
        }
        std::multimap<TString, TCarGenericAttachment> carToAttachment;
        for (const auto& it : AssignmentRecords) {
            TString carId = it.second.GetCarId();
            TString attachmentId = it.second.GetGenericAttachmentId();
            if (idToAttachment.contains(attachmentId)) {
                carToAttachment.insert(std::make_pair(carId, idToAttachment[attachmentId]));
            }
        }
        return carToAttachment;
    }

public:
    TCarAttachmentAssignmentDB(
        NStorage::IDatabase::TPtr db, TCarGenericAttachmentDB& genericAttachmentDB, TCarsDB& carsDB, const IHistoryContext& historyContext
    )
        : TCacheBase("car_document_assignment", historyContext, THistoryConfig()
            .SetDeep(TDuration::Zero())
            .SetUseCache(false)
        )
        , AssignmentsTable(db)
        , GenericAttachmentDB(genericAttachmentDB)
        , CarsDB(carsDB)
    {
    }

    using TCacheBase::GetHistoryManager;
    using TCacheBase::Start;
    using TCacheBase::Stop;
    using TCacheBase::RefreshCache;

    TMaybe<TVector<TString>> GetActiveAttachmentIds(const TString& carId, NDrive::TEntitySession& tx) const;
    TOptionalCarGenericAttachments GetActiveAttachments(const TString& carId, NAttachmentReport::TReportTraits traits, NDrive::TEntitySession& tx) const;
    TOptionalCarAttachmentAssignment GetActiveAttachmentAssignment(const TString& attachmentId, NDrive::TEntitySession& tx) const;

    bool Detach(const TString& assignmentRecordId, const TString& operatorUserId, NDrive::TEntitySession& session, const NDrive::IServer* server) const;
    bool Attach(TCarGenericAttachment& attachment, const TString& carId, const TString& operatorUserId, NDrive::TEntitySession& session, const NDrive::IServer* server, bool wasDetached = false) const;

    template<class TContainer>
    std::multimap<TString, TCarGenericAttachment> GetAssignmentsOfType(const TContainer& ids, EDocumentAttachmentType type, const TInstant actuality = TInstant::Zero()) const {
        Y_ENSURE_BT(RefreshCache(actuality));
        auto rg = MakeObjectReadGuard();
        return GetAssignmentsOfTypeUnsafe(ids, type);
    }

    TVector<TString> GetAssignmentIdsForAttachments(const TVector<TString>& removalCandidates, const TInstant actuality) const;

    TOptionalCarGenericAttachments GetAttachmentsByInstant(const TString& carId, TInstant date, NDrive::TEntitySession& tx, TMaybe<EDocumentAttachmentType> type = {}) const;
    TOptionalCarGenericAttachments GetAttachmentOfType(const TString& carId, EDocumentAttachmentType type, NDrive::TEntitySession& tx) const;
    bool TryGetAttachmentOfType(const TString& carId, EDocumentAttachmentType attachmentType, TCarGenericAttachment& attachment, const TInstant actuality = TInstant::Zero()) const;
    TOptionalAttachmentsByCar GetAttachmentOfType(const TSet<TString>& carIds, EDocumentAttachmentType attachmentType, NDrive::TEntitySession& tx) const;

    TString GetAttachmentOwnerByServiceAppSlug(const TString& slug, const TInstant actuality = TInstant::Zero()) const;
    TString GetCarByHeadId(const TString& headId, const TInstant actuality = TInstant::Zero()) const;
    TString GetCarByTransponderSpbCode(const TString& transponderCode, const TInstant actuality = TInstant::Zero()) const;

    bool GetEffectiveInsurancePolicy(const TString& carId, const TInstant time, TCarInsurancePolicy& policy, const TInstant actuality = TInstant::Zero()) const;

    size_t Size(const TInstant actuality = TInstant::Zero()) const;

private:
    TVector<TCarGenericAttachment> GetActiveAttachments(const TString& carId, const NAttachmentReport::TReportTraits traits, const TInstant actuality = TInstant::Zero()) const;
    bool TryGetActiveAttachmentAssignment(const TString& attachmentId, TCarAttachmentAssignment& assignment, TInstant actuality = TInstant::Zero()) const;
};

TString CarAttachmentType(EDocumentAttachmentType type);
