#include "specification.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/logging/evlog.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/parse.h>

#include <util/digest/multi.h>
#include <util/generic/serialized_enum.h>

namespace {
    TString TransmissionValue(NDriveModelSpecification::ETransmissionType transmission) {
        switch (transmission) {
            case NDriveModelSpecification::ETransmissionType::Manual:
                return "механика";
            case NDriveModelSpecification::ETransmissionType::Automatic:
                return "автомат";
            case NDriveModelSpecification::ETransmissionType::Variable:
                return "вариатор";
            case NDriveModelSpecification::ETransmissionType::Robotic:
                return "робот";
            case NDriveModelSpecification::ETransmissionType::UnknownTransmission:
                return {};
        }
    }

    TString DriveUnitValue(NDriveModelSpecification::EDriveUnitType driveUnit) {
        switch (driveUnit) {
            case NDriveModelSpecification::EDriveUnitType::FourWheel:
                return "полный";
            case NDriveModelSpecification::EDriveUnitType::Front:
                return "передний";
            case NDriveModelSpecification::EDriveUnitType::Rear:
                return "задний";
            case NDriveModelSpecification::EDriveUnitType::UnknownDriveUnit:
                return {};
        }
    }
}

TDriveModelSpecification::TDriveModelSpecification(const TString& id, const TString& name, const TString& value, const ui32 position, const TString& icon)
    : Icon(icon)
    , Id(id)
    , Position(position)
    , Name(name)
    , Value(value)
{
    ParseValueAndType();
}

size_t TDriveModelSpecification::Hash() const {
    return MultiHash(Icon, Name, Value, Position);
}

void TDriveModelSpecification::SetNameAndValue(const TString& name, const TString& value) {
    Name = name;
    Value = value;
    ParseValueAndType();
}

bool TDriveModelSpecification::DeserializeFromJson(const NJson::TJsonValue& json) {
    return
        NJson::ParseField(json["id"], Id) &&
        NJson::ParseField(json["name"], Name) &&
        NJson::ParseField(json["value"], Value) &&
        NJson::ParseField(json["position"], Position) &&
        NJson::ParseField(json["icon"], Icon) &&
        TryParseValueAndType();
}

NJson::TJsonValue TDriveModelSpecification::SerializeToJson() const {
    NJson::TJsonValue result;
    result["id"] = Id;
    result["icon"] = Icon;
    result["name"] = Name;
    result["value"] = Value;
    result["position"] = Position;
    return result;
}

NJson::TJsonValue TDriveModelSpecification::GetReport(ELocalization locale) const {
    NJson::TJsonValue result;
    result["id"] = Id;
    result["name"] = Name;
    result["name_localized"] = GetLocalizedName(locale);
    result["value"] = Value;
    result["value_localized"] = GetLocalizedValue(locale);
    if (Icon) {
        result["icon"] = Icon;
        result["icon_localized"] = Icon;
    } else {
        auto server = NDrive::HasServer() ? &NDrive::GetServer() : nullptr;
        auto localization = server ? server->GetLocalization() : nullptr;
        if (localization) {
            auto key = TStringBuilder() << "model.specification." << Type << '.' << "default_icon";
            auto value = localization->GetLocalString(locale, key, TString{});
            if (value) {
                result["icon_localized"] = std::move(value);
            }
        }
    }
    if (Position) {
        result["position"] = Position;
    }
    return result;
}

NJson::TJsonValue TDriveModelSpecification::GetLegacyReport(ELocalization locale) const {
    NJson::TJsonValue result;
    result["id"] = Id;
    result["value"] = GetLocalizedValue(locale);
    result["name"] = GetLocalizedName(locale);
    result["position"] = Position;
    if (Icon) {
        result["icon"] = Icon;
    } else {
        auto server = NDrive::HasServer() ? &NDrive::GetServer() : nullptr;
        auto localization = server ? server->GetLocalization() : nullptr;
        if (localization) {
            auto key = TStringBuilder() << "model.specification." << Type << '.' << "default_icon";
            auto value = localization->GetLocalString(locale, key, TString{});
            if (value) {
                result["icon"] = std::move(value);
            }
        }
    }
    return result;
}

TString TDriveModelSpecification::GetLocalizedValue(ELocalization locale) const {
    auto server = NDrive::HasServer() ? &NDrive::GetServer() : nullptr;
    auto localization = server ? server->GetLocalization() : nullptr;
    switch (Type) {
        case NDriveModelSpecification::EModelSpecificationType::Transmission:
            if (std::holds_alternative<NDriveModelSpecification::ETransmissionType>(ParsedValue)) {
                auto value = std::get<NDriveModelSpecification::ETransmissionType>(ParsedValue);
                if (locale == LegacyLocale) {
                    return TransmissionValue(value);
                } else {
                    auto key = TStringBuilder() << "model.specification." << Type << '.' << value;
                    return localization->GetLocalString(locale, key);
                }
            }
            break;
        case NDriveModelSpecification::EModelSpecificationType::DriveUnit:
            if (std::holds_alternative<NDriveModelSpecification::EDriveUnitType>(ParsedValue)) {
                auto value = std::get<NDriveModelSpecification::EDriveUnitType>(ParsedValue);
                if (locale == LegacyLocale) {
                    return DriveUnitValue(value);
                } else {
                    auto key = TStringBuilder() << "model.specification." << Type << '.' << value;
                    return localization->GetLocalString(locale, key);
                }
            }
            break;
        case NDriveModelSpecification::EModelSpecificationType::Power:
            if (std::holds_alternative<ui32>(ParsedValue)) {
                auto value = std::get<ui32>(ParsedValue);
                if (locale == LegacyLocale) {
                    return NDrive::TLocalization::PowerValue(value);
                } else {
                    auto key = TStringBuilder() << "model.specification." << Type << '.' << "unit";
                    return localization->FormatNumeral(value, locale, true, " ", key);
                }
            }
            break;
        case NDriveModelSpecification::EModelSpecificationType::AccelerationTime:
            if (std::holds_alternative<float>(ParsedValue)) {
                auto value = std::get<float>(ParsedValue);
                if (locale == LegacyLocale) {
                    return NDrive::TLocalization::AccelerationValue(value);
                } else {
                    auto duration = TDuration::Seconds(value);
                    return localization->FormatDuration(locale, duration, /*withSeconds=*/true, /*allowEmpty=*/true, /*withMilliseconds=*/true);
                }
            }
            break;
        case NDriveModelSpecification::EModelSpecificationType::SeatsNumber:
            if (std::holds_alternative<ui32>(ParsedValue)) {
                auto value = std::get<ui32>(ParsedValue);
                if (locale == LegacyLocale) {
                    return NDrive::TLocalization::SeatsNumberValue(value);
                } else {
                    auto key = TStringBuilder() << "model.specification." << Type << '.' << value;
                    return localization->GetLocalString(locale, key);
                }
            }
            break;
        case NDriveModelSpecification::EModelSpecificationType::FuelTankVolume:
            if (std::holds_alternative<ui32>(ParsedValue)) {
                auto value = std::get<ui32>(ParsedValue);
                if (locale == LegacyLocale) {
                    return NDrive::TLocalization::FuelTankVolumeValue(value);
                } else {
                    auto key = TStringBuilder() << "model.specification." << Type << '.' << "unit";
                    return localization->FormatNumeral(value, locale, true, " ", key);
                }
            }
            break;
        case NDriveModelSpecification::EModelSpecificationType::Range:
            if (std::holds_alternative<ui32>(ParsedValue) && localization) {
                return localization->DistanceFormatKm(locale, std::get<ui32>(ParsedValue), true);
            }
            break;
        case NDriveModelSpecification::EModelSpecificationType::EngineType:
        case NDriveModelSpecification::EModelSpecificationType::ExteriorColor:
        case NDriveModelSpecification::EModelSpecificationType::ExteriorType:
        case NDriveModelSpecification::EModelSpecificationType::FuelType:
        case NDriveModelSpecification::EModelSpecificationType::InteriorColor:
        case NDriveModelSpecification::EModelSpecificationType::InteriorType:
        case NDriveModelSpecification::EModelSpecificationType::AirConditioning:
            if (localization) {
                auto key = TStringBuilder() << "model.specification." << Type << '.' << Value;
                return localization->GetLocalString(locale, key);
            }
            break;
        case NDriveModelSpecification::EModelSpecificationType::Category:
            if (localization) {
                auto key = TStringBuilder() << "model.specification." << Type << '.' << Value;
                return localization->GetLocalString(locale, key);
            }
            break;
        case NDriveModelSpecification::EModelSpecificationType::Other:
            if (localization) {
                auto key = TStringBuilder() << "model.specification." << Type << '.' << Value;
                return localization->GetLocalString(locale, key);
            }
            break;
        case NDriveModelSpecification::EModelSpecificationType::UnknownSpecificationType:
            if (localization) {
                auto key = TStringBuilder() << "model.specification." << Name << '.' << Value;
                return localization->GetLocalString(locale, key, Value);
            }
            break;
        case NDriveModelSpecification::EModelSpecificationType::Year:
            break;
    }
    return Value;
}

TString TDriveModelSpecification::GetLocalizedName(ELocalization locale) const {
    auto server = NDrive::HasServer() ? &NDrive::GetServer() : nullptr;
    auto localization = server ? server->GetLocalization() : nullptr;
    auto defaultValue = TMaybe<TString>();
    auto type = ToString(Type);
    switch (Type) {
        case NDriveModelSpecification::EModelSpecificationType::Transmission:
            if (locale == LegacyLocale) {
                defaultValue = NDrive::TLocalization::TransmissionName();
            }
            break;
        case NDriveModelSpecification::EModelSpecificationType::DriveUnit:
            if (locale == LegacyLocale) {
                defaultValue = NDrive::TLocalization::DriveUnitName();
            }
            break;
        case NDriveModelSpecification::EModelSpecificationType::Power:
            if (locale == LegacyLocale) {
                defaultValue = NDrive::TLocalization::PowerName();
            }
            break;
        case NDriveModelSpecification::EModelSpecificationType::AccelerationTime:
            if (locale == LegacyLocale) {
                defaultValue = NDrive::TLocalization::AccelerationTimeName();
            }
            break;
        case NDriveModelSpecification::EModelSpecificationType::SeatsNumber:
            if (locale == LegacyLocale) {
                defaultValue = NDrive::TLocalization::SeatsNumberName();
            }
            break;
        case NDriveModelSpecification::EModelSpecificationType::FuelTankVolume:
            if (locale == LegacyLocale) {
                defaultValue = NDrive::TLocalization::FuelTankVolumeName();
            }
            break;
        case NDriveModelSpecification::EModelSpecificationType::Range:
        case NDriveModelSpecification::EModelSpecificationType::EngineType:
        case NDriveModelSpecification::EModelSpecificationType::ExteriorColor:
        case NDriveModelSpecification::EModelSpecificationType::ExteriorType:
        case NDriveModelSpecification::EModelSpecificationType::FuelType:
        case NDriveModelSpecification::EModelSpecificationType::InteriorColor:
        case NDriveModelSpecification::EModelSpecificationType::InteriorType:
        case NDriveModelSpecification::EModelSpecificationType::AirConditioning:
        case NDriveModelSpecification::EModelSpecificationType::Category:
        case NDriveModelSpecification::EModelSpecificationType::Other:
        case NDriveModelSpecification::EModelSpecificationType::Year:
            break;
        case NDriveModelSpecification::EModelSpecificationType::UnknownSpecificationType:
            defaultValue = Name;
            type = Name;
            break;
    }
    auto key = TStringBuilder() << "model.specification." << type << ".name";
    return localization ? localization->GetLocalString(locale, key, defaultValue) : key;
}

void TDriveModelSpecification::ParseValueAndType() {
    Y_ENSURE(TryParseValueAndType(), "cannot ParseValueAndType: " << Name << " " << Value);
}

bool TDriveModelSpecification::TryParseValueAndType() {
    TryFromString(Name, Type);
    float floating;
    ui32 integer;
    switch (Type) {
    case NDriveModelSpecification::EModelSpecificationType::Transmission:
        for (auto&& item : GetEnumNames<NDriveModelSpecification::ETransmissionType>()) {
            if (item.second == Value) {
                ParsedValue = item.first;
                return true;
            }
        }
        return false;
    case NDriveModelSpecification::EModelSpecificationType::DriveUnit:
        for (auto&& item : GetEnumNames<NDriveModelSpecification::EDriveUnitType>()) {
            if (item.second == Value) {
                ParsedValue = item.first;
                return true;
            }
        }
        return false;
    case NDriveModelSpecification::EModelSpecificationType::AccelerationTime:
        if (TryFromString(Value, floating)) {
            ParsedValue = floating;
            return true;
        } else {
            return false;
        }
    case NDriveModelSpecification::EModelSpecificationType::Power:
    case NDriveModelSpecification::EModelSpecificationType::SeatsNumber:
    case NDriveModelSpecification::EModelSpecificationType::FuelTankVolume:
    case NDriveModelSpecification::EModelSpecificationType::Range:
    case NDriveModelSpecification::EModelSpecificationType::Year:
        if (TryFromString(Value, integer)) {
            ParsedValue = integer;
            return true;
        } else {
            return false;
        }
    case NDriveModelSpecification::EModelSpecificationType::EngineType:
    case NDriveModelSpecification::EModelSpecificationType::ExteriorColor:
    case NDriveModelSpecification::EModelSpecificationType::ExteriorType:
    case NDriveModelSpecification::EModelSpecificationType::FuelType:
    case NDriveModelSpecification::EModelSpecificationType::InteriorColor:
    case NDriveModelSpecification::EModelSpecificationType::InteriorType:
    case NDriveModelSpecification::EModelSpecificationType::AirConditioning:
    case NDriveModelSpecification::EModelSpecificationType::Category:
    case NDriveModelSpecification::EModelSpecificationType::Other:
    case NDriveModelSpecification::EModelSpecificationType::UnknownSpecificationType:
        return !Value.empty();
    }
}

template <>
NJson::TJsonValue NJson::ToJson(const TDriveModelSpecification& object) {
    return object.SerializeToJson();
}

bool TDriveModelSpecifications::Merge(const TDriveModelSpecifications& other) {
    for (auto&& s : other.GetSpecifications()) {
        auto specification = s;
        if (!Upsert(std::move(specification))) {
            return false;
        }
    }
    return true;
}

bool TDriveModelSpecifications::Upsert(TDriveModelSpecification&& spec) {
    if (!ParseKnownSpecification(spec)) {
        return false;
    }

    bool replaced = false;
    for (auto&& i : Specifications) {
        if (i.GetId() == spec.GetId() || i.GetName() == spec.GetName()) {
            i = std::move(spec);
            replaced = true;
            break;
        }
    }
    if (!replaced) {
        Specifications.emplace_back(std::move(spec));
    }

    std::sort(Specifications.begin(), Specifications.end(), [](const TDriveModelSpecification& left, const TDriveModelSpecification& right) {
        return left.GetPosition() < right.GetPosition();
    });

    return true;
}

bool TDriveModelSpecifications::ParseKnownSpecification(const TDriveModelSpecification& spec) {
    auto type = spec.GetType();
    switch (type) {
    case NDriveModelSpecification::EModelSpecificationType::Transmission:
        if (std::holds_alternative<NDriveModelSpecification::ETransmissionType>(spec.GetParsedValue())) {
            Transmission = std::get<NDriveModelSpecification::ETransmissionType>(spec.GetParsedValue());
            return true;
        }
        return false;
    case NDriveModelSpecification::EModelSpecificationType::DriveUnit:
        if (std::holds_alternative<NDriveModelSpecification::EDriveUnitType>(spec.GetParsedValue())) {
            DriveUnit = std::get<NDriveModelSpecification::EDriveUnitType>(spec.GetParsedValue());
            return true;
        }
        return false;
    case NDriveModelSpecification::EModelSpecificationType::AccelerationTime:
        if (std::holds_alternative<float>(spec.GetParsedValue())) {
            AccelerationTime = std::get<float>(spec.GetParsedValue());
            return true;
        } else {
            return false;
        }
    case NDriveModelSpecification::EModelSpecificationType::Power:
        if (std::holds_alternative<ui32>(spec.GetParsedValue())) {
            Power = std::get<ui32>(spec.GetParsedValue());
            return true;
        } else {
            return false;
        }
    case NDriveModelSpecification::EModelSpecificationType::SeatsNumber:
        if (std::holds_alternative<ui32>(spec.GetParsedValue())) {
            SeatsNumber = std::get<ui32>(spec.GetParsedValue());
            return true;
        } else {
            return false;
        }
    case NDriveModelSpecification::EModelSpecificationType::FuelTankVolume:
        if (std::holds_alternative<ui32>(spec.GetParsedValue())) {
            FuelTankVolume = std::get<ui32>(spec.GetParsedValue());
            return true;
        } else {
            return false;
        }
    case NDriveModelSpecification::EModelSpecificationType::Range:
        if (std::holds_alternative<ui32>(spec.GetParsedValue())) {
            Range = std::get<ui32>(spec.GetParsedValue());
            return true;
        } else {
            return false;
        }
    case NDriveModelSpecification::EModelSpecificationType::Year:
        if (std::holds_alternative<ui32>(spec.GetParsedValue())) {
            Year = std::get<ui32>(spec.GetParsedValue());
            return true;
        } else {
            return false;
        }
    case NDriveModelSpecification::EModelSpecificationType::EngineType:
    case NDriveModelSpecification::EModelSpecificationType::ExteriorColor:
    case NDriveModelSpecification::EModelSpecificationType::ExteriorType:
    case NDriveModelSpecification::EModelSpecificationType::FuelType:
    case NDriveModelSpecification::EModelSpecificationType::InteriorColor:
    case NDriveModelSpecification::EModelSpecificationType::InteriorType:
    case NDriveModelSpecification::EModelSpecificationType::Category:
    case NDriveModelSpecification::EModelSpecificationType::AirConditioning:
    case NDriveModelSpecification::EModelSpecificationType::Other:
    case NDriveModelSpecification::EModelSpecificationType::UnknownSpecificationType:
        return !spec.GetValue().empty();
    }
}

const TDriveModelSpecification* TDriveModelSpecifications::Get(std::string_view name) const {
    for (auto&& specification : Specifications) {
        if (specification.GetName() == name) {
            return &specification;
        }
    }
    return nullptr;
}

NJson::TJsonValue TDriveModelSpecifications::GetReport(ELocalization locale) const {
    NJson::TJsonValue result = NJson::JSON_ARRAY;
    for (auto&& specification : Specifications) {
        result.AppendValue(specification.GetReport(locale));
    }
    return result;
}

NJson::TJsonValue TDriveModelSpecifications::GetLegacyReport(ELocalization locale) const {
    NJson::TJsonValue result = NJson::JSON_ARRAY;
    for (auto&& specification : Specifications) {
        result.AppendValue(specification.GetLegacyReport(locale));
    }
    return result;
}

size_t TDriveModelSpecifications::Hash() const {
    size_t result = 0;
    for (auto&& specification : Specifications) {
        result = CombineHashes(result, specification.Hash());
    }
    return result;
}

bool TDriveModelSpecifications::Deserialize(const NJson::TJsonValue& specificationsArray, TStringBuf objectId) {
    auto evlog = NDrive::GetThreadEventLogger();
    Specifications.clear();
    if (specificationsArray.GetType() != NJson::EJsonValueType::JSON_ARRAY) {
        return false;
    }
    for (auto&& element : specificationsArray.GetArray()) {
        TDriveModelSpecification spec;
        if (!spec.DeserializeFromJson(element)) {
            if (evlog) {
                evlog->AddEvent(NJson::TMapBuilder
                    ("event", "SpecificationDeserializeError")
                    ("value", element)
                );
            }
            return false;
        }
        if (!spec.GetId()) {
            spec.SetId(TStringBuilder() << objectId << '_' << spec.GetName());
        }
        if (!Upsert(std::move(spec))) {
            if (evlog) {
                evlog->AddEvent(NJson::TMapBuilder
                    ("event", "SpecificationUpsertError")
                    ("value", element)
                );
            }
            continue;
        }
    }

    return true;
}

NJson::TJsonValue TDriveModelSpecifications::Serialize() const {
    return NJson::ToJson(Specifications);
}
