#pragma once

#include <drive/backend/abstract/localization.h>

#include <rtline/util/types/accessor.h>

#include <variant>

namespace NDriveModelSpecification {
    enum EModelSpecificationType: ui32 {
        Transmission,
        DriveUnit,
        Power,
        AccelerationTime,
        SeatsNumber,
        FuelTankVolume,
        FuelType,
        EngineType,
        Range,
        ExteriorColor,
        ExteriorType,
        InteriorColor,
        InteriorType,
        Year,
        AirConditioning,
        Category,
        Other,
        UnknownSpecificationType
    };

    enum ETransmissionType: ui32 {
        Manual,
        Automatic,
        Variable,
        Robotic,
        UnknownTransmission
    };

    enum EDriveUnitType: ui32 {
        FourWheel,
        Front,
        Rear,
        UnknownDriveUnit
    };

    using TParsedValue = std::variant<NDriveModelSpecification::ETransmissionType, NDriveModelSpecification::EDriveUnitType, ui32, float>;
};

class TDriveModelSpecification {
    R_FIELD(TString, Icon);
    R_FIELD(TString, Id);
    R_FIELD(ui32, Position, 0);
    R_READONLY(TString, Name);
    R_READONLY(TString, Value);
    R_READONLY(NDriveModelSpecification::EModelSpecificationType, Type, NDriveModelSpecification::EModelSpecificationType::UnknownSpecificationType);
    R_READONLY(NDriveModelSpecification::TParsedValue, ParsedValue, (ui32)(0));

public:
    TDriveModelSpecification() = default;
    TDriveModelSpecification(const TDriveModelSpecification& other) = default;
    TDriveModelSpecification(const TString& id, const TString& name, const TString& value, const ui32 position, const TString& icon = {});

    void SetNameAndValue(const TString& name, const TString& value);

    size_t Hash() const;

    bool DeserializeFromJson(const NJson::TJsonValue& json);
    NJson::TJsonValue GetReport(ELocalization locale) const;
    NJson::TJsonValue GetLegacyReport(ELocalization locale) const;
    NJson::TJsonValue SerializeToJson() const;

    TString GetLocalizedValue(ELocalization locale) const;
    TString GetLocalizedName(ELocalization locale) const;

private:
    void ParseValueAndType();
    bool TryParseValueAndType();
};

class TDriveModelSpecifications {
public:
    R_READONLY(TVector<TDriveModelSpecification>, Specifications);
    R_READONLY(NDriveModelSpecification::ETransmissionType, Transmission, NDriveModelSpecification::ETransmissionType::UnknownTransmission);
    R_READONLY(NDriveModelSpecification::EDriveUnitType, DriveUnit, NDriveModelSpecification::EDriveUnitType::UnknownDriveUnit);
    R_READONLY(ui32, Power, 0);
    R_READONLY(float, AccelerationTime, 0.0);
    R_READONLY(ui32, SeatsNumber, 0);
    R_READONLY(ui32, FuelTankVolume, 0);
    R_READONLY(ui32, Range, 0);
    R_READONLY(ui32, Year, 0);

public:
    TDriveModelSpecifications() = default;
    TDriveModelSpecifications(TVector<TDriveModelSpecification>&& specifications)
        : Specifications(std::move(specifications))
    {
    }

    explicit operator bool() const {
        return !Specifications.empty();
    }

    const TDriveModelSpecification* Get(std::string_view name) const;

    NJson::TJsonValue GetReport(ELocalization locale) const;
    NJson::TJsonValue GetLegacyReport(ELocalization locale) const;

    size_t Hash() const;

    bool Merge(const TDriveModelSpecifications& other);
    bool Upsert(TDriveModelSpecification&& spec);

    bool Deserialize(const NJson::TJsonValue& value, TStringBuf objectId = {});
    NJson::TJsonValue Serialize() const;

private:
    bool ParseKnownSpecification(const TDriveModelSpecification& spec);
};
