#include "state_filters.h"

#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/library/unistat/cache.h>
#include <rtline/util/json_processing.h>
#include <rtline/util/algorithm/container.h>

#include <util/string/join.h>

TStateFilter::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase) {
    StateId = GetFieldDecodeIndex("state_id", decoderBase);
    Description = GetFieldDecodeIndex("state_description", decoderBase);
    Priority = GetFieldDecodeIndex("state_priority", decoderBase);
    Filter = GetFieldDecodeIndex("state_filter", decoderBase);
    SensorFilter = GetFieldDecodeIndex("sensor_filter", decoderBase);
}

bool TStateFilter::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, StateId);
    READ_DECODER_VALUE(decoder, values, Description);
    READ_DECODER_VALUE(decoder, values, Priority);

    if (!StateId) {
        return false;
    }
    {
        NJson::TJsonValue filterJson;
        READ_DECODER_VALUE_JSON(decoder, values, filterJson, Filter);
        if (!Filter.DeserializeFromJson(filterJson)) {
            return false;
        }
    }
    if (decoder.GetSensorFilter() >= 0 && !values[decoder.GetSensorFilter()].empty()) {
        NJson::TJsonValue filterJson;
        READ_DECODER_VALUE_JSON(decoder, values, filterJson, SensorFilter);
        if (!NJson::ParseField(filterJson, SensorFilter)) {
            return false;
        }
    }
    return true;
}

bool TStateFilter::DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    NStorage::TTableRecord record;
    if (!record.DeserializeFromJson(jsonInfo)) {
        return false;
    }
    return DeserializeFromTableRecord(record, nullptr);
}

NJson::TJsonValue TStateFilter::SerializeToReport() const {
    NJson::TJsonValue result = NJson::JSON_MAP;
    result.InsertValue("state_id", StateId);
    result.InsertValue("state_description", Description);
    result.InsertValue("state_priority", Priority);
    result.InsertValue("state_filter", Filter.SerializeToJson());
    if (SensorFilter) {
        result.InsertValue("sensor_filter", NJson::ToJson(SensorFilter));
    }
    return result;
}


bool TStateFilter::DeserializeFromTableRecord(const NStorage::TTableRecord& record, const ITagsHistoryContext* /*context*/) {
    StateId = record.Get("state_id");
    if (!StateId) {
        return false;
    }

    Description = record.Get("state_description");
    if (!record.TryGet("state_priority", Priority)) {
        return false;
    }

    const TString& filterStr = record.Get("state_filter");
    NJson::TJsonValue filterJson;
    if (!NJson::ReadJsonFastTree(filterStr, &filterJson) || !Filter.DeserializeFromJson(filterJson)) {
        return false;
    }

    const auto& sensorFilterString = record.Get("sensor_filter");
    if (sensorFilterString) {
        NJson::TJsonValue sensorFilterJson;
        if (!NJson::ReadJsonFastTree(sensorFilterString, &sensorFilterJson)) {
            return false;
        }
        if (!NJson::ParseField(sensorFilterJson, SensorFilter)) {
            return false;
        }
    }

    return true;
}

NStorage::TTableRecord TStateFilter::SerializeToTableRecord() const {
    NStorage::TTableRecord result;
    result.Set("state_id", StateId)
        .Set("state_description", Description)
        .Set("state_priority", Priority)
        .Set("state_filter", Filter.SerializeToJson().GetStringRobust())
        ;
    if (SensorFilter) {
        result.Set("sensor_filter", NJson::ToJson(SensorFilter));
    }
    return result;
}

TStateFiltersDB::TObjectStates::TObjectStates(TStatesPtr value)
    : Value(value ? value : MakeAtomicShared<TStates>())
{
}

bool TStateFiltersDB::MetricSignal() {
    if (StatesRefreshInstant) {
        auto freshness = Now() - StatesRefreshInstant;
        TUnistatSignalsCache::SignalLastX("state_calcer", "freshness", freshness.MilliSeconds());
    }
    return true;
}

bool TStateFiltersDB::Refresh() {
    if (!TBase::Refresh()) {
        return false;
    }
    RefreshStates();
    return true;
}

void TStateFiltersDB::RefreshFiltersUnsafe() const {
    TVector<TStateFilter> filters = MakeVector(NContainer::Values(TBase::Objects));
    std::sort(filters.begin(), filters.end());
    Filters = std::move(filters);
}

bool TStateFiltersDB::RefreshStates() const {
    TVector<TTaggedDevice> devices;
    if (!TagsManager.GetAllObjectsFromCache(devices)) {
        return true;
    }

    TConstDevicesSnapshot snapshots;
    if (NDrive::HasServer()) {
        const auto& server = NDrive::GetServerAs<NDrive::IServer>();
        const auto& snapshotManager = server.GetSnapshotsManager();
        snapshots = snapshotManager.GetSnapshots();
    }

    auto states = MakeAtomicShared<TStates>();
    TMap<TString, ui32> statesNewCount;
    {
        auto rg = MakeObjectReadGuard();
        for (auto&& i : devices) {
            const auto& snapshot = snapshots.GetSnapshot(i.GetId());
            const auto& sensors = snapshot ? snapshot->GetSensors() : Default<NDrive::TSensors>();
            const auto& status = CalcStatusUnsafe(i.GetTags(), sensors);
            states->emplace(i.GetId(), status);
            ++statesNewCount[status];
        }
    }
    for (auto&& i : statesNewCount) {
        TUnistatSignalsCache::SignalLastX("status-count", i.first, i.second);
    }
    TWriteGuard gMutex(Mutex);
    ObjectStates = std::move(states);
    StatesRefreshInstant = Now();
    return true;
}

TString TStateFiltersDB::CalcStatus(const TVector<TDBTag>& tags, const NDrive::TSensors& sensors) const {
    auto rg = MakeObjectReadGuard();
    return CalcStatusUnsafe(tags, sensors);
}

TString TStateFiltersDB::CalcStatusUnsafe(const TVector<TDBTag>& tags, const NDrive::TSensors& sensors) const {
    TString result = "unknown";
    auto now = Now();
    auto prepared = TTagsFilter::PrepareMatching(tags);
    for (auto&& filter : Filters) {
        const auto& tagFilter = filter.GetFilter();
        if (tagFilter && !tagFilter.IsMatching(prepared)) {
            continue;
        }

        const auto& sensorFilter = filter.GetSensorFilter();
        if (sensorFilter && !sensorFilter.Match(sensors, now)) {
            continue;
        }

        result = filter.GetStateId();
        break;
    }
    return result;
}

bool TStateFiltersDB::DoRebuildCacheUnsafe() const {
    NStorage::TObjectRecordsSet<TStateFilter> records;
    {
        TTransactionPtr transaction = HistoryCacheDatabase->CreateTransaction(true);
        auto tagsTable = HistoryCacheDatabase->GetTable("drive_state_filters");

        TQueryResultPtr queryResult;

        queryResult = tagsTable->GetRows("", records, transaction);

        if (!queryResult->IsSucceed()) {
            return false;
        }
    }
    for (auto&& i : records) {
        Objects.emplace(i.GetStateId(), i);
    }
    RefreshFiltersUnsafe();
    return true;
}

void TStateFiltersDB::AcceptHistoryEventUnsafe(const TObjectEvent<TStateFilter>& ev) const {
    if (ev.GetHistoryAction() == EObjectHistoryAction::Remove) {
        Objects.erase(ev.GetStateId());
    } else {
        Objects[ev.GetStateId()] = ev;
    }
    RefreshFiltersUnsafe();
}

TStateFiltersDB::TStateFiltersDB(const TDeviceTagsManager& tagsManager, const ITagsHistoryContext& context, const THistoryConfig& config)
    : TBase("drive_state_filters", context, config)
    , TagsManager(tagsManager)
    , ObjectStates(MakeAtomicShared<TStates>())
{
    SetMetricPeriod(TDuration::Seconds(1));
}

TSet<TString> TStateFiltersDB::GetAvailableStates(const TInstant reqActuality) const {
    TSet<TString> stateIds;
    const auto action = [&stateIds](const auto& stateId, const auto& /* entity */) { stateIds.insert(stateId); };
    Y_UNUSED(ForObjectsMap(action, reqActuality));
    return stateIds;
}

TString TStateFiltersDB::GetObjectState(const TString& objectId, const TString& fallback) const {
    TReadGuard g(Mutex);
    return ObjectStates->Value(objectId, fallback);
}

TStateFiltersDB::TObjectStates TStateFiltersDB::GetObjectStates() const {
    TReadGuard g(Mutex);
    return ObjectStates;
}

bool TStateFiltersDB::GetHistory(const TInstant& since, TVector<TAtomicSharedPtr<TObjectEvent<TStateFilter>>>& result, const TInstant reqActuality) const {
    return TBase::HistoryManager->GetEventsAll(since, result, reqActuality);
}

bool TStateFiltersDB::Upsert(const TStateFilter& filter, const TString& userId, NDrive::TEntitySession& session) const {
    NStorage::ITransaction::TPtr transaction = session.GetTransaction();
    NStorage::ITableAccessor::TPtr table = HistoryCacheDatabase->GetTable("drive_state_filters");
    const NStorage::TTableRecord record = filter.SerializeToTableRecord();
    NStorage::TRecordBuilder unique("state_id", filter.GetStateId());
    bool isUpdate;
    if (!table->Upsert(record, transaction, unique, &isUpdate)) {
        session.SetErrorInfo("set_state_filter", "Upsert failed", EDriveSessionResult::TransactionProblem);
        return false;
    }
    if (!HistoryManager->AddHistory(filter, userId, isUpdate ? EObjectHistoryAction::UpdateData : EObjectHistoryAction::Add, session)) {
        return false;
    }
    return true;
}

bool TStateFiltersDB::RemoveKeys(const TVector<TString>& ids, const TString& userId, NDrive::TEntitySession& session) const {
    if (ids.empty())
        return true;
    NStorage::ITransaction::TPtr transaction = session.GetTransaction();
    NStorage::ITableAccessor::TPtr table = HistoryCacheDatabase->GetTable("drive_state_filters");
    const TString condition = "state_id IN (" + session->Quote(ids) + ")";
    TRecordsSet records;
    if (!table->RemoveRow(condition, transaction, &records)) {
        session.SetErrorInfo("remove_state_filters", "RemoveRow failed", EDriveSessionResult::TransactionProblem);
        return false;
    }

    TVector<TStateFilter> states;
    for (auto&& i : records) {
        TStateFilter state;
        if (state.DeserializeFromTableRecord(i, nullptr)) {
            states.push_back(state);
        }
    }

    if (!HistoryManager->AddHistory(states, userId, EObjectHistoryAction::Remove, session)) {
        return false;
    }
    return true;
}
