#pragma once

#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/history/cache.h>
#include <drive/backend/database/history/db_entities.h>
#include <drive/backend/database/history/manager.h>

#include <drive/backend/tags/tags_filter.h>

#include <drive/telematics/protocol/sensor_filter.h>

#include <rtline/util/types/accessor.h>

class TStateFilter {
    R_FIELD(TString, StateId);
    R_FIELD(TString, Description);
    R_FIELD(i32, Priority, 0);
    R_FIELD(TTagsFilter, Filter);
    R_FIELD(NDrive::TSensorFilter, SensorFilter);

public:
    class TDecoder: public TBaseDecoder {
        R_FIELD(i32, StateId, -1);
        R_FIELD(i32, Description, -1);
        R_FIELD(i32, Priority, -1);
        R_FIELD(i32, Filter, -1);
        R_FIELD(i32, SensorFilter, -1);

    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase);
    };

public:
    TStateFilter() = default;
    TStateFilter(const TString& id)
        : StateId(id)
    {
    }

    bool operator < (const TStateFilter& item) const {
        return std::tie(Priority, StateId) < std::tie(item.Priority, item.StateId);
    }

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);
    bool DeserializeFromJson(const NJson::TJsonValue& jsonInfo);

    NJson::TJsonValue SerializeToReport() const;
    bool DeserializeFromTableRecord(const NStorage::TTableRecord& record, const ITagsHistoryContext* /*context*/);

    NStorage::TTableRecord SerializeToTableRecord() const;
};

class TStateFiltersHistoryManager: public TIndexedAbstractHistoryManager<TStateFilter> {
private:
    using TBase = TIndexedAbstractHistoryManager<TStateFilter>;

public:
    TStateFiltersHistoryManager(const IHistoryContext& context, const THistoryConfig& config)
        : TBase(context, "drive_state_filters_history", config)
    {
    }
};

class TStateFiltersDB: public TDBCacheWithHistoryOwner<TStateFiltersHistoryManager, TStateFilter> {
private:
    using TBase = TDBCacheWithHistoryOwner<TStateFiltersHistoryManager, TStateFilter>;

public:
    using TStates = TMap<TString, TString>;
    using TStatesPtr = TAtomicSharedPtr<TStates>;

    class TObjectStates {
    public:
        TObjectStates(TStatesPtr value = nullptr);

        auto begin() const {
            return Value->cbegin();
        }
        auto end() const {
            return Value->cend();
        }
        bool empty() const {
            return Value->empty();
        }
        template <class T>
        auto find(T&& key) const {
            return Value->find(std::forward<T>(key));
        }
        template <class T>
        TStates::mapped_type& operator [] (T&& key) {
            return (*Value)[std::forward<T>(key)];
        }

    private:
        TStatesPtr Value;
    };

private:
    const TDeviceTagsManager& TagsManager;
    TRWMutex Mutex;
    mutable TVector<TStateFilter> Filters;
    mutable TStatesPtr ObjectStates;
    mutable TInstant StatesRefreshInstant;

private:
    bool RefreshStates() const;
    void RefreshFiltersUnsafe() const;
    TString CalcStatusUnsafe(const TVector<TDBTag>& tags, const NDrive::TSensors& sensors) const;

    virtual bool DoRebuildCacheUnsafe() const override;

    virtual TStringBuf GetEventObjectId(const TObjectEvent<TStateFilter>& ev) const override {
        return ev.GetStateId();
    }

    virtual void AcceptHistoryEventUnsafe(const TObjectEvent<TStateFilter>& ev) const override;

protected:
    bool MetricSignal() override;
    virtual bool Refresh() override;

public:
    TStateFiltersDB(const TDeviceTagsManager& tagsManager, const ITagsHistoryContext& context, const THistoryConfig& config);

    TString CalcStatus(const TVector<TDBTag>& tags, const NDrive::TSensors& sensors) const;

    TSet<TString> GetAvailableStates(const TInstant reqActuality = TInstant::Zero()) const;
    TString GetObjectState(const TString& objectId, const TString& fallback = {}) const;
    TObjectStates GetObjectStates() const;

    bool GetHistory(const TInstant& since, TVector<TAtomicSharedPtr<TObjectEvent<TStateFilter>>>& result, const TInstant reqActuality) const;

    bool Upsert(const TStateFilter& filter, const TString& userId, NDrive::TEntitySession& session) const;
    bool RemoveKeys(const TVector<TString>& ids, const TString& userId, NDrive::TEntitySession& session) const;
};
