#include "cars_filter.h"

#include <drive/backend/areas/areas.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/cars/status/state_filters.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/library/json/cast.h>

#include <util/string/join.h>

#include <algorithm>

bool TBaseAreaTagsFilter::Filter(bool hasGeo) const {
    if (!hasGeo) {
        if (UndefinedPositionPolicy == EUndefinedPositionUsage::AcceptOnlyUndefined || UndefinedPositionPolicy == EUndefinedPositionUsage::AcceptUndefined) {
            return true;
        }
        return false;
    }
    if (UndefinedPositionPolicy == EUndefinedPositionUsage::AcceptOnlyUndefined) {
        return false;
    }
    return true;
}

bool TBaseAreaTagsFilter::Filter(const NDrive::TLocationTags* geoTags) const {
    if (!geoTags) {
        return Filter(false);
    } else {
        return Filter(true) && Filter(*geoTags, AreaTags);
    }
}

bool TBaseAreaTagsFilter::Filter(const NDrive::TLocationTagsArray* geoTags) const {
    if (!geoTags) {
        return Filter(false);
    } else {
        return Filter(true) && Filter(*geoTags, AreaTags);
    }
}

bool TBaseAreaTagsFilter::Filter(const THistoryDeviceSnapshot* hds) const {
    if (!hds) {
        return Filter(false);
    } else {
        return Filter(*hds);
    }
}

bool TBaseAreaTagsFilter::Filter(const THistoryDeviceSnapshot& hds) const {
    return Filter(hds.HasHistoryLocation() ? &hds.GetLocationTagsArray() : nullptr);
}

bool HasLocationTagImpl(const NDrive::TLocationTags& tags, TStringBuf name) {
    return tags.contains(name);
}

bool HasLocationTagImpl(const NDrive::TLocationTagsArray& tags, TStringBuf name) {
    Y_ASSERT(std::is_sorted(tags.begin(), tags.end()));
    return std::binary_search(tags.begin(), tags.end(), name);
}

template <class T>
bool FilterImpl(const T& areaTags, const NDrive::TLocationTags& tagsExpectation) {
    for (auto&& t : tagsExpectation) {
        if (t.StartsWith("!")) {
            if (HasLocationTagImpl(areaTags, t.substr(1))) {
                return false;
            }
        } else {
            if (!HasLocationTagImpl(areaTags, t)) {
                return false;
            }
        }
    }
    return true;
}

bool TBaseAreaTagsFilter::Filter(const NDrive::TLocationTags& areaTags, const NDrive::TLocationTags& tagsExpectation) {
    return FilterImpl(areaTags, tagsExpectation);
}

bool TBaseAreaTagsFilter::Filter(const NDrive::TLocationTagsArray& areaTags, const NDrive::TLocationTags& tagsExpectation) {
    return FilterImpl(areaTags, tagsExpectation);
}

template <class T>
bool FilterWeakImpl(const T& areaTags, const NDrive::TLocationTags& tagsExpectation) {
    bool hasCheckers = false;
    bool checkAllow = false;
    for (auto&& t : tagsExpectation) {
        if (t.StartsWith("!")) {
            if (HasLocationTagImpl(areaTags, t.substr(1))) {
                return false;
            }
        } else if (!checkAllow) {
            hasCheckers = true;
            if (HasLocationTagImpl(areaTags, t)) {
                checkAllow = true;
            }
        }
    }
    return !hasCheckers || checkAllow;
}

bool TBaseAreaTagsFilter::FilterWeak(const NDrive::TLocationTags& areaTags, const NDrive::TLocationTags& tagsExpectation) {
    return FilterWeakImpl(areaTags, tagsExpectation);
}

bool TBaseAreaTagsFilter::FilterWeak(const NDrive::TLocationTagsArray& areaTags, const NDrive::TLocationTags& tagsExpectation) {
    return FilterWeakImpl(areaTags, tagsExpectation);
}

bool TBaseAreaTagsFilter::FilterWeak(const NDrive::TLocationTags* geoTags) const {
    if (geoTags) {
        return Filter(true) && FilterWeak(*geoTags);
    } else {
        return Filter(false);
    }
}

bool TBaseAreaTagsFilter::FilterWeak(const NDrive::TLocationTags& areaTags) const {
    return FilterWeak(areaTags, AreaTags);
}

TBaseAreaTagsFilter TBaseAreaTagsFilter::BuildFromString(const TString& filterDescription) {
    TBaseAreaTagsFilter areaTagsFilter;
    StringSplitter(filterDescription).SplitBySet(",").SkipEmpty().Collect(&areaTagsFilter.MutableAreaTags());
    areaTagsFilter.Prepare();
    return areaTagsFilter;
}

void TCarsFilter::Init(const TYandexConfig::Section* section) {
    TVector<TString> includeModelCodes;
    section->GetDirectives().FillArray("IncludeModelCodes", includeModelCodes);
    IncludeModelCodes = MakeSet(includeModelCodes);
    TVector<TString> excludeModelCodes;
    section->GetDirectives().FillArray("ExcludeModelCodes", excludeModelCodes);
    ExcludeModelCodes = MakeSet(excludeModelCodes);

    TVector<TString> includeCars;
    section->GetDirectives().FillArray("IncludeCars", includeCars);
    IncludeCars = MakeSet(includeCars);
    TVector<TString> excludeCars;
    section->GetDirectives().FillArray("ExcludeCars", excludeCars);
    ExcludeCars = MakeSet(excludeCars);

    TVector<TString> includeCarStates;
    section->GetDirectives().FillArray("IncludeCarStates", includeCarStates);
    IncludeCarStates = MakeSet(includeCarStates);
    TVector<TString> excludeCarStates;
    section->GetDirectives().FillArray("ExcludeCarStates", excludeCarStates);
    ExcludeCarStates = MakeSet(excludeCarStates);

    TVector<TString> includeDynFilters;
    section->GetDirectives().FillArray("IncludeDynamicFilters", includeDynFilters);
    IncludeDynamicFilters = MakeSet(includeDynFilters);
    TVector<TString> excludeDynFilters;
    section->GetDirectives().FillArray("ExcludeDynamicFilters", excludeDynFilters);
    ExcludeDynamicFilters = MakeSet(excludeDynFilters);

    AssertCorrectConfig(section->GetDirectives().Value<TString>("IncludeCarStatus") == "", "IncludeCarStatus deprecated");
    AssertCorrectConfig(section->GetDirectives().Value<TString>("ExcludeCarStatus") == "", "ExcludeCarStatus deprecated");
    AssertCorrectConfig(section->GetDirectives().Value<TString>("IncludeCarTags") == "", "IncludeCarTags deprecated");
    AssertCorrectConfig(section->GetDirectives().Value<TString>("ExcludeCarTags") == "", "ExcludeCarTags deprecated");

    TVector<TString> excludeAreasVector;
    section->GetDirectives().FillArray("ExcludeAreas", excludeAreasVector);
    ExcludeAreas = MakeSet(excludeAreasVector);

    TVector<TString> includeAreasVector;
    section->GetDirectives().FillArray("IncludeAreas", includeAreasVector);
    IncludeAreas = MakeSet(includeAreasVector);

    SkipWithoutLocation = section->GetDirectives().Value("SkipWithoutLocation", SkipWithoutLocation);

    {
        const TString tagsFilter = section->GetDirectives().Value<TString>("TagsFilter");
        if (!!tagsFilter) {
            IncludeTagsFilter = TTagsFilter::BuildFromString(tagsFilter);
        } else {
            const TString includeTagsFilter = section->GetDirectives().Value<TString>("IncludeTagsFilter");
            if (!!includeTagsFilter) {
                IncludeTagsFilter = TTagsFilter::BuildFromString(includeTagsFilter);
            }
        }
    }
    {
        const TString tagsFilter = section->GetDirectives().Value<TString>("ExcludeTagsFilter");
        if (!!tagsFilter) {
            ExcludeTagsFilter = TTagsFilter::BuildFromString(tagsFilter);
        }
    }
    {
        UseLBSForAreaTags = section->GetDirectives().Value("UseLBSForAreaTags", UseLBSForAreaTags);
        const TString tagsFilter = section->GetDirectives().Value<TString>("AreaTagsFilter");
        if (!!tagsFilter) {
            AreaTagsFilter = TAreaTagsFilter::BuildFromString(tagsFilter);
        }
    }
}

void TCarsFilter::ToString(IOutputStream& os) const {
    os << "IncludeModelCodes: " << JoinSeq(",", IncludeModelCodes) << Endl;
    os << "ExcludeModelCodes: " << JoinSeq(",", ExcludeModelCodes) << Endl;
    os << "IncludeCars: " << JoinSeq(",", IncludeCars) << Endl;
    os << "ExcludeCars: " << JoinSeq(",", ExcludeCars) << Endl;
    os << "IncludeCarStates: " << JoinSeq(",", IncludeCarStates) << Endl;
    os << "ExcludeCarStates: " << JoinSeq(",", ExcludeCarStates) << Endl;
    os << "IncludeDynamicFilters: " << JoinSeq(",", IncludeDynamicFilters) << Endl;
    os << "ExcludeDynamicFilters: " << JoinSeq(",", ExcludeDynamicFilters) << Endl;
    os << "ExcludeAreas: " << JoinSeq(",", ExcludeAreas) << Endl;
    os << "IncludeAreas: " << JoinSeq(",", IncludeAreas) << Endl;
    os << "SkipWithoutLocation: " << SkipWithoutLocation << Endl;
    if (!!IncludeTagsFilter) {
        os << "TagsFilter: " << IncludeTagsFilter->ToString() << Endl;
    }
    if (!!ExcludeTagsFilter) {
        os << "ExcludeTagsFilter: " << ExcludeTagsFilter->ToString() << Endl;
    }
    os << "UseLBSForAreaTags: " << UseLBSForAreaTags << Endl;
    os << "AreaTagsFilter: " << AreaTagsFilter.ToString() << Endl;
}

NJson::TJsonValue TCarsFilter::SerializeToJson() const {
    NJson::TJsonValue result = NJson::JSON_MAP;
    TJsonProcessor::WriteContainerArray(result, "include_models", IncludeModelCodes, false);
    TJsonProcessor::WriteContainerArray(result, "exclude_models", ExcludeModelCodes, false);
    TJsonProcessor::WriteContainerString(result, "include_cars", IncludeCars, false);
    TJsonProcessor::WriteContainerString(result, "exclude_cars", ExcludeCars, false);
    TJsonProcessor::WriteContainerArray(result, "include_car_states", IncludeCarStates, false);
    TJsonProcessor::WriteContainerArray(result, "exclude_car_states", ExcludeCarStates, false);
    TJsonProcessor::WriteContainerArray(result, "include_dynamic_group", IncludeDynamicFilters, false);
    TJsonProcessor::WriteContainerArray(result, "exclude_dynamic_group", ExcludeDynamicFilters, false);
    TJsonProcessor::WriteContainerArray(result, "include_areas", IncludeAreas, false);
    TJsonProcessor::WriteContainerArray(result, "exclude_areas", ExcludeAreas, false);
    AreaTagsFilter.SerializeToJson(result);
    TJsonProcessor::Write(result, "use_LBS_for_area_tags", UseLBSForAreaTags);
    TJsonProcessor::Write(result, "skip_without_location", SkipWithoutLocation);
    if (!!IncludeTagsFilter) {
        TJsonProcessor::Write(result, "include_tags_filter", IncludeTagsFilter->ToString());
    }
    if (!!ExcludeTagsFilter) {
        TJsonProcessor::Write(result, "exclude_tags_filter", ExcludeTagsFilter->ToString());
    }

    TJsonProcessor::Write<TString>(result, "user_id", UserId, "");
    TJsonProcessor::Write<TString>(result, "user_actions", ::ToString(UserAction), ::ToString(EUserActivityFeature::Visibility));
    return result;
}

bool TCarsFilter::DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    JREAD_CONTAINER_OPT(jsonInfo, "include_models", IncludeModelCodes);
    JREAD_CONTAINER_OPT(jsonInfo, "exclude_models", ExcludeModelCodes);
    JREAD_CONTAINER_OPT(jsonInfo, "include_cars", IncludeCars);
    JREAD_CONTAINER_OPT(jsonInfo, "exclude_cars", ExcludeCars);
    JREAD_CONTAINER_OPT(jsonInfo, "include_car_states", IncludeCarStates);
    JREAD_CONTAINER_OPT(jsonInfo, "exclude_car_states", ExcludeCarStates);
    JREAD_CONTAINER_OPT(jsonInfo, "include_dynamic_group", IncludeDynamicFilters);
    JREAD_CONTAINER_OPT(jsonInfo, "exclude_dynamic_group", ExcludeDynamicFilters);
    JREAD_CONTAINER_OPT(jsonInfo, "include_areas", IncludeAreas);
    JREAD_CONTAINER_OPT(jsonInfo, "exclude_areas", ExcludeAreas);
    if (!AreaTagsFilter.DeserializeFromJson(jsonInfo)) {
        return false;
    }
    JREAD_BOOL_OPT(jsonInfo, "use_LBS_for_area_tags", UseLBSForAreaTags);
    JREAD_BOOL_OPT(jsonInfo, "skip_without_location", SkipWithoutLocation);
    {
        TString tagsFilterString;
        JREAD_STRING_OPT(jsonInfo, "include_tags_filter", tagsFilterString);
        if (!!tagsFilterString) {
            IncludeTagsFilter = TTagsFilter::BuildFromString(tagsFilterString);
        }
    }
    {
        TString tagsFilterString;
        JREAD_STRING_OPT(jsonInfo, "exclude_tags_filter", tagsFilterString);
        if (!!tagsFilterString) {
            ExcludeTagsFilter = TTagsFilter::BuildFromString(tagsFilterString);
        }
    }
    JREAD_STRING_OPT(jsonInfo, "user_id", UserId);
    JREAD_FROM_STRING_OPT(jsonInfo, "user_actions", UserAction);
    return true;
}

NDrive::TScheme TCarsFilter::GetScheme(const NDrive::IServer& server) {
    NDrive::TScheme result;
    result.Add<TFSString>("include_tags_filter", "Фильтр для используемых объектов", 80000);
    result.Add<TFSString>("exclude_tags_filter", "Фильтр для не используемых объектов", 80000);
    {
        NDrive::TFSVariants::TCompoundVariants compoundVariant;
        auto models = server.GetDriveAPI()->GetModelsData()->GetCached().GetResult();
        ForEach(models.begin(), models.end(), [&compoundVariant](const std::pair<TString, TDriveModelData>& modelPair) -> void {
            compoundVariant.emplace(
                NDrive::TFSVariants::TCompoundVariant{modelPair.first, modelPair.second.GetName()}
            );
        });

        result.Add<TFSVariable>("include_models", "Использовать модели объектов", 80000).MutableCondition().SetMultiSelect(true).SetCompoundVariants(compoundVariant);
        result.Add<TFSVariable>("exclude_models", "Не использовать модели объектов", 80000).MutableCondition().SetMultiSelect(true).SetCompoundVariants(compoundVariant);
    }
    {
        NDrive::TFSVariants::TCompoundVariants compoundVariant;
        auto action = [&compoundVariant](const TArea& area) -> void {
            compoundVariant.emplace(
                NDrive::TFSVariants::TCompoundVariant{area.GetInternalId(), area.GetZoneName() ? area.GetZoneName() : area.GetInternalId()}
            );
        };
        Y_UNUSED(server.GetDriveAPI()->GetAreasDB()->ForObjectsList(action, TInstant::Now()));

        result.Add<TFSVariable>("include_areas", "Использовать объекты в местности (id1, id2, ...)", 80000).MutableCondition().SetMultiSelect(true).SetCompoundVariants(compoundVariant);
        result.Add<TFSVariable>("exclude_areas", "Не использовать объекты в местности (id1, id2, ...)", 80000).MutableCondition().SetMultiSelect(true).SetCompoundVariants(compoundVariant);
        result.Add<TFSBoolean>("skip_without_location", "Не использовать объекты без локации (фильтрация по именам зон)", 80000).SetDefault(true);
    }
    result.Add<TFSString>("include_cars", "Уточнить идентификаторы объектов для использования: (id1, id2, ...)", 80000);
    result.Add<TFSString>("exclude_cars", "Уточнить идентификаторы объектов для игнорирования: (id1, id2, ...)", 80000);
    {
        NDrive::TFSVariants::TCompoundVariants compoundVariant;
        auto action = [&compoundVariant](const TNamedFilter& namedFilter) -> void {
            compoundVariant.emplace(
                NDrive::TFSVariants::TCompoundVariant{namedFilter.GetInternalId(), namedFilter.GetDisplayName(), namedFilter.GetFilter().ToString()}
            );
        };
        Y_UNUSED(Yensured(server.GetDriveAPI()->GetNamedFiltersDB())->ForObjectsList(action, TInstant::Now()));

        result.Add<TFSVariable>("include_dynamic_group", "Применить для динамических групп").MutableCondition().SetMultiSelect(true).SetCompoundVariants(compoundVariant);
        result.Add<TFSVariable>("exclude_dynamic_group", "Не применять для динамических групп").MutableCondition().SetMultiSelect(true).SetCompoundVariants(compoundVariant);
    }
    const TSet<TString> statusIds = (server.GetDriveAPI()->GetStateFiltersDB()) ? server.GetDriveAPI()->GetStateFiltersDB()->GetAvailableStates() : TSet<TString>();
    result.Add<TFSVariants>("include_car_states", "Использовать статусы объектов").SetMultiSelect(true).SetVariants(statusIds);
    result.Add<TFSVariants>("exclude_car_states", "Не использовать статусы объектов").SetMultiSelect(true).SetVariants(statusIds);

    TAreaTagsFilter::AddScheme(result, server, 80000);
    result.Add<TFSBoolean>("use_LBS_for_area_tags", "Использовать LBS координаты в отсутствии GPS (фильтрация по тегам зон)", 80000).SetDefault(false);

    result.Add<TFSString>("user_id", "Идентификатор пользователя", 80000);
    result.Add<TFSVariants>("user_actions", "Возможные действия пользователя пользователя", 80000).InitVariants<EUserActivityFeature>();
    return result;
}

namespace {
    bool MergeFilter(TTagsFilters&& filters, TTagsFilter& result) {
        TString mergeFilterStr;
        for (auto filterIt =  filters.begin(); filterIt != filters.end(); filterIt++) {
            mergeFilterStr += "(" + filterIt->ToString()  + ")";
            if (filterIt + 1 != filters.end()) {
                mergeFilterStr += "*";
            }
        }
        return result.DeserializeFromString(mergeFilterStr);
    }
}

bool TCarsFilter::GetAllowedCarIds(TSet<TString>& allowedCarIds, const NDrive::IServer* server, const TInstant reqActuality) const {
    allowedCarIds.clear();

    bool hasAreasCondition = false;
    if (!ExcludeAreas.empty() || !IncludeAreas.empty()) {
        hasAreasCondition = true;
    }

    TSet<TString> currentAllowedCarIds;
    if (!ExcludeModelCodes.empty() || !IncludeModelCodes.empty() || !IncludeCars.empty() || !ExcludeCars.empty()) {
        const auto& db = server->GetDriveAPI()->GetDatabase();
        auto transaction = db.CreateTransaction(true);
        TVector<TString> conditions;
        if (!ExcludeModelCodes.empty()) {
            conditions.push_back("model_code NOT IN (" + transaction->Quote(ExcludeModelCodes) + ")");
        }
        if (!IncludeModelCodes.empty()) {
            conditions.push_back("model_code IN (" + transaction->Quote(IncludeModelCodes) + ")");
        }
        if (!IncludeCars.empty()) {
            conditions.push_back("id IN (" + transaction->Quote(IncludeCars) + ")");
        }
        if (!ExcludeCars.empty()) {
            conditions.push_back("id NOT IN (" + transaction->Quote(ExcludeCars) + ")");
        }

        TRecordsSet recordsSet;
        auto result = db.GetTable("car")->GetRows(JoinSeq(" AND ", conditions), recordsSet, transaction);
        if (!result || !result->IsSucceed()) {
            ERROR_LOG << "Cannot select filtered rows" << Endl;
            return false;
        }

        for (auto&& i : recordsSet.GetRecords()) {
            currentAllowedCarIds.insert(i.Get("id"));
        }
    } else {
        TCarsDB::TFetchResult allCars = server->GetDriveAPI()->GetCarsData()->FetchInfo(reqActuality);
        for (const auto& car : allCars) {
            currentAllowedCarIds.insert(car.first);
        }
    }

    TSet<TString> allowedCarsIdsByFilters;
    const IDriveTagsManager& manager = server->GetDriveAPI()->GetTagsManager();
    if (!!IncludeTagsFilter || !!ExcludeTagsFilter || !IncludeDynamicFilters.empty() || !ExcludeDynamicFilters.empty()) {
        {
            TTagsFilters includeFilters;
            if (!IncludeDynamicFilters.empty()) {
                includeFilters = server->GetDriveAPI()->GetNamedFiltersDB()->GetdFiltersByIds(IncludeDynamicFilters);
            }

            if (!!IncludeTagsFilter) {
                includeFilters.push_back(*IncludeTagsFilter);
            }

            TTagsFilter includeFilter;
            if (!MergeFilter(std::move(includeFilters), includeFilter)) {
                return false;
            }


            if (!includeFilter.IsEmpty()) {
                if (!manager.GetDeviceTags().GetIdsFromCache(currentAllowedCarIds, includeFilter, allowedCarsIdsByFilters, reqActuality)) {
                    ERROR_LOG << "Cannot restore objects from cache" << Endl;
                    return false;
                }
            } else {
                allowedCarsIdsByFilters = std::move(currentAllowedCarIds);
            }
        }

        {
            TTagsFilters excludeFilters;
            if (!ExcludeDynamicFilters.empty()) {
                excludeFilters = server->GetDriveAPI()->GetNamedFiltersDB()->GetdFiltersByIds(ExcludeDynamicFilters);
            }

            if (!!ExcludeTagsFilter) {
                excludeFilters.push_back(*ExcludeTagsFilter);
            }

            TTagsFilter excludeFilter;
            if (!MergeFilter(std::move(excludeFilters), excludeFilter)) {
                return false;
            }

            if (!excludeFilter.IsEmpty()) {
                TSet<TString> disallowedCarsIdsByFilters;
                if (!manager.GetDeviceTags().GetIdsFromCache(allowedCarsIdsByFilters, excludeFilter, disallowedCarsIdsByFilters, reqActuality)) {
                    ERROR_LOG << "Cannot restore objects from cache" << Endl;
                    return false;
                }

                if (!disallowedCarsIdsByFilters.empty()) {
                    TSet<TString> candidates;
                    std::set_difference(allowedCarsIdsByFilters.begin(), allowedCarsIdsByFilters.end(),
                        disallowedCarsIdsByFilters.begin(), disallowedCarsIdsByFilters.end(),
                        std::inserter(candidates, candidates.begin()));
                    std::swap(allowedCarsIdsByFilters, candidates);
                }
            }
        }

    } else {
        allowedCarsIdsByFilters = std::move(currentAllowedCarIds);
    }

    if (hasAreasCondition) {
        TConstDevicesSnapshot carsAreas = server->GetSnapshotsManager().GetSnapshots();
        auto itCarAreas = carsAreas.begin();
        auto itAllowedCar = allowedCarsIdsByFilters.begin();

        NDrive::TLocation location;
        while (itCarAreas != carsAreas.end() && itAllowedCar != allowedCarsIdsByFilters.end()) {
            if (itCarAreas->first < *itAllowedCar) {
                itCarAreas++;
            } else if (itCarAreas->first > *itAllowedCar || !itCarAreas->second.GetLocation(location)) {
                if (!SkipWithoutLocation) {
                    allowedCarIds.emplace(*itAllowedCar);
                }
                itAllowedCar++;
            } else {
                bool hasIncludeAreas = false;
                bool hasExcludeAreas = false;
                for (const auto& area : itCarAreas->second.GetAreaIds()) {
                    if (IncludeAreas.contains(area)) {
                        hasIncludeAreas = true;
                        if (ExcludeAreas.empty()) {
                            break;
                        }
                    }
                    if (ExcludeAreas.contains(area)) {
                        hasExcludeAreas = true;
                        break;
                    }
                }
                if ((IncludeAreas.empty() || hasIncludeAreas) && !hasExcludeAreas) {
                    allowedCarIds.emplace(*itAllowedCar);
                }
                itAllowedCar++;
                itCarAreas++;
            }
        }
        if (!SkipWithoutLocation) {
            allowedCarIds.insert(itAllowedCar, allowedCarsIdsByFilters.end());
        }
    } else {
        allowedCarIds = allowedCarsIdsByFilters;
    }

    if (!AreaTagsFilter.Empty()) {
        TConstDevicesSnapshot carsAreas = server->GetSnapshotsManager().GetSnapshots(allowedCarIds);
        for (auto it = allowedCarIds.begin(); it != allowedCarIds.end();) {
            const TRTDeviceSnapshot* snapshot = carsAreas.GetSnapshot(*it);
            bool isAllowed = AreaTagsFilter.Filter(snapshot);
            if (UseLBSForAreaTags && snapshot) {
                TInstant rawLocationTimestamp = (snapshot->HasRawLocation() ? snapshot->GetRawLocation()->Timestamp : TInstant::Zero());
                if (snapshot->HasLBSLocation(Now() - rawLocationTimestamp)) {
                    isAllowed = AreaTagsFilter.Filter(&snapshot->GetLBSLocationTags());
                }
            }
            if (!isAllowed) {
                it = allowedCarIds.erase(it);
            } else {
                ++it;
            }
        }
    }

    if (UserId) {
        TUserPermissions::TPtr permissions = server->GetDriveAPI()->GetUserPermissions(UserId, TUserPermissionsFeatures());
        if (!permissions) {
            ERROR_LOG << "Incorrect user id: " << UserId << Endl;
            return false;
        }
        TVector<TTaggedDevice> objects;
        if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetCustomObjectsFromCache(objects, allowedCarIds)) {
            return false;
        }
        TSet<TString> result;
        for (auto&& i : objects) {
            TUserPermissions::TUnvisibilityInfoSet info;
            if (permissions->GetVisibility(i, NEntityTagsManager::EEntityType::Car, &info) == TUserPermissions::EVisibility::NoVisible) {
                if (UserAction == EUserActivityFeature::VisibilityPotentially && (info & TUserPermissions::EUnvisibilityInfo::Busy)) {
                    result.emplace(i.GetId());
                }
            } else {
                result.emplace(i.GetId());
            }
        }
        allowedCarIds = std::move(result);
    }

    if (server->GetDriveAPI()->GetStateFiltersDB() && (IncludeCarStates || ExcludeCarStates)) {
        auto carStates = server->GetDriveAPI()->GetStateFiltersDB()->GetObjectStates();

        TSet<TString> filteredCarIds;
        for (auto&& carId: allowedCarIds) {
            const TString& carState = carStates[carId];
            bool filtered = true;
            if (!IncludeCarStates.empty()) {
                filtered &= IncludeCarStates.contains(carState);
            }
            if (!ExcludeCarStates.empty()) {
                filtered &= !ExcludeCarStates.contains(carState);
            }
            if (filtered) {
                filteredCarIds.insert(carId);
            }
        }

        allowedCarIds = std::move(filteredCarIds);
    }

    DEBUG_LOG << "Allowed cars: " << allowedCarIds.size() << Endl;
    return true;
}

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& json, TCarsFilter& carsFilter) {
    return carsFilter.DeserializeFromJson(json);
}

template<>
NJson::TJsonValue NJson::ToJson(const TCarsFilter& carsFilter) {
    return carsFilter.SerializeToJson();
}
