#pragma once

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/areas/location.h>
#include <drive/backend/database/drive/named_filters.h>
#include <drive/backend/tags/tags_filter.h>

#include <library/cpp/yconf/conf.h>

#include <rtline/util/types/accessor.h>

#include <util/datetime/base.h>

class THistoryDeviceSnapshot;

class TAreaTagsFilterDefaultNamesConstructor {
public:
    static TString GetFieldNameTags() {
        return "area_tags";
    }
    static TString GetFieldNameUndefinedPolicy() {
        return "undefined_position_policy";
    }
    static TString GetObjectType() {
        return "машина";
    }
};

class TBaseAreaTagsFilter {
public:
    enum class EUndefinedPositionUsage {
        AcceptUndefined,
        AcceptOnlyUndefined,
        DenyUndefined,
    };

private:
    R_FIELD(NDrive::TLocationTags, AreaTags);
    R_FIELD(NDrive::TLocationTags, CleanAreaTags);
    R_FIELD(EUndefinedPositionUsage, UndefinedPositionPolicy, EUndefinedPositionUsage::AcceptUndefined);

private:
    bool Filter(bool hasGeo) const;

protected:
    void Prepare() {
        for (auto&& i : AreaTags) {
            if (i.StartsWith("!")) {
                CleanAreaTags.emplace(i.substr(1));
            } else {
                CleanAreaTags.emplace(i);
            }
        }
    }
public:
    static TBaseAreaTagsFilter BuildFromString(const TString& filterDescription);
    static bool Filter(const NDrive::TLocationTags& areaTags, const NDrive::TLocationTags& tagsExpectation);
    static bool Filter(const NDrive::TLocationTagsArray& areaTags, const NDrive::TLocationTags& tagsExpectation);
    static bool FilterWeak(const NDrive::TLocationTags& areaTags, const NDrive::TLocationTags& tagsExpectation);
    static bool FilterWeak(const NDrive::TLocationTagsArray& areaTags, const NDrive::TLocationTags& tagsExpectation);

public:
    bool Empty() const {
        return AreaTags.empty() && UndefinedPositionPolicy == EUndefinedPositionUsage::AcceptUndefined;
    }

    TString ToString() const {
        return JoinSeq(",", AreaTags);
    }

    bool Filter(const NDrive::TLocationTags* geoTags) const;
    bool Filter(const NDrive::TLocationTagsArray* geoTags) const;
    bool Filter(const THistoryDeviceSnapshot* hds) const;
    bool Filter(const THistoryDeviceSnapshot& hds) const;

    bool FilterWeak(const NDrive::TLocationTags* geoTags) const;
    bool FilterWeak(const NDrive::TLocationTags& areaTags) const;
};

template <class TFieldNamesConstructor = TAreaTagsFilterDefaultNamesConstructor>
class TAreaTagsFilterImpl: public TBaseAreaTagsFilter {
public:
    TAreaTagsFilterImpl() = default;
    TAreaTagsFilterImpl(const TBaseAreaTagsFilter& obj)
        : TBaseAreaTagsFilter(obj)
    {
    }
    TAreaTagsFilterImpl(TBaseAreaTagsFilter&& obj)
        : TBaseAreaTagsFilter(std::move(obj))
    {
    }

    static NDrive::TScheme GetScheme(const NDrive::IServer& server) {
        NDrive::TScheme result;
        AddScheme(result, server);
        return result;
    }

    static void AddScheme(NDrive::TScheme& result, const NDrive::IServer& server, const ui32 priority = 0) {
        result.Add<TFSVariants>(TFieldNamesConstructor::GetFieldNameTags(), "Геотеги фильтрации (" + TFieldNamesConstructor::GetObjectType() + ")", priority).SetVariants(server.GetFullAreaTagsList()).MulVariantsLeft({"", "!"}).SetMultiSelect(true);
        result.Add<TFSVariants>(TFieldNamesConstructor::GetFieldNameUndefinedPolicy(), "Политика при неизвестной позиции (" + TFieldNamesConstructor::GetObjectType() + ") (фильтрация по геотегам)", priority).template InitVariants<EUndefinedPositionUsage>().SetMultiSelect(false);
    }

    bool DeserializeFromJson(const NJson::TJsonValue& info) {
        if (!TJsonProcessor::ReadContainer(info, TFieldNamesConstructor::GetFieldNameTags(), MutableAreaTags())) {
            return false;
        }
        JREAD_FROM_STRING_OPT(info, TFieldNamesConstructor::GetFieldNameUndefinedPolicy(), MutableUndefinedPositionPolicy());
        Prepare();
        return true;
    }

    void SerializeToJson(NJson::TJsonValue& result) const {
        TJsonProcessor::WriteContainerArray(result, TFieldNamesConstructor::GetFieldNameTags(), GetAreaTags());
        JWRITE_ENUM(result, TFieldNamesConstructor::GetFieldNameUndefinedPolicy(), GetUndefinedPositionPolicy());
    }

    NJson::TJsonValue SerializeToJson() const {
        NJson::TJsonValue result = NJson::JSON_MAP;
        SerializeToJson(result);
        return result;
    }
};

using TAreaTagsFilter = TAreaTagsFilterImpl<TAreaTagsFilterDefaultNamesConstructor>;

class TCarsFilter {
public:
    enum class EUserActivityFeature {
        Visibility /* "visibility" */,
        VisibilityPotentially /* "visibility_potentially" */
    };
public:
    R_READONLY(TSet<TString>, IncludeModelCodes);
    R_READONLY(TSet<TString>, ExcludeModelCodes);
    R_READONLY(TSet<TString>, IncludeCars);
    R_READONLY(TSet<TString>, ExcludeCars);
    R_READONLY(TSet<TString>, IncludeCarStates);
    R_READONLY(TSet<TString>, ExcludeCarStates);

    R_FIELD(TSet<TNamedFilter::TId>, IncludeDynamicFilters);
    R_FIELD(TSet<TNamedFilter::TId>, ExcludeDynamicFilters);

    R_READONLY(bool, SkipWithoutLocation, true);
    R_READONLY(TSet<TString>, ExcludeAreas);
    R_READONLY(TSet<TString>, IncludeAreas);

    R_READONLY(bool, UseLBSForAreaTags, false);
    R_READONLY(TAreaTagsFilter, AreaTagsFilter);

    R_OPTIONAL(TTagsFilter, IncludeTagsFilter);
    R_OPTIONAL(TTagsFilter, ExcludeTagsFilter);

    R_READONLY(TString, UserId);
    R_READONLY(EUserActivityFeature, UserAction, EUserActivityFeature::Visibility);

public:
    void Init(const TYandexConfig::Section* section);
    void ToString(IOutputStream& os) const;

    NJson::TJsonValue SerializeToJson() const;
    bool DeserializeFromJson(const NJson::TJsonValue& jsonInfo);
    static NDrive::TScheme GetScheme(const NDrive::IServer& /*server*/);

    bool GetAllowedCarIds(TSet<TString>& allowedCarIds, const NDrive::IServer* server, const TDuration maxAge) const {
        return GetAllowedCarIds(allowedCarIds, server, Now() - maxAge);
    }

    bool GetAllowedCarIds(TSet<TString>& allowedCarIds, const NDrive::IServer* server, const TInstant reqActuality = Now()) const;
};
