#include "chats.h"

bool NDrive::NChat::TChat::DeserializeFromJson(const NJson::TJsonValue& info) {
    JREAD_STRING(info, "search_id", SearchId);
    JREAD_STRING(info, "title", Title);
    JREAD_STRING_OPT(info, "handler_chat_id", HandlerChatId);
    JREAD_STRING_OPT(info, "topic", Topic);
    JREAD_STRING_OPT(info, "object_id", ObjectId);
    JREAD_BOOL_OPT(info, "is_archive", IsArchive);
    JREAD_UINT_OPT(info, "flags", Flags);
    return true;
}

NJson::TJsonValue NDrive::NChat::TChat::SerializeToJson() const {
    NJson::TJsonValue result(NJson::JSON_MAP);
    JWRITE(result, "search_id", SearchId);
    JWRITE(result, "title", Title);
    JWRITE(result, "handler_chat_id", HandlerChatId);
    JWRITE(result, "topic", Topic);
    JWRITE(result, "object_id", ObjectId);
    JWRITE(result, "is_archive", IsArchive);
    JWRITE(result, "flags", Flags);
    return result;
}

bool NDrive::NChat::TChat::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, Id);
    READ_DECODER_VALUE(decoder, values, SearchId);
    READ_DECODER_VALUE(decoder, values, Title);
    READ_DECODER_VALUE(decoder, values, HandlerChatId);
    READ_DECODER_VALUE(decoder, values, Topic);
    READ_DECODER_VALUE(decoder, values, ObjectId);
    READ_DECODER_VALUE_DEF(decoder, values, IsArchive, false);
    READ_DECODER_VALUE_DEF(decoder, values, Flags, 0);
    IdString = ToString(Id);
    return true;
}

bool NDrive::NChat::TChat::DeserializeFromTableRecord(const NStorage::TTableRecord& record, const IHistoryContext* /*hContext*/) {
    SearchId = record.Get("search_id");
    Title = record.Get("title");
    if (!record.TryGet("chat_id", Id)) {
        return false;
    }
    IdString = ToString(Id);
    HandlerChatId = record.Get("handler_chat_id");
    Topic = record.Get("topic");
    ObjectId = record.Get("object_id");
    if (record.Get("is_archive") == "1") {
        IsArchive = true;
    } else {
        IsArchive = false;
    }
    if (record.Get("flags") && !TryFromString(record.Get("flags"), Flags)) {
        return false;
    }
    return true;
}

bool NDrive::NChat::TDBChatEntities::Remove(const TVector<TString>& ids, const TString& userId, NDrive::TEntitySession& session) const {
    if (ids.empty())
        return true;
    NStorage::ITransaction::TPtr transaction = session.GetTransaction();
    NStorage::ITableAccessor::TPtr table = Database->GetTable(GetTableName());
    const TString condition = "search_id IN (" + session->Quote(ids) + ")";
    TRecordsSet records;
    if (!table->RemoveRow(condition, transaction, &records)) {
        session.SetErrorInfo("remove_chats", "RemoveRow failed", EDriveSessionResult::TransactionProblem);
        return false;
    }

    TVector<NDrive::NChat::TChat> chats;
    for (auto&& i : records) {
        TChat chat;
        if (chat.DeserializeFromTableRecord(i, nullptr)) {
            chats.emplace_back(chat);
        }
    }

    if (!HistoryWriter.AddHistory(chats, userId, EObjectHistoryAction::Remove, session)) {
        return false;
    }
    return true;
}

bool NDrive::NChat::TDBChatEntities::Upsert(const TChat& object, const TString& userId, NDrive::TEntitySession& session, TVector<TChat>* chatsPtr) const {
    NStorage::ITransaction::TPtr transaction = session.GetTransaction();
    NStorage::ITableAccessor::TPtr table = Database->GetTable(GetTableName());
    const NStorage::TTableRecord record = object.SerializeToTableRecord();
    const NStorage::TTableRecord unique = object.SerializeUniqueToTableRecord();
    bool isUpdate;
    TRecordsSet records;
    if (!table->Upsert(record, transaction, unique, &isUpdate, &records)) {
        session.SetErrorInfo("upsert_chat", "Upsert failed", EDriveSessionResult::TransactionProblem);
        return false;
    }
    TVector<TChat> chats;
    for (auto&& i : records) {
        TChat chat;
        if (chat.DeserializeFromTableRecord(i, nullptr)) {
            chats.emplace_back(chat);
        }
    }
    if (chatsPtr != nullptr) {
        *chatsPtr = chats;
    }
    if (!HistoryWriter.AddHistory(chats, userId, isUpdate ? EObjectHistoryAction::UpdateData : EObjectHistoryAction::Add, session)) {
        return false;
    }
    return true;
}

bool NDrive::NChat::TChatsMetaManager::Remove(const TVector<TString>& ids, const TString& userId, NDrive::TEntitySession& session) const {
    return ChatsTable.Remove(ids, userId, session);
}

bool NDrive::NChat::TChatsMetaManager::Upsert(const TChat& object, const TString& userId, NDrive::TEntitySession& session, TVector<TChat>* chatsPtr) const {
    return ChatsTable.Upsert(object, userId, session, chatsPtr);
}

bool NDrive::NChat::TChatsMetaManager::GetChatFromCache(const TString& searchId, TChat& result, bool fallback, NStorage::ITransaction::TPtr transactionExt) const {
    {
        auto rg = MakeObjectReadGuard();
        auto chatIt = ChatsBySearchId.find(searchId);
        if (chatIt != ChatsBySearchId.end()) {
            result = *(chatIt->second);
            return true;
        }
    }
    if (fallback) {
        return GetChatFromTable(searchId, result, transactionExt);
    }
    return false;
}

bool NDrive::NChat::TChatsMetaManager::GetChatFromTable(const TString& searchId, TChat& result, NStorage::ITransaction::TPtr transactionExt) const {
    auto chatFetchResult = transactionExt ? ChatsTable.FetchInfo(searchId, transactionExt) : ChatsTable.FetchInfo(searchId);
    auto chatPtr = chatFetchResult.GetResultPtr(searchId);
    if (chatPtr) {
        result = *chatPtr;
        return true;
    }
    return false;
}

TMaybe<TVector<NDrive::NChat::TChat>> NDrive::NChat::TChatsMetaManager::GetChatsFromTable(const TString& chatId, const TString& topic, NDrive::TEntitySession& session) const {
    if (chatId.empty() && topic.empty()) {
        return TVector<NDrive::NChat::TChat>();
    }
    NSQL::TQueryOptions queryOptions;
    if (chatId) {
        queryOptions.AddGenericCondition("handler_chat_id", chatId);
    }
    if (topic) {
        queryOptions.AddGenericCondition("topic", topic);
    }
    auto chatFetchResult = ChatsTable.FetchInfo(session, queryOptions);
    if (!chatFetchResult) {
        return Nothing();
    }
    TVector<NDrive::NChat::TChat> result;
    for (auto&& chat : chatFetchResult) {
        result.emplace_back(chat.second);
    }
    return result;
}

TMaybe<TVector<NDrive::NChat::TChat>> NDrive::NChat::TChatsMetaManager::GetChatsFromTable(const TSet<TString>& searchIds, NDrive::TEntitySession& session) const {
    if (searchIds.empty()) {
        return TVector<NDrive::NChat::TChat>();
    }
    auto chatFetchResult = ChatsTable.FetchInfo(searchIds, session);
    if (!chatFetchResult) {
        return Nothing();
    }
    TVector<NDrive::NChat::TChat> result;
    for (auto&& chat : chatFetchResult) {
        result.emplace_back(chat.second);
    }
    return result;
}

TMaybe<TVector<NDrive::NChat::TChat>> NDrive::NChat::TChatsMetaManager::GetChatsFromTable(const TSet<ui32>& chatIds, NDrive::TEntitySession& session) const {
    if (chatIds.empty()) {
        return TVector<NDrive::NChat::TChat>();
    }
    NSQL::TQueryOptions queryOptions;
    queryOptions.SetGenericCondition("chat_id", chatIds);
    auto chatFetchResult = ChatsTable.FetchInfo(session, queryOptions);
    if (!chatFetchResult) {
        return Nothing();
    }
    TVector<NDrive::NChat::TChat> result;
    for (auto&& chat : chatFetchResult) {
        result.emplace_back(chat.second);
    }
    return result;
}

TVector<NDrive::NChat::TChat> NDrive::NChat::TChatsMetaManager::GetCachedUserChats(const TString& userId, const TString& handlerId) const {
    auto rg = MakeObjectReadGuard();

    TVector<TChat> result;
    auto chatsIt = ChatsByOwner.find(userId);
    if (chatsIt == ChatsByOwner.end()) {
        return result;
    }

    for (auto&& chat : chatsIt->second) {
        if (chat->GetHandlerChatId() == handlerId) {
            result.emplace_back(*chat);
        }
    }

    return result;
}

bool NDrive::NChat::TChatsMetaManager::DoRebuildCacheUnsafe() const {
    NStorage::TObjectRecordsSet<TChat> objects;

    auto table = Database->GetTable("chats");
    {
        auto transaction = Database->CreateTransaction(true);
        auto result = table->GetRows("", objects, transaction);
        if (!result->IsSucceed()) {
            ERROR_LOG << "Can't build snapshot for chats" << Endl;
            return false;
        }
    }

    for (auto&& i : objects) {
        TAtomicSharedPtr<TChat> newChat;
        newChat.Reset(new TChat(i));
        ChatsBySearchId.emplace(i.GetSearchId(), newChat);
        ChatsByOwner[i.GetObjectId()].push_back(newChat);
    }

    return true;
}

bool NDrive::NChat::TChatsMetaManager::DoAcceptHistoryEventUnsafe(const TAtomicSharedPtr<TObjectEvent<TChat>>& ev, const bool isNew) {
    if (!isNew) {
        return true;
    }
    auto chatId = ev->GetId();
    auto searchId = ev->GetSearchId();
    if (ev->GetHistoryAction() == EObjectHistoryAction::Add) {
        ChatsBySearchId.emplace(searchId, ev);
        ChatsByOwner[ev->GetChatObjectId()].emplace_back(ev);
    } else {
        auto chatsIt = ChatsByOwner.find(ev->GetChatObjectId());
        if (ev->GetHistoryAction() == EObjectHistoryAction::Remove) {
            auto it = ChatsBySearchId.find(searchId);
            if (it != ChatsBySearchId.end()) {
                ChatsBySearchId.erase(it);
            }
            if (chatsIt != ChatsByOwner.end()) {
                chatsIt->second.resize(std::remove_if(chatsIt->second.begin(), chatsIt->second.end(), [chatId](const TAtomicSharedPtr<TChat>& c) {return c->GetId() == chatId; }) - chatsIt->second.begin());
            }
        } else if (ev->GetHistoryAction() == EObjectHistoryAction::UpdateData) {
            ChatsBySearchId[searchId] = ev;
            bool chatFound = false;
            if (chatsIt != ChatsByOwner.end()) {
                for (size_t i = 0; i < chatsIt->second.size(); ++i) {
                    if (chatsIt->second[i]->GetId() == chatId) {
                        chatsIt->second[i] = ev;
                        chatFound = true;
                        break;
                    }
                }
            }
            if (!chatFound) {
                ChatsByOwner[ev->GetChatObjectId()].emplace_back(ev);
            }
        }
    }
    return true;
}

TString NDrive::NChat::TChatsMetaManager::GetChatOwnerById(const ui32 chatId) const {
    auto result = GetChatById(chatId);
    if (!result) {
        return "";
    }
    return result->GetObjectId();
}

TMaybe<NDrive::NChat::TChat> NDrive::NChat::TChatsMetaManager::GetChatById(const ui32 chatId, NStorage::ITransaction::TPtr transactionExt) const {
    auto id = ToString(chatId);
    auto fetchResult = ChatsTable.FetchInfoByField(id, "chat_id", transactionExt);
    if (fetchResult.size() != 1) {
        return Nothing();
    }
    return fetchResult.MutableResult().begin()->second;
}

TStringBuf NDrive::NChat::TChatsMetaManager::GetEventObjectId(const TObjectEvent<TChat>& object) const {
    return object.GetSearchId();
}
