#pragma once

#include "message.h"

#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/history/cache.h>
#include <drive/backend/database/history/manager.h>

namespace NDrive::NChat {
    class TChat {
        R_FIELD(ui32, Id, Max<i32>());
        R_READONLY(TString, IdString);
        R_FIELD(TString, SearchId);
        R_FIELD(TString, Title);
        R_FIELD(TString, HandlerChatId);
        R_FIELD(TString, Topic);
        R_FIELD(TString, ObjectId);
        R_FIELD(bool, IsArchive, false);
        R_FIELD(TString, Icon);
        R_FIELD(ui32, Flags, 0);

        // Landing stuff
        R_FIELD(TString, Preview);
        R_FIELD(bool, IsMenuLanding, false);
        R_FIELD(TInstant, CreatedAt);

    public:
        using TChatFlags = ui32;
        using TId = ui32;
        enum class EChatFlags: TChatFlags {
            Muted = 1 << 0 /* "muted" */,
            Hidden = 1 << 1 /* "hidden" */,
        };
        static const TChatFlags HideMessagesFlags = (TChatFlags)EChatFlags::Muted | (TChatFlags)EChatFlags::Hidden;

        virtual ~TChat() = default;
    public:

        class TDecoder: public TBaseDecoder {
            R_FIELD(i32, Id, -1);
            R_FIELD(i32, SearchId, -1);
            R_FIELD(i32, Title, -1);
            R_FIELD(i32, HandlerChatId, -1);
            R_FIELD(i32, Topic, -1);
            R_FIELD(i32, ObjectId, -1);
            R_FIELD(i32, IsArchive, -1);
            R_FIELD(i32, Flags, -1);

        public:
            TDecoder() = default;

            TDecoder(const TMap<TString, ui32>& decoderBase) {
                Id = GetFieldDecodeIndex("chat_id", decoderBase);
                SearchId = GetFieldDecodeIndex("search_id", decoderBase);
                Title = GetFieldDecodeIndex("title", decoderBase);
                HandlerChatId = GetFieldDecodeIndex("handler_chat_id", decoderBase);
                Topic = GetFieldDecodeIndex("topic", decoderBase);
                ObjectId = GetFieldDecodeIndex("object_id", decoderBase);
                IsArchive = GetFieldDecodeIndex("is_archive", decoderBase);
                Flags = GetFieldDecodeIndex("flags", decoderBase);
            }
        };

        TString GetChatObjectId() const {
            return ObjectId;
        }

        bool DeserializeFromJson(const NJson::TJsonValue& info);

        NJson::TJsonValue SerializeToJson() const;

        bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);

        bool Parse(const NStorage::TTableRecord& record) {
            return DeserializeFromTableRecord(record, nullptr);
        }

        bool DeserializeFromTableRecord(const NStorage::TTableRecord& record, const IHistoryContext* /*hContext*/);

        NStorage::TTableRecord SerializeToTableRecord() const {
            NStorage::TTableRecord result;
            if (Id != Max<i32>()) {
                result.Set("chat_id", Id);
            }
            result.Set("search_id", SearchId).Set("title", Title).Set("handler_chat_id", HandlerChatId).Set("topic", Topic).Set("object_id", ObjectId).Set("is_archive", IsArchive).Set("flags", Flags);
            return result;
        }

        NStorage::TTableRecord SerializeUniqueToTableRecord() const {
            NStorage::TTableRecord result;
            result.Set("search_id", SearchId);
            return result;
        }

        TChat& SetFlag(const EChatFlags flag) {
            Flags |= (TChatFlags)flag;
            return *this;
        }

        TChat& DropFlag(const EChatFlags flag) {
            Flags -= Flags & (TChatFlags)flag;
            return *this;
        }

        bool GetAtLeastOneFlag(const TChatFlags requestedFlags) const {
            return Flags & requestedFlags;
        }

        bool GetFlag(const EChatFlags flag) const {
            return GetAtLeastOneFlag((TChatFlags)flag);
        }

        NJson::TJsonValue GetFlagsReport() const {
            NJson::TJsonValue result = NJson::JSON_ARRAY;
            for (auto&& f : GetEnumAllValues<EChatFlags>()) {
                if (Flags & (TChatFlags)f) {
                    result.AppendValue(ToString(f));
                }
            }
            return result;
        }
    };

    class TChatHistoryWriter: public TDatabaseHistoryManager<TChat> {
        using TBase = TDatabaseHistoryManager<TChat>;

    public:
        TChatHistoryWriter(const IHistoryContext& context)
            : TBase(context, "chats_history")
        {
        }
    };

    class TDBChatEntities: public TDBEntities<TChat> {
        using TBase = TDBEntities<TChat>;

    private:
        TChatHistoryWriter HistoryWriter;

    protected:
        virtual TString GetColumnName() const override {
            return "search_id";
        }

        virtual TString GetTableName() const override {
            return "chats";
        }

        virtual TString GetMainId(const TChat& e) const override {
            return e.GetSearchId();
        }
    public:
        TDBChatEntities(const IHistoryContext& context)
            : TBase(context.GetDatabase())
            , HistoryWriter(context)
        {
        }

        bool Remove(const TVector<TString>& ids, const TString& userId, NDrive::TEntitySession& session) const;
        bool Upsert(const TChat& object, const TString& userId, NDrive::TEntitySession& session, TVector<TChat>* chatsPtr = nullptr) const;
    };

    class TChatsMetaManager : public TAutoActualizingSnapshot<TChat> {
        using TBase = TAutoActualizingSnapshot<TChat>;
        using THistoryReader = TAtomicSharedPtr<TCallbackSequentialTableImpl<TObjectEvent<TChat>, TString>>;
        using TDBTable = TDBChatEntities;

    private:
        NStorage::IDatabase::TPtr Database;
        TDBTable ChatsTable;
        mutable TMap<TString, TAtomicSharedPtr<TChat>> ChatsBySearchId;
        mutable TMap<TString, TVector<TAtomicSharedPtr<TChat>>> ChatsByOwner;

        virtual bool DoRebuildCacheUnsafe() const override;
        virtual bool DoAcceptHistoryEventUnsafe(const TAtomicSharedPtr<TObjectEvent<TChat>>& ev, const bool isNew) override;
        virtual TStringBuf GetEventObjectId(const TObjectEvent<TChat>& object) const override;

    public:
        TChatsMetaManager(const THistoryReader& reader, NStorage::IDatabase::TPtr db)
            : TBase("chat_meta", reader)
            , Database(db)
            , ChatsTable(THistoryContext(db))
        {
        }

        bool Remove(const TVector<TString>& ids, const TString& userId, NDrive::TEntitySession& session) const;

        bool Upsert(const TChat& object, const TString& userId, NDrive::TEntitySession& session, TVector<TChat>* chatsPtr = nullptr) const;

        bool GetChatFromCache(const TString& searchId, TChat& result, bool fallback = false, NStorage::ITransaction::TPtr transactionExt = nullptr) const;
        bool GetChatFromTable(const TString& searchId, TChat& result, NStorage::ITransaction::TPtr transactionExt = nullptr) const;
        TMaybe<TVector<TChat>> GetChatsFromTable(const TSet<TString>& searchIds, NDrive::TEntitySession& session) const;
        TMaybe<TVector<TChat>> GetChatsFromTable(const TString& chatId, const TString& topic, NDrive::TEntitySession& session) const;
        TMaybe<TVector<TChat>> GetChatsFromTable(const TSet<ui32>& chatIds, NDrive::TEntitySession& session) const;

        TMaybe<TChat> GetChatById(const ui32 chatId, NStorage::ITransaction::TPtr transactionExt = nullptr) const;
        TString GetChatOwnerById(const ui32 chatId) const;

        TVector<TChat> GetCachedUserChats(const TString& userId, const TString& handlerId) const;
    };
}
