#include "engine.h"

#include <rtline/library/storage/structured.h>

void NDrive::NChat::TEngineConfig::Init(const TYandexConfig::Section* section) {
    DBName = section->GetDirectives().Value("DBName", DBName);
    auto children = section->GetAllChildren();
    {
        auto it = children.find("ChatMetaManager");
        if (it != children.end()) {
            ChatMetaManagerConfig.Init(it->second);
        }
    }
    {
        auto it = children.find("MessagesHistoryConfig");
        if (it != children.end()) {
            MessagesHistoryConfig.Init(it->second);
        }
    }
}

void NDrive::NChat::TEngineConfig::ToString(IOutputStream& os) const {
    os << "DBName: " << DBName << Endl;
    os << "<ChatMetaManager>" << Endl;
    ChatMetaManagerConfig.ToString(os);
    os << "</ChatMetaManager>" << Endl;
    os << "<MessagesHistoryConfig>" << Endl;
    MessagesHistoryConfig.ToString(os);
    os << "</MessagesHistoryConfig>" << Endl;
}

bool NDrive::NChat::TEngine::GetLastMessage(const ui32 chat, const ui32 viewerTraits, NDrive::TEntitySession& session, TMaybe<NDrive::NChat::TMessageEvent>& result, const TString& userId) const {
    return HistoryManager.GetLastMessage(chat, viewerTraits, session, result, userId);
}

NDrive::NChat::TOptionalMessageEvents NDrive::NChat::TEngine::GetMessagesRange(const ui32 chat, const ui32 viewerTraits, NDrive::TEntitySession& session, const ui64 fromId, const ui64 toId) const {
    return HistoryManager.GetMessagesFromDb(NDrive::NChat::TChatMessagesHistoryManager::TMessagesRequestContext(chat), { fromId, toId }, {}, viewerTraits, session);
}

NDrive::NChat::TOptionalMessageEvents NDrive::NChat::TEngine::GetMessagesSinceId(const ui32 chat, const ui32 viewerTraits, const ui64 fromId, NDrive::TEntitySession& session) const {
    return HistoryManager.GetMessagesFromDb(NDrive::NChat::TChatMessagesHistoryManager::TMessagesRequestContext(chat), { fromId }, {}, viewerTraits, session);
}

NDrive::NChat::TOptionalMessageEvents NDrive::NChat::TEngine::GetMessagesSinceTimestamp(const ui32 chat, const ui32 viewerTraits, const TInstant fromTs, NDrive::TEntitySession& session) const {
    return HistoryManager.GetMessagesFromDb(NDrive::NChat::TChatMessagesHistoryManager::TMessagesRequestContext(chat), {}, { fromTs }, viewerTraits, session);
}

TMaybe<size_t> NDrive::NChat::TEngine::GetMessagesCount(const ui32 chat, const ui32 viewerTraits, NDrive::TEntitySession& session, const TSet<TMessage::EMessageType>& excludeTypes, const TMaybe<TSet<TMessage::EMessageType>> includeTypes) const {
    return GetMessagesCountSince(chat, viewerTraits, 0, session, excludeTypes, includeTypes);
}

TMaybe<size_t> NDrive::NChat::TEngine::GetMessagesCountSince(const ui32 chat, const ui32 viewerTraits, const ui64 messageId, NDrive::TEntitySession& session, const TSet<TMessage::EMessageType>& excludeTypes, const TMaybe<TSet<TMessage::EMessageType>> includeTypes) const {
    return HistoryManager.GetMessagesCountSince(chat, viewerTraits, messageId, session, excludeTypes, includeTypes);
}

bool NDrive::NChat::TEngine::AddMessage(const TMessage& message, const TString& userId, NDrive::TEntitySession& session, const TString& chatSearchId) const {
    TRecordsSet addedRecords;
    if (!HistoryManager.AddHistory(message, userId, EObjectHistoryAction::Add, session, &addedRecords)) {
        return false;
    }
    if (chatSearchId) {
        MessagesCache.Invalidate(chatSearchId);
    }
    return true;
}

bool NDrive::NChat::TEngine::AddMessageIfLast(const TMessage& message, const TString& userId, NDrive::TEntitySession& session, const TString& chatSearchId, const ui64 lastMessageId) const {
    NSQL::TQueryOptions queryOptions;
    TSet<ui64> chatIdSet = { message.GetChatId() };
    queryOptions.SetGenericCondition("chat_id", chatIdSet);
    auto messages = HistoryManager.GetEvents({ lastMessageId + 1 }, {}, session, queryOptions);
    if (!messages) {
        return false;
    }
    if (messages->size() > 0) {
        session.SetErrorInfo("AddMessageIfLast", "chat " + ToString(message.GetChatId()) + " is outdated, last message id " + ToString(lastMessageId));
        return false;
    }
    return AddMessage(message, userId, session, chatSearchId);
}

bool NDrive::NChat::TEngine::EditMessage(const ui64 messageId, TMessage message, const TString& userId, NDrive::TEntitySession& session) const {
    TRecordsSet addedRecords;
    message.SetOperatedId(messageId);
    if (!HistoryManager.AddHistory(message, userId, EObjectHistoryAction::UpdateData, session, &addedRecords)) {
        return false;
    }
    return true;
}

bool NDrive::NChat::TEngine::DeleteMessage(const ui64 messageId, const ui32 chatRoomId, const TString& userId, NDrive::TEntitySession& session) const {
    TRecordsSet addedRecords;
    TMessage mockMessage;
    mockMessage.SetOperatedId(messageId).SetChatId(chatRoomId);
    if (!HistoryManager.AddHistory(mockMessage, userId, EObjectHistoryAction::Remove, session, &addedRecords)) {
        return false;
    }
    return true;
}

TMaybe<TObjectEventPtr<NDrive::NChat::TMessage>> NDrive::NChat::TEngine::GetMessage(const ui64 messageId, NDrive::TEntitySession& session) const {
    return HistoryManager.GetMessage(messageId, session);
}

TMaybe<NDrive::NChat::TChatMessages> NDrive::NChat::TEngine::GetMessagesByChats(const TVector<TChat>& chats, const TRange<ui64>& idRange, const TRange<TInstant>& timestampRange, const ui32 viewerTraits, NDrive::TEntitySession& session) const {
    return HistoryManager.GetMessagesByChats(chats, idRange, timestampRange, viewerTraits, session);
}

bool NDrive::NChat::TEngine::UpdateCachedMessages(const TSet<TString>& searchIds, NDrive::TEntitySession& session, const TInstant& requestTime) const {
    return MessagesCache.UpdateCachedMessages(searchIds, session, requestTime);
}

NDrive::NChat::TExpectedMessageEvents NDrive::NChat::TEngine::GetCachedMessages(const TString& searchId) const {
    return MessagesCache.GetCachedMessages(searchId);
}

bool NDrive::NChat::TEngine::RefreshCache(const TInstant actuality, const bool /* doActualizeHistory */) const {
    if (StickerManager && !StickerManager->RefreshCache(actuality)) {
        return false;
    }
    return MessagesCache.RefreshCache();
}

NDrive::NChat::TStickerManager* NDrive::NChat::TEngine::GetStickerManager() const {
    return StickerManager.Get();
}
