#pragma once

#include "messages_cache.h"
#include "sticker_manager.h"

#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/history/cache.h>
#include <drive/backend/database/history/manager.h>

namespace NDrive::NChat {
    class TEngineConfig {
    private:
        R_READONLY(TString, DBName);
        R_READONLY(THistoryConfig, ChatMetaManagerConfig, THistoryConfig().SetDeep(TDuration::Zero()));
        R_READONLY(THistoryConfig, MessagesHistoryConfig, THistoryConfig().SetDeep(TDuration::Days(90)));

    public:
        void Init(const TYandexConfig::Section* section);
        void ToString(IOutputStream& os) const;
    };

    class TEngine: public TDatabaseSessionConstructor {
    private:
        const IHistoryContext& Context;
        TChatMessagesHistoryManager HistoryManager;
        TChatsMetaManager& ChatsMeta;
        const TEngineConfig Config;
        mutable TMessagesCache MessagesCache;

        THolder<TStickerManager> StickerManager;

        TRWMutex MutexEngine;

    public:
        TEngine(TChatsMetaManager& chatsMeta, const TEngineConfig& config, const IHistoryContext& context)
            : TDatabaseSessionConstructor(context.GetDatabase())
            , Context(context)
            , HistoryManager(Context, config.GetMessagesHistoryConfig())
            , ChatsMeta(chatsMeta)
            , Config(config)
            , MessagesCache("messages_history", TDuration::Seconds(100), HistoryManager, ChatsMeta)
            , StickerManager(new TStickerManager(context.GetDatabase()))
        {
        }

        bool StartEngine() {
            return StickerManager->Start() && MessagesCache.Start();
        }

        bool StopEngine() {
            return StickerManager->Stop() && MessagesCache.Stop();
        }

        TString GetDatabaseName() const {
            return Config.GetDBName();
        }

        TChatsMetaManager& GetChats() const {
            return ChatsMeta;
        }

        const THistoryConfig& GetHistoryConfig() const {
            return Config.GetMessagesHistoryConfig();
        }

        bool AddMessage(const TMessage& message, const TString& userId, NDrive::TEntitySession& session, const TString& chatSearchId = "") const;
        bool AddMessageIfLast(const TMessage& message, const TString& userId, NDrive::TEntitySession& session, const TString& chatSearchId, const ui64 lastMessageId) const;
        bool EditMessage(const ui64 messageId, TMessage message, const TString& userId, NDrive::TEntitySession& session) const;
        bool DeleteMessage(const ui64 messageId, const ui32 chatRoomId, const TString& userId, NDrive::TEntitySession& session) const;

        TMaybe<TObjectEventPtr<TMessage>> GetMessage(const ui64 messageId, NDrive::TEntitySession& session) const;

        bool RefreshCache(const TInstant actuality, const bool doActualizeHistory = true) const;
        TStickerManager* GetStickerManager() const;

        TOptionalMessageEvents GetMessagesRange(const ui32 chat, const ui32 viewerTraits, NDrive::TEntitySession& session, const ui64 fromId, const ui64 toId) const;
        TOptionalMessageEvents GetMessagesSinceId(const ui32 chat, const ui32 viewerTraits, const ui64 fromId, NDrive::TEntitySession& session) const;
        TOptionalMessageEvents GetMessagesSinceTimestamp(const ui32 chat, const ui32 viewerTraits, const TInstant fromTs, NDrive::TEntitySession& session) const;
        bool GetLastMessage(const ui32 chat, const ui32 viewerTraits, NDrive::TEntitySession& session, TMaybe<NDrive::NChat::TMessageEvent>& result, const TString& userId = "") const;

        TMaybe<size_t> GetMessagesCount(const ui32 chat, const ui32 viewerTraits, NDrive::TEntitySession& session, const TSet<TMessage::EMessageType>& excludeTypes = {}, const TMaybe<TSet<TMessage::EMessageType>> includeTypes = {}) const;
        TMaybe<size_t> GetMessagesCountSince(const ui32 chat, const ui32 viewerTraits, const ui64 messageId, NDrive::TEntitySession& session, const TSet<TMessage::EMessageType>& excludeTypes = {}, const TMaybe<TSet<TMessage::EMessageType>> includeTypes = {}) const;

        TMaybe<TChatMessages> GetMessagesByChats(const TVector<TChat>& chats, const TRange<ui64>& idRange, const TRange<TInstant>& timestampRange, const ui32 viewerTraits, NDrive::TEntitySession& session) const;

        bool UpdateCachedMessages(const TSet<TString>& searchIds, NDrive::TEntitySession& session, const TInstant& requestTime) const;
        TExpectedMessageEvents GetCachedMessages(const TString& searchId) const;
    };
}
