#include "message.h"

#include <rtline/library/json/parse.h>
#include <rtline/util/algorithm/type_traits.h>

const TString NDrive::NChat::TMessage::OnSendMacroPrefix = "!";
const TString NDrive::NChat::TMessage::OnSendMacroSuffix = "";
const TString NDrive::NChat::TMessage::OnViewMacroPrefix = "$";
const TString NDrive::NChat::TMessage::OnViewMacroSuffix = "";

void NDrive::NChat::ParseEmoji(TString& text) {
    SubstGlobal(text, "\\U0001F3CE", "\U0001F3CE");
    SubstGlobal(text, "\\U0001F46E", "\U0001F46E");
    SubstGlobal(text, "\\U0001F699", "\U0001F699");
    SubstGlobal(text, "\\U0001F6A8", "\U0001F6A8");
    SubstGlobal(text, "\\U0001F4A1", "\U0001F4A1");
}

bool NDrive::NChat::IMessage::IsUnescapeable() const {
    return (
        GetType() == NDrive::NChat::TMessage::EMessageType::ImageLink ||
        GetType() == NDrive::NChat::TMessage::EMessageType::Plaintext ||
        GetType() == NDrive::NChat::TMessage::EMessageType::WelcomeText ||
        GetType() == NDrive::NChat::TMessage::EMessageType::Separator ||
        GetType() == NDrive::NChat::TMessage::EMessageType::FeedbackDialogue
    );
}

bool NDrive::NChat::IMessage::IsVisible(const ui32 observerTraits) const {
    return (GetTraits() & observerTraits) == GetTraits();
}

NDrive::NChat::TMessage::TMessage(const TString& text, const ui32 traits, const EMessageType type)
    : Type(type)
    , Traits(traits)
    , Text(text)
{
}

bool NDrive::NChat::TMessage::DeserializeMeta(const NJson::TJsonValue& jsonMeta) {
    return
        NJson::ParseField(jsonMeta["external_id"], ExternalId, false) &&
        NJson::ParseField(jsonMeta["author_name"], AuthorName, false) &&
        NJson::ParseField(jsonMeta["rating"], Rating, false) &&
        NJson::ParseField(jsonMeta["is_rateable"], IsRateable, false) &&
        NJson::ParseField(jsonMeta["external_status"], NJson::Stringify(ExternalStatus), false);
}

NJson::TJsonValue NDrive::NChat::TMessage::SerializeMeta() const {
    NJson::TJsonValue meta = NJson::JSON_MAP;
    if (ExternalId) {
        meta["external_id"] = ExternalId;
    }
    if (AuthorName) {
        meta["author_name"] = AuthorName;
    }
    if (ExternalStatus != EExternalStatus::NotExternal) {
        meta["external_status"] = ToString(ExternalStatus);
    }
    if (Rating) {
        meta["rating"] = Rating;
    }
    if (IsRateable) {
        meta["is_rateable"] = *IsRateable;
    }
    return meta;
}

bool NDrive::NChat::TMessage::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, ChatId);
    READ_DECODER_VALUE(decoder, values, Text);
    READ_DECODER_VALUE(decoder, values, Icon);
    READ_DECODER_VALUE(decoder, values, Link);

    ui32 tempType = 0;
    READ_DECODER_VALUE_TEMP_OPT(decoder, values, tempType, Type);
    Type = static_cast<EMessageType>(tempType);

    READ_DECODER_VALUE_DEF(decoder, values, OperatedId, 0);
    READ_DECODER_VALUE_DEF(decoder, values, Traits, 0);

    NJson::TJsonValue jsonMeta;
    READ_DECODER_VALUE_JSON(decoder, values, jsonMeta, Meta);

    return DeserializeMeta(jsonMeta);
}

bool NDrive::NChat::TMessage::DeserializeBasicsFromJson(const NJson::TJsonValue& raw) {
    if (!NJson::ParseField(raw, "text", Text, true)) {
        return false;
    }
    ParseEmoji(Text);

    return
        NJson::ParseField(raw, "type", NJson::Stringify(Type), true) &&
        NJson::ParseField(raw, "icon", Icon) &&
        NJson::ParseField(raw, "link", Link) &&
        NJson::ParseField(raw, "traits", Traits, false) &&
        NJson::ParseField(raw, "external_id", ExternalId) &&
        NJson::ParseField(raw, "author_name", AuthorName) &&
        NJson::ParseField(raw, "rating", Rating) &&
        NJson::ParseField(raw, "is_rateable", IsRateable, false) &&
        NJson::ParseField(raw, "external_status", NJson::Stringify(ExternalStatus), false) &&
        DoDeserializeBasicsFromJson(raw);
}

bool NDrive::NChat::TMessage::DoDeserializeBasicsFromJson(const NJson::TJsonValue& /*raw*/) {
    return true;
}

NJson::TJsonValue NDrive::NChat::TMessage::SerializeToJson() const {
    NJson::TJsonValue result = SerializeMeta();
    result["text"] = ToString(GetText());
    result["traits"] = GetTraits();
    result["type"] = ToString(GetType());
    if (GetIcon()) {
        result["icon"] = GetIcon();
    }
    if (GetLink()) {
        result["link"] = GetLink();
    }
    return result;
}

NStorage::TTableRecord NDrive::NChat::TMessage::SerializeToTableRecord() const {
    NStorage::TTableRecord result;
    result.Set("chat_id", ChatId).Set("text", Text).Set("type", (ui32)Type);
    if (Icon) {
        result.Set("icon", Icon);
    }
    if (Link) {
        result.Set("link", Link);
    }
    if (OperatedId) {
        result.Set("operated_id", OperatedId);
    }
    if (Traits) {
        result.Set("traits", Traits);
    }
    result.Set("meta", SerializeMeta().GetStringRobust());
    return result;
}

template struct TExpectedSizeOf<NDrive::NChat::TMessage, 96>;
