#pragma once

#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/history/event.h>

#include <rtline/util/json_processing.h>
#include <rtline/util/types/accessor.h>

#include <util/generic/string.h>

namespace NDrive::NChat {

    void ParseEmoji(TString& text);

    class IMessage {
    public:
        enum class EMessageType : ui32 {
            Plaintext = 0 /* "plaintext" */,
            Link = 1 /* "link" */,
            Mailto = 2 /* "mailto" */,
            Image = 3 /* "image" */,
            Video = 4 /* "video" */,
            UserDocumentPhoto = 5 /* "user_document_photo" */,
            ImageLink = 6 /* "image_link" */,
            CreditCardBinding = 7 /* "CreditCardBinding" */,
            UserDocumentPhotos = 8 /* "user_document_photos" */,
            MediaResources = 9 /* "media_resources" */,
            WelcomeText = 10 /* "welcome_text" */,
            Separator = 11 /* "separator" */,
            Bonus = 12 /* "bonus" */,
            Order = 13 /* "order" */,
            Introscreen = 14 /* "introscreen" */,
            Location = 15 /* "location" */,
            Sticker = 16 /* "sticker" */,
            FeedbackDialogue = 17 /* "feedback_dialogue" */,
            ColorSeparator = 18 /* "color_separator" */,
            ImagesList = 19 /* "images_list" */,
            Custom = 20 /* "custom" */,
            Fine = 21 /* "fine" */,
            PDF = 22 /* "pdf" */,
            Delay = 23 /* "delay" */,
            PhoneNumber = 24 /* "phone_number" */,
            CreditCard = 25 /* "credit_card_binding" */,
            PhoneVerificationCode = 26 /* "phone_verification_code" */,
            NodeId = 27 /* "node_id" */,
            UserDocumentVideo = 28 /* "user_document_video" */,
            Unknown = 1000000000 /* "unknown" */,
        };

        enum class EMessageTraits : ui32 {
            StaffOnly = 1 << 0 /* "staff_only" */,
            DeletedMessage = 1 << 1 /* "deleted_message" */,
        };

        static constexpr ui32 AllKnownTraits = (ui32)EMessageTraits::StaffOnly;
        static constexpr ui32 DefaultUserTraits = 0;

    public:
        virtual EMessageType GetType() const = 0;
        virtual TString GetText() const = 0;
        virtual TString GetIcon() const = 0;
        virtual TString GetLink() const = 0;
        virtual ui32 GetTraits() const = 0;

        bool IsUnescapeable() const;
        bool IsVisible(const ui32 observerTraits) const;

        virtual ~IMessage() = default;
    };

    class TMessage : public IMessage {
    public:
        enum class EExternalStatus: ui32 {
            NotExternal = 0 /* "not_external" */,
            Undefined = 1 /* "undefined" */,
            Sent = 2 /* "sent" */,
            Forwarded = 3 /* "forwarded"*/
        };

    public:
        using TPtr = TAtomicSharedPtr<TMessage>;

    public:
        static const TString OnSendMacroPrefix;
        static const TString OnSendMacroSuffix;
        static const TString OnViewMacroPrefix;
        static const TString OnViewMacroSuffix;

    private:
        R_FIELD(ui32, ChatId, Max<i32>());
        R_FIELD(ui64, OperatedId, 0);
        EMessageType Type = EMessageType::Plaintext;
        ui32 Traits = 0;
        TString Text;
        TString Icon;
        TString Link;
        R_FIELD(TString, ExternalId);
        R_FIELD(EExternalStatus, ExternalStatus, EExternalStatus::NotExternal);
        R_FIELD(TString, AuthorName);
        R_FIELD(TString, Rating);
        R_OPTIONAL(bool, IsRateable);

    public:
        class TDecoder: public TBaseDecoder {
            R_FIELD(i32, ChatId, -1);
            R_FIELD(i32, Text, -1);
            R_FIELD(i32, Link, -1);
            R_FIELD(i32, Type, -1);
            R_FIELD(i32, Icon, -1);
            R_FIELD(i32, OperatedId, -1);
            R_FIELD(i32, Traits, -1);
            R_FIELD(i32, Meta, -1);

        public:
            TDecoder() = default;

            TDecoder(const TMap<TString, ui32>& decoderBase) {
                ChatId = GetFieldDecodeIndex("chat_id", decoderBase);
                Text = GetFieldDecodeIndex("text", decoderBase);
                Type = GetFieldDecodeIndex("type", decoderBase);
                Icon = GetFieldDecodeIndex("icon", decoderBase);
                Link = GetFieldDecodeIndex("link", decoderBase);
                OperatedId = GetFieldDecodeIndex("operated_id", decoderBase);
                Traits = GetFieldDecodeIndex("traits", decoderBase);
                Meta = GetFieldDecodeIndex("meta", decoderBase);
            }
        };

        TMessage() = default;
        TMessage(const TString& text, const ui32 traits, const EMessageType type);

        bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);
        bool DeserializeBasicsFromJson(const NJson::TJsonValue& raw);
        virtual bool DoDeserializeBasicsFromJson(const NJson::TJsonValue& raw);
        bool DeserializeMeta(const NJson::TJsonValue& jsonMeta);
        NJson::TJsonValue SerializeMeta() const;
        NJson::TJsonValue SerializeToJson() const;

        virtual void SetType(const EMessageType type) {
            Type = type;
        }

        EMessageType GetType() const override {
            return Type;
        }

        TMessage& SetTraits(const ui32 extTraits) {
            Traits = extTraits;
            return *this;
        }

        TMessage& AddTraits(const ui32 extTraits) {
            Traits |= extTraits;
            return *this;
        }

        TMessage& SetText(const TString& text) {
            Text = text;
            return *this;
        }

        TMessage& SetIcon(const TString& icon) {
            Icon = icon;
            return *this;
        }

        TMessage& SetLink(const TString& link) {
            Link = link;
            return *this;
        }

        virtual TString GetText() const override {
            return Text;
        }

        virtual TString GetIcon() const override {
            return Icon;
        }

        virtual TString GetLink() const override {
            return Link;
        }

        virtual ui32 GetTraits() const override {
            return Traits;
        }

        NStorage::TTableRecord SerializeToTableRecord() const;
    };

    class TMessageEdit {
        R_OPTIONAL(TString, Text);
        R_OPTIONAL(TMessage::EMessageType, Type);
        R_OPTIONAL(TMessage::EExternalStatus, ExternalStatus);
        R_OPTIONAL(TString, Rating);
        R_OPTIONAL(bool, IsRateable);

    public:
        TMessageEdit() = default;
        TMessageEdit(TMaybe<TString> text = {}, TMaybe<TMessage::EMessageType> type = {}, TMaybe<TMessage::EExternalStatus> externalStatus = {}, TMaybe<TString> rating = {}, TMaybe<bool> isRateable = {})
            : Text(text)
            , Type(type)
            , ExternalStatus(externalStatus)
            , Rating(rating)
            , IsRateable(isRateable)
        {
        }

        void Apply(NDrive::NChat::TMessage& message) const {
            if (Text) {
                message.SetText(*Text);
            }
            if (Type) {
                message.SetType(*Type);
            }
            if (ExternalStatus) {
                message.SetExternalStatus(*ExternalStatus);
            }
            if (Rating) {
                message.SetRating(*Rating);
            }
            if (IsRateable) {
                message.SetIsRateable(*IsRateable);
            }
        }
    };

    using TMessageEvent = TObjectEvent<TMessage>;
    using TMessageEvents = TVector<TMessageEvent>;
    using TOptionalMessageEvents = TMaybe<TVector<TMessageEvent>>;
    using TChatMessages = TMap<TString, TMessageEvents>;
};
