#include "messages_cache.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/logging/evlog.h>

#include <library/cpp/json/writer/json_value.h>
#include <util/string/builder.h>

TMaybe<TDuration> NDrive::NChat::TMessagesCache::GetLifetime() const {
    auto lifetimeKey = TStringBuilder() << "chat_engine.messages.cache_lifetime";
    auto lifetime = NDrive::HasServer()
        ? NDrive::GetServer().GetSettings().GetValue<TDuration>(lifetimeKey)
        : Nothing();
    return lifetime;
}

bool NDrive::NChat::TMessagesCache::UpdateCachedMessages(const TSet<TString>& searchIds, NDrive::TEntitySession& session, const TInstant& requestTime) const {
    auto threshold = requestTime - GetLifetime().GetOrElse(DefaultLifetime);
    TSet<TString> refetch;
    for (auto&& searchId : searchIds) {
        auto optionalObject = ObjectCache.find(searchId);
        if (!optionalObject || optionalObject->GetTimestamp() <= threshold) {
            refetch.emplace(searchId);
        }
    }
    auto chats = Chats.GetChatsFromTable(refetch, session);
    if (!chats) {
        return false;
    }
    auto messages = MessagesHistoryManager.GetMessagesByChats(*chats, {}, {}, NDrive::NChat::TMessage::AllKnownTraits, session);
    if (!messages) {
        return false;
    }
    for (auto& [searchId, messageEvents] : *messages) {
        ObjectCache.update(searchId, {messageEvents, requestTime});
    }
    return true;
}

NDrive::NChat::TMessageEvents NDrive::NChat::TMessagesCache::GetCachedObjectImpl(const TString& searchId, bool& expired, bool& cached) const {
    auto eventLogger = NDrive::GetThreadEventLogger();
    auto now = Now();
    auto threshold = now - GetLifetime().GetOrElse(DefaultLifetime);
    auto optionalObject = ObjectCache.find(searchId);
    if (optionalObject && optionalObject->GetTimestamp() > threshold) {
        if (eventLogger) {
            eventLogger->AddEvent(NJson::TMapBuilder
                ("event", "ChatCacheHit")
                ("search_id", searchId)
                ("timestamp", NJson::ToJson(optionalObject->GetTimestamp()))
            );
        }
        cached = true;
        return std::move(optionalObject->GetEvents());
    }
    if (optionalObject) {
        expired = true;
    }

    if (eventLogger) {
        eventLogger->AddEvent(NJson::TMapBuilder
            ("event", "ChatCacheMiss")
            ("search_id", searchId)
        );
    }

    NDrive::NChat::TChat chat;
    auto session = MessagesHistoryManager.BuildSession(true);
    if (!Chats.GetChatFromCache(searchId, chat, true, session.GetTransaction())) {
        session.Check();
    }
    TInstant updateTime = Now();
    auto restoredMessages =  MessagesHistoryManager.GetMessagesFromDb(NDrive::NChat::TChatMessagesHistoryManager::TMessagesRequestContext(chat.GetId()), {}, {}, NDrive::NChat::TMessage::AllKnownTraits, session);
    if (!restoredMessages) {
        session.Check();
    }
    ObjectCache.update(searchId, {*restoredMessages, updateTime});
    cached = false;
    return std::move(*restoredMessages);
}

NDrive::NChat::TMessagesCache::TMessagesCache(const TString& tableName, TDuration defaultLifetime, const TChatMessagesHistoryManager& messagesHistoryManager, const TChatsMetaManager& chats)
    : IAutoActualization(tableName + "-cache", TDuration::Seconds(1))
    , MessagesHistoryManager(messagesHistoryManager)
    , Chats(chats)
    , DefaultLifetime(defaultLifetime)
    , CacheHit({ tableName + "-cache-hit" }, false)
    , CacheMiss({ tableName + "-cache-miss" }, false)
    , CacheExpired({ tableName + "-cache-expired" }, false)
    , CacheInvalidated({ tableName + "-cache-invalidated" }, false)
    , ObjectCache(64 * 1024)
{
}

NDrive::NChat::TExpectedMessageEvents NDrive::NChat::TMessagesCache::GetCachedMessages(const TString& searchId) const {
    bool expired = false;
    bool cached = false;
    try {
        auto messageEvents = GetCachedObjectImpl(searchId, expired, cached);
        if (cached) {
            CacheHit.Signal(1);
        } else {
            CacheMiss.Signal(1);
        }
        if (expired) {
            CacheExpired.Signal(1);
        }
        return messageEvents;
    } catch (TCodedException& e) {
        return MakeUnexpected(std::move(e));
    }
}

bool NDrive::NChat::TMessagesCache::GetStartFailIsProblem() const {
    return false;
}

bool NDrive::NChat::TMessagesCache::Refresh() {
    auto session = MessagesHistoryManager.BuildSession(true);
    if (!LastEventId) {
        LastEventId = MessagesHistoryManager.GetMaxEventIdOrThrow(session);
    }

    auto since = LastEventId ? *LastEventId + 1 : 0;
    auto optionalEvents = MessagesHistoryManager.GetEvents({ since }, {}, session, NSQL::TQueryOptions().SetLimit(1000));
    if (!optionalEvents) {
        ERROR_LOG << GetName() << ": cannot GetEvents since " << since << ": " << session.GetStringReport() << Endl;
        return false;
    }
    TSet<ui32> chatIds;
    for (auto&& event : *optionalEvents) {
        chatIds.emplace(event.GetChatId());
        LastEventId = std::max(LastEventId.GetOrElse(0), event.GetHistoryEventId());
    }
    auto chats = Chats.GetChatsFromTable(chatIds, session);
    if (!chats) {
        ERROR_LOG << GetName() << ": cannot GetChatsFromTable: " << session.GetStringReport() << Endl;
        return false;
    }
    for (auto&& chat : *chats) {
        bool erased = ObjectCache.erase(chat.GetSearchId());
        if (erased) {
            CacheInvalidated.Signal(1);
            INFO_LOG << GetName() << ": invalidate " << chat.GetSearchId() << Endl;
        }
    }
    return true;
}

bool NDrive::NChat::TMessagesCache::Invalidate(const TString& searchId) const {
    return ObjectCache.erase(searchId);
}
