#include "messages_history_manager.h"

ui64 NDrive::NChat::TChatMessagesHistoryManager::GetMessageId(const NDrive::NChat::TChatMessagesHistoryManager::TEventPtr& event) {
    return event->GetHistoryAction() == EObjectHistoryAction::Add ? event->GetHistoryEventId() : event->GetOperatedId();
}

ui64 NDrive::NChat::TChatMessagesHistoryManager::GetMessageId(const NDrive::NChat::TMessageEvent& event) {
    return event.GetHistoryAction() == EObjectHistoryAction::Add ? event.GetHistoryEventId() : event.GetOperatedId();
}

void NDrive::NChat::TChatMessagesHistoryManager::InsertSorted(TMessageEvents& chatMessages, TMessageEvent& message, const ui32 viewerTraits) const {
    if (!message.IsVisible(viewerTraits) && message.GetOperatedId() == 0) {
        return;
    }
    auto it = chatMessages.rbegin();
    while (it != chatMessages.rend() && GetMessageId(message) < GetMessageId(*it)) {
        ++it;
    }

    if (it != chatMessages.rend() && GetMessageId(*it) == GetMessageId(message)) {
        Y_ENSURE_BT(it->GetHistoryEventId() != message.GetHistoryEventId());
        if (it->GetHistoryEventId() < message.GetHistoryEventId()) {
            if (message.IsVisible(viewerTraits)) {
                auto originalMessageInstant = it->GetHistoryInstant();
                auto originalMessageAuthor = it->GetHistoryUserId();
                *it = message;
                it->SetHistoryInstant(originalMessageInstant);
                it->SetHistoryUserId(originalMessageAuthor);
            } else {
                chatMessages.erase(std::next(it).base());
            }
        }
    } else if (message.GetOperatedId() == 0) {
        chatMessages.insert(it.base(), std::move(message));
    }
}

NDrive::NChat::TMessageEvents NDrive::NChat::TChatMessagesHistoryManager::SortAndFilterMessages(TMessageEvents& events, const ui32 viewerTraits, const TMaybe<ui64> startId, const TMaybe<ui64> finishId, const TMaybe<i64> count) const {
    TVector<TObjectEvent<TMessage>> result;
    int i = 0;
    for (auto&& event : events) {
        if (event.GetHistoryAction() == EObjectHistoryAction::Remove) {
            event.AddTraits((ui32)NDrive::NChat::TMessage::EMessageTraits::DeletedMessage);
        }
        if (startId && GetMessageId(event) < *startId) {
            continue;
        }
        if (finishId && GetMessageId(event) > *finishId) {
            continue;
        }
        if (count && ++i > *count) {
            break;
        }
        InsertSorted(result, event, viewerTraits);
    }
    return result;
}

NDrive::NChat::TOptionalMessageEvents NDrive::NChat::TChatMessagesHistoryManager::GetMessagesFromDb(const TMessagesRequestContext& requestContext, const TRange<ui64>& idRange, const TRange<TInstant>& timestampRange, const ui32 viewerTraits, NDrive::TEntitySession& session) const {
    if (requestContext.HasChatId() && requestContext.GetChatIdUnsafe() == 0) {
        return TMessageEvents();
    }
    auto events = TDatabaseHistoryManager::GetEvents(idRange, timestampRange, session, requestContext.GenerateQueryOptions());
    if (!events) {
        return {};
    }
    return SortAndFilterMessages(*events, viewerTraits, idRange.From, idRange.To);
}

TMaybe<TObjectEventPtr<NDrive::NChat::TMessage>> NDrive::NChat::TChatMessagesHistoryManager::GetMessage(const ui64 messageId, NDrive::TEntitySession& session) const {
    return TDatabaseHistoryManager::GetEvent(messageId, session);
}

TMaybe<size_t> NDrive::NChat::TChatMessagesHistoryManager::GetMessagesCountSince(const ui32 chat, const ui32 viewerTraits, const ui64 startId, NDrive::TEntitySession& session, const TSet<TMessage::EMessageType>& excludeTypes, const TMaybe<TSet<TMessage::EMessageType>>& includeTypes) const {
    NDrive::NChat::TChatMessagesHistoryManager::TMessagesRequestContext requestContext(chat);
    requestContext.SetIncludedTypes(includeTypes);
    auto events = GetMessagesFromDb(requestContext, { startId }, {}, viewerTraits, session);
    if (!events) {
        return {};
    }
    size_t result = 0;
    for (auto&& eventIt = events->rbegin(); eventIt != events->rend(); ++eventIt) {
        if (!excludeTypes.contains(eventIt->GetType())) {
            result += 1;
        }
    }
    return result;
}

bool NDrive::NChat::TChatMessagesHistoryManager::GetLastMessage(const ui32 chat, const ui32 viewerTraits, NDrive::TEntitySession& session, TMaybe<NDrive::NChat::TMessageEvent>& result, const TString& userId) const {
    result = Nothing();
    if (chat == 0) {
        return true;
    }
    ui32 limit = 10;
    ui32 maxLimit = 1001;
    ui32 step = 10;
    bool limitReached = false;
    TQueryOptions queryOptions;
    TSet<ui64> chatId = { chat };
    queryOptions.SetGenericCondition("chat_id", chatId).SetOrderBy({"history_event_id"}).SetDescending(true);
    if (userId) {
        queryOptions.AddGenericCondition("history_user_id", userId);
    }
    while (!result && !limitReached) {
        if (limit < maxLimit) {
            queryOptions.SetLimit(limit);
        } else {
            limitReached = true;
        }
        auto events = TDatabaseHistoryManager::GetEvents({}, {}, session, queryOptions);
        if (!events) {
            return false;
        }
        auto messages = SortAndFilterMessages(*events, viewerTraits);
        if (!messages.empty()) {
            result = messages.back();
        }
        limit *= step;
    }
    return true;
}

TMaybe<NDrive::NChat::TChatMessages> NDrive::NChat::TChatMessagesHistoryManager::GetMessagesByChats(const TMap<ui64, TString>& searchIdsByChatIds, const TRange<ui64>& idRange, const TRange<TInstant>& timestampRange, const ui32 viewerTraits, NDrive::TEntitySession& session) const {
    auto chatIds = MakeSet(NContainer::Keys(searchIdsByChatIds));
    if (chatIds.empty()) {
        return NDrive::NChat::TChatMessages();
    }
    TQueryOptions queryOptions;
    queryOptions.SetGenericCondition("chat_id", chatIds);
    auto messageEvents = TDatabaseHistoryManager::GetEvents(idRange, timestampRange, session, queryOptions);
    if (!messageEvents) {
        return Nothing();
    }
    TChatMessages unsorted, result;
    for (auto&& messageEvent : *messageEvents) {
        auto chatIt = searchIdsByChatIds.find(messageEvent.GetChatId());
        if (chatIt != searchIdsByChatIds.end()) {
            unsorted[chatIt->second].push_back(std::move(messageEvent));
        }
    }
    for (auto&& [searchId, messages] : unsorted) {
        result[searchId] = SortAndFilterMessages(messages, viewerTraits);
    }
    return result;
}

TMaybe<NDrive::NChat::TChatMessages> NDrive::NChat::TChatMessagesHistoryManager::GetMessagesByChats(const TVector<TChat>& chats, const TRange<ui64>& idRange, const TRange<TInstant>& timestampRange, const ui32 viewerTraits, NDrive::TEntitySession& session) const {
    TMap<ui64, TString> searchIdsByChatIds;
    for (auto&& chat : chats) {
        searchIdsByChatIds[chat.GetId()] = chat.GetSearchId();
    }
    return GetMessagesByChats(searchIdsByChatIds, idRange, timestampRange, viewerTraits, session);
}

NSQL::TQueryOptions NDrive::NChat::TChatMessagesHistoryManager::TMessagesRequestContext::GenerateQueryOptions() const {
    TQueryOptions queryOptions;
    if (ChatId) {
        TSet<ui64> chatIdSet = { *ChatId };
        queryOptions.SetGenericCondition("chat_id", chatIdSet);
    }
    if (IncludedTypes) {
        TSet<ui64> uiTypes;
        for (auto&& type : *IncludedTypes) {
            uiTypes.emplace((ui64)type);
        }
        queryOptions.SetGenericCondition("type", uiTypes);
    }
    return queryOptions;
}
