#include "sticker_manager.h"

bool NDrive::NChat::TSticker::DeserializeWithDecoder(const TStickerDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, Id);
    READ_DECODER_VALUE(decoder, values, Url);
    READ_DECODER_VALUE(decoder, values, Position);
    READ_DECODER_VALUE(decoder, values, IsActive);
    READ_DECODER_VALUE(decoder, values, Emoji);
    return true;
}

TString NDrive::NChat::TSticker::GetObjectId(const TSticker& object) {
    return object.GetId();
}

NDrive::NChat::TSticker NDrive::NChat::TSticker::FromHistoryEvent(const TObjectEvent<TSticker>& historyEvent) {
    return historyEvent;
}

NJson::TJsonValue NDrive::NChat::TSticker::SerializeToJson() const {
    NJson::TJsonValue result;
    result["id"] = Id;
    result["url"] = Url;
    result["position"] = Position;
    result["is_active"] = IsActive;
    result["emoji"] = Emoji;
    return result;
}

void NDrive::NChat::TSticker::DoBuildReportItem(NJson::TJsonValue& result) const {
    result["impl"] = SerializeToJson();
}

NStorage::TTableRecord NDrive::NChat::TSticker::SerializeToTableRecord() const {
    NStorage::TTableRecord result;
    result.Set("is_active", IsActive);
    result.Set("url", Url);
    result.Set("id", Id);
    result.Set("position", Position);
    result.Set("emoji", Emoji);
    return result;
}

bool NDrive::NChat::TSticker::ParseFromJson(const NJson::TJsonValue& json) {
    JREAD_STRING_OPT(json, "id", Id);
    JREAD_STRING_OPT(json, "url", Url);
    JREAD_UINT_OPT(json, "position", Position);
    JREAD_BOOL_OPT(json, "is_active", IsActive);
    JREAD_STRING_OPT(json, "emoji", Emoji);
    return true;
}

bool NDrive::NChat::TSticker::Parse(const NStorage::TTableRecord& row) {
    return TBaseDecoder::DeserializeFromTableRecord(*this, row);
}

void NDrive::NChat::TStickerManager::AcceptHistoryEventUnsafe(const TObjectEvent<TRecordType>& ev) const {
    if (ev.GetHistoryAction() != EObjectHistoryAction::Remove) {
        Stickers[ev.GetId()] = ev;
    } else {
        Stickers.erase(ev.GetId());
    }
}

bool NDrive::NChat::TStickerManager::DoRebuildCacheUnsafe() const {
    Stickers.clear();

    NStorage::TObjectRecordsSet<TRecordType> records;
    {
        auto table = Database->GetTable(StickersTable->GetTableName());
        auto transaction = Database->CreateTransaction(true);
        auto result = table->GetRows("", records, transaction);

        if (!result->IsSucceed()) {
            ERROR_LOG << "Cannot refresh data for " << StickersTable->GetTableName() << Endl;
            return false;
        }
    }

    for (auto&& record : records) {
        Stickers.emplace(record.GetId(), record);
    }

    return true;
}

TStringBuf NDrive::NChat::TStickerManager::GetEventObjectId(const TObjectEvent<TRecordType>& ev) const {
    return ev.GetId();
}

bool NDrive::NChat::TStickerManager::AddSticker(const TSticker& sticker, const TString& operatorUserId, NDrive::TEntitySession& session) const {
    NStorage::TObjectRecordsSet<TSticker> insertedData;

    if (!StickersTable->Insert(sticker, session, &insertedData) || insertedData.size() != 1) {
        session.AddErrorMessage("chat_sticker", "can't insert into table");
        return false;
    }

    if (!HistoryManager->AddHistory(*insertedData.begin(), operatorUserId, EObjectHistoryAction::Add, session)) {
        session.AddErrorMessage("chat_sticker_history", "can't write history");
        return false;
    }

    return true;
}

bool NDrive::NChat::TStickerManager::EditSticker(const TSticker& sticker, const TString& operatorUserId, NDrive::TEntitySession& session) const {
    if (!StickersTable->Upsert(sticker, session)) {
        session.AddErrorMessage("chat_sticker", "can't upsert into table");
        return false;
    }

    if (!HistoryManager->AddHistory(sticker, operatorUserId, EObjectHistoryAction::UpdateData, session)) {
        session.AddErrorMessage("chat_sticker_history", "can't write history");
        return false;
    }

    return true;
}

bool NDrive::NChat::TStickerManager::RemoveSticker(const TString& id, const TString& operatorUserId, NDrive::TEntitySession& session) const {
    if (!StickersTable->Remove(id, session)) {
        session.AddErrorMessage("chat_sticker", "can't delete from table");
        return false;
    }

    if (!HistoryManager->AddHistory(TSticker(id), operatorUserId, EObjectHistoryAction::Remove, session)) {
        session.AddErrorMessage("chat_sticker_history", "can't write history");
        return false;
    }

    return true;
}

NJson::TJsonValue NDrive::NChat::TStickerManager::GetStickersReport(const bool isAdminReport) const {
    TVector<TSticker> stickers;
    for (auto&& it : Stickers) {
        if (it.second.GetIsActive() || isAdminReport) {
            stickers.push_back(it.second);
        }
    }
    Sort(stickers.begin(), stickers.end(), [](const TSticker& lhs, const TSticker& rhs){return lhs.GetPosition() < rhs.GetPosition();});
    NJson::TJsonValue report = NJson::JSON_ARRAY;
    for (auto&& sticker : stickers) {
        NJson::TJsonValue stickerReport;
        stickerReport["url"] = sticker.GetUrl();
        stickerReport["id"] = sticker.GetId();
        stickerReport["emoji"] = sticker.GetEmoji();
        if (isAdminReport) {
            stickerReport["is_active"] = sticker.GetIsActive();
            stickerReport["position"] = sticker.GetPosition();
        }
        report.AppendValue(std::move(stickerReport));
    }
    return report;
}

bool NDrive::NChat::TStickerManager::GetSticker(const TString& id, NDrive::NChat::TSticker& result) const {
    auto rg = MakeObjectReadGuard();
    auto stickerIt = Stickers.find(id);
    if (stickerIt != Stickers.end()) {
        result = stickerIt->second;
        return true;
    }
    return false;
}
