#pragma once

#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/history/cache.h>
#include <drive/backend/database/history/manager.h>

#include <rtline/util/json_processing.h>
#include <rtline/util/types/accessor.h>

namespace NDrive::NChat {
    class TSticker {
        R_READONLY(TString, Id, "uuid_generate_v4()");
        R_READONLY(TString, Url);
        R_READONLY(bool, IsActive, false);
        R_READONLY(ui32, Position, 0);
        R_READONLY(TString, Emoji);

    public:
        class TStickerDecoder: public TBaseDecoder {
            R_FIELD(i32, Id, -1);
            R_FIELD(i32, Url, -1);
            R_FIELD(i32, IsActive, -1);
            R_FIELD(i32, Position, -1);
            R_FIELD(i32, Emoji, -1);

        public:
            TStickerDecoder() = default;

            TStickerDecoder(const TMap<TString, ui32>& decoderBase) {
                Id = GetFieldDecodeIndex("id", decoderBase);
                Url = GetFieldDecodeIndex("url", decoderBase);
                IsActive = GetFieldDecodeIndex("is_active", decoderBase);
                Position = GetFieldDecodeIndex("position", decoderBase);
                Emoji = GetFieldDecodeIndex("emoji", decoderBase);
            }
        };

        using TDecoder = TStickerDecoder;

        TSticker() = default;

        TSticker(const TString& id)
            : Id(id)
        {
        }

        bool DeserializeWithDecoder(const TStickerDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);
        static TString GetObjectId(const TSticker& object);

        static TSticker FromHistoryEvent(const TObjectEvent<TSticker>& historyEvent);

        NStorage::TTableRecord SerializeToTableRecord() const;
        NJson::TJsonValue SerializeToJson() const;
        bool Parse(const NStorage::TTableRecord& row);
        bool ParseFromJson(const NJson::TJsonValue& json);

        void DoBuildReportItem(NJson::TJsonValue& result) const;
    };

    class TStickerTable : public TDBEntities<TSticker> {
    private:
        using TBase = TDBEntities<TSticker>;
        using TEntity = TSticker;

    public:
        using TBase::TBase;

        virtual TString GetTableName() const override {
            return "chat_stickers";
        }

        virtual TString GetColumnName() const override {
            return "id";
        }

        virtual TString GetMainId(const TSticker& e) const override {
            return ToString(e.GetId());
        }
    };

    class TStickerHistoryManager : public TIndexedAbstractHistoryManager<TSticker> {
    private:
        using TBase = TIndexedAbstractHistoryManager<TSticker>;

    public:
        TStickerHistoryManager(const IHistoryContext& context, const THistoryConfig& hConfig)
            : TBase(context, "chat_stickers_history", hConfig)
        {
        }
    };

    class TStickerManager : public TDBCacheWithHistoryOwner<TStickerHistoryManager, TSticker> {
    private:
        using TRecordType = TSticker;
        using TCacheBase = TDBCacheWithHistoryOwner<TStickerHistoryManager, TSticker>;
        using TDBTable = TStickerTable;

    private:
        NStorage::IDatabase::TPtr Database;
        mutable TMap<TString, TSticker> Stickers;
        THolder<TDBTable> StickersTable;

    private:
        virtual void AcceptHistoryEventUnsafe(const TObjectEvent<TRecordType>& ev) const override;
        virtual bool DoRebuildCacheUnsafe() const override;
        virtual TStringBuf GetEventObjectId(const TObjectEvent<TRecordType>& ev) const override;

    public:
        TStickerManager(NStorage::IDatabase::TPtr db)
            : TCacheBase("chat_stickers", THistoryContext(db), THistoryConfig().SetDeep(TDuration::Max()))
            , Database(db)
        {
            StickersTable.Reset(new TDBTable(db));
        }

        bool AddSticker(const TSticker& sticker, const TString& operatorUserId, NDrive::TEntitySession& session) const;
        bool EditSticker(const TSticker& sticker, const TString& operatorUserId, NDrive::TEntitySession& session) const;
        bool RemoveSticker(const TString& id, const TString& operatorUserId, NDrive::TEntitySession& session) const;

        NJson::TJsonValue GetStickersReport(const bool isAdminReport = false) const;
        bool GetSticker(const TString& id, NDrive::NChat::TSticker& result) const;
    };
};
