#include "abstract.h"

#include <drive/backend/areas/areas.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/database/drive/landing.h>
#include <drive/backend/offers/actions/abstract.h>
#include <drive/backend/promo_codes/common/manager.h>
#include <drive/backend/user_document_photos/manager.h>

#include <library/cpp/charset/wide.h>

#include <util/generic/string.h>

TChatUserContext::TChatUserContext(const TUserPermissions::TPtr permissions, const NDrive::IServer* server)
    : Server(server)
    , Permissions(permissions)
{
    if (permissions) {
        SetUserId(permissions->GetUserId());
        SetFirstRiding(permissions->IsFirstRiding());
        SetPhone(permissions->GetPhone());

        MutableDefaultEmails().insert(permissions->GetEmail());
        for (const auto& mail : permissions->GetUserFeatures().GetDefaultEmails()) {
            if (!mail.Contains("@yandex.") || mail.EndsWith("ru") || mail.EndsWith("com")) {
                MutableDefaultEmails().insert(mail);
            }
        }
    }
    if (server) {
        auto registrationChatId = TUserPermissions::GetSetting<TString>("user_registration.registration.chat", server->GetSettings(), permissions).GetOrElse("registration");
        auto chatRobot = server->GetChatRobot(registrationChatId);
        TChatRobotScriptItem currentScriptItem;
        if (chatRobot && chatRobot->IsExistsForUser(GetUserId(), "") && chatRobot->GetCurrentScriptItem(GetUserId(), "", currentScriptItem, TInstant::Zero())) {
            SetRegistrationStep(currentScriptItem.GetId());
        }
        if (server->GetDriveAPI()) {
            // Settings tag names
            TString userFlagsTagName;
            {
                Server->GetSettings().GetValueStr(NDrive::UserFlagsTagNameSetting, userFlagsTagName);
            }
            // Debt info
            if (server->GetDriveAPI()->HasBillingManager()) {
                auto session = server->GetDriveAPI()->GetBillingManager().BuildSession(true);
                auto debt = server->GetDriveAPI()->GetBillingManager().GetDebt(GetUserId(), session);
                if (debt) {
                    SetDebtor(*debt);
                }
                session.ClearErrors();
            }
            // Performed car tags info
            {
                TVector<TDBTag> tags;
                server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetPerformedTags(GetUserId(), tags, TInstant::Zero());
                for (auto&& tag : tags) {
                    MutableProtectedPerformedTags().emplace(tag->GetName());
                }
            }
            // User tags info
            {
                auto user = server->GetDriveAPI()->GetTagsManager().GetUserTags().GetCachedObject(GetUserId());
                {
                    for (auto&& tag : user->GetTags()) {
                        if (!tag) {
                            continue;
                        }
                        const auto& tagName = tag->GetName();
                        MutableProtectedUserTags()[tagName] += 1;
                        auto dictImpl = tag.GetTagAs<TUserDictionaryTag>();
                        if (dictImpl) {
                            TMap<TString, TString> mapping;
                            for (auto&& field : dictImpl->GetFields()) {
                                mapping.emplace(field.Key, field.Value);
                            }
                            MutableProtectedDictionaryTagsData()[tagName].emplace_back(std::move(mapping));
                        }
                    }
                }
            }
        }
    }
}

TString TChatUserContext::Unescape(TString pattern, const TChatContext& context) const {
    SubstGlobal(pattern, "[[[!current_user]]]", GetUserId());
    SubstGlobal(pattern, "[[[!message]]]", GetMessage());

    {
        auto chatRobot = GetChatRobot();
        if (chatRobot) {
            auto locale = GetLocale();
            pattern = chatRobot->UnescapeMacroses(locale, pattern, GetUserId(), GetChatTopic(), "[[[!", "]]]", &context);
            pattern = chatRobot->UnescapeMacroses(locale, pattern, GetUserId(), GetChatTopic(), "[[[$", "]]]", &context);
            pattern = chatRobot->UnescapeMacroses(locale, pattern, GetUserId(), GetChatTopic(), "\"[[[&", "]]]\"", &context);
        }
    }

    return pattern;
}

NJson::TJsonValue TChatUserContext::UnescapeJson(const NJson::TJsonValue& json, const TChatContext& context) const {
    TString data = json.GetStringRobust();
    data = Unescape(data, context);

    NJson::TJsonValue result;
    if (!NJson::ReadJsonFastTree(data, &result)) {
        ERROR_LOG << "Could not deserialize patched json from string: " << data << Endl;
    }

    return result;
}

IChatRobot::TPtr TChatUserContext::GetChatRobot() const {
    if (!Server) {
        return nullptr;
    }
    return Server->GetChatRobot(GetChatId());
}

void TChatUserContext::SpecifyGeo(const TGeoCoord& c) {
    if (Server && Server->GetDriveAPI() && Server->GetDriveAPI()->GetAreasDB()) {
        SetProtectedAreaIds(Server->GetDriveAPI()->GetAreasDB()->GetAreaIdsInPoint(c, TInstant::Zero()));
    }
}

const TString TChatUserContext::GetStatus() const {
    return !!Permissions ? Permissions->GetStatus() : NDrive::UserStatusOnboarding;
}

bool TChatUserContext::HasAction(const TString& action) const {
    return !!Permissions ? Permissions->HasAction(action) : false;
}

bool IChatRobotImpl::MoveToStep(const TString& userId, const TString& topic, const TString& stepName, NDrive::TEntitySession* sessionExternal, const bool sendMessages, const TString& newChatTitle, const TString& operatorId, const TMaybe<ui64> lastMessageId, TVector<TContextMapInfo> contextMap) const {
    if (sessionExternal) {
        return MoveToStep(userId, topic, stepName, *sessionExternal, sendMessages, newChatTitle, operatorId, lastMessageId, contextMap);
    } else {
        auto session = BuildChatEngineSession();
        if (!MoveToStep(userId, topic, stepName, session, sendMessages, newChatTitle, operatorId, lastMessageId, contextMap)) {
            ERROR_LOG << "MoveToStep: " << session.GetStringReport() << Endl;
            return false;
        }
        if (!session.Commit()) {
            ERROR_LOG << "MoveToStep: cannot commit session: " << session.GetStringReport() << Endl;
            return false;
        }
        return true;
    }
}

TString IChatRobotImpl::GetChatSearchId(const TString& userId, const TString& topic) const {
    if (!topic) {
        return "user." + userId + "." + ChatConfig.GetChatId();
    } else {
        return "user." + userId + "." + ChatConfig.GetChatId() + "." + topic;
    }
}

bool IChatRobotImpl::GetChatRoom(const TString& userId, const TString& topic, ui32& chatRoomId, NDrive::TEntitySession* sessionExt) const {
    auto searchId = GetChatSearchId(userId, topic);
    TChat chat;
    bool isChatFound = false;
    if (!sessionExt) {
        isChatFound = ChatEngine.GetChats().GetChatFromCache(searchId, chat, true);
    } else {
        isChatFound = ChatEngine.GetChats().GetChatFromCache(searchId, chat, true, sessionExt->GetTransaction());
    }
    if (!isChatFound) {
        return false;
    }
    chatRoomId = chat.GetId();
    return true;
}

TMaybe<TVector<NDrive::NChat::TChat>> IChatRobotImpl::GetChats(const TSet<TString> searchIds, NDrive::TEntitySession& session) const {
    return ChatEngine.GetChats().GetChatsFromTable(searchIds, session);
}

TMaybe<TVector<NDrive::NChat::TChat>> IChatRobotImpl::GetChats(const TString& chatId, const TString& topic, NDrive::TEntitySession& session) const {
    return ChatEngine.GetChats().GetChatsFromTable(chatId, topic, session);
}

bool IChatRobotImpl::GetOrCreateChat(const TString& userId, const TString& topic, ui32& resultChatRoom, TChatContext& resultContext, NDrive::TEntitySession& session) const {
    Y_UNUSED(resultContext);
    if (!GetOrCreateChatRoom(userId, topic, resultChatRoom, &session)) {
        return false;
    }
    return true;
}

bool IChatRobotImpl::UpdateChat(const TChatContext& /*chatContext*/, const TString& /*userId*/, const TString& /*topic*/, const TString& /*currentStep*/, NDrive::TEntitySession& /*session*/) const {
    return true;
}

bool IChatRobotImpl::ArchiveChat(const TString& userId, const TString& topic) const {
    auto searchId = GetChatSearchId(userId, topic);

    TChat chat;
    bool isChatFound = ChatEngine.GetChats().GetChatFromTable(searchId, chat);
    if (!isChatFound || chat.GetIsArchive()) {
        return true;
    }
    chat.SetIsArchive(true);

    auto session = BuildChatEngineSession();
    return ChatEngine.GetChats().Upsert(chat, "robot-frontend", session) && session.Commit();
}

TSet<TString> IChatRobotImpl::GetNonRobotParticipants(const NDrive::NChat::TMessageEvents& messages) const {
    TSet<TString> result;
    for (auto&& message : messages) {
        if (!IsUserRobot(message.GetHistoryUserId())) {
            result.insert(message.GetHistoryUserId());
        }
    }
    return result;
}

bool IChatRobotImpl::RefreshEngine(const TInstant actuality) const {
    auto eg = NDrive::BuildEventGuard("refresh_cache");
    return ChatEngine.RefreshCache(actuality);
}

bool IChatRobotImpl::GetOrCreateChatRoom(const TString& userId, const TString& topic, ui32& chatRoomId, NDrive::TEntitySession* sessionExternal, const TString& titleOverride) const {
    if (GetChatRoom(userId, topic, chatRoomId, sessionExternal)) {
        return true;
    }
    auto eg = NDrive::BuildEventGuard("robot_create_chat_room");
    TChat chatRoom;
    chatRoom.SetSearchId(GetChatSearchId(userId, topic));
    chatRoom.SetTitle(titleOverride ? titleOverride : GetChatTitle(userId, topic));
    chatRoom.SetHandlerChatId(ChatConfig.GetChatId());
    chatRoom.SetTopic(topic);
    chatRoom.SetObjectId(userId);
    chatRoom.SetIsArchive(false);
    TVector<TChat> addedChats;
    if (!sessionExternal) {
        auto session = BuildChatEngineSession();
        if (!ChatEngine.GetChats().Upsert(chatRoom, "robot-frontend", session, &addedChats) || !session.Commit() || addedChats.size() != 1) {
            ERROR_LOG << "ChatRobotImpl::GetOrCreateChatRoom: session failure: " << session.GetStringReport() << Endl;
            session.ClearErrors();
            return false;
        }
    } else {
        if (!ChatEngine.GetChats().Upsert(chatRoom, "robot-frontend", *sessionExternal, &addedChats) || addedChats.size() != 1) {
            return false;
        }
    }
    chatRoomId = addedChats.front().GetId();
    return true;
}

bool IChatRobotImpl::DeleteChatRoom(const TString& userId, const TString& topic, const TString& operatorId, NDrive::TEntitySession& session) const {
    auto searchId = GetChatSearchId(userId, topic);
    return ChatEngine.GetChats().Remove({searchId}, operatorId, session);
    return true;
}

bool IChatRobotImpl::DoRefresh(const TInstant /*actuality*/) const {
    return true;
}

TString IChatRobotImpl::FormatPoints(const ui32 amount) const {
    if (amount % 10 == 1 && (amount % 100) != 11) {
        return ToString(amount) + " балл";
    }
    if (amount % 10 > 1 && amount % 10 < 5 && (amount % 100) / 10 != 1) {
        return ToString(amount) + " балла";
    }
    return ToString(amount) + " баллов";
}

NDrive::NChat::TMessage IChatRobotImpl::CreatePlaintextMessage(const TString& text) const {
    NDrive::NChat::TMessage result;
    result.SetType(TMessage::EMessageType::Plaintext);
    result.SetText(text);
    return result;
}

bool IChatRobotImpl::GetInferredMessages(const TPreActionMessage& message, const TString& userId, const TString& topic, TVector<TMessage>& messages, NDrive::TEntitySession& /*tx*/, const TMaybe<TChatContext>& chatContext) const {
    auto locale = ChatConfig.GetLocale();
    messages = TVector<TMessage>({message});
    messages.front().SetText(UnescapeMacroses(locale, message.GetText(), userId, topic, NDrive::NChat::TMessage::OnSendMacroPrefix, NDrive::NChat::TMessage::OnSendMacroSuffix, chatContext.Get()));
    if (message.GetText() == "$random" && message.GetChoices().size() > 0) {
        messages.front().SetText(message.GetChoices()[rand() % message.GetChoices().size()]);
    }
    return true;
}

bool IChatRobotImpl::SendPreActionMessages(const ui32 chatRoomId, const TString& chatItemId, NDrive::TEntitySession& session, const TString& operatorId, const TMaybe<TChatContext>& chatContext, const TMaybe<ui64> lastMessageId) const {
    auto eg = NDrive::BuildEventGuard("send_pre_action");
    TChatRobotScriptItem scriptItem;
    if (!ChatConfig.GetChatScript().GetScriptItemById(chatItemId, scriptItem)) {
        session.SetErrorInfo("ChatRobotImpl::SendPreActionMessages", "cannot GetScriptItemById: " + chatItemId);
        return false;
    }

    auto chat = Server->GetChatEngine()->GetChats().GetChatById(chatRoomId, session.GetTransaction());
    if (!chat) {
        session.SetErrorInfo("ChatRobotImpl::SendPreActionMessages", "cannot GetChatById: " + ToString(chatRoomId));
        return false;
    }

    bool firstMessage = true;
    for (auto&& preMessage : scriptItem.GetPreActionMessages()) {
        TVector<TMessage> messages;
        if (!GetInferredMessages(preMessage, chat->GetObjectId(), chat->GetTopic(), messages, session, chatContext)) {
            session.AddErrorMessage("ChatRobotImpl::SendPreActionMessages", "cannot GetInferredMessages");
            return false;
        }
        for (auto&& newMessage : messages) {
            newMessage.SetChatId(chatRoomId);
            newMessage.SetAuthorName(ChatConfig.GetRobotName());
            if (!newMessage.HasIsRateable() && newMessage.GetType() == NDrive::NChat::IMessage::EMessageType::Plaintext) {
                newMessage.SetIsRateable(ChatConfig.GetChatScript().HasRatings());
            }
            if (lastMessageId && firstMessage) {
                if (!ChatEngine.AddMessageIfLast(newMessage, operatorId, session, chat->GetSearchId(), *lastMessageId)) {
                    session.AddErrorMessage("ChatRobotImpl::SendPreActionMessages", "cannot AddMessageIfLast");
                    return false;
                }
            } else if (!ChatEngine.AddMessage(newMessage, operatorId, session, chat->GetSearchId())) {
                session.AddErrorMessage("ChatRobotImpl::SendPreActionMessages", "cannot AddMessage");
                return false;
            }
            firstMessage = false;
        }
    }
    return true;
}

NDrive::TEntitySession IChatRobotImpl::BuildChatEngineSession(const bool readOnly) const {
    return ChatEngine.BuildSession(readOnly);
}

TString IChatRobotImpl::GetChatTitle(const TString& userId, const TString& topic) const {
    auto searchId = GetChatSearchId(userId, topic);
    TChat chat;
    auto isChatFetched = ChatEngine.GetChats().GetChatFromCache(searchId, chat, false);
    return isChatFetched ? chat.GetTitle() : ChatConfig.GetTitle();
}

bool IChatRobotImpl::OverrideTitle(const TString& userId, const TString& topic, const TString& newTitle, NDrive::TEntitySession& session) const {
    NDrive::NChat::TChat chatRoom;
    auto searchId = GetChatSearchId(userId, topic);
    if (!ChatEngine.GetChats().GetChatFromCache(searchId, chatRoom, true, session.GetTransaction())) {
        return false;
    }
    chatRoom.SetTitle(newTitle);
    return ChatEngine.GetChats().Upsert(chatRoom, userId, session);
}

TString IChatRobotImpl::GetChatTitle(const NDrive::NChat::TChat& chat) const {
    return chat.GetTitle();
}

TString IChatRobotImpl::GetChatIcon(const NDrive::NChat::TChat& /*chat*/) const {
    return "https://carsharing.s3.yandex.net/drive/static/chat/chat-icon.png";
}


TMaybe<ui64> IChatRobotImpl::GetFirstEventIdByTime(const TString& userId, const TString& topic, const TInstant since, const bool isGoingBack, NDrive::TEntitySession& session) const {
    ui32 chatRoomId;
    if (!EnsureChat(userId, topic, session, true) || !GetOrCreateChatRoom(userId, topic, chatRoomId)) {
        return false;
    }

    auto messages = ChatEngine.GetMessagesSinceId(chatRoomId, Max<ui32>(), 0, session);
    if (!messages) {
        return false;
    }

    ui64 lastIdBefore = 0;
    ui64 firstIdAfter = 0;

    for (auto&& message : *messages) {
        if (message.GetHistoryInstant() == since) {
            lastIdBefore = message.GetHistoryEventId();
            firstIdAfter = message.GetHistoryEventId();
        } else if (message.GetHistoryInstant() < since) {
            lastIdBefore = message.GetHistoryEventId();
        } else if (!firstIdAfter) {
            firstIdAfter = message.GetHistoryEventId();
        }
    }

    if (!firstIdAfter) {
        firstIdAfter = lastIdBefore;
    }

    return isGoingBack ? lastIdBefore : firstIdAfter;
}

bool IChatRobotImpl::GetChat(const TString& userId, const TString& topic, NDrive::NChat::TChat& chat, const bool fallback, NStorage::ITransaction::TPtr transactionExt) const {
    auto searchId = GetChatSearchId(userId, topic);
    return ChatEngine.GetChats().GetChatFromCache(searchId, chat, fallback, transactionExt);
}

TString IChatRobotImpl::GetFaqUrl() const {
    return ChatConfig.GetChatScript().GetFaqUrl();
}

TString IChatRobotImpl::GetSupportUrl() const {
    return ChatConfig.GetChatScript().GetSupportUrl();
}

const TChatRobotConfig& IChatRobotImpl::GetChatConfig() const {
    return ChatConfig;
}

bool IChatRobotImpl::IsExistsForUser(const TString& userId, const TString& topic) const {
    TChat chat;
    return GetChat(userId, topic, chat);
}

bool IChatRobotImpl::DeduceChatContinuation(const TString& /*userId*/, const TString& /*topic*/, const ui32 /*chatRoomId*/, NDrive::TEntitySession& /*session*/) const {
    return false;
}

bool IChatRobotImpl::UpdateLastViewedMessageId(const TString& userId, const TString& topic, const TString& viewerId, const ui64 sentMessageId, NDrive::TEntitySession& session) const {
    auto eg = NDrive::BuildEventGuard("update_last_viewewed_message");
    ui32 chatRoomId;
    if (!EnsureChat(userId, topic, session, true) || !GetOrCreateChatRoom(userId, topic, chatRoomId)) {
        return false;
    }
    return MessageViewTracker->UpdateLastViewedMessageId(viewerId, chatRoomId, sentMessageId, session);
}

bool IChatRobotImpl::SendArbitraryMessage(const TString& userId, const TString& topic, const TString& operatorId, TMessage& message, NDrive::TEntitySession& session, const TChatContext* externalContext) const {
    if (message.GetAuthorName().empty() && userId != operatorId) {
        if (IsUserRobot(operatorId)) {
            message.SetAuthorName(ChatConfig.GetRobotName());
        } else if (externalContext) {
            message.SetAuthorName(externalContext->GetOperatorName());
        }
    }
    ui32 chatRoomId;
    if (!EnsureChat(userId, topic, session, false) || !GetChatRoom(userId, topic, chatRoomId, &session)) {
        return false;
    }
    message.SetChatId(chatRoomId);
    return ChatEngine.AddMessage(message, operatorId, session, GetChatSearchId(userId, topic));
}

bool IChatRobotImpl::AcceptMessages(const TString& userId, const TString& topic, TVector<TMessage>& messages, const TString& operatorId, NDrive::TEntitySession& session) const {
    ui32 chatRoomId;
    if (!EnsureChat(userId, topic, session, false) || !GetChatRoom(userId, topic, chatRoomId, &session)) {
        return false;
    }
    for (auto&& message : messages) {
        message.SetChatId(chatRoomId);
        if (!ChatEngine.AddMessage(message, operatorId, session, GetChatSearchId(userId, topic))) {
            return false;
        }
    }
    return true;
}

bool IChatRobotImpl::EditMessage(const TString& operatorId, const ui64 messageId, const NDrive::NChat::TMessageEdit messageEdit, NDrive::TEntitySession& session) const {
    auto messageMaybe = ChatEngine.GetMessage(messageId, session);
    if (!messageMaybe || !messageMaybe.GetRef()) {
        if (messageMaybe) {
            session.SetErrorInfo("EditMessage", "cannot find message with id " + ToString(messageId));
        }
        return false;
    }
    NDrive::NChat::TMessage& message = *(messageMaybe.GetRef());
    messageEdit.Apply(message);
    return ChatEngine.EditMessage(messageId, message, operatorId, session);
}

bool IChatRobotImpl::DeleteMessage(const TString& operatorId, const ui64 messageId, NDrive::TEntitySession& session) const {
    auto message = ChatEngine.GetMessage(messageId, session);
    if (!message || !message.GetRef()) {
        if (message) {
            session.SetErrorInfo("DeleteMessage", "cannot find message with id " + ToString(messageId));
        }
        return false;
    }
    return ChatEngine.DeleteMessage(messageId, (*message)->GetChatId(), operatorId, session);
}

bool IChatRobotImpl::GetLastMessage(const ui32& chatRoomId, NDrive::TEntitySession& session, const ui32 messageTraits, TMaybe<NDrive::NChat::TMessageEvent>& result, const TString& messageFromUserId) const {
    if ((ChatConfig.IsCommon() || ChatConfig.GetIsStatic()) && !chatRoomId) {
        if (messageFromUserId && !IsUserRobot(messageFromUserId)) {
            result = Nothing();
            return true;
        }
        TChatRobotScriptItem introItem;
        NDrive::NChat::TMessage lastMessage;
        if (ChatConfig.GetChatScript().GetScriptItemById(ChatConfig.GetInitialStepId(), introItem) && !introItem.GetPreActionMessages().empty()) {
            lastMessage = introItem.GetPreActionMessages().back();
        } else {
            result = Nothing();
            return true;
        }
        TString messageActor = messageFromUserId.empty() ? "robot-frontend" : messageFromUserId;
        NDrive::NChat::TMessageEvent message = TObjectEvent<NDrive::NChat::TMessage>(lastMessage, EObjectHistoryAction::Add, ChatConfig.GetCommonInstant(), messageActor, session.GetOriginatorId(), "");
        message.SetHistoryEventId(0);
        result = message;
        return true;
    }
    return ChatEngine.GetLastMessage(chatRoomId, messageTraits, session, result, messageFromUserId);
}

bool IChatRobotImpl::GetLastMessage(const TString& chatUserId, const TString& topic, NDrive::TEntitySession& session, const ui32 messageTraits, TMaybe<NDrive::NChat::TMessageEvent>& result, const TString& messageFromUserId) const {
    ui32 chatRoomId;
    if (!GetChatRoom(chatUserId, topic, chatRoomId)) {
        return {};
    }
    return GetLastMessage(chatRoomId, session, messageTraits, result, messageFromUserId);
}

NDrive::NChat::TOptionalMessageEvents IChatRobotImpl::GetUnreadMessages(const ui32 chatId, const TString& userId, const ui32 viewerTraits, NDrive::TEntitySession& session) const {
    size_t lastReadMessageId = MessageViewTracker->GetLastViewedMessageId(userId, chatId);
    return ChatEngine.GetMessagesSinceId(chatId, viewerTraits, lastReadMessageId + 1, session);
}

size_t IChatRobotImpl::GetIntroSize() const {
    TChatRobotScriptItem introItem;
    return ChatConfig.GetChatScript().GetScriptItemById(ChatConfig.GetInitialStepId(), introItem) ? introItem.GetPreActionMessages().size() : 0;
}

TMaybe<size_t> IChatRobotImpl::GetUnreadMessagesCount(const ui32 chatId, const TString& userId, const ui32 viewerTraits, NDrive::TEntitySession& session) const {
    if (ChatConfig.IsCommon() && !chatId) {
        return GetIntroSize();
    }
    size_t lastReadMessageId = MessageViewTracker->GetLastViewedMessageId(userId, chatId);
    return ChatEngine.GetMessagesCountSince(chatId, viewerTraits, lastReadMessageId + 1, session, /*excludeTypes = */  { NDrive::NChat::TMessage::EMessageType::Separator });
}

TMaybe<size_t> IChatRobotImpl::GetMessagesCount(const ui32 chatId, const ui32 viewerTraits, NDrive::TEntitySession& session) const {
    if (ChatConfig.IsCommon() && !chatId) {
        return GetIntroSize();
    }
    return ChatEngine.GetMessagesCount(chatId, viewerTraits, session, /*excludeTypes = */ { NDrive::NChat::TMessage::EMessageType::Separator });
}

TMaybe<size_t> IChatRobotImpl::GetMessagesCount(const TString& userId, const TString& topic, const NDrive::NChat::TMessage::EMessageType type, NDrive::TEntitySession& session) const {
    TChat existingChat;
    if (ChatConfig.IsCommon() && !GetChat(userId, topic, existingChat, false)) {
        return GetIntroSize();
    }
    ui32 chatRoomId;
    if (!GetChatRoom(userId, topic, chatRoomId, &session)) {
        return Nothing();
    }
    return ChatEngine.GetMessagesCount(chatRoomId, 0, session, /*excludeTypes = */ {}, /*includeTypes = */ TSet<TMessage::EMessageType>{ type });
}

NDrive::NChat::TOptionalMessageEvents IChatRobotImpl::GetChatMessagesRange(const TString& userId, const TString& topic, const ui32 viewerTraits, const ui64 fromId, const ui64 untilId, NDrive::TEntitySession& session) const {
    ui32 chatRoomId;
    if (!EnsureChat(userId, topic, session, true) || !GetOrCreateChatRoom(userId, topic, chatRoomId)) {
        return {};
    }
    return ChatEngine.GetMessagesRange(chatRoomId, viewerTraits, session, fromId, untilId);
}

NDrive::NChat::TOptionalMessageEvents IChatRobotImpl::GetChatMessages(const TString& userId, const TString& topic, NDrive::TEntitySession& session, const ui32 viewerTraits, const ui64 startId) const {
    ui32 chatRoomId;
    if (!EnsureChat(userId, topic, session, true) || !GetOrCreateChatRoom(userId, topic, chatRoomId)) {
        return {};
    }
    return ChatEngine.GetMessagesSinceId(chatRoomId, viewerTraits, startId, session);
}

NDrive::NChat::TOptionalMessageEvents IChatRobotImpl::GetChatMessagesSinceTimestamp(const TString& userId, const TString& topic, const TInstant startTs, NDrive::TEntitySession& session, const ui32 viewerTraits) const {
    ui32 chatRoomId;
    if (!EnsureChat(userId, topic, session, true) || !GetOrCreateChatRoom(userId, topic, chatRoomId)) {
        return {};
    }
    return ChatEngine.GetMessagesSinceTimestamp(chatRoomId, viewerTraits, startTs, session);
}

NDrive::NChat::TExpectedMessageEvents IChatRobotImpl::GetCachedMessages(const TString& searchId) const {
    return ChatEngine.GetCachedMessages(searchId);
}

bool IChatRobotImpl::UpdateCachedMessages(const TSet<TString>& searchIds, NDrive::TEntitySession& session, const TInstant& requestTime) const {
    return ChatEngine.UpdateCachedMessages(searchIds, session, requestTime);
}

bool IChatRobotImpl::MarkMessagesRead(const NDrive::NChat::TMessageEvent& lastMessage, const TString& userId, NDrive::TEntitySession& session) const {
    auto eg = NDrive::BuildEventGuard("mark_messages_read");
    return MessageViewTracker->UpdateLastViewedMessageId(userId, lastMessage.GetChatId(), NDrive::NChat::TChatMessagesHistoryManager::GetMessageId(lastMessage), session);
}

TMaybe<NDrive::NChat::TChatMessages> IChatRobotImpl::GetUserChatsMessages(const TVector<TChat>& chats, NDrive::TEntitySession& session, const TRange<ui64>& idRange, const TRange<TInstant>& timestampRange, const ui32 viewerTraits) const {
    return ChatEngine.GetMessagesByChats(chats, idRange, timestampRange, viewerTraits, session);
}

bool IChatRobotImpl::UnescapePromoMacroses(TString& rawText, const TString& userId, const TString& topic, NDrive::TEntitySession& session, const TString& prefix, const TString& suffix) const {
    if (!Server->GetPromoCodesManager()) {
        session.SetErrorInfo("UnescapePromoMacroses", "PromoCodesManager does not exist", EDriveSessionResult::InternalError);
        return false;
    }

    SubstGlobal(rawText, prefix + "promo_code" + suffix, topic);
    TVector<TString> contextTags;
    {
        auto metaInfos = Server->GetPromoCodesManager()->GetMetaByCode(topic, session);
        if (!metaInfos) {
            ERROR_LOG << "cannot get promo code meta " << userId << ": " << session.GetStringReport() << Endl;
            return false;
        }
        if (!metaInfos->empty() && metaInfos->front().GetTagDictionaryContext().HasTagName()) {
            contextTags.emplace_back(metaInfos->front().GetTagDictionaryContext().GetTagNameUnsafe());
        }

        TAtomicSharedPtr<IPromoProfitBase> type;
        if (!Server->GetPromoCodesManager()->GetPromoProfit(topic, type, session)) {
            ERROR_LOG << "cannot get promo profit " << userId << ": " << session.GetStringReport() << Endl;
            return false;
        }
        if (type) {
            auto promoProfit = dynamic_cast<IPromoTagProfit*>(type.Get());
            if (promoProfit && promoProfit->GetTagName()) {
                SubstGlobal(rawText, prefix + "promo_dictionary_tag" + suffix, promoProfit->GetTagName());
                contextTags.emplace_back(promoProfit->GetTagName());
            }

            if (type->GetTagDictionaryContext().HasTagName()) {
                contextTags.emplace_back(type->GetTagDictionaryContext().GetTagNameUnsafe());
            }
        }
    }

    const auto& userTagManager = Server->GetDriveAPI()->GetTagsManager().GetUserTags();
    auto optionalUser = userTagManager.RestoreObject(userId, session);
    if (!optionalUser) {
        ERROR_LOG << "cannot RestoreObject " << userId << ": " << session.GetStringReport() << Endl;
        return false;
    }

    TMap<TString, TString> fieldsMap;
    for (const auto& tagName : contextTags) {
        TConstDBTag dictionaryTag;
        {
            if (optionalUser) {
                for (auto&& tag : optionalUser->GetTags()) {
                    if (tag->GetName() == tagName) {
                        dictionaryTag = tag;
                        break;
                    }
                }
            }
            if (!dictionaryTag.HasData()) {
                TVector<TDBTag> tags;
                bool restored = Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, { tagName }, tags, session);
                if (!restored) {
                    ERROR_LOG << "ChatRobotImpl::UnescapePromoMacroses: cannot restore tags " << session.GetStringReport() << Endl;
                    return false;
                }
                if (tags.size() == 1) {
                    dictionaryTag = tags.front();
                }
            }
        }
        if (!dictionaryTag.HasData()) {
            continue;
        }

        auto tagImpl = dictionaryTag.GetTagAs<TUserDictionaryTag>();
        if (!tagImpl) {
            continue;
        }

        for (auto&& f : tagImpl->GetFields()) {
            fieldsMap.emplace(f.Key, f.Value);
        }

        auto description = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(tagName);
        if (!description) {
            continue;
        }
        const TDictionaryTagDescription* descriptionImpl = dynamic_cast<const TDictionaryTagDescription*>(description.Get());
        if (!descriptionImpl) {
            continue;
        }
        for (auto&& f : descriptionImpl->GetFields()) {
            fieldsMap.emplace(f.Id, f.DefaultValue);
        }
    }

    TVector<std::pair<TString, TString>> fields;
    for (auto&&[k, v] : fieldsMap) {
        fields.emplace_back(std::make_pair(k, v));
    }

    Sort(fields.begin(), fields.end(), [](const auto& lhs, const auto& rhs) {return lhs.first.size() > rhs.first.size(); });
    for (auto&&[k, v] : fields) {
        SubstGlobal(rawText, prefix + "promo_dictionary." + k + suffix, v);
    }
    return true;
}

TString IChatRobotImpl::UnescapeMacroses(ELocalization locale, const TString& rawText, const TString& userId, const TString& topic, const TString& prefix, const TString& suffix, const TChatContext* context) const {
    if (rawText.find(prefix) == TString::npos) {
        return rawText;
    }

    TString result = rawText;
    auto localization = Server->GetLocalization();
    if (localization) {
        auto startMarker = prefix + "(resource:";
        auto finishMarker = ")" + suffix;
        while (true) {
            auto startMarkerPosition = result.find(startMarker);
            if (startMarkerPosition == TString::npos) {
                break;
            }
            auto keyStartPosition = startMarkerPosition + startMarker.size();
            auto keyFinishPosition = result.find(finishMarker, keyStartPosition);
            if (keyFinishPosition == TString::npos) {
                break;
            }
            auto key = TString(result.begin() + keyStartPosition, result.begin() + keyFinishPosition);
            auto value = localization->GetLocalString(locale, key);
            result.replace(startMarkerPosition, startMarker.size() + key.size() + finishMarker.size(), value);
        }
    }
    if (context) {
        for (auto&& [key, value] : context->GetContextMap()) {
            SubstGlobal(result, prefix + key + suffix, value);
        }
    }
    auto userPtr = Server->GetDriveAPI()->GetUsersData()->GetCachedObject(userId);
    {
        TString displayName = userPtr ? userPtr->GetDisplayName() : "";
        SubstGlobal(result, prefix + "name" + suffix, displayName);
    }
    {
        TString displayPhone = userPtr ? userPtr->GetPhone() : "";
        SubstGlobal(result, prefix + "phone" + suffix, displayPhone);
    }
    {
        TString displayEmail = userPtr ? userPtr->GetEmail() : "";
        SubstGlobal(result, prefix + "email" + suffix, displayEmail);
    }
    {
        TString login = userPtr ? userPtr->GetLogin() : "";
        SubstGlobal(result, prefix + "login" + suffix, login);
    }
    {
        SubstGlobal(result, prefix + "current_user" + suffix, userId);
    }

    {
        const auto now = ModelingNow();
        auto replaceTime = [&](const TString& key, std::function<TString(TInstant)> format) {
            const TString startPattern = prefix + key;
            size_t startPos = result.find(startPattern);
            while (startPos < result.size()) {
                size_t endPos = result.find(suffix, startPos);
                TDuration delta;
                if (endPos != TString::npos && (endPos - startPos == startPattern.size() || TDuration::TryParse(result.substr(startPos + startPattern.size() + 1, endPos - (startPos + startPattern.size() + 1)), delta))) {
                    result = result.replace(startPos, (endPos - startPos) + suffix.size(), format(now + delta));
                    startPos = result.find(startPattern);
                } else {
                    startPos = result.find(startPattern, startPos + 1);
                }
            }
        };
        replaceTime("current_time", [&](TInstant time) -> TString {
            return !!Server->GetLocalization() ? Server->GetLocalization()->FormatInstant(locale, time) : ::ToString(time);
        });
        replaceTime("current_timestamp", [&](TInstant time) -> TString {
            return ::ToString(time.Seconds());
        });
    }

    TVector<ISession::TConstPtr> userSessions;
    TString carId;
    if (Server->GetDriveAPI()->GetCurrentUserSessions(userId, userSessions, TInstant::Zero()) && userSessions.size() == 1) {
        if (userSessions.front()->GetObjectId()) {
            carId = userSessions.front()->GetObjectId();
            SubstGlobal(result, prefix + "current_session" + suffix, userSessions.front()->GetSessionId());
        }
    }
    if (carId) {
        SubstGlobal(result, prefix + "current_car" + suffix, carId);
        TMaybe<TDriveCarInfo> carObject = Server->GetDriveAPI()->GetCarsData()->GetObject(carId);
        if (carObject) {
            TString modelCode = carObject->GetModel();
            auto modelFetchResult = Server->GetDriveAPI()->GetModelsData()->GetCached(modelCode);
            auto model = modelFetchResult.GetResultPtr(modelCode);
            if (!!model) {
                SubstGlobal(result, prefix + "current_model_hr" + suffix, model->GetName());
            }
            SubstGlobal(result, prefix + "current_model" + suffix, modelCode);
            SubstGlobal(result, prefix + "current_number" + suffix, carObject->GetNumber());
        }
    }

    if (context && context->GetSessionId() && localization && result.find(prefix + "offer.") != TString::npos) {
        TAtomicSharedPtr<const ISession> userSession;
        if (Server->GetDriveAPI()->GetUserSession(userId, userSession, context->GetSessionId(), TInstant::Zero()) && userSession) {
            const IEventsSession<TCarTagHistoryEvent>* eventsSession = dynamic_cast<const IEventsSession<TCarTagHistoryEvent>*>(userSession.Get());
            if (eventsSession) {
                if (auto compilation = eventsSession->GetCompilationAs<TBillingSession::TBillingCompilation>()) {
                    auto offer = compilation->GetCurrentOffer();
                    if (offer) {
                        result = offer->FormDescriptionElement(result, locale, localization);
                    }
                    auto state = compilation->GetCurrentOfferState();
                    if (state && offer) {
                        result = state->FormDescriptionElement(result, offer->GetCurrency(), locale, *localization);
                    }
                }
            }
        }
        SubstGlobal(result, prefix + "offer.", "");
    }

    auto session = Server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
    if (result.find(prefix + "referral_code") != TString::npos) {
        TString promoCode = "";
        auto permissions = Server->GetDriveAPI()->GetUserPermissions(userId, TUserPermissionsFeatures());
        if (auto permissions = Server->GetDriveAPI()->GetUserPermissions(userId, TUserPermissionsFeatures())) {
            if (!Server->GetDriveAPI()->CheckReferralProgramParticipation(promoCode, *permissions, session)) {
                ERROR_LOG << "Cannot get referral code: " << session.GetStringReport() << Endl;
            }
        }
        SubstGlobal(result, prefix + "referral_code" + suffix, promoCode);
    }
    if (result.find(prefix + "promo_dictionary.") != TString::npos || result.find(prefix + "promo_dictionary_tag" + suffix) != TString::npos || result.find(prefix + "promo_code") != TString::npos) {
        if (!UnescapePromoMacroses(result, userId, topic, session, prefix, suffix)) {
            ERROR_LOG << "Cannot unescape promo macroses: " << session.GetStringReport() << Endl;
            return result;
        }
    }
    if (result.find(prefix + "promocode.") != TString::npos) {
        auto links = DriveAPI.GetBillingManager().GetPromocodeAccountLinks().GetAccountsByPromocodesFromDB({ topic }, session);
        if (links && links->size()) {
            SubstGlobal(result, prefix + "promocode.account_id" + suffix, ::ToString(links->front().GetAccountId()));
            if (result.find(prefix + "promocode.") != TString::npos) {
                auto account = DriveAPI.GetBillingManager().GetAccountsManager().GetAccountById(links->front().GetAccountId());
                if (account) {
                    SubstGlobal(result, prefix + "promocode.wallet_name" + suffix, account->GetName());
                    SubstGlobal(result, prefix + "promocode.wallet_id" + suffix, account->GetUniqueName());
                }
            }
        }
    }
    return result;
}

NJson::TJsonValue IChatRobotImpl::GetMessageReport(ELocalization locale, const NDrive::NChat::TMessageEvent& message, const TString& userId, const TString& topic) const {
    NJson::TJsonValue messageReport;
    messageReport["id"] = message.GetHistoryEventId();
    messageReport["timestamp"] = message.GetHistoryInstant().Seconds();
    messageReport["type"] = ToString(message.GetType());
    messageReport["author"] = message.GetHistoryUserId();
    messageReport["is_rateable"] = message.GetIsRateableDef(false);
    if (auto icon = message.GetIcon()) {
        messageReport["icon"] = std::move(icon);
    }
    if (auto link = message.GetLink()) {
        messageReport["link"] = UnescapeMacroses(locale, link, userId, topic, NDrive::NChat::TMessage::OnViewMacroPrefix, NDrive::NChat::TMessage::OnViewMacroSuffix);
    }
    if (message.GetTraits()) {
        messageReport["traits"] = NJson::JSON_ARRAY;
        for (auto&& trait : GetEnumNames<NDrive::NChat::TMessage::EMessageTraits>()) {
            if ((ui32)trait.first & message.GetTraits()) {
                messageReport["traits"].AppendValue(trait.second);
            }
        }
    }
    if (message.GetAuthorName() && Server->GetSettings().GetValueDef<bool>(ChatConfig.GetChatId() + ".report_author_name", false)) {
        messageReport["author_name"] = message.GetAuthorName();
    }
    if (message.GetExternalId()) {
        messageReport["external_id"] = message.GetExternalId();
    }
    if (message.GetRating()) {
        messageReport["rating"] = message.GetRating();
    }
    if (message.GetExternalStatus() != NDrive::NChat::TMessage::EExternalStatus::NotExternal) {
        messageReport["external_status"] = ToString(message.GetExternalStatus());
    }
    if (message.IsUnescapeable()) {
        messageReport["text"] = UnescapeMacroses(locale, message.GetText(), userId, topic, NDrive::NChat::TMessage::OnViewMacroPrefix, NDrive::NChat::TMessage::OnViewMacroSuffix);
    } else {
        messageReport["text"] = message.GetText();
    }

    if (message.GetType() == NDrive::NChat::TMessage::EMessageType::UserDocumentPhoto || message.GetType() == NDrive::NChat::TMessage::EMessageType::UserDocumentPhotos) {
        auto tokens = SplitString(message.GetText(), ",");
        const auto& udpManager = DriveAPI.GetDocumentPhotosManager();
        TString messageClean = "";
        for (auto&& token : tokens) {
            if (messageClean) {
                messageClean += ",";
            }
            messageClean += udpManager.BuildPhotoAccessPath(token);
        }
        messageReport["text"] = messageClean;
    } else if (message.GetType() == NDrive::NChat::TMessage::EMessageType::Bonus) {
        messageReport["bg_link"] = ChatConfig.GetTheme().GetBonusBackground().GetPath();
        messageReport["bg_width"] = ChatConfig.GetTheme().GetBonusBackground().GetWidth();
        messageReport["bg_height"] = ChatConfig.GetTheme().GetBonusBackground().GetHeight();
        ui32 bonusAmount;
        if (TryFromString(message.GetText(), bonusAmount)) {
            messageReport["text"] = "[color=#FFFFFF][b=26]" + message.GetText() + "[/b][r=12]\n" + NDrive::TLocalization::BonusRubles(bonusAmount) + "[/r][/color]";
        }
    } else if (message.GetType() == NDrive::NChat::TMessage::EMessageType::Introscreen) {
        TUtf16String publicName = CharToWide(userId, CODES_UTF8);

        auto userPtr = Server->GetDriveAPI()->GetUsersData()->GetCachedObject(userId);
        if (userPtr) {
            TUtf16String publicName = CharToWide(userPtr->GetFirstName(), CODES_UTF8);
            if (!publicName) {
                publicName = CharToWide(userPtr->GetLogin(), CODES_UTF8);
            }
        }

        NJsonWriter::TBuf writer;
        writer.WriteString("<name>", NJsonWriter::HEM_DONT_ESCAPE_HTML);
        TUtf16String fs = CharToWide(writer.Str(), CODES_UTF8);
        fs = fs.substr(1, fs.size() - 2);

        // Get landing
        TMaybe<TLanding> landing = Server->GetDriveAPI()->GetLandingsDB()->GetObject(message.GetText());
        if (landing) {
            TUtf16String text = CharToWide(landing->GetJsonLanding().GetStringRobust(), CODES_UTF8);
            for (size_t pos = text.find(fs); pos != TString::npos; pos = text.find(fs, pos)) {
                text = text.replace(pos, fs.size(), publicName);
                pos = pos + (publicName.size());
            }

            const TString textForJson = WideToChar(text, CODES_UTF8);
            const TString localizedTextForJson = Server->GetLocalization()->ApplyResourcesForJson(textForJson, locale);

            NJson::TJsonValue jsonNew;
            if (!NJson::ReadJsonFastTree(localizedTextForJson, &jsonNew)) {
                ERROR_LOG << "Cannot parse landing: " << WideToChar(text, CODES_UTF8) << Endl;
                return messageReport;
            }

            landing->SetJsonLanding(std::move(jsonNew));
            messageReport["landing"] = landing->GetPublicReport(locale);
        }
    } else if (message.GetType() == NDrive::NChat::TMessage::EMessageType::FeedbackDialogue) {
        messageReport["type"] = ToString(NDrive::NChat::TMessage::EMessageType::Plaintext);
        messageReport["bg_color"] = ChatConfig.GetTheme().GetFeedbackBackgroundColor();
    } else if (message.GetType() == NDrive::NChat::TMessage::EMessageType::ColorSeparator) {
        messageReport["type"] = ToString(NDrive::NChat::TMessage::EMessageType::Separator);
        messageReport["bg_color"] = ChatConfig.GetTheme().GetColorSeparatorBackgroundColor();
    } else if (message.GetType() == NDrive::NChat::TMessage::EMessageType::Sticker) {
        auto stickerManager = ChatEngine.GetStickerManager();
        NDrive::NChat::TSticker sticker;
        if (stickerManager && stickerManager->GetSticker(message.GetText(), sticker)) {
            messageReport["emoji"] = sticker.GetEmoji();
            messageReport["text"] = sticker.GetUrl();
        }
    }

    return messageReport;
}

bool IChatRobotImpl::Refresh(const TInstant actuality) const {
    if (!ChatEngine.RefreshCache(actuality)) {
        return false;
    }
    return DoRefresh(actuality);
}

TVector<NDrive::NChat::TChat> IChatRobotImpl::GetTopics(IChatUserContext::TPtr ctx, const bool onlyActive) const {
    if (ChatConfig.IsCommon()) {
        if (!ChatConfig.GetCommonShardingPolicy().CheckMatching(ctx->GetUserId())) {
            TVector<NDrive::NChat::TChat> emptyResult;
            return emptyResult;
        }
        TChatContext emptyContext;
        if (ChatConfig.GetAuditoryCondition() && !ChatConfig.GetAuditoryCondition()->IsMatching(ctx, emptyContext)) {
            TVector<NDrive::NChat::TChat> emptyResult;
            return emptyResult;
        }
    }

    auto chats = ChatEngine.GetChats().GetCachedUserChats(ctx->GetUserId(), ChatConfig.GetChatId());

    TVector<NDrive::NChat::TChat> result;
    bool hadDefaultTopic = false;

    for (auto&& chat : chats) {
        if (chat.GetTopic() == "") {
            hadDefaultTopic = true;
        }
        if (onlyActive && chat.GetIsArchive()) {
            continue;
        }
        result.push_back(std::move(chat));
    }

    TChat defaultTopic;
    if (!hadDefaultTopic && GetChat(ctx->GetUserId(), "", defaultTopic, false)) {
        result.push_back(std::move(defaultTopic));
    }

    if (ChatConfig.IsCommon() && result.empty()) {
        TChat commonChat;
        commonChat.SetHandlerChatId(ChatConfig.GetChatId());
        commonChat.SetId(0);
        commonChat.SetIsArchive(false);
        commonChat.SetObjectId(ctx->GetUserId());
        commonChat.SetSearchId(GetChatSearchId(ctx->GetUserId(), ""));
        commonChat.SetTopic("");
        result.emplace_back(std::move(commonChat));
    }

    for (auto&& chat : result) {
        chat.SetTitle(GetChatTitle(chat));
        chat.SetIcon(GetChatIcon(chat));
    }

    return result;
}

bool IChatRobotImpl::DoPreEntryActions(const IChatUserContext::TPtr userContext, const TChatRobotScriptItem& item, TChatContext& context, NDrive::TEntitySession& tagsSession, NDrive::TEntitySession& chatsSession) const {
    for (auto&& action : item.GetOnEntryActions()) {
        if (!action->Perform(userContext, context, Server, tagsSession, chatsSession, item)) {
            return false;
        }
    }
    return true;
}

bool IChatRobotImpl::MoveToStep(const TString& userId, const TString& nodeId, const TString& topicLink, const NDrive::IServer& server, NDrive::TEntitySession& chatSession, TVector<TContextMapInfo> contextMap) {
    TString chatId, topic;
    IChatRobot::ParseTopicLink(topicLink, chatId, topic);
    if (chatId.empty()) {
        chatSession.SetErrorInfo("IChatRobotImpl::MoveToStep", "empty chat id in topic link " + topicLink);
        return false;
    }
    auto chatRobot = server.GetChatRobot(chatId);
    if (!chatRobot) {
        chatSession.SetErrorInfo("IChatRobotImpl::MoveToStep", "cannot find chat robot " + chatId);
        return false;
    }
    if (!chatRobot->MoveToStep(userId, topic, nodeId, &chatSession, true, "", chatRobot->GetChatConfig().GetRobotUserId(), {}, contextMap)) {
        return false;
    }
    return true;
}

TMaybe<TDBTag> IChatRobotImpl::GetTagByTopicLink(const TString& topicLink, const TString& userId, NDrive::TEntitySession& session, const NDrive::IServer& server) {
    auto tags = server.GetDriveAPI()->GetTagsManager().GetUserTags().RestoreEntityTags(userId, {}, session);
    TDBTag result;
    if (!tags) {
        return Nothing();
    }
    for (auto&& tag : *tags) {
        auto tagImpl = tag.GetTagAs<ITopicLinkOwner>();
        if (tagImpl && tagImpl->GetTopicLink() == topicLink) {
            return tag;
        }
    }
    return TDBTag();
}

bool IChatRobotImpl::IsUserRobot(const TString& userId) const {
    return userId.StartsWith("robot-") || userId == ChatConfig.GetRobotUserId();
}
