#pragma once

#include "ifaces.h"
#include "state_storage.h"
#include "view_tracker.h"

#include <drive/backend/chat_robots/configuration/chat_script.h>
#include <drive/backend/chat_robots/configuration/config.h>
#include <drive/backend/chat_robots/state/robot_state.pb.h>
#include <drive/backend/chat_robots/script_actions/script_action.h>

#include <drive/backend/chat/engine.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/logging/evlog.h>

#include <rtline/library/storage/abstract.h>
#include <rtline/util/json_processing.h>
#include <rtline/util/types/accessor.h>

class TChatUserContext : public IChatUserContext {
public:
    TChatUserContext() = default;
    TChatUserContext(const TUserPermissions::TPtr permissions, const NDrive::IServer* server);

    virtual void SpecifyGeo(const TGeoCoord& c) override;
    virtual const TString GetStatus() const override;
    virtual bool HasAction(const TString& action) const override;

    virtual IChatRobot::TPtr GetChatRobot() const override;

    virtual TString Unescape(TString pattern, const TChatContext& context) const override;
    virtual NJson::TJsonValue UnescapeJson(const NJson::TJsonValue& json, const TChatContext& context) const override;

    TUserPermissions::TPtr GetUserPermissions() const override {
        return Permissions;
    }

    virtual const NDrive::IServer& GetServer() const override {
        return *Server;
    }

private:
    const NDrive::IServer* Server;
    const TUserPermissions::TPtr Permissions;
};

class IChatRobotImpl : public IChatRobot {
protected:
    using TChat = NDrive::NChat::TChat;
    using TMessage = NDrive::NChat::TMessage;

protected:
    const NDrive::IServer* Server;
    const TDriveAPI& DriveAPI;
    const NDrive::NChat::TEngine& ChatEngine;
    const TChatRobotConfig ChatConfig;
    const TAtomicSharedPtr<TChatViewTracker> MessageViewTracker;

protected:
    size_t GetIntroSize() const;
    virtual bool GetChatRoom(const TString& userId, const TString& topic, ui32& chatRoomId, NDrive::TEntitySession* sessionExt = nullptr) const;
    virtual bool GetOrCreateChatRoom(const TString& userId, const TString& topic, ui32& chatRoomId, NDrive::TEntitySession* sessionExternal = nullptr, const TString& titleOverride = "") const;
    bool DeleteChatRoom(const TString& userId, const TString& topic, const TString& operatorId, NDrive::TEntitySession& session) const;

    virtual bool DoRefresh(const TInstant actuality) const;

    bool UnescapePromoMacroses(TString& rawText, const TString& userId, const TString& topic, NDrive::TEntitySession& session, const TString& prefix = "", const TString& suffix = "") const;

    // Game
    TString FormatPoints(const ui32 amount) const;
    TMessage CreatePlaintextMessage(const TString& text) const;

    bool GetInferredMessages(const TPreActionMessage& message, const TString& userId, const TString& topic, TVector<TMessage>& messages, NDrive::TEntitySession& tx, const TMaybe<TChatContext>& chatContext = {}) const;

public:
    IChatRobotImpl(const NDrive::IServer* server, const TChatRobotConfig& chatConfig, const TAtomicSharedPtr<TChatViewTracker> messageViewTracker)
        : Server(server)
        , DriveAPI(*server->GetDriveAPI())
        , ChatEngine(*server->GetChatEngine())
        , ChatConfig(chatConfig)
        , MessageViewTracker(messageViewTracker)
    {
    }

    virtual ~IChatRobotImpl() = default;

    virtual TMaybe<TVector<NDrive::NChat::TChat>> GetChats(const TSet<TString> searchIds, NDrive::TEntitySession& session) const override;
    virtual TMaybe<TVector<NDrive::NChat::TChat>> GetChats(const TString& chatId, const TString& topic, NDrive::TEntitySession& session) const override;
    bool EnsureChat(const TString& userId, const TString& topic, NDrive::TEntitySession& session, const bool isRead = false, TVector<TContextMapInfo> contextMap = {}) const override = 0;
    bool RemoveChat(const TString& userId, const TString& topic, const TString& operatorId) const override = 0;
            bool MoveToStep(const TString& userId, const TString& topic, const TString& stepName, NDrive::TEntitySession* sessionExternal = nullptr, const bool sendMessages = false, const TString& newChatTitle = "", const TString& operatorId = "robot-frontend", const TMaybe<ui64> lastMessageId = {}, TVector<TContextMapInfo> contextMap = {}) const override;
    virtual bool MoveToStep(const TString& userId, const TString& topic, const TString& stepName, NDrive::TEntitySession& session, const bool sendMessages = false, const TString& newChatTitle = "", const TString& operatorId = "robot-frontend", const TMaybe<ui64> lastMessageId = {}, TVector<TContextMapInfo> contextMap = {}) const = 0;
    bool GetIsChatCompleted(const TString& userId, const TString& topic, const TInstant actuality) const override = 0;
    bool GetIsChatInClosedNode(const TString& userId, const TString& topic, const TInstant actuality) const override = 0;
    bool GetCurrentScriptItem(const TString& userId, const TString& topic, TChatRobotScriptItem& result, const TInstant actuality) const override = 0;
    TMaybe<TNextActionInfo> GetFirstResubmitStep(const IChatUserContext::TPtr context, TChatContext& stateContext, NDrive::TEntitySession& chatSession, ui32 maskOverride) const override = 0;
    TMaybe<TNextActionInfo> GetNextResubmitStep(const IChatUserContext::TPtr context, const TChatRobotScriptItem& currentScriptItem, TChatContext& stateContext, NDrive::TEntitySession& chatSession) const override = 0;
    ui64 GetLastViewedMessageIdExcept(const TString& userId, const TString& topic, const TString& exceptUserId) const override = 0;
    ui64 GetLastViewedMessageId(const TString& userId, const TString& topic, const TString& viewerId) const override = 0;
    NJson::TJsonValue GetCurrentActionMessagesReport(ELocalization locale, const TString& userId, const TString& topic, const bool withMetaInfo = false, const TInstant timestampOverride = TInstant::Zero()) const override = 0;
    bool GetActualStateId(const TString& userId, const TString& topic, TString& result) const override = 0;
    bool RefreshStates(const TInstant actiality) const override = 0;

    virtual TString UnescapeMacroses(ELocalization locale, const TString& rawText, const TString& userId, const TString& topic, const TString& prefix = "", const TString& suffix = "", const TChatContext* context = nullptr) const override;

    // The only methods to be implemented in actual chat
    NThreading::TFuture<void> AcceptUserResponse(const IChatUserContext::TPtr context, const TMessage& message, const TVector<TMessageAttachment>& attachments, const TString& operatorId) const override = 0;
    bool MaybeContinueChat(const IChatUserContext::TPtr context, const TString& operatorId, TChatContext& stateContext) const override = 0;
    void AcceptMediaResource(const IChatUserContext::TPtr context, const TString& resourceId, const TString& contentType, const TString& content, TAtomicSharedPtr<IChatMediaResourcePostUploadCallback> callback) const override = 0;
    NThreading::TFuture<TChatResourceAcquisitionResult> GetMediaResource(const IChatUserContext::TPtr context, const TString& resourceId, const bool needsFurtherProcessing) const override = 0;
    bool RegisterMediaResource(const TString& userId, const TString& resourceId, const TString& contentType, const bool shared, NDrive::TEntitySession& session) const override = 0;
    bool UpdateMediaResourceDescription(const TMediaResourceDescription description, const TString& actorUserId, NDrive::TEntitySession& session) const override = 0;
    NThreading::TFuture<void> UploadResourcePreview(const TString& content, const TMediaResourceDescription& description, const TString& contentType) const override = 0;

    // Override for actual previews
    virtual NThreading::TFuture<TChatResourceAcquisitionResult> GetMediaResourcePreview(const IChatUserContext::TPtr context, const TString& resourceId, const bool needsFurtherProcessing, const TMap<TString, TString>& typeResourceOverrides) const override {
        Y_UNUSED(context);
        Y_UNUSED(resourceId);
        Y_UNUSED(typeResourceOverrides);
        return GetMediaResource(context, resourceId, needsFurtherProcessing);
    }

    virtual TSet<TString> GetNonRobotParticipants(const NDrive::NChat::TMessageEvents& messages) const override;
    virtual bool DoPreEntryActions(const IChatUserContext::TPtr userContext, const TChatRobotScriptItem& item, TChatContext& context, NDrive::TEntitySession& tagsSession, NDrive::TEntitySession& chatsSession) const override;
    bool SendPreActionMessages(const ui32 chatRoomId, const TString& chatItemId, NDrive::TEntitySession& session, const TString& operatorId, const TMaybe<TChatContext>& chatContext = {}, const TMaybe<ui64> lastMessageId = {}) const override;

    bool ArchiveChat(const TString& userId, const TString& topic) const override;

    NDrive::TEntitySession BuildChatEngineSession(const bool readOnly = false) const override;

    virtual TString GetChatTitle(const TString& userId, const TString& topic) const override;
    virtual TString GetChatTitle(const NDrive::NChat::TChat& chat) const override;
    virtual bool OverrideTitle(const TString& userId, const TString& topic, const TString& newTitle, NDrive::TEntitySession& session) const override;
    virtual TString GetChatIcon(const NDrive::NChat::TChat& chat) const override;

    virtual bool GetOrCreateChat(const TString& userId, const TString& topic, ui32& resultState, TChatContext& resultContext, NDrive::TEntitySession& session) const override;
    virtual bool UpdateChat(const TChatContext& chatContext, const TString& userId, const TString& topic, const TString& currentStep, NDrive::TEntitySession& session) const override;

    bool GetChat(const TString& userId, const TString& topic, NDrive::NChat::TChat& chat, const bool fallback = true, NStorage::ITransaction::TPtr transactionExt = nullptr) const override;
    TString GetFaqUrl() const override;
    TString GetSupportUrl() const override;

    const TChatRobotConfig& GetChatConfig() const override;

    virtual bool IsExistsForUser(const TString& userId, const TString& topic) const override;
    virtual bool DeduceChatContinuation(const TString& /*userId*/, const TString& /*topic*/, const ui32 /*chatRoomId*/, NDrive::TEntitySession& /*session*/) const override;

    virtual bool SetChatOperatorName(const TString& /*userId*/, const TString& /*topic*/, const TString& /*operatorName*/, NDrive::TEntitySession& /*session*/) const override {
        return true;
    }

    virtual bool AddToContext(const TVector<TContextMapInfo>& /*values*/, const TString& /*userId*/, const TString& /*topic*/, NDrive::TEntitySession& /*session*/) const override {
        return true;
    }

    virtual bool SendArbitraryMessage(const TString& userId, const TString& topic, const TString& operatorId, TMessage& message, NDrive::TEntitySession& session, const TChatContext* externalContext = nullptr) const override;
    virtual bool EditMessage(const TString& operatorId, const ui64 messageId, const NDrive::NChat::TMessageEdit messageEdit, NDrive::TEntitySession& session) const override;
    virtual bool DeleteMessage(const TString& operatorId, const ui64 message, NDrive::TEntitySession& session) const override;
    virtual bool AcceptMessages(const TString& userId, const TString& topic, TVector<TMessage>& messages, const TString& operatorId, NDrive::TEntitySession& session) const override;

    virtual NDrive::NChat::TOptionalMessageEvents GetChatMessagesRange(const TString& userId, const TString& topic, const ui32 viewerTraits, const ui64 startId, const ui64 finishId, NDrive::TEntitySession& session) const override;
    virtual NDrive::NChat::TOptionalMessageEvents GetChatMessages(const TString& userId, const TString& topic, NDrive::TEntitySession& session, const ui32 viewerTraits = NDrive::NChat::TMessage::AllKnownTraits, const ui64 startId = 0) const override;
    virtual NDrive::NChat::TOptionalMessageEvents GetChatMessagesSinceTimestamp(const TString& userId, const TString& topic, const TInstant startTs, NDrive::TEntitySession& session, const ui32 viewerTraits = NDrive::NChat::TMessage::AllKnownTraits) const override;
    virtual bool MarkMessagesRead(const NDrive::NChat::TMessageEvent& lastMessage, const TString& userId, NDrive::TEntitySession& session) const override;

    virtual NDrive::NChat::TExpectedMessageEvents GetCachedMessages(const TString& searchId) const override;
    virtual bool UpdateCachedMessages(const TSet<TString>& searchIds, NDrive::TEntitySession& session, const TInstant& requestTime) const override;

    virtual TMaybe<NDrive::NChat::TChatMessages> GetUserChatsMessages(const TVector<TChat>& chats, NDrive::TEntitySession& session, const TRange<ui64>& idRange = {}, const TRange<TInstant>& timestampRange = {}, const ui32 viewerTraits = NDrive::NChat::TMessage::AllKnownTraits) const override;

    virtual NDrive::NChat::TOptionalMessageEvents GetUnreadMessages(const ui32 chatId, const TString& userId, const ui32 viewerTraits, NDrive::TEntitySession& session) const override;

    virtual bool GetLastMessage(const ui32& chatRoomId, NDrive::TEntitySession& session, const ui32 viewerTraits, TMaybe<NDrive::NChat::TMessageEvent>& result, const TString& messageFromUserId = "") const override;
    virtual bool GetLastMessage(const TString& chatUserId, const TString& topic, NDrive::TEntitySession& session, const ui32 messageTraits, TMaybe<NDrive::NChat::TMessageEvent>& result, const TString& messageFromUserId = "") const override;

    virtual NJson::TJsonValue GetMessageReport(ELocalization locale, const NDrive::NChat::TMessageEvent& message, const TString& userId, const TString& topic) const override;
    virtual TMaybe<ui64> GetFirstEventIdByTime(const TString& userId, const TString& topic, const TInstant since, const bool isGoingBack, NDrive::TEntitySession& session) const override;

    virtual TMaybe<size_t> GetMessagesCount(const ui32 chatId, const ui32 viewerTraits, NDrive::TEntitySession& session) const override;
    TMaybe<size_t> GetMessagesCount(const TString& userId, const TString& topic, const NDrive::NChat::TMessage::EMessageType type, NDrive::TEntitySession& session) const override;
    virtual TMaybe<size_t> GetUnreadMessagesCount(const ui32 chatId, const TString& userId, const ui32 viewerTraits, NDrive::TEntitySession& session) const override;
    virtual bool UpdateLastViewedMessageId(const TString& userId, const TString& topic, const TString& viewerId, const ui64 sentMessageId, NDrive::TEntitySession& session) const override;

    virtual bool IsUserRobot(const TString& userId) const;

    virtual bool RefreshEngine(const TInstant actiality) const override;
    virtual bool Refresh(const TInstant actuality) const override;
    virtual TVector<TChat> GetTopics(IChatUserContext::TPtr ctx, bool onlyActive = false) const override;
    virtual TString GetChatSearchId(const TString& userId, const TString& topic) const override;

    virtual void CheckIn(const TString& /*userId*/, const TString& /*operatorUserId*/, const TString& /*topic*/, const TString& /*origin*/, const TString& /*externalUserId*/, const TGeoCoord* /*coord*/, const TInstant /*actuality*/) const override {
    }

public:
    static TMaybe<TDBTag> GetTagByTopicLink(const TString& topicLink, const TString& userId, NDrive::TEntitySession& session, const NDrive::IServer& server);
    static bool MoveToStep(const TString& userId, const TString& nodeId, const TString& topicLink, const NDrive::IServer& server, NDrive::TEntitySession& chatSession, TVector<TContextMapInfo> contextMap);
};

template<class TChatRobotState>
class IStatefulChatRobot : public IChatRobotImpl {
private:
    using TBase = IChatRobotImpl;
    using TBase::TBase;

protected:
    const TAtomicSharedPtr<TChatRobotStatePostgresStorage> RobotStateStorage;

    virtual TExpected<TChatRobotState, EChatError> GetChatRobotState(const TString& userId, const TString& topic, const TInstant actuality = TInstant::Zero(), NDrive::TEntitySession* sessionExt = nullptr, bool isDirect = false) const {
        TChatRobotState result;
        TString stateKey = userId + "-" + ChatConfig.GetChatId();
        if (topic) {
            stateKey += "-" + topic;
        }
        auto resultSerialized = isDirect ? RobotStateStorage->DirectGetSerializedState(stateKey, sessionExt) : RobotStateStorage->GetSerializedState(stateKey, actuality, sessionExt);
        if (!resultSerialized) {
            return MakeUnexpected(resultSerialized.GetError());
        }
        try {
            Y_ENSURE(result.ParseFromString(Base64Decode(*resultSerialized)));
        } catch (const std::exception& e) {
            ERROR_LOG << "cannot parse ChatRobotState from " << *resultSerialized << ": " << FormatExc(e) << Endl;
            return MakeUnexpected(EChatError::ParsingFailed);
        }
        return result;
    }

    virtual bool GetChatRobotState(const TString& userId, const TString& topic, TChatRobotState& result, const TInstant actuality = TInstant::Zero(), NDrive::TEntitySession* sessionExt = nullptr, bool isDirect = false) const {
        auto eg = NDrive::BuildEventGuard("get_chat_robot_state");
        auto expectedResult = GetChatRobotState(userId, topic, actuality, sessionExt, isDirect);
        if (!expectedResult) {
            return false;
        }
        result = std::move(*expectedResult);
        return true;
    }

    virtual bool RemoveChatRobotState(const TString& userId, const TString& topic, const TString& operatorId, NDrive::TEntitySession& session) const {
        TString stateKey = userId + "-" + ChatConfig.GetChatId();
        if (topic) {
            stateKey += "-" + topic;
        }
        auto serializedState = RobotStateStorage->GetSerializedState(stateKey, Now());
        if (serializedState) {
            return RobotStateStorage->RemoveState(stateKey, operatorId, session);
        }
        return true;
    }

    virtual TString BuildStateKey(const TString& userId, const TString& topic) const {
        TString stateKey = userId + "-" + ChatConfig.GetChatId();
        if (topic) {
            stateKey += "-" + topic;
        }
        return stateKey;
    }

    virtual bool UpdateChatRobotState(const TString& userId, const TString& topic, const TChatRobotState& state, NDrive::TEntitySession& session) const {
        return RobotStateStorage->SetSerializedState(BuildStateKey(userId, topic), Base64Encode(state.SerializeAsString()), state.GetCurrentStep(), session);
    }

    virtual bool AddChatRobotStateHistoryEvent(const TString& userId, const TString& topic, const TChatRobotState& state, NDrive::TEntitySession& session) const {
        return RobotStateStorage->AddSerializedStateHistoryEvent(BuildStateKey(userId, topic), Base64Encode(state.SerializeAsString()), state.GetCurrentStep(), session);
    }

    virtual bool GetOrCreateRobotState(const TString& userId, const TString& topic, TChatRobotState& resultState, NDrive::TEntitySession& session, TVector<TContextMapInfo> contextMap = {}) const {
        auto eg = NDrive::BuildEventGuard("get_or_create_chat_robot_state");
        if (GetChatRobotState(userId, topic, resultState, TInstant::Zero(), &session, true)) {
            return true;
        }
        if (!SetInitialState(userId, topic, resultState, session, contextMap)) {
            ERROR_LOG << "could not set initial state for (" << userId << ", " << topic << ")" << Endl;
            return false;
        }
        return ChatEngine.RefreshCache(Now());
    }

    virtual bool GetOrCreateChat(const TString& userId, const TString& topic, ui32& resultChatRoom, TChatContext& resultContext, NDrive::TEntitySession& session) const override {
        TChatRobotState resultState;
        if (!GetOrCreateRobotState(userId, topic, resultState, session)) {
            return false;
        }
        if (!GetChatRoom(userId, topic, resultChatRoom, &session)) {
            return false;
        }
        if (!GetChatContext(resultState, resultContext)) {
            return false;
        }
        return true;
    }

    virtual bool DoGetIsChatCompleted(const TString& /*userId*/, const TString& /*topic*/, const TInstant /*actuality*/) const {
        return true;
    }

public:
    IStatefulChatRobot(const NDrive::IServer* server, const TChatRobotConfig& chatConfig, const TAtomicSharedPtr<TChatViewTracker> messageViewTracker, const TAtomicSharedPtr<TChatRobotStatePostgresStorage> robotStateStorage)
        : TBase(server, chatConfig, messageViewTracker)
        , RobotStateStorage(robotStateStorage)
    {
    }

    ~IStatefulChatRobot() = default;

    virtual bool SendArbitraryMessage(const TString& userId, const TString& topic, const TString& operatorId, TMessage& message, NDrive::TEntitySession& session, const TChatContext* externalContext = nullptr) const override {
        if (message.GetAuthorName().empty() && userId != operatorId && !IsUserRobot(operatorId) && !externalContext) {
            TChatContext chatContext(nullptr, &session);
            auto currentState = GetChatRobotState(userId, topic, TInstant::Zero(), &session, true);
            if (!currentState && currentState.GetError() == EChatError::FetchFailed) {
                return false;
            } else if (currentState && GetChatContext(*currentState, chatContext)) {
                return TBase::SendArbitraryMessage(userId, topic, operatorId, message, session, &chatContext);
            }
        }
        return TBase::SendArbitraryMessage(userId, topic, operatorId, message, session, externalContext);
    }

    virtual bool SetChatOperatorName(const TString& userId, const TString& topic, const TString& operatorName, NDrive::TEntitySession& session) const override {
        TContextMapInfo operatorInfo("operator_name", operatorName, EContextDataType::OperatorName);
        return AddToContext({ operatorInfo }, userId, topic, session);
    }

    bool AddToContext(const TVector<TContextMapInfo>& values, const TString& userId, const TString& topic, NDrive::TEntitySession& session) const override {
        TChatContext chatContext(nullptr, &session);
        auto currentState = GetChatRobotState(userId, topic, TInstant::Zero(), &session, true);
        if (!currentState) {
            if (currentState.GetError() != EChatError::FetchFailed) {
                session.SetErrorInfo("SetChatOperatorName", ToString(currentState.GetError()));
            }
            return false;
        }
        if (!GetChatContext(*currentState, chatContext)) {
            session.SetErrorInfo("SetChatOperatorName", "could not get chat context");
            return false;
        }
        chatContext.AddMapData(values);
        return UpdateChatRobotStateContext(chatContext, userId, topic, *currentState, session);
    }

    virtual bool HasChat(const TString& userId, const TString& topic, NDrive::TEntitySession* sessionExt = nullptr) const override {
        if (ChatConfig.GetIsStatic()) {
            return true;
        }
        ui32 roomId;
        return GetChatRoom(userId, topic, roomId, sessionExt);
    }

    virtual bool EnsureChat(const TString& userId, const TString& topic, NDrive::TEntitySession& session, const bool isRead = false, TVector<TContextMapInfo> contextMap = {}) const override {
        if (ChatConfig.GetIsStatic()) {
            return true;
        }

        if (contextMap) {
            auto state = GetChatRobotState(userId, topic, Now(), &session);
            TChatContext chatContext(nullptr, &session);
            if (state && GetChatContext(*state, chatContext)) {
                chatContext.AddMapData(contextMap);
                if (!UpdateChatRobotStateContext(chatContext, userId, topic, *state, session)) {
                    return false;
                }
            } else if (state.GetError() == EChatError::FetchFailed) {
                return false;
            }
        }

        if (isRead && !ChatConfig.GetCreateOnRead()) {
            ui32 roomId;
            if (!GetChatRoom(userId, topic, roomId, &session)) {
                return false;
            }
        }
        if (HasChat(userId, topic, &session)) {
            return true;
        }
        TChatRobotState currentState;
        return GetOrCreateRobotState(userId, topic, currentState, session, contextMap);
    }

    virtual bool ResetChat(const TString& userId, const TString& topic) const override {
        TChatRobotState state;
        auto session = BuildChatEngineSession();
        if (!SetInitialState(userId, topic, state, session, {})) {
            ERROR_LOG << "SetState: " << session.GetStringReport() << Endl;
            return false;
        }
        if (!session.Commit()) {
            ERROR_LOG << "SetState: " << session.GetStringReport() << Endl;
            return false;
        }
        return true;
    }

    virtual bool CreateCleanChat(const TString& userId, const TString& topic) const override {
        auto session = BuildChatEngineSession();
        ui32 chatRoomId;
        if (!GetOrCreateChatRoom(userId, topic, chatRoomId, &session)) {
            return false;
        }
        TChatRobotState state;
        state.SetCurrentStep(ChatConfig.GetCleanStateId());
        return UpdateChatRobotState(userId, topic, state, session) && session.Commit();
    }

    bool RemoveChat(const TString& userId, const TString& topic, const TString& operatorId) const override {
        auto session = BuildChatEngineSession();
        if (!DeleteChatRoom(userId, topic, operatorId, session)) {
            return false;
        }
        return RemoveChatRobotState(userId, topic, operatorId, session) && session.Commit();
    }

    virtual bool GetChatContext(const TChatRobotState& /*state*/, TChatContext& /*chatContext*/) const {
        return true;
    }

    virtual bool UpdateChat(const TChatContext& chatContext, const TString& userId, const TString& topic, const TString& currentStep, NDrive::TEntitySession& session) const override {
        TChatRobotState state;
        if (!GetChatRobotState(userId, topic, state, Now(), &session)) {
            return false;
        }
        if (currentStep) {
            state.SetCurrentStep(currentStep);
        }
        return UpdateChatRobotStateContext(chatContext, userId, topic, state, session);
    }

    virtual bool UpdateChatRobotStateContext(const TChatContext& /*chatContext*/, const TString& /*userId*/, const TString& /*topic*/, const TChatRobotState& /*state*/, NDrive::TEntitySession& /*session*/) const {
        return true;
    }

    using TBase::MoveToStep;

    virtual bool MoveToStep(const TString& userId, const ui32 chatRoomId, const TString& topic, const TString& stepName, TChatRobotState& currentState, NDrive::TEntitySession& chatSession, const bool sendMessages = false, const TString& operatorId = "robot-frontend", const TMaybe<ui64> lastMessageId = {}, TVector<TContextMapInfo> contextMap = {}) const {
        auto eg = NThreading::BuildEventGuard("MoveToStep");
        TString currentStep = currentState.GetCurrentStep();
        if (currentStep != stepName) {
            TChatContext chatContext(nullptr, &chatSession);
            if (!GetChatContext(currentState, chatContext)) {
                return false;
            }
            chatContext.AddMapData(contextMap);
            TChatRobotScriptItem scriptItem;
            if (!ChatConfig.GetChatScript().GetScriptItemById(stepName, scriptItem)) {
                chatSession.SetErrorInfo("ChatRobotImpl::MoveToStep", "cannot GetScriptItemById: " + stepName);
                return false;
            }
            auto permissions = Server->GetDriveAPI()->GetUserPermissions(userId, TUserPermissionsFeatures());
            IChatUserContext::TPtr context = MakeAtomicShared<TChatUserContext>(permissions, Server);
            context->SetUserId(userId);
            context->SetChatId(ChatConfig.GetChatId());
            context->SetChatTopic(topic);
            IChatScriptAction::TPtr scriptAction = IChatScriptAction::Construct(context, scriptItem, operatorId);
            if (!scriptAction) {
                chatSession.SetErrorInfo("ChatRobotImpl::MoveToStep", "cannot IChatScriptAction::Construct: " + stepName);
                return false;
            }
            TString nextNodeId = stepName;
            auto execute = [&](NDrive::TEntitySession& tx) {
                return scriptAction->OnEnter(chatRoomId, chatContext, chatSession, tx, nextNodeId, sendMessages, lastMessageId);
            };
            if (ChatEngine.GetDatabaseName() != DriveAPI.GetDatabaseName()) {
                auto tagsSession = DriveAPI.BuildTx<NSQL::Writable | NSQL::Deferred>();
                if (!execute(tagsSession) || !tagsSession.Commit()) {
                    if (!chatSession.GetMessages().HasMessages()) {
                        chatSession.SetErrorInfo("IChatRobotImpl::MoveToStep", "cannot do pre-entry actions");
                    }
                    chatSession.MergeErrorMessages(tagsSession.GetMessages(), "MoveToStep");
                    return false;
                }
            } else {
                if (!execute(chatSession)) {
                    return false;
                }
            }
            currentState.SetCurrentStep(nextNodeId);
            if (!UpdateChatRobotStateContext(chatContext, userId, topic, currentState, chatSession)) {
                return false;
            }

            auto evlog = NDrive::GetThreadEventLogger();
            if (evlog) {
                evlog->AddEvent(NJson::TMapBuilder
                    ("source", "MoveToStep")
                    ("from_step", currentStep)
                    ("to_step", stepName)
                );
            }
            return true;
        }
        return true;
    }

    bool MoveToStep(const TString& userId, const TString& topic, const TString& stepName, NDrive::TEntitySession& chatSession, const bool sendMessages, const TString& newChatTitle, const TString& operatorId, const TMaybe<ui64> lastMessageId, TVector<TContextMapInfo> contextMap) const override {
        ui32 chatRoomId;
        if (!GetOrCreateChatRoom(userId, topic, chatRoomId, &chatSession, newChatTitle)) {
            return false;
        }
        TChatRobotState currentState;
        auto currentStateExp = GetChatRobotState(userId, topic, Now(), &chatSession);
        if (!currentStateExp) {
            if (currentStateExp.GetError() != EChatError::NotFound) {
                return false;
            }
        } else {
            currentState = std::move(*currentStateExp);
        }
        return MoveToStep(userId, chatRoomId, topic, stepName, currentState, chatSession, sendMessages, operatorId, lastMessageId, contextMap);
    }

    virtual bool GetIsChatCompleted(const TString& userId, const TString& topic, const TInstant actuality) const override {
        TChatRobotState currentState;
        if (!GetChatRobotState(userId, topic, currentState, actuality)) {
            return false;
        }

        TChatRobotScriptItem currentScriptItem;
        if (!ChatConfig.GetChatScript().GetScriptItemById(currentState.GetCurrentStep(), currentScriptItem)) {
            return false;
        }

        return (!currentScriptItem.HasFurtherSteps() || currentScriptItem.IsSkippable()) && DoGetIsChatCompleted(userId, topic, actuality);
    }

    virtual bool GetIsChatInClosedNode(const TString& userId, const TString& topic, const TInstant actuality) const override {
        TChatRobotState currentState;
        if (!GetChatRobotState(userId, topic, currentState, actuality)) {
            return false;
        }
        TChatRobotScriptItem currentScriptItem;
        if (!ChatConfig.GetChatScript().GetScriptItemById(currentState.GetCurrentStep(), currentScriptItem)) {
            return false;
        }
        return currentScriptItem.GetActionType() == NChatRobot::EUserAction::ChatClosed;
    }

    TMaybe<TNextActionInfo> GetFirstResubmitStep(const IChatUserContext::TPtr /*context*/, TChatContext& /*stateContext*/, NDrive::TEntitySession& /*chatSession*/, ui32 /*maskOverride*/) const override {
        return TNextActionInfo();
    }

    TMaybe<TNextActionInfo> GetNextResubmitStep(const IChatUserContext::TPtr context, const TChatRobotScriptItem& currentScriptItem, TChatContext& stateContext, NDrive::TEntitySession& chatSession) const override {
        Y_UNUSED(chatSession);
        TChatRobotScriptItem newScriptItem;
        if (!ChatConfig.GetChatScript().GetNextScriptItem(currentScriptItem, context, stateContext, newScriptItem)) {
            return TNextActionInfo{};
        }
        return { newScriptItem.GetId() };
    }

    virtual bool GetCurrentScriptItem(const TString& userId, const TString& topic, TChatRobotScriptItem& result, const TInstant actuality) const override {
        auto eg = NDrive::BuildEventGuard("get_current_script_item");
        if (ChatConfig.GetIsStatic() || (ChatConfig.IsCommon() && !IsExistsForUser(userId, topic))) {
            auto stepId = topic ? topic : ChatConfig.GetInitialStepId();
            return ChatConfig.GetChatScript().GetScriptItemById(stepId, result);
        }

        if (!IsExistsForUser(userId, topic)) {
            return false;
        }

        TChatRobotState userState;
        if (!GetChatRobotState(userId, topic, userState, actuality)) {
            return false;
        }

        return ChatConfig.GetChatScript().GetScriptItemById(userState.GetCurrentStep(), result);
    }

    virtual ui64 GetLastViewedMessageId(const TString& userId, const TString& topic, const TString& viewerId) const override {
        auto eg = NDrive::BuildEventGuard("last_viewed_message_id");
        ui32 chatRoomId;
        if (!GetChatRoom(userId, topic, chatRoomId)) {
            return 0;
        }
        return MessageViewTracker->GetLastViewedMessageId(viewerId, chatRoomId);
    }

    virtual ui64 GetLastViewedMessageIdExcept(const TString& userId, const TString& topic, const TString& exceptUserId) const override {
        ui32 chatRoomId;
        if (!GetChatRoom(userId, topic, chatRoomId)) {
            return 0;
        }
        return MessageViewTracker->GetLastViewedMessageIdExcept(exceptUserId, chatRoomId);
    }

    virtual TVector<std::pair<TString, TString>> GetUsersInNodes(const TString& reqTopic, const TSet<TString>& nodeNames) const override {
        auto rawStatesMap = RobotStateStorage->GetRawCachedStates();
        TVector<std::pair<TString, TString>> result;
        for (auto&& userStateIt : rawStatesMap) {
            if (userStateIt.first.Size() <= 37) {
                continue;
            }

            TChatRobotState state;
            try {
                Y_ENSURE(state.ParseFromString(Base64Decode(userStateIt.second)));
            } catch (const std::exception& e) {
                ERROR_LOG << "cannot parse ChatRobotState from " << userStateIt.second << ": " << FormatExc(e) << Endl;
                continue;
            }
            if (!nodeNames.contains(state.GetCurrentStep())) {
                continue;
            }

            TString objectId = userStateIt.first.substr(0, 36);
            auto stateKeyName = objectId + "-" + ChatConfig.GetChatId();
            if (reqTopic != "*") {
                if (reqTopic) {
                    stateKeyName += "-" + reqTopic;
                }
                if (stateKeyName == userStateIt.first) {
                    result.emplace_back(std::make_pair(std::move(objectId), state.GetCurrentStep()));
                }
            } else {
                if (userStateIt.first.StartsWith(stateKeyName)) {
                    result.emplace_back(std::make_pair(std::move(objectId), state.GetCurrentStep()));
                }
            }
        }
        return result;
    }

    virtual bool SetInitialState(const TString userId, const TString& topic, TChatRobotState& resultState, NDrive::TEntitySession& session, TVector<TContextMapInfo> contextMap = {}) const {
        // Create chat room along with robot state
        ui32 chatRoomId;
        if (!GetOrCreateChatRoom(userId, topic, chatRoomId, &session)) {
            return false;
        }

        resultState = TChatRobotState();
        if (ChatConfig.GetStartFromCleanChat() || !DeduceChatContinuation(userId, topic, chatRoomId, session)) {
            return MoveToStep(userId, chatRoomId, topic, ChatConfig.GetInitialStepId(), resultState, session, true, "robot-frontend", {}, contextMap);
        }
        return GetChatRobotState(userId, topic, resultState, TInstant::Zero(), &session);
    }

    virtual bool GetHistoryStatsByNodes(const TDuration& windowSize, TMap<TString, size_t>& result) const override {
        TVector<TAtomicSharedPtr<TObjectEvent<TChatRobotSerializedState>>> statesHistory;
        if (!RobotStateStorage->GetHistoryManager().GetEventsAll(Now() - windowSize, statesHistory, TInstant::Zero())) {
            return false;
        }

        TMap<TString, TSet<TString>> objectsByState;
        for (auto&& event : statesHistory) {
            if (event->GetTalkId().Size() <= 37) {
                continue;
            }
            TString objectId = event->GetTalkId().substr(0, 36);
            bool isMatching = event->GetTalkId() == objectId + "-" + ChatConfig.GetChatId() || event->GetTalkId().StartsWith(objectId + "-" + ChatConfig.GetChatId() + "-");
            if (!isMatching) {
                continue;
            }
            TChatRobotState state;
            try {
                Y_ENSURE(state.ParseFromString(Base64Decode(event->GetSerializedState())));
            } catch (const std::exception& e) {
                ERROR_LOG << "cannot parse ChatRobotState from " << event->GetSerializedState() << ": " << FormatExc(e) << Endl;
                continue;
            }
            objectsByState[state.GetCurrentStep()].emplace(objectId);
        }

        result.clear();
        for (auto&& it : objectsByState) {
            result.emplace(it.first, it.second.size());
        }

        return true;
    }

    virtual NJson::TJsonValue GetCurrentActionMessagesReport(ELocalization locale, const TString& userId, const TString& topic, const bool /*withMetaInfo*/, const TInstant timestampOverride) const override {
        NJson::TJsonValue report = NJson::JSON_ARRAY;
        TString currentStep;
        if (ChatConfig.GetIsStatic()) {
            currentStep = topic ? topic : ChatConfig.GetInitialStepId();
        } else {
            TChatRobotState currentState;
            if (!GetChatRobotState(userId, topic, currentState)) {
                return report;
            }
            currentStep = currentState.GetCurrentStep();
        }

        TChatRobotScriptItem scriptItem;
        if (!ChatConfig.GetChatScript().GetScriptItemById(currentStep, scriptItem)) {
            return report;
        }

        auto currentInstant = Now();
        size_t currentMessageId = 0;
        for (auto&& message : scriptItem.GetPreActionMessages()) {
            NDrive::NChat::TMessageEvent wrap = TObjectEvent<NDrive::NChat::TMessage>(message, EObjectHistoryAction::Add, timestampOverride == TInstant::Zero() ? currentInstant : timestampOverride, "robot-frontend", "", "");
            wrap.SetHistoryEventId(++currentMessageId);
            auto messageReport = GetMessageReport(locale, wrap, userId, topic);
            report.AppendValue(std::move(messageReport));
        }

        return report;
    }

    virtual bool GetActualStateId(const TString& userId, const TString& topic, TString& result) const override {
        TChatRobotState resultState;
        auto session = BuildChatEngineSession(true);
        if (!GetChatRobotState(userId, topic, resultState, Now(), &session)) {
            return false;
        }
        result = resultState.GetCurrentStep();
        return true;
    }

    virtual bool RefreshStates(const TInstant actuality) const override {
        auto eg = NDrive::BuildEventGuard("refresh_states");
        return RobotStateStorage->ForceRefresh(actuality);
    }

    virtual bool Refresh(const TInstant actuality) const override {
        auto eg = NDrive::BuildEventGuard("refresh_states_and_view_tracker");
        if (ChatConfig.GetIsStatic()) {
            return true;
        }
        TBase::Refresh(actuality);
        return RobotStateStorage->ForceRefresh(actuality) && MessageViewTracker->ForceRefresh(actuality);
    }
};
