#include "condition.h"

#include <drive/backend/billing/manager.h>
#include <drive/backend/database/drive/private_data.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/experiments/context.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/user_document_photos/manager.h>

TAndCondition::TFactory::TRegistrator<TAndCondition> TAndCondition::Registrator(ICondition::EType::And);
TOrCondition::TFactory::TRegistrator<TOrCondition> TOrCondition::Registrator(ICondition::EType::Or);
TNotCondition::TFactory::TRegistrator<TNotCondition> TNotCondition::Registrator(ICondition::EType::Not);
TPerformsCondition::TFactory::TRegistrator<TPerformsCondition> TPerformsCondition::Registrator(ICondition::EType::Performs);
THasTagCondition::TFactory::TRegistrator<THasTagCondition> THasTagCondition::Registrator(ICondition::EType::HasTag);
TUserInStatusCondition::TFactory::TRegistrator<TUserInStatusCondition> TUserInStatusCondition::Registrator(ICondition::EType::UserInStatus);
TPlatformCondition::TFactory::TRegistrator<TPlatformCondition> TPlatformCondition::Registrator(ICondition::EType::Platform);
TAppBuildCondition::TFactory::TRegistrator<TAppBuildCondition> TAppBuildCondition::Registrator(ICondition::EType::AppBuild);
THasActionCondition::TFactory::TRegistrator<THasActionCondition> THasActionCondition::Registrator(ICondition::EType::HasAction);
TRegistrationStepCondition::TFactory::TRegistrator<TRegistrationStepCondition> TRegistrationStepCondition::Registrator(ICondition::EType::RegistrationStep);
THasDebtCondition::TFactory::TRegistrator<THasDebtCondition> THasDebtCondition::Registrator(ICondition::EType::HasDebt);
TInAreaCondition::TFactory::TRegistrator<TInAreaCondition> TInAreaCondition::Registrator(ICondition::EType::InArea);
TEnteredMessageCondition::TFactory::TRegistrator<TEnteredMessageCondition> TEnteredMessageCondition::Registrator(ICondition::EType::Message);
TDictionaryTagValueCondition::TFactory::TRegistrator<TDictionaryTagValueCondition> TDictionaryTagValueCondition::Registrator(ICondition::EType::DictionaryTagValue);
THadRidesCondition::TFactory::TRegistrator<THadRidesCondition> THadRidesCondition::Registrator(ICondition::EType::HadRides);
THandlerAnswerCondition::TFactory::TRegistrator<THandlerAnswerCondition> THandlerAnswerCondition::Registrator(ICondition::EType::HandlerAnswer);
THandlerCodeCondition::TFactory::TRegistrator<THandlerCodeCondition> THandlerCodeCondition::Registrator(ICondition::EType::HandlerCode);
THasWalletCondition::TFactory::TRegistrator<THasWalletCondition> THasWalletCondition::Registrator(ICondition::EType::HasWallet);
TPhoneVerifiedCondition::TFactory::TRegistrator<TPhoneVerifiedCondition> TPhoneVerifiedCondition::Registrator(ICondition::EType::PhoneVerified);
TOriginCondition::TFactory::TRegistrator<TOriginCondition> TOriginCondition::Registrator(ICondition::EType::Origin);
TPhotoStatusCondition::TFactory::TRegistrator<TPhotoStatusCondition> TPhotoStatusCondition::Registrator(ICondition::EType::PhotoStatus);
TEmailVerifiedCondition::TFactory::TRegistrator<TEmailVerifiedCondition> TEmailVerifiedCondition::Registrator(ICondition::EType::EmailVerified);
TContextMapEqualsCondition::TFactory::TRegistrator<TContextMapEqualsCondition> TContextMapEqualsCondition::Registrator(ICondition::EType::ContextMapEquals);
TContextMapContainsKeyCondition::TFactory::TRegistrator<TContextMapContainsKeyCondition> TContextMapContainsKeyCondition::Registrator(ICondition::EType::ContextMapContainsKey);
TCarTagsCondition::TFactory::TRegistrator<TCarTagsCondition> TCarTagsCondition::Registrator(ICondition::EType::CarTags);
THasReferralCondition::TFactory::TRegistrator<THasReferralCondition> THasReferralCondition::Registrator(ICondition::EType::HasReferral);
TTagFieldValueCondition::TFactory::TRegistrator<TTagFieldValueCondition> TTagFieldValueCondition::Registrator(ICondition::EType::TagFieldValue);

bool TAndCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const {
    for (auto&& sc : SubConditions) {
        if (!sc->IsMatching(ctx, chatContext)) {
            return false;
        }
    }
    return true;
}

ICondition::TPtr TAndCondition::GetTemplateImpl(const TVector<TString>& params) const {
    ICondition::TPtr result = new TAndCondition();
    for (auto&& sc : SubConditions) {
        result->AddSubCondition(sc->GetTemplateImpl(params));
    }
    return result;
}

bool TOrCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const {
    for (auto&& sc : SubConditions) {
        if (sc->IsMatching(ctx, chatContext)) {
            return true;
        }
    }
    return false;
}

ICondition::TPtr TOrCondition::GetTemplateImpl(const TVector<TString>& params) const {
    ICondition::TPtr result = new TOrCondition();
    for (auto&& sc : SubConditions) {
        result->AddSubCondition(sc->GetTemplateImpl(params));
    }
    return result;
}

bool TNotCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const {
    return !SubConditions[0]->IsMatching(ctx, chatContext);
}

ICondition::TPtr TNotCondition::GetTemplateImpl(const TVector<TString>& params) const {
    ICondition::TPtr result = new TNotCondition();
    result->AddSubCondition(SubConditions[0]->GetTemplateImpl(params));
    return result;
}

bool TPerformsCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    if (!raw.Has("tag_name")) {
        errors.AddMessage("performs_condition", "don't have tag_name");
        return false;
    }
    if (!raw["tag_name"].IsString()) {
        errors.AddMessage("performs_condition", "tag_name is not string");
        return false;
    }
    TagName = raw["tag_name"].GetString();
    return true;
}

bool TPerformsCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const {
    return ctx->GetPerformedTags().contains(ctx->Unescape(TagName, chatContext));
}

ICondition::TPtr TPerformsCondition::GetTemplateImpl(const TVector<TString>& params) const {
    THolder<TPerformsCondition> result(new TPerformsCondition());
    result->SetTagName(NChatScript::GetMaybeParametrizedField(TagName, params));
    return result.Release();
}

bool THasTagCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    if (!raw.Has("tag_name")) {
        errors.AddMessage("has_tag_condition", "don't have tag_name");
        return false;
    }
    if (!raw["tag_name"].IsString()) {
        errors.AddMessage("has_tag_condition", "tag_name is not string");
        return false;
    }
    TagName = raw["tag_name"].GetString();
    return true;
}

bool THasTagCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const {
    return ctx->GetUserTags().contains(ctx->Unescape(TagName, chatContext));
}

ICondition::TPtr THasTagCondition::GetTemplateImpl(const TVector<TString>& params) const {
    THolder<THasTagCondition> result(new THasTagCondition());
    result->SetTagName(NChatScript::GetMaybeParametrizedField(TagName, params));
    return result.Release();
}

ICondition::TPtr THasWalletCondition::GetTemplateImpl(const TVector<TString>& params) const {
    THolder<THasWalletCondition> result(new THasWalletCondition());
    result->SetName(NChatScript::GetMaybeParametrizedField(Name, params));
    result->SetActive(Active);
    return result.Release();
}

bool THasWalletCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    if (raw.Has("active") && !NJson::ParseField(raw, "active", Active, true, errors)) {
        return false;
    }
    return NJson::ParseField(raw, "name", Name, true, errors);
}

bool THasWalletCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const {
    auto name = ctx->Unescape(Name, chatContext);
    for (const auto& account : ctx->GetServer().GetDriveAPI()->GetBillingManager().GetAccountsManager().GetUserAccounts(ctx->GetUserId(), ModelingNow())) {
        if (account && account->GetUniqueName() == name) {
            if (!Active.Defined() || *Active == account->IsActive()) {
                return true;
            }
        }
    }
    return false;
}

bool TUserInStatusCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    if (!raw.Has("status")) {
        errors.AddMessage("status_condition", "don't have status");
        return false;
    }
    if (!raw["status"].IsString()) {
        errors.AddMessage("status_condition", "status is not string");
        return false;
    }
    Status = raw["status"].GetString();
    return true;
}

bool TUserInStatusCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const {
    return ctx->GetStatus() == ctx->Unescape(Status, chatContext);
}

ICondition::TPtr TUserInStatusCondition::GetTemplateImpl(const TVector<TString>& params) const {
    THolder<TUserInStatusCondition> result(new TUserInStatusCondition());
    result->SetStatus(NChatScript::GetMaybeParametrizedField(Status, params));
    return result.Release();
}

bool TPlatformCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    if (!raw.Has("name")) {
        errors.AddMessage("platform_condition", "doesn't have name");
        return false;
    }
    if (!raw["name"].IsString()) {
        errors.AddMessage("platform_condition", "name is not string");
        return false;
    }
    Name = raw["name"].GetString();
    if (!TryFromString(Name, EnumName)) {
        errors.AddMessage("platform_condition", "name is neither variable not element from {'ios', 'android_client', 'android_service', 'unknown'}");
        return false;
    }
    return true;
}

bool TPlatformCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const {
    return ctx->GetPlatform() == EnumName;
}

ICondition::TPtr TPlatformCondition::GetTemplateImpl(const TVector<TString>& /*params*/) const {
    THolder<TPlatformCondition> result(new TPlatformCondition());
    result->SetName(Name);
    result->SetEnumName(EnumName);
    return result.Release();
}

bool TAppBuildCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    if (!raw.Has("min_value")) {
        errors.AddMessage("app_build_condition", "doesn't have min_value");
        return false;
    }
    if (!raw["min_value"].IsUInteger()) {
        errors.AddMessage("app_build_condition", "min_value is not uint");
        return false;
    }
    MinValue = raw["min_value"].GetUInteger();
    return true;
}

bool TAppBuildCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const {
    return ctx->GetAppBuild() >= MinValue;
}

ICondition::TPtr TAppBuildCondition::GetTemplateImpl(const TVector<TString>& /*params*/) const {
    THolder<TAppBuildCondition> result(new TAppBuildCondition());
    result->SetMinValue(MinValue);
    return result.Release();
}

bool THasActionCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    if (!raw.Has("name")) {
        errors.AddMessage("has_action_condition", "doesn't have name");
        return false;
    }
    if (!raw["name"].IsString()) {
        errors.AddMessage("has_action_condition", "name is not string");
        return false;
    }
    Name = raw["name"].GetString();
    return true;
}

bool THasActionCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const {
    return ctx->HasAction(ctx->Unescape(Name, chatContext));
}

ICondition::TPtr THasActionCondition::GetTemplateImpl(const TVector<TString>& params) const {
    THolder<THasActionCondition> result(new THasActionCondition());
    result->SetName(NChatScript::GetMaybeParametrizedField(Name, params));
    return result.Release();
}

bool THasDebtCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    if (!raw.Has("debtor") || !raw["debtor"].IsBoolean()) {
        errors.AddMessage("has_debt_condition", "doesn't have bool 'debtor' field");
        return false;
    }
    Debtor = raw["debtor"].GetBoolean();
    return true;
}

bool THasDebtCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const {
    return ctx->IsDebtor() == Debtor;
}

ICondition::TPtr THasDebtCondition::GetTemplateImpl(const TVector<TString>& /*params*/) const {
    THolder<THasDebtCondition> result(new THasDebtCondition());
    result->SetDebtor(Debtor);
    return result.Release();
}

bool TInAreaCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    if (!raw.Has("name") || !raw["name"].IsString()) {
        errors.AddMessage("in_area_condition", "doesn't have string 'name' field");
        return false;
    }
    Name = raw["name"].GetString();
    return true;
}

bool TInAreaCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const {
    return ctx->GetAreaIds().contains(ctx->Unescape(Name, chatContext));
}

ICondition::TPtr TInAreaCondition::GetTemplateImpl(const TVector<TString>& params) const {
    THolder<TInAreaCondition> result(new TInAreaCondition());
    result->SetName(NChatScript::GetMaybeParametrizedField(Name, params));
    return result.Release();
}

bool TRegistrationStepCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    if (!raw.Has("step") || !raw["step"].IsString()) {
        errors.AddMessage("registration_step_condition", "doesn't have string 'step' field");
        return false;
    }
    Step = raw["step"].GetString();
    return true;
}

bool TRegistrationStepCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const {
    return ctx->GetRegistrationStep() == ctx->Unescape(Step, chatContext);
}

ICondition::TPtr TRegistrationStepCondition::GetTemplateImpl(const TVector<TString>& params) const {
    THolder<TRegistrationStepCondition> result(new TRegistrationStepCondition());
    result->SetStep(NChatScript::GetMaybeParametrizedField(Step, params));
    return result.Release();
}

bool TEnteredMessageCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    if (!raw.Has("message") || !raw["message"].IsString()) {
        errors.AddMessage("entered_message_condition", "doesn't have string 'message' field");
        return false;
    }
    Message = raw["message"].GetString();
    return true;
}

bool TEnteredMessageCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const {
    return ctx->GetMessage() == ctx->Unescape(Message, chatContext);
}

ICondition::TPtr TEnteredMessageCondition::GetTemplateImpl(const TVector<TString>& params) const {
    THolder<TEnteredMessageCondition> result(new TEnteredMessageCondition());
    result->SetMessage(NChatScript::GetMaybeParametrizedField(Message, params));
    return result.Release();
}

bool TDictionaryTagValueCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    return
        NJson::ParseField(raw, "tag_name", TagName, true, errors) &&
        NJson::ParseField(raw, "field", Field, true, errors) &&
        NJson::ParseField(raw, "policy", NJson::Stringify(Policy), true, errors) &&
        NJson::ParseField(raw, "value", Value, Policy == NChatRobot::ETagCheckPolicy::Equals, errors);
}

bool TDictionaryTagValueCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const {
    auto tagsData = ctx->GetDictionaryTagsData();
    auto tagIt = tagsData.find(ctx->Unescape(GetTagName(), chatContext));
    if (tagIt == tagsData.end()) {
        return false;
    }

    auto soughtField = ctx->Unescape(Field, chatContext);
    auto soughtValue = ctx->Unescape(Value, chatContext);

    for (auto&& mapping : tagIt->second) {
        auto valueIt = mapping.find(soughtField);
        if (valueIt != mapping.end()) {
            if (Policy == NChatRobot::ETagCheckPolicy::Defined) {
                return true;
            } else if (Policy == NChatRobot::ETagCheckPolicy::Equals && valueIt->second == soughtValue) {
                return true;
            }
        }
    }
    return false;
}

ICondition::TPtr TDictionaryTagValueCondition::GetTemplateImpl(const TVector<TString>& params) const {
    THolder<TDictionaryTagValueCondition> result(new TDictionaryTagValueCondition());
    result->SetTagName(NChatScript::GetMaybeParametrizedField(TagName, params));
    result->SetPolicy(Policy);
    result->SetField(NChatScript::GetMaybeParametrizedField(Field, params));
    result->SetValue(NChatScript::GetMaybeParametrizedField(Value, params));
    return result.Release();
}

bool THadRidesCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    if (!raw.Has("value") || !raw["value"].IsBoolean()) {
        errors.AddMessage("had_rides_condition", "doesn't have boolean 'value' field");
        return false;
    }
    TargetState = raw["value"].GetBoolean();
    return true;
}

bool THadRidesCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const {
    auto hadRidesActual = !ctx->IsFirstRiding();
    return (hadRidesActual == TargetState);
}

ICondition::TPtr THadRidesCondition::GetTemplateImpl(const TVector<TString>& /*params*/) const {
    THolder<THadRidesCondition> result(new THadRidesCondition());
    result->SetTargetState(TargetState);
    return result.Release();
}

bool THandlerAnswerCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    if (!NJson::ParseField(raw, "path", Path, true)) {
        errors.AddMessage("handler_answer_condition", "doesn't have 'path' field");
        return false;
    }

    if (!NJson::ParseField(raw, "value", Value, true)) {
        errors.AddMessage("handler_answer_condition", "doesn't have 'value' field");
        return false;
    }
    return true;
}

bool THandlerAnswerCondition::IsMatching(const IChatUserContext::TPtr /*ctx*/, const TChatContext& chatContext) const {
    auto valueByPath = chatContext.GetLastRequestData().GetValueByPath(Path);
    return !!valueByPath && valueByPath->GetStringRobust() == Value;
}

ICondition::TPtr THandlerAnswerCondition::GetTemplateImpl(const TVector<TString>& params) const {
    THolder<THandlerAnswerCondition> result(new THandlerAnswerCondition());
    result->SetPath(NChatScript::GetMaybeParametrizedField(Path, params));
    result->SetValue(NChatScript::GetMaybeParametrizedField(Value, params));
    return result.Release();
}

bool THandlerCodeCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    if (!NJson::ParseField(raw, "code", Code, true)) {
        errors.AddMessage("handler_answer_condition", "doesn't have 'code' field");
        return false;
    }
    return true;
}

bool THandlerCodeCondition::IsMatching(const IChatUserContext::TPtr /*ctx*/, const TChatContext& chatContext) const {
    ui32 code = 0;
    if (!TryFromString(Code, code)) {
        return false;
    }
    return code == chatContext.GetLastRequestCode();
}

ICondition::TPtr THandlerCodeCondition::GetTemplateImpl(const TVector<TString>& params) const {
    THolder<THandlerCodeCondition> result(new THandlerCodeCondition());
    result->SetCode(NChatScript::GetMaybeParametrizedField(Code, params));
    return result.Release();
}

bool TPhoneVerifiedCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    Y_UNUSED(raw);
    Y_UNUSED(errors);
    return true;
}

bool TPhoneVerifiedCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const {
    Y_UNUSED(chatContext);
    if (!ctx) {
        return false;
    }
    auto userData = ctx->GetServer().GetDriveAPI()->GetUsersData();

    NDrive::TEntitySession localSession;
    if (!chatContext.GetTagsSession()) {
        ERROR_LOG << "No tags session" << Endl;
        localSession = userData->BuildSession(NSQL::ReadOnly);
    }
    auto& session = chatContext.GetTagsSession() ? *chatContext.GetTagsSession() : localSession;

    auto user = userData->RestoreUser(ctx->GetUserId(), session);
    if (!user) {
        ERROR_LOG << "cannot RestoreUser: " << session.GetStringReport() << Endl;
        return false;
    }
    return user->IsPhoneVerified();
}

ICondition::TPtr TPhoneVerifiedCondition::GetTemplateImpl(const TVector<TString>& params) const {
    Y_UNUSED(params);
    return MakeHolder<TPhoneVerifiedCondition>();
}

bool TOriginCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    return NJson::ParseField(raw["origin"], Origin, errors);
}

bool TOriginCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const {
    return ctx->GetOrigin() == Origin;
}

ICondition::TPtr TOriginCondition::GetTemplateImpl(const TVector<TString>& params) const {
    auto result = MakeHolder<TOriginCondition>();
    result->SetOrigin(NChatScript::GetMaybeParametrizedField(Origin, params));
    return result.Release();
}

bool TPhotoStatusCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    return
        NJson::ParseField(raw["max_age"], MaxAge, errors) &&
        NJson::ParseField(raw["statuses"], Statuses, errors) &&
        NJson::ParseField(raw["photo_type"], NJson::Stringify(PhotoType), errors);
}

bool TPhotoStatusCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const {
    Y_UNUSED(chatContext);
    auto driveApi = ctx->GetServer().GetDriveAPI();
    Y_ENSURE_BT(driveApi);
    Y_ENSURE_BT(driveApi->HasDocumentPhotosManager());

    const auto& photosDb = driveApi->GetDocumentPhotosManager().GetUserPhotosDB();
    const auto photos = photosDb.GetPhotosForUsers({ ctx->GetUserId() }, PhotoType);
    const auto now = Now();
    for (auto&& [_, photo] : photos) {
        if (photo.GetType() != PhotoType) {
            continue;
        }
        if (!Statuses.contains(photo.GetVerificationStatus())) {
            continue;
        }
        auto age = now - photo.GetSubmittedAt();
        if (age > MaxAge) {
            continue;
        }
        return true;
    }
    return false;
}

ICondition::TPtr TPhotoStatusCondition::GetTemplateImpl(const TVector<TString>& params) const {
    Y_UNUSED(params);
    auto result = MakeHolder<TPhotoStatusCondition>();
    result->SetMaxAge(MaxAge);
    result->SetStatuses(Statuses);
    result->SetPhotoType(PhotoType);
    return result;
}

bool TEmailVerifiedCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /*chatContext*/) const {
    if (!ctx) {
        return false;
    }
    auto userData = ctx->GetServer().GetDriveAPI()->GetUsersData();
    auto session = userData->BuildSession(true);
    auto user = userData->RestoreUser(ctx->GetUserId(), session);
    if (!user) {
        ERROR_LOG << "cannot RestoreUser: " << session.GetStringReport() << Endl;
        return false;
    }
    return user->IsEMailVerified();
}

ICondition::TPtr TEmailVerifiedCondition::GetTemplateImpl(const TVector<TString>& /*params*/) const {
    return MakeHolder<TEmailVerifiedCondition>();
}

bool TContextMapEqualsCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    return NJson::ParseField(raw["key"], Key, true, errors) &&
            NJson::ParseField(raw["value"], Value, errors);
}

ICondition::TPtr TContextMapEqualsCondition::GetTemplateImpl(const TVector<TString>& /*params*/) const {
    auto result = MakeHolder<TContextMapEqualsCondition>();
    result->SetKey(Key);
    result->SetValue(Value);
    return result.Release();
}

bool TContextMapEqualsCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const {
    auto unescapedKey = ctx->Unescape(GetKey(), chatContext);
    auto unescapedValue = ctx->Unescape(GetValue(), chatContext);
    auto it = chatContext.GetContextMap().find(unescapedKey);
    if (it == chatContext.GetContextMap().end()) {
        return false;
    }
    return it->second == unescapedValue;
}

ICondition::TPtr TContextMapContainsKeyCondition::GetTemplateImpl(const TVector<TString>& /*params*/) const {
    auto result = MakeHolder<TContextMapContainsKeyCondition>();
    result->SetKey(Key);
    return result.Release();
}

bool TContextMapContainsKeyCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const {
    auto unescapedKey = ctx->Unescape(GetKey(), chatContext);
    return chatContext.GetContextMap().contains(unescapedKey);
}

bool TContextMapContainsKeyCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    return NJson::ParseField(raw["key"], Key, true, errors);
}

ICondition::TPtr TCarTagsCondition::GetTemplateImpl(const TVector<TString>& /*params*/) const {
    auto result = MakeHolder<TCarTagsCondition>();
    result->SetHasAll(HasAll);
    result->SetHasAny(HasAny);
    result->SetHasNone(HasNone);
    result->SetCarId(CarId);
    return result.Release();
}

bool TCarTagsCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const {
    auto unescapedId = ctx->Unescape(CarId, chatContext);
    TVector<TString> toRestore;
    std::copy(HasAll.begin(), HasAll.end(), std::back_inserter(toRestore));
    std::copy(HasAny.begin(), HasAny.end(), std::back_inserter(toRestore));
    std::copy(HasNone.begin(), HasNone.end(), std::back_inserter(toRestore));
    TVector<TDBTag> tags;
    {
        auto session = ctx->GetServer().GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        if (!ctx->GetServer().GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreEntityTags(unescapedId, toRestore, tags, session)) {
            return false;
        }
    }
    bool hasAny = HasAny.empty();
    TSet<TString> actualHasAll;
    for (auto tag : tags) {
        if (HasNone.contains(tag->GetName())) {
            return false;
        }
        if (!hasAny && HasAny.contains(tag->GetName())) {
            hasAny = true;
        }
        if (HasAll.contains(tag->GetName())) {
            actualHasAll.emplace(tag->GetName());
        }
    }
    return hasAny && actualHasAll.size() == HasAll.size();
}

bool TCarTagsCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    return
        NJson::ParseField(raw["has_all"], HasAll, false, errors) &&
        NJson::ParseField(raw["has_any"], HasAny, false, errors) &&
        NJson::ParseField(raw["has_none"], HasNone, false, errors) &&
        NJson::ParseField(raw["car_id"], CarId, true, errors);
}

bool THasReferralCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /*chatContext*/) const {
    if (!ctx) {
        return false;
    }
    TString code = "";
    auto permissions = ctx->GetServer().GetDriveAPI()->GetUserPermissions(ctx->GetUserId(), TUserPermissionsFeatures());
    if (!permissions) {
        return false;
    }
    auto session = ctx->GetServer().GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
    if (!ctx->GetServer().GetDriveAPI()->CheckReferralProgramParticipation(code, *permissions, session)) {
        return false;
    }
    return !code.empty();
}

ICondition::TPtr THasReferralCondition::GetTemplateImpl(const TVector<TString>& /*params*/) const {
    return MakeHolder<THasReferralCondition>();
}

bool TTagFieldValueCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    return
        NJson::ParseField(raw, "tag_name", TagName, true, errors) &&
        NJson::ParseField(raw, "field_path", FieldPath, true, errors) &&
        NJson::ParseField(raw, "policy", NJson::Stringify(Policy), true, errors) &&
        NJson::ParseField(raw, "values", Values, Policy == NChatRobot::ETagCheckPolicy::Equals, errors);
}

bool TTagFieldValueCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const {
    IEntityTagsManager::TOptionalTags tags;
    {
        auto session = ctx->GetServer().GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        tags = ctx->GetServer().GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({ ctx->GetUserId() }), { ctx->Unescape(GetTagName(), chatContext) }, session);
    }
    if (!tags) {
        return false;
    }

    auto fieldPath = ctx->Unescape(FieldPath, chatContext);
    TSet<TString> soughtValues;
    for (auto&& value : Values) {
        soughtValues.emplace(ctx->Unescape(value, chatContext));
    }

    for (auto&& tag : *tags) {
        auto json = tag.BuildJsonReport();
        NJson::TJsonValue value;
        if (json.GetValueByPath(FieldPath, value)) {
            const TString& foundValue = value.GetStringRobust();
            if (Policy == NChatRobot::ETagCheckPolicy::Defined) {
                return true;
            } else if (Policy == NChatRobot::ETagCheckPolicy::Equals && soughtValues.contains(foundValue)) {
                return true;
            }
        }
    }
    return false;
}

ICondition::TPtr TTagFieldValueCondition::GetTemplateImpl(const TVector<TString>& params) const {
    THolder<TTagFieldValueCondition> result = MakeHolder<TTagFieldValueCondition>();
    result->SetTagName(NChatScript::GetMaybeParametrizedField(TagName, params));
    result->SetPolicy(Policy);
    result->SetFieldPath(NChatScript::GetMaybeParametrizedField(FieldPath, params));
    TSet<TString> resultValues;
    for (auto&& value : Values) {
        resultValues.emplace(NChatScript::GetMaybeParametrizedField(value, params));
    }
    result->SetValues(std::move(resultValues));
    return result.Release();
}

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& json, TDocumentsRecognitionResultCondition::TFieldWeights& weights) {
    if (!json.IsMap()) {
        return false;
    }

    for (auto&& [key, value] : json.GetMap()) {
        TRecognitionConfidenceData::EDocumentField field;
        if (!NJson::TryFromJson(key, NJson::Stringify(field)) ||
            !NJson::TryFromJson(value, weights[field])) {
            return false;
        }
    }

    return true;
}

bool TDocumentsRecognitionResultCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    return
        NJson::ParseField(raw["photo_type"], PhotoType, true, errors) &&
        NJson::ParseField(raw["recognizer_meta_filled"], RecognizerMetaFilled, false, errors) &&
        NJson::ParseField(raw["recognition_failed_value"], RecognitionFailedValue, false, errors) &&
        NJson::ParseField(raw["min_field_value"], MinFieldValue, false, errors) &&
        NJson::ParseField(raw["min_average_value"], MinAverageValue, false, errors) &&
        NJson::ParseField(raw["recognized_field_names"], RecognizedFieldNames, false, errors) &&
        NJson::ParseField(raw["field_weights"], FieldWeights, false, errors) &&
        NJson::ParseField(raw["bad_format_score_range"], NJson::Range(BadFormatModelScoreRange), false, errors) &&
        NJson::ParseField(raw["from_screen_score_range"], NJson::Range(FromScreenModelScoreRange), false, errors) &&
        NJson::ParseField(raw["quasi_fms_score_range"], NJson::Range(QuasiFmsScoreRange), false, errors) &&
        NJson::ParseField(raw["quasi_gibdd_score_range"], NJson::Range(QuasiGibddScoreRange), false, errors);
}

bool IsMaybeInRange(TMaybe<double> value, TRange<double> range) {
    if (!range) {
        return true;
    }
    if (!value) {
        return false;
    }
    if (range.From && *value < *range.From) {
        return false;
    }
    if (range.To && *value > *range.To) {
        return false;
    }
    return true;
}

bool TDocumentsRecognitionResultCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /*chatContext*/) const {
    auto evlog = NDrive::GetThreadEventLogger();
    auto typeToPhotoMap = ctx->GetServer().GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB().GetTypeToRecentPhotoMapping(ctx->GetUserId(), { PhotoType });
    auto photoIt = typeToPhotoMap.find(PhotoType);
    if (photoIt == typeToPhotoMap.end()) {
        return false;
    }
    auto photo = std::move(photoIt->second);
    if (HasRecognizerMetaFilled() && GetRecognizerMetaFilledRef() != photo.HasRecognizerMeta()) {
        return false;
    }
    if (!photo.HasRecognizerMeta()) {
        return !HasRecognizerMetaFilled();
    }
    if (HasRecognitionFailedValue() && photo.GetRecognizerMetaRef().IsRecognitionFailed() != GetRecognitionFailedValueRef()) {
        return false;
    }
    if (!IsMaybeInRange(photo.GetRecognizerMetaRef().OptionalQuasiGibddScore(), QuasiGibddScoreRange)) {
        return false;
    }
    if (!IsMaybeInRange(photo.GetRecognizerMetaRef().OptionalQuasiFmsScore(), QuasiFmsScoreRange)) {
        return false;
    }
    if (!IsMaybeInRange(photo.GetRecognizerMetaRef().OptionalFromScreenScore(), FromScreenModelScoreRange)) {
        return false;
    }
    if (!IsMaybeInRange(photo.GetRecognizerMetaRef().OptionalBadFormatScore(), BadFormatModelScoreRange)) {
        return false;
    }
    if (!RecognizedFieldNames.empty()) {
        double confidencesSum = 0;
        for (auto&& documentField : RecognizedFieldNames) {
            auto fieldValue = photo.GetRecognizerMetaRef().GetField(documentField);
            if (fieldValue == TRecognitionConfidenceData::FIELD_DOES_NOT_EXIST || fieldValue == 0) {
                if (evlog) {
                    evlog->AddEvent(NJson::TMapBuilder
                        ("event", "TDocumentsRecognitionResultCondition")
                        ("topic", "registration")
                        ("message", "Field " + ::ToString(documentField) + " is missing or equals to 0. Anyway, it will be treated as 0")
                    );
                }
                fieldValue = 0;
            }

            if (fieldValue < MinFieldValue) {
                return false;
            }

            double weight = FieldWeights.contains(documentField) ? FieldWeights.at(documentField) : 1;
            confidencesSum += weight * fieldValue;
        }
        if (HasMinAverageValue() && GetMinAverageValueRef() > confidencesSum / RecognizedFieldNames.size()) {
            return false;
        }
    }
    return true;
}

ICondition::TPtr TDocumentsRecognitionResultCondition::GetTemplateImpl(const TVector<TString>& /*params*/) const {
    auto result = MakeHolder<TDocumentsRecognitionResultCondition>();
    result->SetRecognizerMetaFilled(RecognizerMetaFilled);
    result->SetRecognitionFailedValue(RecognitionFailedValue);
    result->SetMinFieldValue(MinFieldValue);
    result->SetMinAverageValue(MinAverageValue);
    result->SetBadFormatModelScoreRange(BadFormatModelScoreRange);
    result->SetFromScreenModelScoreRange(FromScreenModelScoreRange);
    result->SetQuasiFmsScoreRange(QuasiFmsScoreRange);
    result->SetQuasiGibddScoreRange(QuasiGibddScoreRange);
    result->SetRecognizedFieldNames(RecognizedFieldNames);
    result->SetFieldWeights(FieldWeights);
    result->SetPhotoType(PhotoType);
    return result.Release();
}

bool TDocumentsCheckStatusCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    return
        NJson::ParseField(raw["main_types"], MainTypes, true, errors) &&
        NJson::ParseField(raw["main_statuses"], MainStatuses, true, errors) &&
        NJson::ParseField(raw["main_aggregation"], NJson::Stringify(MainAggregation), true, errors) &&
        NJson::ParseField(raw["additional_types"], AdditionalTypes, false, errors) &&
        NJson::ParseField(raw["additional_statuses"], AdditionalStatuses, false, errors) &&
        NJson::ParseField(raw["additional_aggregation"], NJson::Stringify(AdditionalAggregation), false, errors);
}

bool TDocumentsCheckStatusCondition::CheckMain(const TVector<TUserDocumentsCheck>& checks) const {
    auto checker = [this](const TUserDocumentsCheck& check) -> bool {
        return MainStatuses.contains(check.GetStatus());
    };
    return CheckAggregation(checks, MainAggregation, checker);
}

bool TDocumentsCheckStatusCondition::CheckAdditional(const TVector<TUserDocumentsCheck>& checks) const {
    if (!checks) {
        return true;
    }
    auto checker = [this](const TUserDocumentsCheck& check) -> bool {
        return AdditionalStatuses.contains(check.GetStatus());
    };
    return CheckAggregation(checks, AdditionalAggregation, checker);
}

TVector<TUserDocumentsCheck> TDocumentsCheckStatusCondition::GetChecks(const TString& userId, const TSet<TString>& types, const TUserDocumentsChecksManager* documentsChecksManager, NDrive::TEntitySession& session) const {
    if (!types) {
        return {};
    }
    auto optionalChecks = documentsChecksManager->GetChecks(userId, types, session);
    if (!optionalChecks) {
        return {};
    }
    return *optionalChecks;
}

bool TDocumentsCheckStatusCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const {
    auto evlog = NDrive::GetThreadEventLogger();
    auto documentsChecksManager = ctx->GetServer().GetUserDocumentsChecksManager();
    if (!documentsChecksManager) {
        return false;
    }

    NDrive::TEntitySession localSession;
    if (!chatContext.GetChatSession()) {
        if (evlog) {
            evlog->AddEvent(NJson::TMapBuilder
                ("event", "IPrivateDataFetcher")
                ("topic", "registration")
                ("error", "No chat session")
            );
        }
        localSession = ctx->GetServer().GetChatEngine()->BuildSession(true);
    }
    auto& session = chatContext.GetChatSession() ? *chatContext.GetChatSession() : localSession;

    TVector<TUserDocumentsCheck> mainChecks = GetChecks(ctx->GetUserId(), MainTypes, documentsChecksManager, session);
    TVector<TUserDocumentsCheck> additionalChecks = GetChecks(ctx->GetUserId(), AdditionalTypes, documentsChecksManager, session);

    auto status = CheckMain(mainChecks) && CheckAdditional(additionalChecks);

    if (auto eventLogger = NDrive::GetThreadEventLogger()) {
        TString currentStep;
        auto chatRobot = ctx->GetChatRobot();
        if (chatRobot) {
            chatRobot->GetActualStateId(ctx->GetUserId(), ctx->GetChatTopic(), currentStep);
        }

        eventLogger->AddEvent(NJson::TMapBuilder
            ("topic", "registration")
            ("event", "DocumentsCheckStatusCondition")
            ("condition", NJson::TMapBuilder
                ("type", "documents_check_status")
                ("main_aggregation", ::ToString(MainAggregation))
                ("main_types", NJson::ToJson(MainTypes))
                ("main_statuses", NJson::ToJson(MainStatuses))
                ("additional_aggregation", ::ToString(AdditionalAggregation))
                ("additional_types", NJson::ToJson(AdditionalTypes))
                ("additional_statuses", NJson::ToJson(AdditionalStatuses))
                ("status", status)
                ("step", currentStep)
            )
        );
    }
    return status;
}

ICondition::TPtr TDocumentsCheckStatusCondition::GetTemplateImpl(const TVector<TString>& /*params*/) const {
    auto result = MakeHolder<TDocumentsCheckStatusCondition>();
    result->SetMainTypes(MainTypes);
    result->SetMainStatuses(MainStatuses);
    result->SetMainAggregation(MainAggregation);
    result->SetAdditionalTypes(AdditionalTypes);
    result->SetAdditionalStatuses(AdditionalStatuses);
    result->SetAdditionalAggregation(AdditionalAggregation);
    return result.Release();
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, IPrivateDataFetcher::TDocumentField& result) {
    return
        NJson::ParseField(value, "document", NJson::Stringify(result.Document), true) &&
        NJson::ParseField(value, "field", result.Field, true);
}

bool IPrivateDataFetcher::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    return NJson::ParseField(raw["timeout"], NJson::Hr(Timeout), false, errors);
}

bool IPrivateDataFetcher::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const {
    auto evlog = NDrive::GetThreadEventLogger();
    TDriveUserData userData;
    auto userId = Yensured(ctx)->GetUserId();

    NDrive::TEntitySession localSession;
    if (!chatContext.GetTagsSession()) {
        if (evlog) {
            evlog->AddEvent(NJson::TMapBuilder
                ("event", "IPrivateDataFetcher")
                ("topic", "registration")
                ("error", "No tags session")
            );
        }
        localSession = ctx->GetServer().GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
    }
    auto& session = chatContext.GetTagsSession() ? *chatContext.GetTagsSession() : localSession;

    auto optionalUser = ctx->GetServer().GetDriveAPI()->GetUsersData()->FetchInfo(userId, session);
    auto userPtr = optionalUser.GetResultPtr(userId);
    if (userPtr) {
        userData = std::move(*userPtr);
    } else {
        if (evlog) {
            evlog->AddEvent(NJson::TMapBuilder
                ("event", "IPrivateDataFetcher")
                ("topic", "registration")
                ("error", "No user data is found")
                ("user", userId)
            );
        }
        return false;
    }

    NThreading::TFuture<TUserPassportData> passportDataFuture = NThreading::MakeFuture(TUserPassportData());
    NThreading::TFuture<TUserDrivingLicenseData> licenseDataFuture = NThreading::MakeFuture(TUserDrivingLicenseData());
    if (!userData.GetPassportDatasyncRevision().empty()) {
        passportDataFuture = ctx->GetServer().GetDriveAPI()->GetPrivateDataClient().GetPassport(userData, userData.GetPassportDatasyncRevision());
    }
    if (!userData.GetDrivingLicenseDatasyncRevision().empty()) {
        licenseDataFuture = ctx->GetServer().GetDriveAPI()->GetPrivateDataClient().GetDrivingLicense(userData, userData.GetDrivingLicenseDatasyncRevision());
    }
    TVector<NThreading::TFuture<void>> futures = { passportDataFuture.IgnoreResult(), licenseDataFuture.IgnoreResult() };
    auto eventLogState = NDrive::TEventLog::CaptureState();
    auto compareFuture = NThreading::WaitExceptionOrAll(futures).Apply([
        this,
        eventLogState = std::move(eventLogState),
        &userId,
        passportDataFuture = std::move(passportDataFuture),
        licenseDataFuture = std::move(licenseDataFuture)
    ] (const auto& waiter) mutable {
        auto elsg = NDrive::TEventLog::Guard(eventLogState);
        auto evlog = NDrive::GetThreadEventLogger();
        if (waiter.HasException() || !waiter.HasValue()) {
            if (evlog) {
                evlog->AddEvent(NJson::TMapBuilder
                    ("event", "IPrivateDataFetcher")
                    ("topic", "registration")
                    ("error", "Failed to get data from Datasync: " + NThreading::GetExceptionMessage(waiter))
                    ("user", userId)
                );
            }
            return false;
        }
        auto passportJson = passportDataFuture.GetValue().SerializeToDatasyncJson();
        auto licenseJson = licenseDataFuture.GetValue().SerializeToDatasyncJson();
        return IsPrivateDataMatching(passportJson, licenseJson);
    });
    if (!compareFuture.Wait(Timeout)) {
        auto evlog = NDrive::GetThreadEventLogger();
        Log(NJson::TMapBuilder
                ("event", "IPrivateDataFetcher")
                ("topic", "registration")
                ("error", "Failed to get data from Datasync: timeout")
                ("user", userData.GetUserId()),
            evlog);
        return false;
    }
    return compareFuture.HasValue() ? compareFuture.GetValue() : false;
}

template <typename TDatasyncCondition>
TAtomicSharedPtr<TDatasyncCondition> IPrivateDataFetcher::GetTemplateImplBase(const TVector<TString>& /* params */) const {
    auto result = MakeHolder<TDatasyncCondition>();
    result->SetTimeout(GetTimeout());
    return result;
}

bool TPrivateDataEqualsCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    return
        TBase::DoParse(raw, errors) &&
        NJson::ParseField(raw["compare_parameters"], CompareParameters, true, errors);
}

bool TPrivateDataEqualsCondition::IsPrivateDataMatching(const NJson::TJsonValue& passportJson, const NJson::TJsonValue& licenseJson) const {
    NJson::TJsonValue value;
    for (auto&& compareParameter : CompareParameters) {
        TSet<TString> strFields;
        for (auto&& fieldName : compareParameter) {
            const NJson::TJsonValue& comparedDocument = fieldName.Document == TBase::EDocumentType::Passport ? passportJson : licenseJson;
            if (!comparedDocument.GetValueByPath(fieldName.Field, value)) {
                auto evlog = NDrive::GetThreadEventLogger();
                Log(NJson::TMapBuilder
                        ("event", "TPrivateDataEqualsCondition::IsPrivateDataMatching")
                        ("error", "Field is missing. Aborting with false")
                        ("field", fieldName.Field)
                        ("topic", "registration"),
                    evlog);
                return false;
            }
            const TString& foundValue = value.GetStringRobust();
            strFields.insert(ToLowerUTF8(foundValue));
        }
        if (strFields.size() > 1) {
            return false;
        }
    }
    return true;
}

ICondition::TPtr TPrivateDataEqualsCondition::GetTemplateImpl(const TVector<TString>& params) const {
    auto result = GetTemplateImplBase<TPrivateDataEqualsCondition>(params);
    result->SetCompareParameters(CompareParameters);
    return result;
}

bool TPrivateDataFieldMatchesCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    TString matchCondition;
    TString aggregatedType;
    return
        TBase::DoParse(raw, errors) &&
        NJson::ParseField(raw["check_parameters"], CheckParameters, true, errors) &&
        NJson::ParseField(raw["match_type"], NJson::Stringify(MatchType), true, errors) &&
        NJson::ParseField(raw["aggregation_type"], aggregatedType, false, errors) &&
        NJson::ParseField(raw["match_condition"], matchCondition, false, errors) &&
        BuildMatcher(matchCondition, aggregatedType);
}

bool TPrivateDataFieldMatchesCondition::BuildMatcher(const TString& matchCondition, const TString& aggregatedType) {
    if (matchCondition.empty() && MatchType != EMatchType::HasLatin) {
        return false;
    }

    switch(MatchType) {
        case EMatchType::HasLatin:
            AggregationType = EAggregatedConditionType::AnyOf;
            Matcher = TRegExMatch("[a-zA-Z]");
            return true;
        case EMatchType::HasAnyOfSymbols:
            AggregationType = EAggregatedConditionType::AnyOf;
            Matcher = TRegExMatch("[" + matchCondition + "]");
            return true;
        case EMatchType::HasOnlySymbols:
            AggregationType = EAggregatedConditionType::AllOf;
            Matcher = TRegExMatch("^[" + matchCondition + "]+$");
            return true;
        case EMatchType::Regex:
            if (!aggregatedType || !TryFromString(aggregatedType, AggregationType)) {
                return false;
            }
            Matcher = TRegExMatch(matchCondition);
            return true;
        default:
            return false;
    }
}

bool TPrivateDataFieldMatchesCondition::IsPrivateDataMatching(const NJson::TJsonValue& passportJson, const NJson::TJsonValue& licenseJson) const {
    TVector<TString> values;
    NJson::TJsonValue value;
    auto evlog = NDrive::GetThreadEventLogger();
    for (auto&& fieldName : CheckParameters) {
        const NJson::TJsonValue& checkedDocument = fieldName.Document == TBase::EDocumentType::Passport ? passportJson : licenseJson;
        if (!checkedDocument.GetValueByPath(fieldName.Field, value)) {
            Log(NJson::TMapBuilder
                    ("event", "TPrivateDataFieldMatchesCondition::IsPrivateDataMatching")
                    ("error", "Field is missing. Skipping it")
                    ("field", fieldName.Field)
                    ("topic", "registration"),
                evlog);
            continue;
        }
        const TString& fieldValue = value.GetStringRobust();
        values.push_back(fieldValue);
    }

    auto checker = [this](const TString& value) -> bool {
        return Matcher.Match(value.data());
    };
    return CheckAggregation(values, AggregationType, checker);
}


ICondition::TPtr TPrivateDataFieldMatchesCondition::GetTemplateImpl(const TVector<TString>& params) const {
    auto result = GetTemplateImplBase<TPrivateDataFieldMatchesCondition>(params);
    result->SetCheckParameters(CheckParameters);
    result->SetMatchType(MatchType);
    result->SetMatcher(Matcher);
    return result;
}

bool THasRoleCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const {
    if (!ctx || !ctx->GetUserPermissions()) {
        return false;
    }

    TSet<TString> roles;
    const auto& rolesByUser = ctx->GetUserPermissions()->GetRolesByUser();
    for (const auto& role : rolesByUser) {
        roles.insert(role.GetRoleId());
    }

    auto checker = [&roles](const TString& role) -> bool {
        return roles.contains(role);
    };
    return CheckAggregation(RoleNames, AggregationType, checker);
}

bool THasRoleCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    return
        TBase::DoParse(raw, errors) &&
        NJson::ParseField(raw, "aggregation_type", NJson::Stringify(AggregationType), true) &&
        NJson::ParseField(raw, "role_names", RoleNames, true);
}

ICondition::TPtr THasRoleCondition::GetTemplateImpl(const TVector<TString>& params) const {
    auto result = MakeHolder<THasRoleCondition>();
    for (const auto& role : RoleNames) {
        result->RoleNames.push_back(NChatScript::GetMaybeParametrizedField(role, params));
    }
    return result.Release();
}

bool TRidesStatisticCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext & chatContext) const {
    auto evlog = NDrive::GetThreadEventLogger();
    if (!ctx || !ctx->GetServer().GetDriveAPI()) {
        Log(NJson::TMapBuilder
                ("event", "TRidesStatisticCondition::IsMatching")
                ("error", "Failed to get DriveAPI"),
            evlog
        );
        return false;
    }

    const TString& userId = ctx->GetUserId();
    if (!userId) {
        Log(NJson::TMapBuilder
                ("event", "TRidesStatisticCondition::IsMatching")
                ("error", "Empty userId"),
            evlog
        );
        return false;
    }

    NDrive::TEntitySession localSession;
    if (!chatContext.GetTagsSession()) {
        localSession = Yensured(ctx->GetServer().GetChatEngine())->template BuildTx<NSQL::ReadOnly>();
    }
    auto& session = chatContext.GetTagsSession() ? *chatContext.GetTagsSession() : localSession;

    THistoryRidesContext context(ctx->GetServer(), Now() - Period);
    auto ydbTx = ctx->GetServer().GetDriveAPI()->BuildYdbTx<NSQL::ReadOnly>("rides_statistic_condition", &ctx->GetServer());
    if (!context.InitializeUser(userId, session, ydbTx, TInstant::Max(), RidesAmount)) {
        Log(NJson::TMapBuilder
                ("event", "TRidesStatisticCondition::IsMatching")
                ("error", "Cannot initialize HistoryRidesContext")
                ("user", userId),
            evlog
        );
        return false;
    }

    auto rideSessions = context.GetSessions(TInstant::Max(), RidesAmount);
    if (rideSessions.size() < RidesAmount) {
        return false;
    }

    if (!RequiredRidePrice && !RequiredRidesSum) {
        return true;
    }

    double totalSum = 0;
    for (const auto& rideSession : rideSessions) {
        auto price = rideSession.GetSumPrice();
        if (RequiredRidePrice && price < *RequiredRidePrice) {
            return false;
        }
        totalSum += price;
    }

    if (RequiredRidesSum && totalSum < *RequiredRidesSum) {
        return false;
    }

    return true;
}

bool TRidesStatisticCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    return
        TBase::DoParse(raw, errors) &&
        NJson::ParseField(raw, "rides_amount", RidesAmount, true) &&
        NJson::ParseField(raw, "period", Period) &&
        NJson::ParseField(raw, "ride_price", RequiredRidePrice, false) &&
        NJson::ParseField(raw, "rides_sum", RequiredRidesSum, false);
}

ICondition::TPtr TRidesStatisticCondition::GetTemplateImpl(const TVector<TString>& /* params */) const {
    auto result = MakeHolder<TRidesStatisticCondition>();
    result->SetRidesAmount(RidesAmount);
    result->SetPeriod(Period);
    result->SetRequiredRidePrice(RequiredRidePrice);
    result->SetRequiredRidesSum(RequiredRidesSum);
    return result;
}

const NDrive::TUserEventsApi* TUserFeatureCondition::GetFeatureClient(const IChatUserContext::TPtr ctx) const {
    const auto& server = Yensured(ctx)->GetServer();
    return UseFeatureClient ? server.GetFeaturesClient() : server.GetUserEventsApi();
}

bool TUserFeatureCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const {
    auto evlog = NDrive::GetThreadEventLogger();
    const NDrive::TUserEventsApi* featureClient = GetFeatureClient(ctx);
    if (!featureClient) {
        Log(NJson::TMapBuilder
                ("event", "TUserFeatureCondition::IsMatching")
                ("error", "Feature client is undefined")
                ("topic", "registration"),
            evlog
        );
        return false;
    }

    const TString& userId = ctx->GetUserId();
    if (!userId) {
        Log(NJson::TMapBuilder
                ("event", "TUserFeatureCondition::IsMatching")
                ("error", "Empty userId")
                ("topic", "registration")
                ("user", userId),
            evlog
        );
        return false;
    }

    auto featuresFuture = featureClient->GetUserFeatures(userId);
    if (!featuresFuture.Wait(Timeout) || featuresFuture.HasException() || !featuresFuture.HasValue()) {
        Log(NJson::TMapBuilder
                ("event", "TUserFeatureCondition::IsMatching")
                ("error", "Failed to get features: " + NThreading::GetExceptionMessage(featuresFuture))
                ("user", userId),
            evlog
        );
        return false;
    }
    auto features = featuresFuture.ExtractValue();

    const IFactorsInfo* factorsInfo = NDrive::GetOfferFactorsInfo2();
    if (!factorsInfo) {
        Log(NJson::TMapBuilder
                ("event", "TUserFeatureCondition::IsMatching")
                ("error", "factorInfo is null")
                ("user", userId),
            evlog
        );
    }
    for (size_t i = 0; i < std::min(factorsInfo->GetFactorCount(), features.Factors2.size()); ++i) {
        if (factorsInfo->GetFactorName(i) == FeatureName) {
            return IsMaybeInRange(features.Factors2[i], FeatureRange);
        }
    }

    return false;
}

bool TUserFeatureCondition::DoParse(const NJson::TJsonValue &raw, TMessagesCollector &errors) {
    return
        TBase::DoParse(raw, errors) &&
        NJson::ParseField(raw, "feature_range", NJson::Range(FeatureRange), true, errors) &&
        NJson::ParseField(raw, "feature_name",  FeatureName, true, errors) &&
        NJson::ParseField(raw, "use_feature_client", UseFeatureClient, false, errors) &&
        NJson::ParseField(raw, "timeout", Timeout, false);
}

ICondition::TPtr TUserFeatureCondition::GetTemplateImpl(const TVector<TString>& params) const {
    auto result = MakeHolder<TUserFeatureCondition>();
    result->SetFeatureName(NChatScript::GetMaybeParametrizedField(FeatureName, params));
    result->SetFeatureRange(GetFeatureRange());
    result->SetUseFeatureClient(IsUseFeatureClient());
    result->SetTimeout(GetTimeout());
    return result;
}

bool TAreaMessageCondition::IsMatching(const IChatUserContext::TPtr ctx, const TChatContext & /* chatContext */) const {
    auto evlog = NDrive::GetThreadEventLogger();
    if (!ctx || !ctx->GetMessage()) {
        Log(NJson::TMapBuilder
                ("event", "TAreaMessageCondition::IsMatching")
                ("error", "Can't get the message"),
            evlog
        );
        return false;
    }
    auto message = ctx->GetMessage();

    if (!ctx->GetServer().GetDriveAPI() || !ctx->GetServer().GetDriveAPI()->GetAreasDB()) {
        Log(NJson::TMapBuilder
                ("event", "TAreaMessageCondition::IsMatching")
                ("error", "AreasDB is not configured"),
            evlog
        );
        return false;
    }

    TGeoCoord coord;
    if (!TryFromString<TGeoCoord>(message, coord)) {
        Log(NJson::TMapBuilder
                ("event", "TAreaMessageCondition::IsMatching")
                ("error", "Failed to parse geo coordinates")
                ("message", message),
            evlog
        );
        return false;
    }

    const auto& areaManager = ctx->GetServer().GetDriveAPI()->GetAreasDB();
    switch(FetchType) {
        case EFetchType::AreaIds: {
            const auto closeAreasIds = areaManager->GetAreaIdsInPoint(coord);
            for (const auto& areaId : AreaIds) {
                if (closeAreasIds.contains(areaId)) {
                    return true;
                }
            }
            return false;
        }
        case EFetchType::GeoTags: {
            const auto closeGeoTagNames = areaManager->CheckGeoTags(coord, GeoTagNames, TInstant::Zero());
            return !closeGeoTagNames.empty();
        }
    }
}

bool TAreaMessageCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    return
        TBase::DoParse(raw, errors) &&
        NJson::ParseField(raw, "fetch_type", NJson::Stringify(FetchType), true) &&
        NJson::ParseField(raw, "area_ids", AreaIds, false) &&
        NJson::ParseField(raw, "geo_tags_names", GeoTagNames, false) &&
        ((FetchType == EFetchType::AreaIds) && AreaIds || (FetchType == EFetchType::GeoTags) && GeoTagNames);
}

ICondition::TPtr TAreaMessageCondition::GetTemplateImpl(const TVector<TString>& params) const {
    auto result = MakeHolder<TAreaMessageCondition>();
    result->SetAreaIds(NChatScript::GetMaybeParametrizedStringContainer(AreaIds, params));
    result->SetGeoTagNames(NChatScript::GetMaybeParametrizedStringContainer(GeoTagNames, params));
    return result;
}

bool TContextMapCompareCondition::IsMatching(const IChatUserContext::TPtr /* ctx */, const TChatContext& chatContext) const {
    auto evlog = NDrive::GetThreadEventLogger();

    double targetVal;
    if (!chatContext.GetContextMap().TryToGet(Value, targetVal)) {
        Log(NJson::TMapBuilder
            ("event", "TContextMapCompareCondition")
            ("error", "Failed to get value " + Value + " from the context map"),
        evlog);
        return false;
    }

    if (UseFloat) {
        return CompareValues(targetVal, Limit);
    } else {
        return CompareValues(std::lround(targetVal), std::lround(Limit));
    }
}

bool TContextMapCompareCondition::DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    return
        TBase::DoParse(raw, errors) &&
        NJson::ParseField(raw, "use_float", UseFloat, false) &&
        NJson::ParseField(raw, "limit", Limit, true) &&
        NJson::ParseField(raw, "value", Value, true) &&
        NJson::ParseField(raw, "compare_type", NJson::Stringify(CompareType), true);
}

ICondition::TPtr TContextMapCompareCondition::GetTemplateImpl(const TVector<TString>& params) const {
    auto result = MakeHolder<TContextMapCompareCondition>();
    result->SetValue(NChatScript::GetMaybeParametrizedField(Value, params));
    result->SetLimit(Limit);
    result->SetUseFloat(GetUseFloat());
    result->SetCompareType(CompareType);
    return result;
}
