#pragma once

#include "ifaces.h"

#include <drive/backend/chat_robots/configuration/chat_script.h>

#include <drive/backend/areas/areas.h>
#include <drive/backend/chat/engine.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/history_iterator/history_iterator.h>
#include <drive/backend/offers/factors/factors_info.h>
#include <drive/backend/roles/permissions.h>
#include <drive/backend/users/user_documents.h>
#include <drive/backend/users/user_documents_check.h>

#include <drive/library/cpp/user_events_api/client.h>

#include <rtline/util/json_processing.h>
#include <rtline/util/types/accessor.h>

namespace NChatRobot {
    enum ETagCheckPolicy {
        Defined /* "defined" */,
        Equals /* "equals" */,
    };
}

class TAndCondition : public ICondition {
    static TFactory::TRegistrator<TAndCondition> Registrator;

protected:
    virtual size_t GetMinSubConditions() const override {
        return 2;
    }

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

class TOrCondition : public ICondition {
    static TFactory::TRegistrator<TOrCondition> Registrator;

protected:
    virtual size_t GetMinSubConditions() const override {
        return 2;
    }

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

class TNotCondition : public ICondition {
    static TFactory::TRegistrator<TNotCondition> Registrator;

protected:
    virtual size_t GetMinSubConditions() const override {
        return 1;
    }

    virtual size_t GetMaxSubConditions() const override {
        return 1;
    }

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

class ITerminalCondition : public ICondition {
private:
    virtual size_t GetMinSubConditions() const override {
        return 0;
    }

    virtual size_t GetMaxSubConditions() const override {
        return 0;
    }
};

class TPerformsCondition : public ITerminalCondition {
    static TFactory::TRegistrator<TPerformsCondition> Registrator;

    R_FIELD(TString, TagName);

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

class THasTagCondition : public ITerminalCondition {
    static TFactory::TRegistrator<THasTagCondition> Registrator;

    R_FIELD(TString, TagName);

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

class THasWalletCondition : public ITerminalCondition {
    static TFactory::TRegistrator<THasWalletCondition> Registrator;

    R_FIELD(TString, Name);
    R_OPTIONAL(bool, Active);
protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

class TUserInStatusCondition : public ITerminalCondition {
    static TFactory::TRegistrator<TUserInStatusCondition> Registrator;

    R_FIELD(TString, Status);

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

class TPlatformCondition : public ITerminalCondition {
    static TFactory::TRegistrator<TPlatformCondition> Registrator;

    R_FIELD(TString, Name);
    R_FIELD(IRequestProcessor::EApp, EnumName, IRequestProcessor::EApp::Unknown);

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

class TAppBuildCondition : public ITerminalCondition {
    static TFactory::TRegistrator<TAppBuildCondition> Registrator;

    R_FIELD(ui32, MinValue, 0);

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

class THasActionCondition : public ITerminalCondition {
    static TFactory::TRegistrator<THasActionCondition> Registrator;

    R_FIELD(TString, Name);

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

class THasDebtCondition : public ITerminalCondition {
    static TFactory::TRegistrator<THasDebtCondition> Registrator;

    R_FIELD(bool, Debtor, false);

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

class TInAreaCondition : public ITerminalCondition {
    static TFactory::TRegistrator<TInAreaCondition> Registrator;

    R_FIELD(TString, Name);

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

class TRegistrationStepCondition : public ITerminalCondition {
    static TFactory::TRegistrator<TRegistrationStepCondition> Registrator;

    R_FIELD(TString, Step);

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

class TEnteredMessageCondition : public ITerminalCondition {
    static TFactory::TRegistrator<TEnteredMessageCondition> Registrator;

    R_FIELD(TString, Message);

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

class TDictionaryTagValueCondition : public ITerminalCondition {
    static TFactory::TRegistrator<TDictionaryTagValueCondition> Registrator;

public:
    R_FIELD(TString, TagName);
    R_FIELD(NChatRobot::ETagCheckPolicy, Policy);
    R_FIELD(TString, Field);
    R_FIELD(TString, Value);

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

class THadRidesCondition : public ITerminalCondition {
    static TFactory::TRegistrator<THadRidesCondition> Registrator;

    R_FIELD(bool, TargetState, false);

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

class THandlerAnswerCondition : public ITerminalCondition {
    static TFactory::TRegistrator<THandlerAnswerCondition> Registrator;

    R_FIELD(TString, Path);
    R_FIELD(TString, Value);

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

class THandlerCodeCondition : public ITerminalCondition {
    static TFactory::TRegistrator<THandlerCodeCondition> Registrator;

    R_FIELD(TString, Code);

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

class TPhoneVerifiedCondition: public ITerminalCondition {
public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

private:
    static TFactory::TRegistrator<TPhoneVerifiedCondition> Registrator;
};

class TOriginCondition : public ITerminalCondition {
public:
    R_FIELD(TString, Origin);

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;

private:
    static TFactory::TRegistrator<TOriginCondition> Registrator;
};

class TPhotoStatusCondition : public ITerminalCondition {
public:
    R_FIELD(TDuration, MaxAge, TDuration::Max());
    R_FIELD(TSet<NUserDocument::EVerificationStatus>, Statuses);
    R_FIELD(NUserDocument::EType, PhotoType, NUserDocument::EType::Unknown);

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

private:
    static TFactory::TRegistrator<TPhotoStatusCondition> Registrator;
};

class TEmailVerifiedCondition: public ITerminalCondition {
public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;

private:
    static TFactory::TRegistrator<TEmailVerifiedCondition> Registrator;
};

class TContextMapEqualsCondition: public ITerminalCondition {
private:
    R_FIELD(TString, Key);
    R_FIELD(TString, Value);

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

private:
    static TFactory::TRegistrator<TContextMapEqualsCondition> Registrator;
};

class TContextMapContainsKeyCondition: public ITerminalCondition {
private:
    R_FIELD(TString, Key);
    R_FIELD(TString, Value);

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

private:
    static TFactory::TRegistrator<TContextMapContainsKeyCondition> Registrator;
};

class TCarTagsCondition: public ITerminalCondition {
private:
    R_FIELD(TSet<TString>, HasAll);
    R_FIELD(TSet<TString>, HasAny);
    R_FIELD(TSet<TString>, HasNone);
    R_FIELD(TString, CarId);

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;

private:
    static TFactory::TRegistrator<TCarTagsCondition> Registrator;
};

class THasReferralCondition : public ITerminalCondition {
public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;

private:
    static TFactory::TRegistrator<THasReferralCondition> Registrator;
};

class TTagFieldValueCondition : public ITerminalCondition {
    static TFactory::TRegistrator<TTagFieldValueCondition> Registrator;

public:
    R_FIELD(TString, TagName);
    R_FIELD(NChatRobot::ETagCheckPolicy, Policy);
    R_FIELD(TString, FieldPath);
    R_FIELD(TSet<TString>, Values);

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

class TDocumentsRecognitionResultCondition: public ITerminalCondition {
public:
    using TFieldWeights = TMap<TRecognitionConfidenceData::EDocumentField, double>;

private:
    static inline TFactory::TRegistrator<TDocumentsRecognitionResultCondition> Registrator{ICondition::EType::DocumentsRecognitionResult};

    R_OPTIONAL(bool, RecognizerMetaFilled);
    R_OPTIONAL(bool, RecognitionFailedValue);
    R_OPTIONAL(double, MinFieldValue, 0);
    R_OPTIONAL(double, MinAverageValue, 0);
    R_FIELD(TRange<double>, BadFormatModelScoreRange);
    R_FIELD(TRange<double>, FromScreenModelScoreRange);
    R_FIELD(TRange<double>, QuasiFmsScoreRange);
    R_FIELD(TRange<double>, QuasiGibddScoreRange);
    R_FIELD(TVector<TRecognitionConfidenceData::EDocumentField>, RecognizedFieldNames);
    R_FIELD(TFieldWeights, FieldWeights);
    R_FIELD(NUserDocument::EType, PhotoType);

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

enum class EAggregatedConditionType {
    AllOf /* "all_of" */,
    AnyOf /* "any_of" */,
    NoneOf /* "none_of" */,
};

template <typename TCheckedObject>
class IAggregatedCondition {
protected:
    bool CheckAggregation(const TVector<TCheckedObject>& objects, EAggregatedConditionType aggregationType, std::function<bool (const TCheckedObject&)> checker) const {
        for (auto&& object : objects) {
            switch (aggregationType)
            {
            case EAggregatedConditionType::AllOf:
                if (!checker(object)) {
                    return false;
                }
                break;
            case EAggregatedConditionType::AnyOf:
                if (checker(object)) {
                    return true;
                }
                break;
            case EAggregatedConditionType::NoneOf:
                if (checker(object)) {
                    return false;
                }
                break;
            }
        }
        return aggregationType != EAggregatedConditionType::AnyOf;
    }

};

class TDocumentsCheckStatusCondition: public ITerminalCondition, public IAggregatedCondition<TUserDocumentsCheck> {
private:
    static inline TFactory::TRegistrator<TDocumentsCheckStatusCondition> Registrator{ICondition::EType::DocumentsCheckStatus};

    R_FIELD(TSet<TString>, MainTypes);
    R_FIELD(TSet<TString>, MainStatuses);
    R_FIELD(EAggregatedConditionType, MainAggregation);
    R_FIELD(TSet<TString>, AdditionalTypes);
    R_FIELD(TSet<TString>, AdditionalStatuses);
    R_FIELD(EAggregatedConditionType, AdditionalAggregation);

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

private:
    bool CheckMain(const TVector<TUserDocumentsCheck>& checks) const;
    bool CheckAdditional(const TVector<TUserDocumentsCheck>& checks) const;
    TVector<TUserDocumentsCheck> GetChecks(const TString& userId, const TSet<TString>& types, const TUserDocumentsChecksManager* documentsChecksManager, NDrive::TEntitySession& session) const;
};

class IPrivateDataFetcher : public ITerminalCondition  {
public:
    enum class EDocumentType {
        Passport /* "passport" */,
        License /* "license" */,
    };

    struct TDocumentField {
        EDocumentType Document;
        TString Field;
    };

    using TParameters = TVector<TDocumentField>;

protected:
    R_FIELD(TDuration, Timeout, TDuration::Seconds(10));

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const final;

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;
    virtual bool IsPrivateDataMatching(const NJson::TJsonValue& passportJson, const NJson::TJsonValue& licenseJson) const = 0;

    template <typename TDatasyncCondition>
    TAtomicSharedPtr<TDatasyncCondition> GetTemplateImplBase(const TVector<TString>& params) const;
};

class TPrivateDataEqualsCondition: public IPrivateDataFetcher {
public:
    using TBase = IPrivateDataFetcher;

private:
    static inline TFactory::TRegistrator<TPrivateDataEqualsCondition> Registrator{ICondition::EType::PrivateDataEquals};

    R_FIELD(TVector<TParameters>, CompareParameters);

public:
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;
    virtual bool IsPrivateDataMatching(const NJson::TJsonValue& passportJson, const NJson::TJsonValue& licenseJson) const override;
};

class TPrivateDataFieldMatchesCondition: public IPrivateDataFetcher, public IAggregatedCondition<TString> {
public:
    using TBase = IPrivateDataFetcher;

    enum class EMatchType {
        Regex /* "regex" */,
        HasLatin /* "has_latin" */,
        HasAnyOfSymbols /* "has_any_of_symbols" */,
        HasOnlySymbols /* "has_only_symbols" */
    };

private:
    static inline TFactory::TRegistrator<TPrivateDataFieldMatchesCondition> Registrator{ICondition::EType::PrivateDataMatches};

    R_FIELD(TParameters, CheckParameters);
    R_FIELD(EMatchType, MatchType);
    R_FIELD(EAggregatedConditionType, AggregationType);
    R_FIELD(TRegExMatch, Matcher);

public:
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;
    virtual bool IsPrivateDataMatching(const NJson::TJsonValue& passportJson, const NJson::TJsonValue& licenseJson) const override;

private:
    bool BuildMatcher(const TString& matchCondition, const TString& aggregatedType);
};

class THasRoleCondition : public ITerminalCondition, public IAggregatedCondition<TString> {
public:
    using TBase = ITerminalCondition;

private:
    static inline TFactory::TRegistrator<THasRoleCondition> Registrator{ICondition::EType::HasRole};

    R_FIELD(TVector<TString>, RoleNames);
    R_FIELD(EAggregatedConditionType, AggregationType);

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

class TRidesStatisticCondition: public ITerminalCondition {
public:
    using TBase = ITerminalCondition;

private:
    static inline TFactory::TRegistrator<TRidesStatisticCondition> Registrator{ICondition::EType::RidesStatisticCondition};

    R_FIELD(ui64, RidesAmount, 3);
    R_FIELD(TDuration, Period);
    R_OPTIONAL(ui64, RequiredRidePrice);
    R_OPTIONAL(ui64, RequiredRidesSum);

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

class TUserFeatureCondition: public ITerminalCondition {
public:
    using TBase = ITerminalCondition;

private:
    static inline TFactory::TRegistrator<TUserFeatureCondition> Registrator{ICondition::EType::UserFeatureCondition};

    R_FIELD(TRange<double>, FeatureRange);
    R_FIELD(TString, FeatureName);
    R_FIELD(bool, UseFeatureClient, false);
    R_FIELD(TDuration, Timeout, TDuration::Seconds(10));

private:
    const NDrive::TUserEventsApi* GetFeatureClient(const IChatUserContext::TPtr ctx) const;

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

class TAreaMessageCondition: public ITerminalCondition {
public:
    using TBase = ITerminalCondition;

    enum class EFetchType {
        AreaIds /* "area_ids" */,
        GeoTags /* "geo_tags" */,
    };

private:
    static inline TFactory::TRegistrator<TAreaMessageCondition> Registrator{ICondition::EType::AreaMessageCondition};

    R_FIELD(EFetchType, FetchType);
    R_FIELD(TVector<TString>, AreaIds);
    R_FIELD(NDrive::TLocationTags, GeoTagNames);

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

class TContextMapCompareCondition: public ITerminalCondition {
public:
    using TBase = ITerminalCondition;

    enum class ECompareType {
        Ge /* "ge" */,
        Gt /* "gt" */,
        Le /* "le" */,
        Lt /* "lt" */
    };

private:
    static inline TFactory::TRegistrator<TContextMapCompareCondition> Registrator{ICondition::EType::ContextMapCompare};

    R_FIELD(ECompareType, CompareType);
    R_FIELD(double, Limit, 0);
    R_FIELD(TString, Value);
    R_FIELD(bool, UseFloat, false);

protected:
    virtual bool DoParse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;

    template <typename TType>
    bool CompareValues(TType target, TType limit) const {
        switch(CompareType) {
            case ECompareType::Ge:
                return target >= limit;
            case ECompareType::Gt:
                return target > limit;
            case ECompareType::Le:
                return target <= limit;
            case ECompareType::Lt:
                return target < limit;
        }

        auto evlog = NDrive::GetThreadEventLogger();
        Log(NJson::TMapBuilder
            ("event", "TContextMapCompareCondition")
            ("error", "Unknown compare type"),
        evlog);
        return false;
    }

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& /* chatContext */) const override;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};
