#include "chat_script.h"

#include <util/generic/hash_set.h>
#include <util/generic/queue.h>
#include <util/string/subst.h>
#include <util/string/vector.h>

#include <random>

bool TChatRobotScript::BuildParametrizedNode(const TString& nodeName, TMessagesCollector& errors) {
    auto params = SplitString(nodeName, "|", 0, KEEP_EMPTY_TOKENS);
    auto templateNodeIt = Items.find(params[0]);
    if (templateNodeIt == Items.end()) {
        errors.AddMessage(nodeName, "template node not found: " + params[0]);
        return false;
    }
    TChatRobotScriptItem result;
    if (!templateNodeIt->second.GetTemplateImpl(nodeName, params, result)) {
        errors.AddMessage(nodeName, "troubles in templateNodeIt->second.GetTemplateImpl");
        return false;
    }
    Items[nodeName] = std::move(result);
    return true;
}

bool TChatRobotScript::Parse(const NJson::TJsonValue& rawScript, TMessagesCollector& errors) {
    if (!rawScript["items"].IsArray()) {
        errors.AddMessage("script", "items is not an array");
        return false;
    }

    auto nodeResolver = INodeResolver::Construct(rawScript["classification"]);
    if (!nodeResolver) {
        errors.AddMessage("classification", nodeResolver.GetError());
        return false;
    }

    THashSet<TString> knownItems;
    for (auto&& rawItem : rawScript["items"].GetArray()) {
        TChatRobotScriptItem item;
        item.SetNodeResolver(*nodeResolver);
        if (!item.Parse(rawItem, errors)) {
            errors.AddMessage(item.GetId(), "parsing failed");
            return false;
        }
        if (Items.contains(item.GetId())) {
            errors.AddMessage(item.GetId(), "listed twice or more");
            return false;
        }
        knownItems.emplace(item.GetId());
        Items[item.GetId()] = std::move(item);
    }

    TQueue<TString> itemsBuildQueue;
    for (auto&& itemIt : Items) {
        auto adjacentItems = itemIt.second.GetAdjacentItems();
        for (auto&& i : adjacentItems) {
            if (i.find("%") == TString::npos && knownItems.emplace(i).second) {
                itemsBuildQueue.emplace(i);
            }
        }
    }

    while (!itemsBuildQueue.empty() && knownItems.size() <= 222222) {
        auto itemName = itemsBuildQueue.front();
        itemsBuildQueue.pop();

        if (!BuildParametrizedNode(itemName, errors)) {
            errors.AddMessage(itemName, "failed to build parametrized node");
            return false;
        }

        auto adjacentItems = Items[itemName].GetAdjacentItems();
        for (auto&& i : adjacentItems) {
            if (i.find("%") == TString::npos && knownItems.emplace(i).second) {
                itemsBuildQueue.emplace(i);
            }
        }
    }
    if (!itemsBuildQueue.empty()) {
        errors.AddMessage("script", "it was hard, but you did it. the total number of nodes after deduction exceeds 222222");
        return false;
    }

    TQueue<TString> traverseQueue;
    THashSet<TString> visitedNodes;
    traverseQueue.emplace(StartNode);
    visitedNodes.emplace(StartNode);
    while (!traverseQueue.empty()) {
        auto currentNode = traverseQueue.front();
        traverseQueue.pop();

        auto nodeIt = Items.find(currentNode);
        if (nodeIt == Items.end()) {
            errors.AddMessage(currentNode, "you refer this node but it does not exist");
            return false;
        }
        for (auto&& node : nodeIt->second.GetAdjacentItems()) {
            if (visitedNodes.emplace(node).second) {
                traverseQueue.emplace(node);
            }
        }
    }

    if (rawScript.Has("faq_url") && rawScript["faq_url"].IsString()) {
        FaqUrl = rawScript["faq_url"].GetString();
    }
    if (rawScript.Has("support_url") && rawScript["support_url"].IsString()) {
        SupportUrl = rawScript["support_url"].GetString();
    }
    if (rawScript.Has("missing_nodes_redirect") && rawScript["missing_nodes_redirect"].IsString()) {
        MissingNodesRedirect = rawScript["missing_nodes_redirect"].GetString();
        if (!Items.contains(MissingNodesRedirect)) {
            errors.AddMessage("missing nodes redirect", "node " + MissingNodesRedirect + " does not exist");
        }
    }
    if (!NJson::ParseField(rawScript, "message_ratings", MessageRatings, false, errors)) {
        return false;
    }

    if (!Items.contains("!intro_clean")) {
        TChatRobotScriptItem item;
        item.SetId("!intro_clean");
        Items.emplace("!intro_clean", std::move(item));
    }

    return true;
}

TMaybe<TChatRobotScriptItem> TChatRobotScript::FindNextItem(const TString& nextStepId) const {
    auto it = Items.find(nextStepId);
    if (it == Items.end()) {
        return Nothing();
    }
    return it->second;
}

bool TChatRobotScript::GetNextScriptItemWithSuggest(const TChatRobotScriptItem& currentScriptItem, const IChatUserContext::TPtr ctx, const TChatContext& chatContext, TChatRobotScriptItem& nextItem, const NDrive::NChat::TMessageEvents& messages) const {
    TString nextStepId;
    if (currentScriptItem.GetUseClassifier()) {
        if (!currentScriptItem.GetNodeResolver()) {
            return false;
        }
        TString nextStepId = currentScriptItem.GetNodeResolver()->GetNextNode(ctx, messages);
        auto nextItemMaybe = FindNextItem(nextStepId);
        if (!nextItemMaybe) {
            return false;
        } else {
            nextItem = std::move(nextItemMaybe.GetRef());
            return true;
        }
    }
    return GetNextScriptItem(currentScriptItem, ctx, chatContext, nextItem);
}

bool TChatRobotScript::GetNextScriptItem(const TChatRobotScriptItem& currentScriptItem, const IChatUserContext::TPtr ctx, const TChatContext& chatContext, TChatRobotScriptItem& nextItem) const {
    auto nextItemMaybe = GetNextScriptItem(currentScriptItem, ctx, chatContext);
    if (!nextItemMaybe) {
        return false;
    }
    nextItem = std::move(nextItemMaybe.GetRef());
    return true;
}

TMaybe<TChatRobotScriptItem> TChatRobotScript::GetNextScriptItem(const TChatRobotScriptItem& currentScriptItem, const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const {
    if (!currentScriptItem.GetNextSteps().HasFurtherSteps()) {
        return Nothing();
    }
    TString nextStepId = currentScriptItem.GetNextSteps().GetNextNode(ctx, chatContext);
    return FindNextItem(nextStepId);
}

bool TChatRobotScript::GetScriptItemById(const TString& id, TChatRobotScriptItem& item) const {
    auto scriptItemMaybe = GetScriptItemById(id);
    if (!scriptItemMaybe) {
        return false;
    }
    item = std::move(scriptItemMaybe.GetRef());
    return true;
}

TMaybe<TChatRobotScriptItem> TChatRobotScript::GetScriptItemById(const TString& id) const {
    auto itemIt = Items.find(id);
    if (itemIt != Items.end()) {
        return itemIt->second;
    }
    if (!!MissingNodesRedirect) {
        auto itemIt = Items.find(MissingNodesRedirect);
        if (itemIt != Items.end()) {
            return itemIt->second;
        }
    }
    return Nothing();
}

TChatRobotScript TChatRobotScript::FromFilename(const TString& fileName) {
    NJson::TJsonValue json;
    TFileInput in(fileName);
    ReadJsonTree(&in, &json);

    TChatRobotScript result;
    TMessagesCollector errors;
    CHECK_WITH_LOG(result.Parse(json, errors)) << errors.GetStringReport();

    return result;
}

TVector<TChatResourceInfo> TChatRobotScript::GetLinkedResources() const {
    TVector<TChatResourceInfo> result;
    for (const auto& [_, scriptItem] : Items) {
        for (auto&& preActionMessage : scriptItem.GetPreActionMessages()) {
            if (preActionMessage.GetResourceInfo().Link) {
                result.emplace_back(preActionMessage.GetResourceInfo());
            }
        }
    }
    return result;
}
