#pragma once

#include "chat_script_item.h"

#include <drive/backend/chat_robots/ifaces.h>
#include <drive/backend/chat_robots/suggest/node_resolver.h>

#include <drive/backend/chat/engine.h>

#include <rtline/util/json_processing.h>
#include <rtline/util/types/accessor.h>

#include <util/generic/map.h>
#include <util/generic/vector.h>
#include <util/stream/file.h>

class TChatRobotScript {
    R_READONLY(TString, FaqUrl);
    R_READONLY(TString, SupportUrl);
    R_READONLY(TString, MissingNodesRedirect);
    R_FIELD(TString, StartNode, "intro");
    R_FIELD(TSet<TString>, MessageRatings);

    R_FIELD(NJson::TJsonValue, RawScript);

protected:
    TMap<TString, TChatRobotScriptItem> Items;

public:
    TChatRobotScript() = default;
    static TChatRobotScript FromFilename(const TString& fileName);

    bool Parse(const NJson::TJsonValue& rawScript, TMessagesCollector& errors);
    bool BuildParametrizedNode(const TString& nodeName, TMessagesCollector& errors);
    bool GetScriptItemById(const TString& id, TChatRobotScriptItem& item) const;
    TMaybe<TChatRobotScriptItem> GetScriptItemById(const TString& id) const;
    bool GetNextScriptItem(const TChatRobotScriptItem& currentScriptItem, const IChatUserContext::TPtr ctx, const TChatContext& chatContext, TChatRobotScriptItem& nextItem) const;
    TMaybe<TChatRobotScriptItem> GetNextScriptItem(const TChatRobotScriptItem& currentScriptItem, const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const;
    bool GetNextScriptItemWithSuggest(const TChatRobotScriptItem& currentScriptItem, const IChatUserContext::TPtr ctx, const TChatContext& chatContext, TChatRobotScriptItem& nextItem, const NDrive::NChat::TMessageEvents& messages) const;

    NJson::TJsonValue DumpToJson() const {
        return RawScript;
    }

    TSet<TString> GetNodeNames() const {
        TSet<TString> result;
        for (auto&& it : Items) {
            result.emplace(it.first);
        }
        return result;
    }

    bool HasRatings() const {
        return !MessageRatings.empty();
    }

    TVector<TChatResourceInfo> GetLinkedResources() const;

private:
    TMaybe<TChatRobotScriptItem> FindNextItem(const TString& nextStepId) const;
};
