#include "chat_script_item.h"

#include <util/generic/hash_set.h>
#include <util/generic/queue.h>
#include <util/string/subst.h>
#include <util/string/vector.h>

#include <random>

namespace NChatScript {
    TString GetMaybeParametrizedSingleFieldString(const TString& text, const TVector<TString>& params) {
        TStringBuf sbText(text);
        if (!text.StartsWith("%")) {
            return text;
        }
        sbText.Skip(1);
        size_t index;
        if (!TryFromString(sbText, index) || index >= params.size()) {
            return text;
        }
        return params[index];
    }
};

bool TPreActionMessage::DoDeserializeBasicsFromJson(const NJson::TJsonValue& raw) {
    bool hasChoices = raw.Has("choices");
    bool hasResource = raw.Has("resource_link");
    if (hasChoices && hasResource) {
        return false;
    }
    if (hasChoices) {
        if (GetText() != "$random" || !NJson::ParseField(raw, "choices", Choices, false)) {
            return false;
        }
        return !Choices.empty();
    }
    if (hasResource) {
        if (!NJson::ParseField(raw, "resource_link", ResourceInfo.Link, true)
            || !NJson::ParseField(raw, "resource_content_type", ResourceInfo.ContentType, true)) {
            return false;
        }
        ResourceInfo.Name = GetText();
    }
    return true;
}

bool TChatRobotScriptItem::Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    if (!NJson::ParseField(raw, "id", Id, true, errors)) {
        return false;
    }
    if (Id.find("|") != TString::npos) {
        errors.AddMessage(Id, "script item names can't contain | delimiter in id, but this one does");
        return false;
    }

    TSet<TString> processedJsonChildren;
    processedJsonChildren.emplace("id");
    if (raw.Has("action_type") && raw["action_type"].IsString()) {
        processedJsonChildren.emplace("action_type");
        if (!NJson::ParseField(raw, "action_type", NJson::Stringify(ActionType), true, errors)) {
            return false;
        }
    } else if (raw.Has("action_type") && raw["action_type"].IsNull()) {
        processedJsonChildren.emplace("action_type");
        ActionType = NChatRobot::EUserAction::NoAction;
    } else {
        errors.AddMessage(Id, "can't parse action type: it is not string and not null");
        return false;
    }

    if (raw.Has("action_type_interface")) {
        processedJsonChildren.emplace("action_type_interface");
        TString actionTypeInterfaceRaw;
        if (raw["action_type_interface"].IsString()) {
            if (!TryFromString(raw["action_type_interface"].GetString(), ActionTypeInterface)) {
                errors.AddMessage(Id, "can't restore interface action type from string");
                return false;
            }
        } else if (raw["action_type_interface"].IsNull()) {
            ActionTypeInterface = NChatRobot::EUserAction::NoAction;
        } else {
            errors.AddMessage(Id, "can't restore interface action type: it is not string and not null");
            return false;
        }
    } else {
        ActionTypeInterface = ActionType;
    }

    if (ActionType == NChatRobot::EUserAction::DeeplinkAction) {
        if (!NJson::ParseField(raw, "link", Link, true, errors)) {
            return false;
        }
        processedJsonChildren.emplace("link");
    }

    if (raw.Has("action_button_text") && !raw["action_button_text"].IsNull()) {
        processedJsonChildren.emplace("action_button_text");
        if (!raw["action_button_text"].IsString()) {
            errors.AddMessage(Id, "action button text is neither null nor string");
            return false;
        }
        ActionButtonText = raw["action_button_text"].GetString();
    } else if (raw.Has("action_button_text")) {
        processedJsonChildren.emplace("action_button_text");
    }

    if (ActionTypeInterface == NChatRobot::EUserAction::Button || ActionTypeInterface == NChatRobot::EUserAction::ChatClosed) {
        if (!ActionButtonText) {
            errors.AddMessage(Id, "empty action button text");
            return false;
        }
    }

    NJson::TJsonValue::TArray preMessagesRaw;
    if (raw.Has("pre_action_messages") && raw["pre_action_messages"].GetArray(&preMessagesRaw)) {
        processedJsonChildren.emplace("pre_action_messages");
        size_t msgId = 0;
        for (auto&& item : preMessagesRaw) {
            msgId += 1;
            TPreActionMessage message;
            if (!message.DeserializeBasicsFromJson(item)) {
                errors.AddMessage(Id, "can't deserialize pre_action_message number " + ToString(msgId));
                return false;
            }
            PreActionMessages.push_back(std::move(message));
        }
    } else {
        errors.AddMessage(Id, "pre_action_messages are given not by array");
        return false;
    }

    if (raw.Has("fallback_node")) {
        processedJsonChildren.emplace("fallback_node");
        if (!raw["fallback_node"].IsString()) {
            errors.AddMessage(Id, "fallback node is not string");
            return false;
        }
        FallbackNode = raw["fallback_node"].GetString();
    }

    if (raw.Has("support_line_tag")) {
        processedJsonChildren.emplace("support_line_tag");
        if (!raw["support_line_tag"].IsString()) {
            errors.AddMessage(Id, "support_line_tag is not string");
            return false;
        }
        SupportLineTag = raw["support_line_tag"].GetString();
    }

    if (raw.Has("put_tag_on_entry")) {
        processedJsonChildren.emplace("put_tag_on_entry");
        if (!raw["put_tag_on_entry"].IsBoolean()) {
            errors.AddMessage(Id, "put_tag_on_entry is not boolean");
            return false;
        }
        PutTagOnEntry = raw["put_tag_on_entry"].GetBoolean();
    }

    if (raw.Has("is_skippable")) {
        processedJsonChildren.emplace("is_skippable");
        if (!raw["is_skippable"].IsBoolean()) {
            errors.AddMessage(Id, "is_skippable is not boolean");
            return false;
        }
        Skippable = raw["is_skippable"].GetBoolean();
    }

    if (raw.Has("suppress_support_call")) {
        processedJsonChildren.emplace("suppress_support_call");
        if (!raw["suppress_support_call"].IsBoolean()) {
            errors.AddMessage(Id, "suppress_support_call is not boolean");
            return false;
        }
        SuppressSupportCall = raw["suppress_support_call"].GetBoolean();
    }
    if (raw.Has("move_support_call")) {
        processedJsonChildren.emplace("move_support_call");
        if (!raw["move_support_call"].IsBoolean()) {
            errors.AddMessage(Id, "move_support_call is not boolean");
            return false;
        }
        MoveSupportCall = raw["move_support_call"].GetBoolean();
    }
    if ((PutTagOnEntry || MoveSupportCall) && SuppressSupportCall) {
        errors.AddMessage(Id, "both put_tag_on_entry or move_support_call and suppress_support_call are true");
        return false;
    }
    if (raw.Has("allowed_message_types")) {
        processedJsonChildren.emplace("allowed_message_types");
        if (!raw["allowed_message_types"].IsArray()) {
            errors.AddMessage(Id, "allowed message types are not an array");
            return false;
        }
        AllowedMessageTypes.clear();
        for (auto&& type : raw["allowed_message_types"].GetArray()) {
            if (!type.IsString()) {
                errors.AddMessage(Id, "one of allowed message types is not defined by string");
                return false;
            }
            NDrive::NChat::TMessage::EMessageType msgType;
            if (!TryFromString(type.GetString(), msgType)) {
                errors.AddMessage(Id, "message type " + type.GetString() + " is not enum element in ");
                return false;
            }
            AllowedMessageTypes.push_back(msgType);
        }
    }

    if (raw.Has("schema")) {
        processedJsonChildren.emplace("schema");
        if (ActionType != NChatRobot::EUserAction::Tree && ActionType != NChatRobot::EUserAction::ContextButtons) {
            errors.AddMessage(Id, "specifying schema for user action which doesn't require it");
            return false;
        }
        if (!raw["schema"].IsMap()) {
            errors.AddMessage(Id, "schema is not map");
            return false;
        }
        if (!Schema.ConstructFromJson(raw["schema"], errors)) {
            errors.AddMessage(Id, "unable to parse schema");
            return false;
        }
    } else if (ActionType == NChatRobot::EUserAction::Tree || ActionType == NChatRobot::EUserAction::ContextButtons) {
        errors.AddMessage(Id, "no schema specified for item");
        return false;
    }

    if (raw.Has("next_step")) {
        processedJsonChildren.emplace("next_step");
    } else {
        errors.AddMessage(Id, "next_step is not specified");
        return false;
    }

    if (raw.Has("next_step_error")) {
        if (!NextStepsError.Parse(raw["next_step_error"], errors)) {
            errors.AddMessage(Id, "cannot parse next_step_error");
            return false;
        }
        processedJsonChildren.emplace("next_step_error");
    }

    if (raw.Has("on_entry_actions")) {
        processedJsonChildren.emplace("on_entry_actions");
        if (!raw["on_entry_actions"].IsArray()) {
            errors.AddMessage(Id, "on_entry_actions are not an array");
            return false;
        }
        OnEntryActions.clear();
        for (auto&& actionRaw : raw["on_entry_actions"].GetArray()) {
            IHookAction::EType type;
            if (!actionRaw.Has("type") || !actionRaw["type"].IsString() || !TryFromString(actionRaw["type"].GetString(), type)) {
                errors.AddMessage(Id, "can't parse type in on_entry_action");
                return false;
            }
            IHookAction::TPtr action = IHookAction::TFactory::Construct(type);
            if (!action) {
                errors.AddMessage(Id, "can't construct action with type " + ToString(type));
                return false;
            }
            if (!action->Parse(actionRaw, errors)) {
                errors.AddMessage(Id, "can't deserialize action in on_entry_action, type: " + ToString(type));
                return false;
            }
            OnEntryActions.emplace_back(action);
        }
    }

    if (ActionType == NChatRobot::EUserAction::Resubmit) {
        if (!NJson::ParseField(raw, "resubmit_documents", ResubmitDocuments, true, errors)) {
            errors.AddMessage(Id, "resubmit_documents not specified");
            return false;
        }
        processedJsonChildren.emplace("resubmit_documents");
    }

    if (ActionType == NChatRobot::EUserAction::FeedbackCurrent || ActionType == NChatRobot::EUserAction::FeedbackPast || ActionType == NChatRobot::EUserAction::EnterPromocode || ActionType == NChatRobot::EUserAction::ValidatePass) {
        if (!raw.Has("next_step_incorrect") || !NextStepsIncorrect.Parse(raw["next_step_incorrect"], errors)) {
            errors.AddMessage(Id, "next_step_incorrect not specified");
            return false;
        }
        processedJsonChildren.emplace("next_step_incorrect");
    }

    if (ActionType == NChatRobot::EUserAction::FeedbackCurrent || ActionType == NChatRobot::EUserAction::FeedbackPast) {
        if (!raw.Has("next_step_no_bonus") || !NextStepsNoBonus.Parse(raw["next_step_no_bonus"], errors)) {
            errors.AddMessage(Id, "next_step_no_bonus not specified");
            return false;
        }
        if (!raw.Has("feedback_tag") || !raw["feedback_tag"].IsString()) {
            errors.AddMessage(Id, "no feedback_tag or it is not string");
            return false;
        }
        FeedbackTag = raw["feedback_tag"].GetString();
        processedJsonChildren.emplace("next_step_no_bonus");
        processedJsonChildren.emplace("feedback_tag");
    }

    if (raw["classification_override"].IsMap()) {
        if (ActionType != NChatRobot::EUserAction::UserMessage) {
            errors.AddMessage("classification_override", "can be used only in user message action");
            return false;
        }
        auto nodeResolver = INodeResolver::Construct(raw["classification_override"], NodeResolver);
        if (!nodeResolver) {
            errors.AddMessage("classification_override", nodeResolver.GetError());
            return false;
        }
        if (*nodeResolver) {
            NodeResolver = *nodeResolver;
        }
        processedJsonChildren.emplace("classification_override");
    }
    if (raw["use_classifier"].IsBoolean()) {
        if (!NJson::ParseField(raw, "use_classifier", UseClassifier, false, errors)) {
            return false;
        }
        processedJsonChildren.emplace("use_classifier");
    }
    if (raw["use_recognition"].IsBoolean()) {
        if (!NJson::ParseField(raw, "use_recognition", UseRecognition, false, errors)) {
            return false;
        }
        processedJsonChildren.emplace("use_recognition");
    }

    if (raw["context_map"].IsMap()) {
        TString key, value;
        if (!NJson::ParseField(raw["context_map"], "key", key, true, errors) ||
            !NJson::ParseField(raw["context_map"], "value", value, true, errors)) {
            return false;
        }
        ContextMapInfo.push_back(TContextMapInfo(key, value));
        processedJsonChildren.emplace("context_map");
    } else if (raw["context_map"].IsArray()) {
        if (!NJson::ParseField(raw["context_map"], ContextMapInfo, true, errors)) {
            return false;
        }
        processedJsonChildren.emplace("context_map");
    }

    if (raw.Has("json_mapper")) {
        if (JsonMapper.DeserializeFromJson(raw["json_mapper"], errors)) {
            processedJsonChildren.emplace("json_mapper");
        } else {
            return false;
        }
    }

    if (raw.Has("action_report")) {
        if (!raw["action_report"].IsMap()) {
            errors.AddMessage(Id, "action report should be json map");
            return false;
        }
        if (NJson::ParseField(raw, "action_report", ActionReport, true, errors)) {
            processedJsonChildren.emplace("action_report");
        } else {
            return false;
        }
    }

    if (raw.Has("comment")) {
        if (!raw["comment"].IsString()) {
            errors.AddMessage(Id, "comment should be string");
            return false;
        }
        processedJsonChildren.emplace("comment");
    }

    if (processedJsonChildren.size() != raw.GetMap().size()) {
        for (auto&& it : raw.GetMap()) {
            if (!processedJsonChildren.contains(it.first)) {
                errors.AddMessage(Id, "there is unrecognized element " + it.first + " in node description");
            }
        }
        errors.AddMessage(Id, "there are unrecognized elements in schema");
        return false;
    }

    if (raw.Has("next_step") && !NextSteps.Parse(raw["next_step"], errors)) {
        errors.AddMessage(Id, "could not parse next_step");
        return false;
    }

    return true;
}

bool TChatRobotScriptItem::GetTemplateImpl(const TString& nodeName, const TVector<TString>& params, TChatRobotScriptItem& result) const {
    result.SetId(nodeName);
    result.SetActionType(ActionType);
    result.SetActionTypeInterface(ActionTypeInterface);
    result.SetActionButtonText(NChatScript::GetMaybeParametrizedSingleFieldString(ActionButtonText, params));
    {
        for (auto&& message : PreActionMessages) {
            TPreActionMessage paMessage = message;
            paMessage.SetText(NChatScript::GetMaybeParametrizedSingleFieldString(message.GetText(), params));
            TVector<TString> choicesImpl;
            for (auto&& choice : paMessage.GetChoices()) {
                choicesImpl.push_back(NChatScript::GetMaybeParametrizedSingleFieldString(choice, params));
            }
            paMessage.SetChoices(std::move(choicesImpl));
            result.MutablePreActionMessages().push_back(std::move(paMessage));
        }
    }
    result.SetSchema(*Schema.GetTemplateImpl(params));
    result.SetFallbackNode(NChatScript::GetMaybeParametrizedSingleFieldString(FallbackNode, params));
    result.SetSupportLineTag(NChatScript::GetMaybeParametrizedSingleFieldString(SupportLineTag, params));
    result.SetPutTagOnEntry(PutTagOnEntry);
    result.SetSuppressSupportCall(SuppressSupportCall);
    result.SetMoveSupportCall(MoveSupportCall);
    {
        for (auto&& onEntryAction : OnEntryActions) {
            auto newAction = onEntryAction->GetTemplateImpl(params);
            result.MutableOnEntryActions().push_back(newAction);
        }
    }
    result.SetAllowedMessageTypes(AllowedMessageTypes);
    result.SetNextSteps(NextSteps.GetTemplateImpl(params));
    result.SetLink(NChatScript::GetMaybeParametrizedSingleFieldString(Link, params));
    result.SetSkippable(IsSkippable());
    result.SetActionReport(GetActionReport());
    result.SetNodeResolver(GetNodeResolver());
    result.SetContextMapInfo(GetContextMapInfo());
    result.SetJsonMapper(GetJsonMapper());
    result.SetUseClassifier(GetUseClassifier());
    result.SetUseRecognition(GetUseRecognition());
    result.SetResubmitDocuments(NChatScript::GetMaybeParametrizedSingleFieldString(ResubmitDocuments, params));

    result.SetNextStepsIncorrect(GetNextStepsIncorrect());
    result.SetNextStepsNoBonus(GetNextStepsNoBonus());
    result.SetFeedbackTag(GetFeedbackTag());
    result.SetNextStepsError(GetNextStepsError());
    result.SetNextStepsIncorrectDate(GetNextStepsIncorrectDate());
    result.SetNextStepsIncorrectPass(GetNextStepsIncorrectPass());
    return true;
}

bool TChatRobotScriptItem::SaveFieldsToContext(TChatContext& chatContext, const IChatUserContext::TPtr ctx) const {
    for (auto&& field : ContextMapInfo) {
        if (field.Value && field.Type == EContextDataType::MathExpression) {
            auto result = chatContext.CalculateExpression(field);
            if (!result) {
                return false;
            }
            chatContext.AddMapData(field.Key, *result);
        } else if (field.Value) {
            chatContext.AddMapData(field.Key, ctx->Unescape(field.Value, chatContext));
        } else if (field.Type != EContextDataType::None) {
            chatContext.AddMapData(field.Key, chatContext.GetSpecialData(field.Type));
        }
    }
    return true;
}

void TChatRobotScriptItem::AddSpecialValueToContext(TChatContext& chatContext, const EContextDataType type, const TString& value) const {
    for (auto&& field : ContextMapInfo) {
        if (field.Type == type) {
            chatContext.AddMapData(field.Key, value);
        }
    }
}

void TChatRobotScriptItem::ProcessAdjacentItem(const TString& raw, const TVector<TString>& params, TSet<TString>& result) const {
    auto candidate = NChatScript::GetMaybeParametrizedField(raw, params);
    if (candidate) {
        result.emplace(std::move(candidate));
    }
}

bool TChatRobotScriptItem::GetBareId(TString& result) const {
    const auto idParts = SplitString(Id, "|");
    if (idParts.empty()) {
        return false;
    }

    result = idParts.front();
    return true;
}

TSet<TString> TChatRobotScriptItem::GetAdjacentItems() const {
    auto tokens = SplitString(Id, "|");
    TSet<TString> result;

    {
        auto adjacentBySchema = Schema.GetAdjacentItems(tokens);
        for (auto&& i : adjacentBySchema) {
            ProcessAdjacentItem(i, tokens, result);
        }
    }

    if (FallbackNode) {
        ProcessAdjacentItem(FallbackNode, tokens, result);
    }

    if (NextSteps.GetDefaultNode()) {
        ProcessAdjacentItem(NextSteps.GetDefaultNode(), tokens, result);
        for (auto&& part : NextSteps.GetParts()) {
            ProcessAdjacentItem(part.NodeName, tokens, result);
        }
    }

    return result;
}
