#include "condition_interface.h"

#include <util/string/subst.h>
#include <random>

namespace NChatScript {
    TStringBuf GetMaybeParametrizedSingleField(const TStringBuf& text, const TVector<TString>& params) {
        if (!text.StartsWith("%")) {
            return text;
        }
        size_t index;
        if (!TryFromString(text.substr(1, text.size() - 1), index) || index >= params.size()) {
            return text;
        }
        return params[index];
    }

    TString GetMaybeParametrizedField(const TString& text, const TVector<TString>& params) {
        if (text.find("%") == TString::npos) {
            return text;
        }
        TStringBuf localText = text;
        TStringBuf currentToken;
        TStringBuilder sb;
        bool first = true;
        while (localText.NextTok('|', currentToken)) {
            if (first) {
                first = false;
            } else {
                sb << '|';
            }
            sb << NChatScript::GetMaybeParametrizedSingleField(currentToken, params);
        }
        return sb;
    }
};

bool ICondition::Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    if (!DoParse(raw, errors)) {
        errors.AddMessage("condition", "condition-specific parse failed");
        return false;
    }
    if (raw.Has("subconditions")) {
        if (!raw["subconditions"].IsArray()) {
            errors.AddMessage("condition", "subconditions are not an array");
            return false;
        }

        auto subConditionsNumber = raw["subconditions"].GetArray().size();
        if (subConditionsNumber < GetMinSubConditions() || subConditionsNumber > GetMaxSubConditions()) {
            errors.AddMessage(
                "condition",
                "number of subconditions " + ToString(subConditionsNumber) + " doesn't match range [" + ToString(GetMinSubConditions()) + "; " + ToString(GetMaxSubConditions()) + "]"
            );
            return false;
        }

        SubConditions.clear();
        for (auto&& c : raw["subconditions"].GetArray()) {
            EType type;
            if (!NJson::ParseField(c, "type", NJson::Stringify(type), true, errors)) {
                errors.AddMessage("condition", "cannot parse subcondition type");
                return false;
            }
            ICondition::TPtr cond = ICondition::TFactory::Construct(type);
            if (!cond || !cond->Parse(c, errors)) {
                errors.AddMessage("condition", "subcondition parse failed");
                return false;
            }
            SubConditions.emplace_back(cond);
        }
    } else if (GetMinSubConditions() > 0) {
        errors.AddMessage("condition", "number of subconditions 0 doesn't match range [" + ToString(GetMinSubConditions()) + "; " + ToString(GetMaxSubConditions()) + "]");
        return false;
    }

    return true;
}

ICondition::TPtr ICondition::DeserializeFromJson(const NJson::TJsonValue& raw, TMessagesCollector* errors) {
    if (!raw.Has("type") || !raw["type"].IsString()) {
        return nullptr;
    }
    ICondition::EType type;
    TString typeStr = raw["type"].GetString();
    if (!TryFromString(typeStr, type)) {
        return nullptr;
    }

    TMessagesCollector* errorsCollector;
    THolder<TMessagesCollector> collectorHolder;
    if (errors) {
        errorsCollector = errors;
    } else {
        collectorHolder.Reset(new TMessagesCollector);
        errorsCollector = collectorHolder.Get();
    }

    ICondition::TPtr cond = ICondition::TFactory::Construct(type);
    if (!cond || !cond->Parse(raw, *errorsCollector)) {
        if (!errors) {
            ERROR_LOG << errorsCollector->GetStringReport() << Endl;
        }
        return nullptr;
    }

    return cond;
}

void ICondition::Log(NJson::TJsonValue&& message, NThreading::IEventLogger* evlog) const {
    if (evlog) {
        evlog->AddEvent(std::move(message));
    } else {
        NDrive::TEventLog::Log(message["event"].GetString(), std::move(message));
    }
}

TCaseCondition TCaseCondition::GetTemplateImpl(const TVector<TString>& params) const {
    TCaseCondition result;
    result.SetDefaultNode(NChatScript::GetMaybeParametrizedField(DefaultNode, params));
    TVector<TConditionPart> inferedParts;
    for (auto&& part : Parts) {
        auto inferedCondition = part.Condition->GetTemplateImpl(params);
        if (!inferedCondition) {
            continue;
        }
        inferedParts.emplace_back(TConditionPart(inferedCondition, NChatScript::GetMaybeParametrizedField(part.NodeName, params)));
    }
    result.SetParts(std::move(inferedParts));
    return result;
}

TString TCaseCondition::GetNextNode(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const {
    for (auto&& part : Parts) {
        if (part.Condition && part.Condition->IsMatching(ctx, chatContext)) {
            return part.NodeName;
        }
    }
    return DefaultNode;
}

bool TCaseCondition::Parse(const NJson::TJsonValue& rawJson, TMessagesCollector& errors) {
    if (rawJson.IsNull()) {
        DefaultNode = "";
        return true;
    }

    if (rawJson.IsString()) {
        DefaultNode = rawJson.GetString();
        return true;
    }

    if (!rawJson.IsMap()) {
        errors.AddMessage("case_condition", "neither string nor map");
        return false;
    }

    // check old format of next action to ensure backwards compatibility
    {
        bool isSimpleMapping = true;
        for (auto&& it : rawJson.GetMap()) {
            if (!it.first || !it.second.IsString()) {
                isSimpleMapping = false;
                break;
            }
        }
        if (isSimpleMapping) {
            if (!rawJson.Has("!default")) {
                errors.AddMessage("case_condition", "old-style case condition does not have !default");
                return false;
            }
            DefaultNode = rawJson["!default"].GetString();
            size_t seqNum = 0;
            for (auto&& it : rawJson.GetMap()) {
                ++seqNum;
                auto message = it.first;
                auto nextNode = it.second.GetString();
                if (message == "!default") {
                    continue;
                }

                ICondition::TPtr cond = ICondition::TFactory::Construct(ICondition::EType::Message);
                if (!cond) {
                    errors.AddMessage("case_condition", "could not construct condition for message");
                    return false;
                }
                {
                    NJson::TJsonValue messageConditionRaw;
                    messageConditionRaw["message"] = message;
                    if (!cond->Parse(messageConditionRaw, errors)) {
                        errors.AddMessage("case_condition", "could not parse one of conditions: " + ToString(seqNum));
                        return false;
                    }
                }
                Parts.emplace_back(TCaseCondition::TConditionPart(cond, nextNode));
            }
            return true;
        }
    }

    if (!rawJson.Has("default") || !rawJson["default"].IsString()) {
        errors.AddMessage("case_condition", "no default node, or it's not string");
        return false;
    }
    DefaultNode = rawJson["default"].GetString();

    if (!rawJson.Has("conditions") || !rawJson["conditions"].IsArray()) {
        errors.AddMessage("case_condition", "conditions is not array or just missing");
        return false;
    }
    for (auto&& it : rawJson["conditions"].GetArray()) {
        if (!it.IsMap()) {
            errors.AddMessage("case_condition", "one of conditions is defined not by map");
            return false;
        }
        if (!it.Has("node") || !it["node"].IsString()) {
            errors.AddMessage("case_condition", "one of conditions doesn't have node or it's not string");
            return false;
        }
        if (!it.Has("condition")) {
            errors.AddMessage("case_condition", "there is no condition in case part");
            return false;
        }

        ICondition::EType type;
        if (!NJson::ParseField(it["condition"], "type", NJson::Stringify(type), true, errors)) {
            return false;
        }
        ICondition::TPtr cond = ICondition::TFactory::Construct(type);
        if (!cond || !cond->Parse(it["condition"], errors)) {
            errors.AddMessage("condition", "subcondition parse failed");
            return false;
        }
        Parts.emplace_back(TConditionPart(cond, it["node"].GetString()));
    }

    return true;
}
