#pragma once

#include "context.h"

#include <drive/backend/chat_robots/ifaces.h>
#include <drive/backend/chat_robots/suggest/node_resolver.h>


#include <rtline/util/json_processing.h>
#include <rtline/util/types/accessor.h>

#include <util/generic/map.h>
#include <util/generic/vector.h>

namespace NChatScript {
    TString GetMaybeParametrizedField(const TString& text, const TVector<TString>& params);

    template<class TStringContainer>
    TStringContainer GetMaybeParametrizedStringContainer(
        const TStringContainer& container,
        const TVector<TString>& params
    ) {
        TStringContainer result;
        for (const TString& entry: container) {
            result.insert(result.end(), GetMaybeParametrizedField(entry, params));
        }
        return result;
    }
};

class ICondition {
public:
    using TPtr = TAtomicSharedPtr<ICondition>;
    class TConditionException : public yexception {
    };

    enum class EType: ui32 {
        And = 0 /* "and" */,
        Or = 1 /* "or" */,
        Not = 2 /* "not" */,
        Performs = 3 /* "performs" */,
        HasTag = 4 /* "has_tag" */,
        UserInStatus = 5 /* "user_in_status" */,
        Platform = 6 /* "platform" */,
        AppBuild = 7 /* "min_app_build" */,
        HasAction = 8 /* "has_action" */,
        HasDebt = 9 /* "has_debt" */,
        RegistrationStep = 10 /* "registration_step" */,
        InArea = 11 /* "in_area" */,
        Message = 12 /* "message" */,
        DictionaryTagValue = 13 /* "dictionary_tag_value" */,
        HadRides = 14 /* "had_rides" */,
        HandlerAnswer = 15 /* "hardler_answer" */,
        HandlerCode = 16 /* "hardler_code" */,
        HasWallet = 17 /* "has_wallet" */,
        PhoneVerified = 18 /* "phone_verified" */,
        Origin = 19 /* "origin" */,
        EmailVerified = 20 /* "email_verified" */,
        PhotoStatus = 21 /* "photo_status" */,
        ContextMapEquals = 22 /* "context_map_equals" */,
        ContextMapContainsKey = 23 /* "context_map_contains_key" */,
        CarTags = 24 /* "car_tags" */,
        HasReferral = 25 /* "has_referral" */,
        TagFieldValue = 26 /* "tag_field_value" */,
        DocumentsRecognitionResult = 27 /* "documents_recognition_result" */,
        DocumentsCheckStatus = 28 /* "documents_check_status" */,
        PrivateDataEquals = 29 /* "private_data_equals" */,
        PrivateDataMatches = 30 /* "private_data_matches" */,
        HasRole = 31 /* "has_role" */,
        RidesStatisticCondition = 32 /* "rides_statistic" */,
        UserFeatureCondition = 33 /* "user_feature_condition" */,
        AreaMessageCondition = 34 /* "area_message_condition" */,
        ContextMapCompare = 35 /* "context_map_compare" */
    };

    using TFactory = NObjectFactory::TParametrizedObjectFactory<ICondition, ICondition::EType>;

public:
    virtual bool IsMatching(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const = 0;
    virtual ICondition::TPtr GetTemplateImpl(const TVector<TString>& params) const = 0;

    virtual bool Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors);
    static ICondition::TPtr DeserializeFromJson(const NJson::TJsonValue& raw, TMessagesCollector* errors = nullptr);

    virtual void AddSubCondition(ICondition::TPtr sc) {
        SubConditions.emplace_back(sc);
    }

    virtual ~ICondition() = default;

protected:
    TVector<ICondition::TPtr> SubConditions;

    virtual bool DoParse(const NJson::TJsonValue& /*raw*/, TMessagesCollector& /*errors*/) {
        return true;
    }

    virtual size_t GetMinSubConditions() const {
        return 0;
    }

    virtual size_t GetMaxSubConditions() const {
        return Max<size_t>() / 3;
    }

    virtual void Log(NJson::TJsonValue&& message, NThreading::IEventLogger* evlog = nullptr) const;
};

class TCaseCondition {
    struct TConditionPart {
        ICondition::TPtr Condition;
        TString NodeName;

        TConditionPart() = default;
        TConditionPart(const ICondition::TPtr condition, const TString nodeName)
            : Condition(condition)
            , NodeName(nodeName)
        {
        }
    };

    R_FIELD(TVector<TConditionPart>, Parts);
    R_FIELD(TString, DefaultNode);

public:
    bool Parse(const NJson::TJsonValue& rawJson, TMessagesCollector& errors);
    TCaseCondition GetTemplateImpl(const TVector<TString>& parts) const;

    TString GetNextNode(const IChatUserContext::TPtr ctx, const TChatContext& context) const;

    bool HasFurtherSteps() const {
        return !(Parts.empty() && !DefaultNode);
    }
};
