#include "config.h"


template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TCustomMessageBackground& result) {
    return result.DeserializeFromJson(value);
}

template <>
NJson::TJsonValue NJson::ToJson(const TCustomMessageBackground& object) {
    return object.SerializeToJson();
}

template <>
NJson::TJsonValue NJson::ToJson(const TChatStyles::TListItem& object) {
    NJson::TJsonValue result;
    NJson::InsertNonNull(result, "substrate_color", object.GetSubstrateColor());
    NJson::InsertField(result, "title_color", object.GetTitleColor());
    NJson::InsertField(result, "subtitle_color", object.GetSubtitleColor());
    NJson::InsertField(result, "date_color", object.GetDateColor());
    NJson::InsertField(result, "unread_indicator_color", object.GetUnreadIndicatorColor());
    NJson::InsertField(result, "pinned", object.GetPinned());
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TChatStyles::TListItem& result) {
    return
        NJson::ParseField(value, "substrate_color", result.MutableSubstrateColor(), false) &&
        NJson::ParseField(value, "title_color", result.MutableTitleColor(), false) &&
        NJson::ParseField(value, "date_color", result.MutableDateColor(), false) &&
        NJson::ParseField(value, "unread_indicator_color", result.MutableUnreadIndicatorColor(), false) &&
        NJson::ParseField(value, "subtitle_color", result.MutableSubtitleColor(), false) &&
        NJson::ParseField(value, "pinned", result.MutablePinned(), false);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TChatStyles& result) {
    return
        NJson::ParseField(value, "name", result.MutableName()) &&
        NJson::ParseField(value, "bg_color", result.MutableBgColor()) &&
        NJson::ParseField(value, "elem_color", result.MutableElemColor()) &&
        NJson::ParseField(value, "list_item", result.MutableListItem());
}

template <>
NJson::TJsonValue NJson::ToJson(const TChatStyles& object) {
    NJson::TJsonValue result;
    result["name"] = object.GetName();
    result["bg_color"] = object.GetBgColor();
    result["elem_color"] = object.GetElemColor();
    NJson::InsertField(result, "list_item", object.GetListItem());
    return result;
}

void TChatRobotConfig::DoInit(const TYandexConfig::Section* section) {
    ChatId = section->GetDirectives().Value<TString>("ChatId", ChatId);
    AssertCorrectConfig(ChatId.find(".") == TString::npos, "dot ('.') is not allowed in chat robot id, troublesome chat id: " + ChatId);

    TString botClassNameStr = "";
    botClassNameStr = section->GetDirectives().Value<TString>("BotClassName", botClassNameStr);
    CHECK_WITH_LOG(TryFromString(botClassNameStr, BotClassName));

    ChatScriptFilename = section->GetDirectives().Value<TString>("ChatScriptFilename", ChatScriptFilename);
    if (ChatScriptFilename) {
        ChatScript = TChatRobotScript::FromFilename(ChatScriptFilename);
    }

    InitialStepId = section->GetDirectives().Value<TString>("InitialStepId", InitialStepId);
    Title = section->GetDirectives().Value<TString>("Title", Title);
    Icon = section->GetDirectives().Value<TString>("Icon", Icon);
    StartFromCleanChat = section->GetDirectives().Value<bool>("StartFromCleanChat", StartFromCleanChat);
    CreateOnRead = section->GetDirectives().Value<bool>("CreateOnRead", CreateOnRead);
    IsHidden = section->GetDirectives().Value<bool>("IsHidden", IsHidden);
    MarkAllRead = section->GetDirectives().Value<bool>("MarkAllRead", MarkAllRead);
    PersistVisitedStates = section->GetDirectives().Value("PersistVisitedStates", PersistVisitedStates);
    IsStatic = section->GetDirectives().Value<bool>("IsStatic", IsStatic);

    Pinned = section->GetDirectives().Value<bool>("Pinned", Pinned);

    RobotName = section->GetDirectives().Value<TString>("RobotName", RobotName);
    RobotUserId = section->GetDirectives().Value<TString>("RobotUserId", RobotUserId);

    Locale = section->GetDirectives().Value<ELocalization>("Locale", Locale);

    Common = section->GetDirectives().Value<bool>("IsCommon", Common);
    if (Common) {
        ui64 commonInstantTS = 0;
        commonInstantTS = section->GetDirectives().Value<ui64>("CommonInstant", commonInstantTS);
        CHECK_WITH_LOG(commonInstantTS);
        CommonInstant = TInstant::Seconds(commonInstantTS);
    }
}

void TChatRobotConfig::DoToString(IOutputStream& os) const {
    os << "ChatId: " << ChatId << Endl;
    os << "BotClassName: " << BotClassName << Endl;
    os << "ChatScriptFilename: " << ChatScriptFilename << Endl;
    os << "InitialStepId: " << InitialStepId << Endl;
    os << "Title: " << Title << Endl;
    os << "Icon: " << Icon << Endl;
    os << "StartFromCleanChat: " << StartFromCleanChat << Endl;
    os << "IsStatic: " << IsStatic << Endl;
    os << "CreateOnRead: " << CreateOnRead << Endl;
    os << "IsHidden: " << IsHidden << Endl;
    os << "MarkAllRead: " << MarkAllRead << Endl;
    os << "PersistVisitedStates: " << PersistVisitedStates << Endl;
    os << "IsCommon: " << Common << Endl;
    os << "CommonInstant: " << CommonInstant << Endl;
    os << "Pinned: " << Pinned << Endl;
    os << "RobotName: " << RobotName << Endl;
    os << "RobotUserId: " << RobotUserId << Endl;
    os << "Locale: " << Locale << Endl;
}

TChatRobotConfig TChatRobotConfig::ParseFromString(const TString& configStr) {
    TChatRobotConfig result;
    TAnyYandexConfig config;
    CHECK_WITH_LOG(config.ParseMemory(configStr.data()));
    result.DoInit(config.GetRootSection());
    return result;
}

void TChatRobotsConfig::Init(const TYandexConfig::Section* section) {
    StateStorageDatabase = section->GetDirectives().Value<TString>("StateStorageDatabase", StateStorageDatabase);
    auto allChildren = section->GetAllChildren();
    auto p = allChildren.find("Robots");
    if (p != allChildren.end()) {
        const TYandexConfig::TSectionsMap children = p->second->GetAllChildren();
        for (auto&& itSection : children) {
            TChatRobotConfig chatConfig;
            chatConfig.DoInit(itSection.second);
            BotConfigs.push_back(std::move(chatConfig));
        }
    }
    {
        auto it = allChildren.find("StatesHistoryConfig");
        if (it != allChildren.end()) {
            StatesHistoryConfig.Init(it->second);
        }
    }
    {
        auto it = allChildren.find("ViewsHistoryConfig");
        if (it != allChildren.end()) {
            ViewsHistoryConfig.Init(it->second);
        }
    }
    {
        auto it = allChildren.find("MediaHistoryConfig");
        if (it != allChildren.end()) {
            MediaHistoryConfig.Init(it->second);
        }
    }
    {
        auto it = allChildren.find("MediaStorageConfig");
        if (it != allChildren.end()) {
            MediaStorageConfig.Init(it->second);
        }
    }
}

void TChatRobotsConfig::ToString(IOutputStream& os) const {
    os << "StateStorageDatabase: " << StateStorageDatabase << Endl;
    for (auto&& botConfig : BotConfigs) {
        os << "<" << botConfig.GetChatId() << ">" << Endl;
        botConfig.DoToString(os);
        os << "</" << botConfig.GetChatId() << ">" << Endl;
    }

    os << "<MediaStorageConfig>" << Endl;
    MediaStorageConfig.ToString(os);
    os << "</MediaStorageConfig>" << Endl;

    os << "<StatesHistoryConfig>" << Endl;
    StatesHistoryConfig.ToString(os);
    os << "</StatesHistoryConfig>" << Endl;

    os << "<ViewsHistoryConfig>" << Endl;
    ViewsHistoryConfig.ToString(os);
    os << "</ViewsHistoryConfig>" << Endl;

    os << "<MediaHistoryConfig>" << Endl;
    MediaHistoryConfig.ToString(os);
    os << "</MediaHistoryConfig>" << Endl;
}

TChatRobotsConfig TChatRobotsConfig::ParseFromString(const TString& configStr) {
    TChatRobotsConfig result;
    TAnyYandexConfig config;
    CHECK_WITH_LOG(config.ParseMemory(configStr.data()));
    result.Init(config.GetRootSection());
    return result;
}

void TMediaStorageConfig::Init(const TYandexConfig::Section* section) {
    BucketName = section->GetDirectives().Value<TString>("BucketName", BucketName);
    FFMpegPath = section->GetDirectives().Value<TString>("FFMpegPath", FFMpegPath);
    PreviewResizeMaxSide = section->GetDirectives().Value<ui32>("PreviewResizeMaxSide", PreviewResizeMaxSide);
}

void TMediaStorageConfig::ToString(IOutputStream& os) const {
    os << "BucketName: " << BucketName << Endl;
    os << "FFMpegPath: " << FFMpegPath << Endl;
    os << "PreviewResizeMaxSide: " << PreviewResizeMaxSide << Endl;
}

TMediaStorageConfig TMediaStorageConfig::ParseFromString(const TString& configStr) {
    TMediaStorageConfig result;
    TAnyYandexConfig config;
    CHECK_WITH_LOG(config.ParseMemory(configStr.data()));
    result.Init(config.GetRootSection());
    return result;
}

NDrive::TScheme TChatStyles::GetScheme(const IServerBase& server) {
    NDrive::TScheme scheme;
    scheme.Add<TFSString>("name", "Имя стиля").SetDefault("chat");
    scheme.Add<TFSString>("bg_color", "Цвет фона");
    scheme.Add<TFSString>("elem_color", "Цвет элементов");
    scheme.Add<TFSStructure>("list_item", "Стили в списке чатов").SetStructure(TChatStyles::TListItem::GetScheme(server));
    return scheme;
}

NDrive::TScheme TChatStyles::TListItem::GetScheme(const IServerBase& /*server*/) {
    NDrive::TScheme scheme;
    scheme.Add<TFSString>("substrate_color", "Цвет подложки").SetDefault("");
    scheme.Add<TFSString>("title_color", "Цвет заголовка").SetDefault("#FF3B3B");
    scheme.Add<TFSString>("subtitle_color", "Цвет подзаголовка").SetDefault("#FF3B3BB3");
    scheme.Add<TFSString>("date_color", "Цвет даты").SetDefault("#FF3B3BB3");
    scheme.Add<TFSString>("unread_indicator_color", "Цвет индикатора непрочитанных сообщений").SetDefault("#241FFF");
    scheme.Add<TFSBoolean>("pinned", "Запинен сверху").SetDefault(false);
    return scheme;
}

bool TChatTheme::DeserializeFromJson(const NJson::TJsonValue& info, TMessagesCollector& errors) {
    return
        NJson::ParseField(info, "feedback_bg_color", FeedbackBackgroundColor, false, errors) &&
        NJson::ParseField(info, "color_separator_bg_color", ColorSeparatorBackgroundColor, false, errors) &&
        NJson::ParseField(info, "color_separator_bg_color", ColorSeparatorBackgroundColor, false, errors) &&
        NJson::ParseField(info, "styles", Styles, false, errors) &&
        NJson::ParseField(info, "bonus_bg", BonusBackground, false, errors);
}

NJson::TJsonValue TChatTheme::SerializeToJson() const {
    NJson::TJsonValue result;
    NJson::InsertField(result, "feedback_bg_color", FeedbackBackgroundColor);
    NJson::InsertField(result, "color_separator_bg_color", ColorSeparatorBackgroundColor);
    NJson::InsertField(result, "styles", Styles);
    NJson::InsertField(result, "bonus_bg", BonusBackground);
    return result;
}

NDrive::TScheme TChatTheme::GetScheme(const IServerBase& server) {
    NDrive::TScheme scheme;
    scheme.Add<TFSString>("feedback_bg_color", "Цвет фона для сообщений фидбека").SetDefault("#434343");
    scheme.Add<TFSString>("color_separator_bg_color", "Цвет сепаратора между диалогом и фидбеком").SetDefault("#F5F5F5");
    scheme.Add<TFSStructure>("styles", "Стили для отрисовки на клиенте").SetStructure<NDrive::TScheme>(TChatStyles::GetScheme(server));
    scheme.Add<TFSStructure>("bonus_bg", "Фон для бонусов").SetStructure<NDrive::TScheme>(TCustomMessageBackground::GetScheme(server));
    return scheme;
}

bool TCustomMessageBackground::DeserializeFromJson(const NJson::TJsonValue& info) {
    return
        NJson::ParseField(info, "path", Path, false) &&
        NJson::ParseField(info, "width", Width, false) &&
        NJson::ParseField(info, "height", Height, false);
}

NJson::TJsonValue TCustomMessageBackground::SerializeToJson() const {
    NJson::TJsonValue result;
    result["path"] = Path;
    result["width"] = Width;
    result["height"] = Height;
    return result;
}

NDrive::TScheme TCustomMessageBackground::GetScheme(const IServerBase& /*server*/) {
    NDrive::TScheme scheme;
    scheme.Add<TFSString>("path", "Путь к картинке фона").SetDefault("https://carsharing.s3.yandex.net/drive/static/chat/bonus-3x.png");
    scheme.Add<TFSNumeric>("width", "Ширина").SetMin(0).SetDefault(171);
    scheme.Add<TFSNumeric>("height", "Высота").SetMin(0).SetDefault(108);
    return scheme;
}
