#pragma once

#include "chat_script.h"

#include <drive/backend/abstract/notifier.h>
#include <drive/backend/database/history/config.h>
#include <drive/backend/abstract/localization.h>

#include <drive/library/cpp/common/object_sharding.h>

#include <kernel/daemon/config/daemon_config.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/mediator/global_notifications/system_status.h>

#include <rtline/util/types/accessor.h>

namespace NChatRobot {
    enum ELogic {
        LightRegistration /* "TLightRegistrationChatBot" */,
        Registration /* "TRegistrationChatBot" */,
        Support /* "TSupportChatBot" */,
        Channel /* "TChannelChatBot" */,
    };
};

class TCustomMessageBackground {
    R_READONLY(TString, Path);
    R_READONLY(size_t, Width, 0);
    R_READONLY(size_t, Height, 0);

public:
    TCustomMessageBackground() = default;

    TCustomMessageBackground(const TString& path, const size_t width, const size_t height)
        : Path(path)
        , Width(width)
        , Height(height)
    {
    }

public:
    bool DeserializeFromJson(const NJson::TJsonValue& info);
    NJson::TJsonValue SerializeToJson() const;
    static NDrive::TScheme GetScheme(const IServerBase& server);
};

class TChatStyles {
    R_FIELD(TString, Name);
    R_FIELD(TString, BgColor);
    R_FIELD(TString, ElemColor);

public:
    class TListItem {
        R_FIELD(TString, SubstrateColor);
        R_FIELD(TString, TitleColor, "#000000");
        R_FIELD(TString, SubtitleColor, "#000000B3");
        R_FIELD(TString, DateColor, "#0000004D");
        R_FIELD(TString, UnreadIndicatorColor, "#241FFF");
        R_FIELD(bool, Pinned, false);

    public:
        static NDrive::TScheme GetScheme(const IServerBase& server);
    };

    R_FIELD(TListItem, ListItem);

public:
    static NDrive::TScheme GetScheme(const IServerBase& server);
};

class TChatTheme {
    R_FIELD(TCustomMessageBackground, BonusBackground, TCustomMessageBackground("https://carsharing.s3.yandex.net/drive/static/chat/bonus-3x.png", 171, 108));
    R_FIELD(TString, FeedbackBackgroundColor, "#434343");
    R_FIELD(TString, ColorSeparatorBackgroundColor, "#F5F5F5");
    R_FIELD(TChatStyles, Styles);

public:
    bool DeserializeFromJson(const NJson::TJsonValue& info, TMessagesCollector& errors);
    NJson::TJsonValue SerializeToJson() const;
    static NDrive::TScheme GetScheme(const IServerBase& server);
};

class TChatRobotConfig : public NDrive::INotifierConfig {
    R_FIELD(TString, ChatId);
    R_FIELD(NChatRobot::ELogic, BotClassName);
    R_FIELD(TString, ChatScriptFilename);

    R_FIELD(TChatRobotScript, ChatScript);

    R_FIELD(TString, InitialStepId, "intro");
    R_FIELD(TString, CleanStateId, "!intro_clean");
    R_FIELD(TString, Title);
    R_FIELD(TString, Icon);
    R_FIELD(bool, StartFromCleanChat, false);
    R_FIELD(bool, CreateOnRead, true);
    R_FIELD(bool, IsHidden, false);
    R_FIELD(bool, MarkAllRead, false);
    R_FIELD(bool, PersistVisitedStates, false);

    R_FIELD(bool, Common, false);
    R_FIELD(TInstant, CommonInstant, TInstant::Zero());
    R_FIELD(TObjectSharding, CommonShardingPolicy);

    R_FIELD(bool, Pinned, false);
    R_FIELD(TChatTheme, Theme);

    R_FIELD(bool, IsStatic, false);       // same as static function: don't create separate instance of chat for every user


    R_FIELD(ICondition::TPtr, AuditoryCondition);
    R_FIELD(NJson::TJsonValue, AuditoryConditionRaw, NJson::JSON_NULL);

    R_FIELD(TString, RobotName);
    R_FIELD(TString, RobotUserId, "robot-frontend");

    R_FIELD(ELocalization, Locale, ELocalization::Rus);

public:
    virtual void DoInit(const TYandexConfig::Section* section) override;
    virtual void DoToString(IOutputStream& os) const override;
    static TChatRobotConfig ParseFromString(const TString& configStr);

    virtual NDrive::INotifier::TPtr Construct() const override {
        Y_UNREACHABLE();
    }

    virtual ~TChatRobotConfig() = default;
};

class TMediaStorageConfig {
    R_READONLY(TString, BucketName, "carsharing-chat");
    R_READONLY(TString, FFMpegPath, "ffmpeg");
    R_READONLY(ui32, PreviewResizeMaxSide, 300);

public:
    void Init(const TYandexConfig::Section* section);
    void ToString(IOutputStream& os) const;
    static TMediaStorageConfig ParseFromString(const TString& configStr);
};

class TChatRobotsConfig {
    R_READONLY(TVector<TChatRobotConfig>, BotConfigs);
    R_READONLY(TString, StateStorageDatabase);
    R_READONLY(TMediaStorageConfig, MediaStorageConfig);

    R_READONLY(THistoryConfig, StatesHistoryConfig, THistoryConfig().SetDeep(TDuration::Zero()));
    R_READONLY(THistoryConfig, ViewsHistoryConfig, THistoryConfig().SetDeep(TDuration::Zero()));
    R_READONLY(THistoryConfig, MediaHistoryConfig, THistoryConfig().SetDeep(TDuration::Zero()));

public:
    void Init(const TYandexConfig::Section* section);

    void ToString(IOutputStream& os) const;

    static TChatRobotsConfig ParseFromString(const TString& configStr);
};
