#include "context.h"

#include <drive/library/cpp/exp_calculator/exp_calculator.h>
#include <rtline/library/json/parse.h>

TString TContextMapInfo::StartMarker = "${";
TString TContextMapInfo::FinishMarker = "}";

template <>
NJson::TJsonValue NJson::ToJson(const TContextMapInfo& object) {
    NJson::TJsonValue result;
    NJson::InsertField(result, "key", object.Key);
    NJson::InsertField(result, "value", object.Value);
    NJson::InsertField(result, "type", ToString(object.Type));
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TContextMapInfo& result) {
    if (!NJson::ParseField(value, "key", result.Key, true) ||
        !NJson::ParseField(value, "value", result.Value) ||
        !NJson::ParseField(value, "type", NJson::Stringify(result.Type)))
    {
        return false;
    }

    if (result.Type == EContextDataType::MathExpression) {
        const auto tokens = result.GetTokens();
        if (!tokens) {
            return false;
        }
    }

    return true;
}

void TChatContext::AddMapData(const TVector<TContextMapInfo>& values) {
    for (auto&& field : values) {
        if (field.Key) {
            AddMapData(field.Key, field.Value);
        }
        if (field.Type != EContextDataType::None && field.Value) {
            MutableContextMap()[ToString(field.Type)] = field.Value;
        }
    }
}

TMaybe<TSet<TString>> TContextMapInfo::GetTokens() const {
    if (Type != EContextDataType::MathExpression || !Value) {
        return Nothing();
    }

    TSet<TString> result;

    size_t startPos = 0;
    while (startPos != TString::npos) {
        auto tokenStartPos = Value.find(StartMarker, startPos);
        auto tokenFinishPos = Value.find(FinishMarker, startPos);

        if (tokenFinishPos <= (tokenStartPos + StartMarker.size()) ||
            ((tokenStartPos == Value.npos) != (tokenFinishPos == Value.npos)))
        {
            return Nothing();
        }
        if (tokenStartPos == Value.npos) {
            break;
        }
        tokenStartPos += StartMarker.size();

        auto token = Value.substr(
            tokenStartPos,
            tokenFinishPos - tokenStartPos
        );
        result.insert(token);

        startPos = tokenFinishPos + 1;
    }

    return result;
}

TMaybe<TString> TChatContext::CalculateExpression(const TContextMapInfo& expression) {
    const auto tokens = expression.GetTokens();
    if (!tokens) {
        return Nothing();
    }
    TExpressionCalculator calculator(expression.Value);
    TMap<TString, double> values;
    for (const auto& valueName : *tokens) {
        double value;
        if (!ContextMap.TryToGet(valueName, value)) {
            return Nothing();
        }
        values.insert({
            TContextMapInfo::StartMarker + valueName + TContextMapInfo::FinishMarker,
            value
        });
    }

    auto result = calculator.Calc(values);
    if (!result) {
        ERROR_LOG << "TChatContext::CalculateExpression: " << result.GetError() << Endl;
        return Nothing();
    }

    return ToString(*result);
}

TString TChatContext::GetSpecialData(EContextDataType key) const {
    auto it = GetContextMap().find(ToString(key));
    if (it == GetContextMap().end()) {
        return "";
    }
    return it->second;
}

TString TChatContext::GetOperatorName() const {
    return GetSpecialData(EContextDataType::OperatorName);
}

bool TChatContext::SetOperatorName(const TString& value) {
    return MutableContextMap().emplace(ToString(EContextDataType::OperatorName), value).second;
}

TString TChatContext::GetSessionId() const {
    return GetSpecialData(EContextDataType::SessionId);
}

bool TChatContext::SetSessionId(const TString& value) {
    return MutableContextMap().emplace(ToString(EContextDataType::SessionId), value).second;
}

void TJsonMapper::SaveFieldsToContext(TChatContext& chatContext, const NJson::TJsonValue& json) const {
    for (auto&& [key, fieldPath] : ContextKeyToFieldPath) {
        NJson::TJsonValue value;
        if (json.GetValueByPath(fieldPath, value)) {
            chatContext.AddMapData(key, value.GetStringRobust());
        }
    }
}

bool TChatContext::InitResubmitDataFromState(const TChatRobotState& state) {
    ResubmitMask = state.GetResubmitMask();
    OriginalResubmitMask = state.GetOriginalResubmitMask();
    QueuedResubmitMask = state.GetQueuedResubmitMask();
    auto optionalResubmitOverride = TDocumentResubmitOverride::GetFromProto(state);
    if (!optionalResubmitOverride) {
        return false;
    }
    ResubmitOverrides = std::move(*optionalResubmitOverride);
    ResubmitInitialized = true;
    return true;
}

void TChatContext::WriteResubmitDataToState(NChatRobotState::TChatRobotState &state) const {
    if (!ResubmitInitialized) {
        return;
    }
    state.SetResubmitMask(ResubmitMask);
    state.SetOriginalResubmitMask(OriginalResubmitMask);
    state.SetQueuedResubmitMask(QueuedResubmitMask);
    TDocumentResubmitOverride::AddToProto(ResubmitOverrides, state);
}

NJson::TJsonValue TJsonMapper::SerializeToJson() const {
    NJson::TJsonValue result = NJson::ToJson(NJson::KeyValue(ContextKeyToFieldPath, "key", "field_path"));
    return result;
}

bool TJsonMapper::DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector& errors) {
    return NJson::ParseField(json, NJson::KeyValue(ContextKeyToFieldPath, "key", "field_path"), true, errors);
}

const TString TChatContext::UserPrefix = "custom";
