#pragma once

#include <drive/backend/chat_robots/state/robot_state.pb.h>
#include <drive/backend/database/transaction/tx.h>
#include <drive/backend/users/user_documents.h>

#include <rtline/util/types/accessor.h>
#include <rtline/util/types/messages_collector.h>

#include <library/cpp/json/json_value.h>
#include <library/cpp/regex/pcre/regexp.h>

#include <util/generic/map.h>
#include <util/string/vector.h>

namespace NChatScript {
    using TStringMap = TMap<TString, TString>;
};

enum class EContextDataType {
    None /* "none" */,
    CreditCardId /* "credit_card_id" */,
    CreditCardMask /* "credit_card_mask" */,
    SessionId /* "session_id" */,
    CarId /* "car_id" */,
    EnteredMessageType /* "entered_message_type" */,
    OperatorName /* "operator_name" */,
    PhoneVerificationError /* "phone_verification_error" */,
    MathExpression /* "math_expression" */,
};

struct TContextMapInfo {
public:
    TString Key;
    TString Value;
    EContextDataType Type;

    static TString StartMarker;
    static TString FinishMarker;

public:
    TContextMapInfo() = default;

    TContextMapInfo(const TString& key, const TString& value)
        : Key(key)
        , Value(value)
        , Type(EContextDataType::None)
    {
    }

    TContextMapInfo(const TString& key, const EContextDataType type)
        : Key(key)
        , Value("")
        , Type(type)
    {
    }

    TContextMapInfo(const TString& key, const TString& value, const EContextDataType type)
        : Key(key)
        , Value(value)
        , Type(type)
    {
    }

    TMaybe<TSet<TString>> GetTokens() const;
};

class TChatContext {
public:
    using TChatRobotState = NChatRobotState::TChatRobotState;

private:
    class TContextDataMap : public TMap<TString, TString> {
    private:
        using TBase =  TMap<TString, TString>;

    public:
        using TBase::TBase;
        using TBase::operator=;

        template <typename TType>
        bool TryToGet(const TString& key, TType& result) const {
            auto iter = this->find(key);
            return iter == this->end()
                ? false
                : TryFromString(iter->second, result);
        }
    };

private:
    static const TString UserPrefix;

public:
    R_FIELD(ui32, LastRequestCode, 0);
    R_FIELD(NJson::TJsonValue, LastRequestData);
    R_FIELD(TContextDataMap, ContextMap);

    R_FIELD(NDrive::TEntitySession*, TagsSession, nullptr);
    R_FIELD(NDrive::TEntitySession*, ChatSession, nullptr);

    R_FIELD(TVector<TString>, VisitedStates);

    /* Consider to remove this when resubmits will be handled via chats scenarios */
    R_READONLY(bool, ResubmitInitialized, false);
    R_FIELD(i32, ResubmitMask, 0);
    R_FIELD(i32, OriginalResubmitMask, 0);
    R_FIELD(i32, QueuedResubmitMask, 0);
    R_FIELD(TVector<TDocumentResubmitOverride>, ResubmitOverrides);

public:
    TChatContext() = default;

    TChatContext(NDrive::TEntitySession* tagsSession, NDrive::TEntitySession* chatSession)
    : TagsSession(tagsSession)
    , ChatSession(chatSession)
    {
    }

    virtual ~TChatContext() = default;

    virtual void AddMapData(const TString& key, const TString& value) {
        auto prefixedKey = UserPrefix + "." + key;
        ContextMap[prefixedKey] = value;
    }
    virtual TMaybe<TString> CalculateExpression(const TContextMapInfo& expression);

    bool IsEmpty() const {
        return LastRequestCode == 0 && !LastRequestData.IsDefined() && ContextMap.empty();
    }

    virtual TString GetOperatorName() const;
    virtual TString GetSessionId() const;

public:
    virtual void AddMapData(const TVector<TContextMapInfo>& values);

    virtual bool SetOperatorName(const TString& value);
    virtual bool SetSessionId(const TString& value);

    TString GetSpecialData(EContextDataType key) const;

    bool InitResubmitDataFromState(const NChatRobotState::TChatRobotState& state);
    void WriteResubmitDataToState(NChatRobotState::TChatRobotState& state) const;
};

class TJsonMapper {
private:
    R_FIELD(NChatScript::TStringMap, ContextKeyToFieldPath);

public:
    void SaveFieldsToContext(TChatContext& chatContext, const NJson::TJsonValue& json) const;
    bool Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors);
    NJson::TJsonValue SerializeToJson() const;
    bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector& errors);

    static TMaybe<TJsonMapper> Construct(TString str);
};
