#include "item_types.h"

#include <util/string/subst.h>
#include <random>

THolder<TChatTreeSchema> TChatTreeSchema::GetTemplateImpl(const TVector<TString>& params) const {
    THolder<TChatTreeSchema> result(new TChatTreeSchema());
    result->SetType(Type);
    result->SetText(NChatScript::GetMaybeParametrizedField(Text, params));
    result->SetHistoryText(NChatScript::GetMaybeParametrizedField(HistoryText, params));
    result->SetMessageText(NChatScript::GetMaybeParametrizedField(MessageText, params));
    result->SetNextNode(NextNode.GetTemplateImpl(params));
    result->SetMetrikaEventName(NChatScript::GetMaybeParametrizedField(MetrikaEventName, params));
    if (!!ShowCondition) {
        result->SetShowCondition(ShowCondition->GetTemplateImpl(params));
    }
    for (auto&& option : Options) {
        result->MutableOptions().emplace_back(option->GetTemplateImpl(params));
    }
    result->SetPickRandom(PickRandom);
    return result;
}

THolder<TChatTreeSchema> TChatTreeSchema::ApplyTemplate(const TMap<TString, TString>& params) const {
    THolder<TChatTreeSchema> result(new TChatTreeSchema());
    result->SetType(Type);

    result->SetText(Text);
    result->SetHistoryText(HistoryText);
    result->SetMessageText(MessageText);
    for (const auto& param : params) {
        SubstGlobal(result->MutableText(), param.first, param.second);
        SubstGlobal(result->MutableHistoryText(), param.first, param.second);
        SubstGlobal(result->MutableMessageText(), param.first, param.second);
    }
    result->SetNextNode(NextNode);
    result->SetMetrikaEventName(MetrikaEventName);
    if (!!ShowCondition) {
        result->SetShowCondition(ShowCondition);
    }
    for (auto&& option : Options) {
        result->MutableOptions().emplace_back(option->ApplyTemplate(params));
    }
    return result;
}

bool TChatTreeSchema::ConstructFromJson(const NJson::TJsonValue& schema, TMessagesCollector& errors, const bool isRootElement, const TString& trace) {
    Options.clear();

    if (!schema.Has("type")) {
        errors.AddMessage("schema", "schema element doesn't have type");
        return false;
    }
    if (!isRootElement && !schema.Has("text")) {
        errors.AddMessage("schema", "schema element doesn't have text");
        return false;
    }

    auto typeStr = schema["type"].GetString();
    Text = schema["text"].GetString();

    if (!TryFromString(typeStr, Type)) {
        errors.AddMessage("schema", "can't parse type");
        return false;
    }

    if (schema.Has("history_text")) {
        if (!schema["history_text"].IsString()) {
            errors.AddMessage("schema", "schema element has non-string history_text");
            return false;
        }
        HistoryText = schema["history_text"].GetString();
    } else {
        HistoryText = Text;
    }

    if (Type == NChatRobot::ETreeSchema::Message) {
        if (schema.Has("options")) {
            errors.AddMessage("schema", "nodes of type 'message' can't have options");
            return false;
        }
        if (schema.Has("message_text")) {
            if (!schema["message_text"].IsString()) {
                errors.AddMessage("schema", "schema has message_text but it is not string");
                return false;
            }
            MessageText = schema["message_text"].GetString();
        } else {
            MessageText = "!auto" + trace + "-" + Text;
        }
        if (!NextNode.Parse(schema["node"], errors)) {
            errors.AddMessage("schema", "could not parse node");
            return false;
        }
    } else if (Type == NChatRobot::ETreeSchema::Options || Type == NChatRobot::ETreeSchema::DefaultEmails) {
        if (!schema.Has("options") || !schema["options"].IsArray()) {
            errors.AddMessage("schema", "nodes of type 'options' should have options section and it should be an array");
            return false;
        }
        for (auto&& item : schema["options"].GetArray()) {
            TChatTreeSchema::TPtr child = new TChatTreeSchema();
            if (!child->ConstructFromJson(item, errors, false, trace + "-" + Text)) {
                errors.AddMessage("schema", "can't construct child from json");
                return false;
            }
            Options.emplace_back(child);
        }
        if (schema.Has("pick_only")) {
            if (!schema["pick_only"].IsUInteger()) {
                errors.AddMessage("schema", "pick_only is not uint");
                return false;
            }
            PickRandom = schema["pick_only"].GetUInteger();
        }
    } else if (Type == NChatRobot::ETreeSchema::Logout || Type == NChatRobot::ETreeSchema::SupportCenter) {
        if (schema.GetMap().size() != 2) {
            errors.AddMessage("schema", "logout/support_center section has more than 2 children");
            return false;
        }
    } else if (Type == NChatRobot::ETreeSchema::Deeplink) {
        if (!schema.Has("link") || !schema["link"].IsString()) {
            errors.AddMessage("schema", "no link in schema or it is not string");
            return false;
        }
        Link = schema["link"].GetString();
    } else if (Type == NChatRobot::ETreeSchema::PhoneNumber || Type == NChatRobot::ETreeSchema::CreditCardSchema) {
        if (schema.Has("options")) {
            errors.AddMessage("schema", "nodes of type 'phone_number' and 'credit_card_schema' can't have options");
            return false;
        }
        if (schema.Has("message_text")) {
            if (!schema["message_text"].IsString()) {
                errors.AddMessage("schema", "schema has message_text but it is not string");
                return false;
            }
            MessageText = schema["message_text"].GetString();
        }
        if (!NextNode.Parse(schema["node"], errors)) {
            errors.AddMessage("schema", "could not parse node");
            return false;
        }
        if (schema["link"].IsString()) {
            Link = schema["link"].GetString();
        }
    }

    if (schema.Has("metrika_event_name")) {
        if (!schema["metrika_event_name"].IsString()) {
            errors.AddMessage("schema", "metrika_event_name is not string");
            return false;
        }
        MetrikaEventName = schema["metrika_event_name"].GetString();
    }

    if (schema.Has("show_if")) {
        if (!schema["show_if"].IsMap()) {
            errors.AddMessage("show_if", "show_if in tree schema is not map");
            return false;
        }
        ICondition::EType type;
        if (!NJson::ParseField(schema["show_if"], "type", NJson::Stringify(type), true, errors)) {
            return false;
        }
        ICondition::TPtr cond = ICondition::TFactory::Construct(type);
        if (!cond || !cond->Parse(schema["show_if"], errors)) {
            errors.AddMessage("show_if", "can't parse show_if condition");
            return false;
        }
        ShowCondition = cond;
    }

    return true;
}

NJson::TJsonValue TChatTreeSchema::SerializeToJson(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const {
    NJson::TJsonValue result;
    result["type"] = ToString(Type);
    result["text"] = Text;
    if (!!MetrikaEventName) {
        result["metrika_event_name"] = MetrikaEventName;
    }
    if (Type == NChatRobot::ETreeSchema::Deeplink || Type == NChatRobot::ETreeSchema::PhoneNumber || Type == NChatRobot::ETreeSchema::CreditCardSchema) {
        result["link"] = Link;
    } else if (Type == NChatRobot::ETreeSchema::Message || Type == NChatRobot::ETreeSchema::PhoneNumber || Type == NChatRobot::ETreeSchema::CreditCardSchema) {
        result["message_text"] = MessageText;
    } else if (Type == NChatRobot::ETreeSchema::Options) {
        NJson::TJsonValue optionsJson = NJson::JSON_ARRAY;

        // Quarantine promo. To be buried ASAP.
        TVector<size_t> pickIndexes;
        for (size_t i = 0; i < Options.size(); ++i) {
            pickIndexes.push_back(i);
        }
        if (PickRandom && Options.size()) {
            size_t lastMessageId = 0;
            auto chatRobot = ctx->GetChatRobot();
            if (chatRobot) {
                lastMessageId = chatRobot->GetLastViewedMessageId(ctx->GetUserId(), ctx->GetChatTopic(), ctx->GetUserId());
            }
            std::mt19937 rng(lastMessageId);
            for (size_t i = 0; i < Options.size(); ++i) {
                ui32 random = rng();
                std::swap(pickIndexes[i], pickIndexes[i + random % (Options.size() - i)]);
            }
            pickIndexes.resize(Min(PickRandom, pickIndexes.size()));
        }

        for (auto&& i : pickIndexes) {
            auto& option = Options[i];
            if (option->GetShowCondition() && !option->GetShowCondition()->IsMatching(ctx, chatContext)) {
                continue;
            }
            optionsJson.AppendValue(option->SerializeToJson(ctx, chatContext));
        }
        result["options"] = std::move(optionsJson);
    } else if (Type == NChatRobot::ETreeSchema::DefaultEmails) {
        NJson::TJsonValue optionsJson = NJson::JSON_ARRAY;
        for (auto&& option : Options) {
            if (!ctx) {
                optionsJson.AppendValue(option->SerializeToJson(ctx, chatContext));
            } else {
                auto optionJson = option->SerializeToJson(ctx, chatContext);
                if (optionJson.GetStringRobust().Contains("$default_email")) {
                    for (const auto& mail : ctx->GetDefaultEmails()) {
                        optionsJson.AppendValue(option->ApplyTemplate({ { "$default_email", mail } })->SerializeToJson(ctx, chatContext));
                    }
                } else {
                    optionsJson.AppendValue(optionJson);
                }
            }
        }
        result["options"] = std::move(optionsJson);
    }
    return result;
}

void TChatTreeSchema::DoGetAdjacentItems(TChatTreeSchema::TPtr node, const TVector<TString>& params, TSet<TString>& result) const {
    if (!node) {
        return;
    }
    if (node->GetType() == NChatRobot::ETreeSchema::Message) {
        result.emplace(NChatScript::GetMaybeParametrizedField(node->GetNextNode().GetDefaultNode(), params));
        for (auto&& part : node->GetNextNode().GetParts()) {
            result.emplace(NChatScript::GetMaybeParametrizedField(part.NodeName, params));
        }
    }
    if (node->GetType() == NChatRobot::ETreeSchema::Options) {
        for (auto&& option : node->GetOptions()) {
            DoGetAdjacentItems(option, params, result);
        }
    }
}

TSet<TString> TChatTreeSchema::GetAdjacentItems(const TVector<TString>& params) const {
    TSet<TString> result;
    for (auto&& option : Options) {
        DoGetAdjacentItems(option, params, result);
    }
    return result;
}

bool TChatTreeSchema::FillTreeMessagesRoute(const IChatUserContext::TPtr ctx, const NDrive::NChat::TMessage& message, const TChatContext& chatContext, TVector<TTreeRouteText>& route, TString& nextStepId) const {
    if (Type == NChatRobot::ETreeSchema::Message && MessageText == message.GetText() ||
        Type == NChatRobot::ETreeSchema::PhoneNumber && message.GetType() == NDrive::NChat::TMessage::EMessageType::PhoneNumber ||
        Type == NChatRobot::ETreeSchema::CreditCardSchema && (message.GetType() == NDrive::NChat::TMessage::EMessageType::CreditCardBinding || message.GetType() == NDrive::NChat::TMessage::EMessageType::CreditCard)) {
        nextStepId = NextNode.GetNextNode(ctx, chatContext);
        return true;
    }
    for (auto&& option : Options) {
        route.push_back(TTreeRouteText(option->GetHistoryText(), option->GetText(), option->GetMessageText(), option->GetType()));
        if (option->FillTreeMessagesRoute(ctx, message, chatContext, route, nextStepId)) {
            return true;
        }
        route.pop_back();
    }
    return false;
}
