#pragma once

#include "context.h"
#include "condition_interface.h"

#include <drive/backend/chat_robots/ifaces.h>
#include <drive/backend/chat_robots/suggest/node_resolver.h>


#include <rtline/util/json_processing.h>
#include <rtline/util/types/accessor.h>

#include <util/generic/map.h>
#include <util/generic/vector.h>

namespace NChatRobot {
    enum ETreeSchema {
        Message /* "message" */,
        Options /* "options" */,
        Logout /* "logout" */,
        SupportCenter /* "support_center" */,
        Deeplink /* "deeplink" */,
        DefaultEmails /* "default_emails" */,
        PhoneNumber /* "phone_number" */,
        CreditCardSchema /* "credit_card_schema" */,
        CarComplaint /* "car_complaint" */,
    };
};

class IHookAction {
public:
    using TPtr = TAtomicSharedPtr<IHookAction>;
    enum class EType: ui32 {
        ProposeTag = 0 /* "propose_tag" */,
        FeedbackOverall = 1 /* "feedback_overall" */,
        FeedbackDetails = 2 /* "feedback_details" */,
        SetDocumentStatuses = 3 /* "set_document_statuses" */,
        DeviceVerify = 4 /* "device_verify" */,
        AddTag = 5 /* "add_tag" */,
        AlterFlag = 6 /* "alter_flag" */,
        ClassifyRequest = 7 /* "classify_request" */,
        EndpointCall = 8 /* "endpoint_call" */,
        EditDictionary = 9 /* "edit_dictionary" */,
        ModifyTags = 10 /* "modify_tag" */,
        SendComplaintPhotos = 11 /* "send_complaint_photos" */,
        SetDocumentsCheckStatus = 12 /* "set_documents_check_status" */,
        PhoneVerify = 13 /* "phone_verify" */,
    };
    using TFactory = NObjectFactory::TParametrizedObjectFactory<IHookAction, IHookAction::EType>;

public:
    virtual bool Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) = 0;
    virtual bool Perform(const IChatUserContext::TPtr userContext, TChatContext& context, const NDrive::IServer* server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatSession, const TChatRobotScriptItem& item) const = 0;
    virtual IHookAction::TPtr GetTemplateImpl(const TVector<TString>& params) const = 0;

    virtual ~IHookAction() = default;
};

class TTreeRouteText {
    R_READONLY(TString, HistoryText);
    R_READONLY(TString, MenuText);
    R_READONLY(TString, MessageText);
    R_READONLY(NChatRobot::ETreeSchema, Type);

public:
    TTreeRouteText() = default;

    TTreeRouteText(const TString& historyText, const TString& menuText, const TString& messageText, const NChatRobot::ETreeSchema type)
        : HistoryText(historyText)
        , MenuText(menuText)
        , MessageText(messageText)
        , Type(type)
    {
    }
};

class TChatTreeSchema {
public:
    using TPtr = TAtomicSharedPtr<TChatTreeSchema>;
private:
    R_FIELD(NChatRobot::ETreeSchema, Type, NChatRobot::ETreeSchema::Message);
    R_FIELD(TString, Text);
    R_FIELD(TString, HistoryText);
    R_FIELD(TString, MessageText);
    R_FIELD(TCaseCondition, NextNode);
    R_FIELD(TVector<TChatTreeSchema::TPtr>, Options);
    R_FIELD(TString, Link);
    R_FIELD(TString, MetrikaEventName);
    R_FIELD(size_t, PickRandom, 0);

    ICondition::TPtr ShowCondition;

public:
    TChatTreeSchema() = default;

    bool ConstructFromJson(const NJson::TJsonValue& schema, TMessagesCollector& errors, const bool isRootElement = true, const TString& trace = "");
    NJson::TJsonValue SerializeToJson(const IChatUserContext::TPtr ctx, const TChatContext& chatContext) const;
    THolder<TChatTreeSchema> GetTemplateImpl(const TVector<TString>& params) const;
    THolder<TChatTreeSchema> ApplyTemplate(const TMap<TString, TString>& params) const;
    TSet<TString> GetAdjacentItems(const TVector<TString>& params) const;

    const ICondition* GetShowCondition() const {
        return ShowCondition.Get();
    }

    void SetShowCondition(ICondition::TPtr condition) {
        ShowCondition = condition;
    }

    bool FillTreeMessagesRoute(const IChatUserContext::TPtr ctx, const NDrive::NChat::TMessage& message, const TChatContext& chatContext, TVector<TTreeRouteText>& route, TString& nextStepId) const;

private:
    void DoGetAdjacentItems(TChatTreeSchema::TPtr node, const TVector<TString>& params, TSet<TString>& result) const;
};
